/**********************************************************************
 ** Commands: This is a listing of all the commands that a player can
 **           call in the game. The functions here must pass back void
 **           and must accept a Parse pointer and a Player pointer, like
 **           this: void chatcom(Parse *this_parse, MudObject *this_user)
 **
 **    
 ** Reviewed through: 
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Kelly Gerke
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the
 **   License, or any later version.
 **
 **   This program is distributed in the hope that it will be useful, but
 **   WITHOUT ANY WARRANTY; without even the implied warranty of
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 **   General Public License for more details.
 **
 **   You should have received a copy of the GNU General Public License
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef BCOMMANDS_C
#define BCOMMANDS_C

#include "config.h"
#include "sysdep.h"
#include "newfuncts.h"
#include "strings.h"
#include "global.h"
#include "mudobject.h"
#include "objtype.h"
#include "text.h"
#include "spell.h"
#include "skill.h"
#include "shop.h"
#include "adminflags.h"
#include "actflags.h"
#include "indflags.h"
#include "locflags.h"
#include "spellflags.h"
#include "areaflags.h"
#include "skillflags.h"
#include "food.h"
#include "door.h"
#include "boat.h"
#include "mobile.h"
#include "book.h"
#include "wearable.h"
#include "key.h"
#include "inp_funct.h"
#include "action.h"
#include "memchk.h"
#include "bcommands.h"
#include "mask.h"
#include "chatline.h"
#include "race.h"
#include "inclination.h"
#include "talent.h"
#include "utils.h"
#include "dirread.h"

extern char *depend_name[];

/***********************************************************************
 ** bldrchatcom - sends a message to all builders who have chat set
 **
 **
 ***********************************************************************/

void bldrchatcom(Parse *the_parsed, MudObject *the_user) {
   Builder *builders_list;   /* points to a player object */ 
   Builder *the_builder = NULL;  
   Strings formatted;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;
   the_builder = (Builder *) the_user;


   /* if there is nothing after chat, send a message to the player */
   if ((the_parsed->get_speech() == NULL) || 
       (strlen(the_parsed->get_speech()) <= 0))
   {
      the_builder->send_bldr(_("Format: chat <phrase>\n"));
      return;
   }
   
   /* send a message to the player that chatted */
   /* send a message to the player that chatted */
   formatted.format_for_comm(the_parsed->get_speech(), 10,
                                          the_config.max_line_len);
   the_builder->send_bldr(_("&+BYou chat &+W'%s&+W'&*\n"), formatted.str_show());


   builders_list = mainstruct->get_first_builder();

   /* put it in the correct format */
   formatted.format_for_comm(the_parsed->get_speech(), 
               (strlen(the_builder->get_title()) + 8), the_config.max_line_len);

   /* go through all players sending the message */
   while (builders_list != NULL)
   {
      /* if the players is not the originator of the message, send to them */
      if (STRCASECMP(the_builder->get_name(), builders_list->get_name()))
      {
         builders_list->send_bldr(_("&+B%s chats &+W'%s&+W'\n"), 
                           the_builder->get_title(), formatted.str_show());
      }

      /* go to next player */
      builders_list = builders_list->get_next_builder();
   }
}



/***********************************************************************
 **  bldrloadcom - loads a zone into memory, assigning it to the user
 **
 ***********************************************************************/

void bldrloadcom(Parse *the_parsed, MudObject *the_user) {
   Builder *the_builder;
   char    *the_area;
   Strings the_filename;
   FILE    *areafile = NULL;
   int     results;
   int     build_zone = 1;  /* are we reading from a .area or .bldr */
   int     text_type = 0;
   int     is_bulletin = 0;
   Strings dirname;
   Strings holder;
   DirRead *dirptr = NULL;
   Strings pattern;
#ifdef WIN32
   CFileFind dir;
#endif

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if ((the_area = the_parsed->get_target1()) == NULL)
   {
      the_builder->send_bldr(_("Load what?\n"));
      return;
   }

   if (!STRCASECMP(the_parsed->get_target1(), _("help")))
   {
      text_type = TEXT_TYPE_HELP;
      dirname.sprintf("%s%s%s", the_config.basedir.str_show(), DSEP, HELPDIR);
      pattern = HELPEXTENTION;
   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("info")))
   {
      text_type = TEXT_TYPE_INFO;
      dirname.sprintf("%s%s%s", the_config.basedir.str_show(), DSEP, INFODIR);
      pattern = INFOEXTENTION;
   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("bulletins")))
   {
      is_bulletin = 1;
      dirname.sprintf("%s%s%s", the_config.basedir.str_show(), DSEP, BULLETINDIR);
      pattern = BULLEXTENTION;
   }
   else if (!STRCASECMP(the_parsed->get_target1(), _("banner")))
   {
      text_type = TEXT_TYPE_BANNER;
      dirname.sprintf("%s%s%s", the_config.basedir.str_show(), DSEP, BANNERDIR);
      pattern = BANNEREXTENTION;
   }


   if (text_type || is_bulletin)
   {
      dirptr = new DirRead(dirname.str_show());
      if (!dirptr->is_valid())
      { 
		 holder.sprintf("Could not open directory '%s', failed.\n",
		                                         dirname.str_show());
		 mainstruct->log_error(holder.str_show(), "bldrloadcom");
		 return;
      }

      build_zone = 0;
 
   }
   else
   {
      the_filename.sprintf("%s%s%s%s%s.bldr", the_config.basedir.str_show(), 
                                                DSEP, BLDRDIR, DSEP, the_area);

      the_filename.lowercase();

      /* open the area file for reading */
      if ((areafile = xfopen(the_filename.str_show(), "r", 
                                               "read_area_file")) == NULL)
      {
         if (!STRCASECMP(the_area, _("actions")))
         {
            the_filename.sprintf("%s%s%s%sactions.dat", 
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("abilities")))
	 {
            the_filename.sprintf("%s%s%s%sabilities.dat",
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("levels")))
	 {
            the_filename.sprintf("%s%s%s%slevels.dat",
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
	 }
         else if (!STRCASECMP(the_area, _("quests")))
	 {
            the_filename.sprintf("%s%s%s%squests.dat",
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("races")))
	 {
            the_filename.sprintf("%s%s%s%sraces.dat", 
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("inclinations")))
	 {
            the_filename.sprintf("%s%s%s%sinclinations.dat", 
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("talents")))
	 {
            the_filename.sprintf("%s%s%s%stalents.dat", 
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("masks")))
	 {
            the_filename.sprintf("%s%s%s%smasks.dat",
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }
         else if (!STRCASECMP(the_area, _("chatlines")))
	 {
            the_filename.sprintf("%s%s%s%schatlines.dat",
                                    the_config.basedir.str_show(), DSEP, DATA_DIR, DSEP);
         }

         else
            the_filename.sprintf("%s%s%s%s%s.area",  
                            the_config.basedir.str_show(), DSEP, AREADIR, DSEP, the_area);

         /* open the area file for reading */
         if ((areafile = xfopen(the_filename.str_show(), "r", 
                                              "read_area_file")) == NULL)
         {
            the_builder->
                   send_bldr(_("I could not find the area '%s'\n"), the_area);

            return;
         }
         build_zone = 0;
      }
   }

   if (dirptr == NULL)
   {
      if ((results = the_builder->load_area(the_area, areafile, 
                                 mainstruct->get_log(), build_zone)) < 0)
      {
         if (results == -2)
            the_builder->send_bldr(_("That area is already loaded.\n"));
         else if (results == -3)
            the_builder->
              send_bldr(_("You do not have read permissions for that area.\n"));
         else
            the_builder->send_bldr(_("Error loading areafile, failed.\n"));
         return;
      }
   }
   else
   {
      if ((results = the_builder->load_area(dirptr, the_area, pattern.str_show(), 
                                 mainstruct->get_log(), build_zone)) < 0)
      {
         if (results == -2)
            the_builder->send_bldr(_("That area is already loaded.\n"));
         else if (results == -3)
            the_builder->
              send_bldr(_("You do not have read permissions for that area.\n"));
         else
            the_builder->send_bldr(_("Error loading areafile, failed.\n"));
         return;
      }
      delete dirptr;
   }
   the_builder->send_bldr(_("Successfully loaded area '%s'\n"), the_area);
}


/***********************************************************************
 **  bldrmodifycom - points the builder to an object for modification, and
 **                  sets appropriate mutexes (not yet)
 **
 ***********************************************************************/

void bldrmodifycom(Parse *the_parsed, MudObject *the_user) {
   Builder *the_builder;
   char *the_name;
   int results;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if ((the_name = the_parsed->get_target1()) == NULL) 
      return;
   
   if ((results = the_builder->modify_object(the_parsed->get_target1())) < 0)
   {
      if (results == -1)
         the_builder->send_bldr(_("You have not loaded a zone yet.\n"));
      else if (results == -3)
         the_builder->send_bldr(_("Somebody is already modifying '%s'.\n"),
                                                   the_parsed->get_target1());
      else
         the_builder->send_bldr(_("Object '%s' not found in zone '%s'.\n"),
                   the_parsed->get_target1(), the_builder->get_areaname());
      return;
   }
   the_builder->send_bldr(_("Object '%s' set to modify.\n"), 
                                             the_parsed->get_target1());
}


/***********************************************************************
 **  bldrsavecom - saves the modified zone to a file temporarily, till
 **                it can be 'committed' to the original area file
 **
 ***********************************************************************/

void bldrsavecom(Parse *the_parsed, MudObject *the_user) {
   Builder     *the_builder;
   Strings     filename;
   FILE        *the_file = NULL;
   Area_Dbase  *the_area = NULL;
   Timespan    the_time;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;
 
   if ((the_area = the_builder->get_area()) == NULL)
   {
      the_builder->send_bldr(_("You must first load in order to save.\n"));
      return;
   }
   if ((!STRCASECMP(the_area->get_areaname(), _("info"))) ||
       (!STRCASECMP(the_area->get_areaname(), _("banner"))) ||
       (!STRCASECMP(the_area->get_areaname(), _("help"))))
   {
      the_builder->
               send_bldr(_("You must use commit to store changes on texts.\n"));
      return;
   }
   if (!STRCASECMP(the_area->get_areaname(), _("bulletins")))
   {
      the_builder->
       send_bldr(_("You must use commit to store changes to bulletin boards.\n"));
      return;
   }

   /* open the file to write to */
   filename.sprintf("%s%s%s%s%s.bldr", the_config.basedir.str_show(), DSEP,
                                       BLDRDIR, DSEP, the_builder->get_areaname());
   if ((the_file = xfopen(filename.str_show(), "w", 
                                               "write_area_file")) == NULL)
   {
      the_builder->send_bldr(_("Error opening write file, could not save.\n"));
      return;
   }

   the_builder->send_bldr(_("Saving data to temporary builder file.\n"));

   the_area->write_area(the_file, 1);


   xfclose(the_file, "write_area_file");

   the_builder->send_bldr(_("Saved.\n"));
   the_parsed = NULL;
}


/***********************************************************************
 **  bldrsetcom - sets an attribute of an object
 **
 ***********************************************************************/

void bldrsetcom(Parse *the_parsed, MudObject *the_user)
{
   Builder *the_builder;
   Entity *the_object;
   Area_Dbase *tmp_area;
   Flags *admflags;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   tmp_area = the_builder->get_area();
   admflags = the_builder->get_adminflags();

   if (tmp_area == NULL)
   {
      the_builder->send_bldr(_("You must load an area first.\n"));
      return;
   }

   if ((the_parsed->get_target1() == NULL) || 
       (strlen(the_parsed->get_target1()) <= 0))
   {
      the_builder->send_bldr(_("Set what?\n"));
      return;
   }

   if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(_("area"), the_parsed->get_target1())))
   {
      Strings holder;    

      if (!admflags->get_flag(ADMINFLAG_SETAREAS))
      {
         the_builder->send_bldr(_("You do not have permissions to modify "
                                                      "area attributes.\n"));
         return;
      }

      if (tmp_area == NULL)
      {
         the_builder->send_bldr(_("You must load an area first.\n"));
         return;
      }

      if ((the_parsed->get_speech() == NULL) || 
          (strlen(the_parsed->get_speech()) == 0))
      {
         the_builder->send_bldr(_("Set what area attribute?\n"));
         return;
      }

      holder.assign_word(the_parsed->get_speech(), 1);
      if (!STRNCASECMP(_("writeallow"), holder.str_show(), holder.str_len()))
      {
         holder.assign_phrase(the_parsed->get_speech(), 1);      
         tmp_area->set_write_allow(holder.str_show());
         the_builder->send_bldr(_("Write allow list has been set to: %s\n"), 
                                        tmp_area->get_write_allow());
         return;
      }
      else if (!STRNCASECMP(_("readdeny"), holder.str_show(), holder.str_len()))
      {
         holder.assign_phrase(the_parsed->get_speech(), 1);
         tmp_area->set_read_deny(holder.str_show());
         the_builder->send_bldr(_("Read deny list has been set to: %s\n"), 
                                        tmp_area->get_read_deny());
         return;
      }
      else if (!STRNCASECMP(_("reloadinterval"), holder.str_show(), 
                                                       holder.str_len()))
      {
         holder.assign_phrase(the_parsed->get_speech(), 1);

         if (holder.str_show() == NULL)
         {
            the_builder->send_bldr(_("You need to specify a number.\n"));
            return;
         }
 
         if (!isdigit(*(holder.str_show()))) 
         {
            the_builder->
              send_bldr(_("You need to specify a number as ReloadInterval.\n"));
            return;
         }

         tmp_area->set_reload_interval(atol(holder.str_show()));
         the_builder->send_bldr(_("ReloadInterval set to %ld.\n"), 
                                          tmp_area->get_reload_interval());
         return;
      }
      else
      {
         the_builder->send_bldr(_("That is not an area attribute.\n"));
         return;
      }

   }
   if ((the_parsed->get_target1() != NULL) && 
       (!STRNCASECMP(_("areaflags"), the_parsed->get_target1(), 
                                       strlen(the_parsed->get_target1()))))
   {
      Strings holder;
      Strings flagname;
      int     flagnum;
      Flags   *tmp_areaflags;

      if (!admflags->get_flag(ADMINFLAG_SETAREAS))
      {
         the_builder->send_bldr(_("You do not have permissions to modify "
                                                      "area attributes.\n"));
         return;
      }

      if (tmp_area == NULL)
      {
         the_builder->send_bldr(_("You must load an area first.\n"));
         return;
      }

      if ((the_parsed->get_speech() == NULL) || 
          (strlen(the_parsed->get_speech()) == 0))
      {
         the_builder->send_bldr(_("Set what areaflag?\n"));
         return;
      }

      tmp_areaflags = tmp_area->get_areaflags();

      flagname.assign_word(the_parsed->get_speech(), 1);

      if ((flagnum = tmp_areaflags->
                   get_by_name(flagname.str_show(), areaflagnames)) == -1)
      {
         the_builder->send_bldr(_("That areaflag does not exist.\n"));
         return;
      }

      holder.assign_word(the_parsed->get_speech(), 2);

      if ((holder.str_show() == NULL) || 
          (holder.str_len() == 0))
      {
         the_builder->send_bldr(_("Set '%s' areaflag to what? (On/Off)\n"), 
                                    areaflagnames[flagnum]);
         return;
      }

      if (holder.str_n_cmp(_("On"), holder.str_len()))
         tmp_areaflags->set_flag(flagnum);
      else if (holder.str_n_cmp(_("Off"), holder.str_len()))
         tmp_areaflags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr(_("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n"));
         return;
      }
      the_builder->send_bldr(_("Flag &+M%s&* has been set to: %s\n"),
           areaflagnames[flagnum], (tmp_areaflags->get_flag(flagnum)) ? 
			     _("&+GOn&*") : _("&+ROff&*"));

      return;
   }
   if ((!((tmp_area != NULL) && (!tmp_area->can_write(the_builder)))) &&
       !(admflags->get_flag(ADMINFLAG_SETAREAS)))
   {
      the_builder->
          send_bldr(_("You do not have permissions to modify this area\n"));
      return;
   }

   if ((the_object = the_builder->get_modify()) == NULL)
   {
      the_builder->send_bldr(_("You have not set an object to modify yet.\n"));
      return;
   }

   the_object->set_attrib(the_builder, the_parsed);
}



/***********************************************************************
 **  bldrlistcom - does various lists of mudobjects
 **
 ***********************************************************************/

void bldrlistcom(Parse *the_parsed, MudObject *the_user)
{
   Builder     *the_builder;
   Entity      *the_object;
   int         to_list;    /* the types to list */
   Area_Dbase  *tmp_dbase;
   int         count = 0;
   Connection  *the_connection;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   the_connection = the_builder->get_connection();
   the_connection->start_paging();

   if ((the_parsed->get_target1() != NULL) && 
       (!STRNCASECMP(the_parsed->get_target1(), _("areas"),
                               strlen(the_parsed->get_target1()))))
   {
      Strings dirname;
      Strings holder;
      FILE    *lsfile = NULL;
      char    areaname[129];
      char    *charptr; 
      char    *prevptr;
      DirRead *dirptr;

      the_builder->send_bldr(_("\n&+WHardcoded areas available for "
                             "loading&*\n"));
      the_builder->send_bldr("&+G________________________________"
                             "___________&*\n");

      the_builder->send_bldr(_("&+gabilities&*\n")); 
      the_builder->send_bldr(_("&+gactions&*\n"));
      the_builder->send_bldr(_("&+gbanner&*\n"));
      the_builder->send_bldr(_("&+ghelp&*\n"));
      the_builder->send_bldr(_("&+ginfo&*\n"));
      the_builder->send_bldr(_("&+glevels&*\n"));
    
      if (the_config.conf_flags->get_flag(CF_USEQUESTS))
      {
         the_builder->send_bldr(_("&+gquests&*\n"));
      }
      the_builder->send_bldr(_("&+graces&*\n"));
      the_builder->send_bldr(_("&+ginclinations&*\n"));
      the_builder->send_bldr(_("&+gtalents&*\n"));
      the_builder->send_bldr(_("&+gmasks&*\n"));
      the_builder->send_bldr(_("&+gchatlines&*\n"));

      the_builder->send_bldr("&+G________________________________"
                          "___________&*\n\n");


      the_builder->send_bldr(_("\n&+WArea files available in the area "
                             "directory&*\n"));
      the_builder->send_bldr("&+G________________________________"
                             "___________&*\n");
      dirname.sprintf("%s%s%s%s", the_config.basedir.str_show(), DSEP, AREADIR, DSEP);
      dirptr = new DirRead(dirname.str_show());
      if (!dirptr->is_valid())
      {
		 holder.sprintf("Could not read directory '%s'", dirname.str_show());
		 mainstruct->log_error(holder.str_show(), "bldrlistcom");
		 the_connection->end_paging(the_builder);
		 return;
      }

      while ((charptr = dirptr->get_next(".area")) != NULL)
      {
         /* format the string to a correct filename */
		 strncpy(areaname, charptr, 128);
         charptr = &areaname[0];
         prevptr = charptr;
         while ((*charptr) && ((*charptr != '\n') && (*charptr != '\0') && 
                                                     (*charptr != '.')))
         {
            if (*charptr == '/')
               prevptr = charptr+1;
            charptr++;
         }
         if (*charptr)
            *charptr = '\0';
         the_builder->send_bldr("&+g%s&*\n", prevptr);
         count++;
      }
      delete dirptr;
 
      the_builder->send_bldr("&+G________________________________"
                          "___________&*\n");
      the_builder->send_bldr(_("&+GTotal Areas Listed: &+W%d\n\n"), count);

      the_connection->end_paging(the_builder);
      return;
   }


   if ((tmp_dbase = the_builder->get_area()) == NULL)
   {
      the_builder->send_bldr(_("You do not have an area loaded.\n"));
      the_connection->end_paging(the_builder);
      return;
   }

   if ((the_parsed->get_target1() == NULL) ||
      (!STRNCASECMP(the_parsed->get_target1(), _("all"), 
                               strlen(the_parsed->get_target1()))))
   {
      to_list = 0;
      the_builder->send_bldr(_("\n&+WAll MudObjects in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("locations"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_LOCATION;
      the_builder->send_bldr(_("\n&+WAll Locations in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("actions"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_ACTION;
      the_builder->send_bldr(_("\n&+WAll Actions in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("quests"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_QUEST;
      the_builder->send_bldr(_("\n&+WAll Quests in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("races"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_RACE;
      the_builder->send_bldr(_("\n&+WAll Races in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("inclinations"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_INCLINAT;
      the_builder->send_bldr(_("\n&+WAll Inclinations in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("talents"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_TALENT;
      the_builder->send_bldr(_("\n&+WAll Talents in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }

   else if (!STRNCASECMP(the_parsed->get_target1(), _("masks"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_MASK;
      the_builder->send_bldr(_("\n&+WAll Masks in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("chatlines"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_CHATLINE;
      the_builder->send_bldr(_("\n&+WAll ChatLines in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }

   else if (!STRNCASECMP(the_parsed->get_target1(), _("levels"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_LEVEL;
      the_builder->send_bldr(_("\n&+WAll Levels in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("markers"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_MARKER;
      the_builder->send_bldr(_("\n&+WAll Markers in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("doors"), 
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_DOOR;
      the_builder->send_bldr(_("\n&+WAll Doors in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("moveables"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_MOVEABLE;
      the_builder->send_bldr(_("\n&+WAll moveables in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("wearables"),
                               strlen(the_parsed->get_target1())))
   {
      to_list = OBJ_TYPE_WEARABLE;
      the_builder->send_bldr(_("\n&+WAll wearables in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("weapons"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_WEAPON;
      the_builder->send_bldr(_("\n&+WAll weapons in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("mergers"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_MERGER;
      the_builder->send_bldr(_("\n&+WAll mergers in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("money"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_MONEY;
      the_builder->send_bldr(_("\n&+WAll moneys in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("food"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_FOOD;
      the_builder->send_bldr(_("\n&+WAll food in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("keys"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_KEY;
      the_builder->send_bldr(_("\n&+WAll keys in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("boat"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_BOAT;
      the_builder->send_bldr(_("\n&+WAll boats in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("books"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_BOOK;
      the_builder->send_bldr(_("\n&+WAll books in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("rope"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_ROPE;
      the_builder->send_bldr(_("\n&+WAll ropes in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("mobiles"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_MOBILE;
      the_builder->send_bldr(_("\n&+WAll mobiles in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("specials"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_SPECIAL;
      the_builder->send_bldr(_("\n&+WAll specials in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("spells"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_SPELL;
      the_builder->send_bldr(_("\n&+WAll spells in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("skills"),
                               strlen(the_parsed->get_target1())))
   {  to_list = OBJ_TYPE_SKILL;
      the_builder->send_bldr(_("\n&+WAll skills in area: &+m%s&*\n"),
                             the_builder->get_areaname());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("modifieds"),
                               strlen(the_parsed->get_target1())))
   {
     the_builder->send_bldr(_("\n&+WAll Entities recently modified "
                             "in area: &+m%s.&*\n"), 
                                            the_builder->get_areaname());
      to_list = -1;
   }
   else
   {
      the_builder->send_bldr(_("I don't know the '%s' list.\n"), 
         the_parsed->get_target1());
      the_connection->end_paging(the_builder);
      return;
   }
  
   the_builder->send_bldr(_("&+GName                            "
                          "&+GObject Type&*\n"));
   the_builder->send_bldr("&+G________________________________"
                          "___________&*\n");

   tmp_dbase->reset_list();
   the_object = tmp_dbase->get_next_obj();
   while (the_object != NULL)
   {
      if ((!to_list) || 
          ((the_object->get_type() == to_list) ||
          ((the_object->is_modified()) && (to_list == -1))))
      {
         
         the_builder->send_bldr("%-31s %s\n", the_object->get_name(), 
                              objtype_name[the_object->get_type()]);
         count++;
      }
      the_object = tmp_dbase->get_next_obj();
   }
   the_builder->send_bldr("&+G________________________________"
                          "___________&*\n");
   the_builder->send_bldr(_("&+GTotal Objects Listed: &+W%d\n\n"), count);

   the_connection->end_paging(the_builder);   
}

/***********************************************************************
 **  commitcom - commits all changes to a zone to the actual zonefile.
 **              It then becomes a part of the game
 **
 ***********************************************************************/

void commitcom(Parse *the_parsed, MudObject *the_user){
   Builder *the_builder;
   Builder *builder_list;
   Area_Dbase *tmp_dbase;
   FILE       *write_file = NULL;
   Strings    filename;
   Strings    cmnd;
   int        is_action = 0;
   int        is_text = 0;
   int        is_bulletin = 0;
   int        is_ability = 0;
   int        is_quest = 0;
   int        is_race = 0;
   int        is_inclination = 0;
   int        is_talent = 0;
   int        is_mask = 0;
   int        is_chatline = 0;
   int        is_level = 0;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if ((tmp_dbase = the_builder->get_area()) == NULL)

   {
      the_builder->send_bldr(_("You do not have an area loaded.\n"));
      return;
   }

   if (!STRCASECMP(tmp_dbase->get_areaname(), _("actions")))
      is_action = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("quests")))
      is_quest = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("races")))
      is_race = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("inclinations")))
      is_inclination = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("talents")))
      is_talent = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("abilities")))
      is_ability = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("levels")))
      is_level = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("masks")))
      is_mask = 1;
   else if (!STRCASECMP(tmp_dbase->get_areaname(), _("chatlines")))
      is_chatline = 1;

   /* see if somebody else is pointing to it, if so, dont allow commit */
   builder_list = mainstruct->get_first_builder();
   while (builder_list != NULL)
   {
      if ((builder_list->get_areaname() != NULL) && 
                                 (!STRCASECMP(the_builder->get_areaname(), 
                                            builder_list->get_areaname())))
      {
         if (STRCASECMP(the_builder->get_name(), builder_list->get_name()))
	 {
            the_builder->send_bldr(_("Someone else has the area loaded. "
                                      "You can't commit yet.\n"));
            return;
         }
      }
      builder_list = builder_list->get_next_builder();
   }

   /* first delete all the old info files */
   if (!STRCASECMP(the_builder->get_areaname(), _("info")))
   {
      filename.sprintf("%s/%s*%s", the_config.basedir.str_show(), 
                                                      INFODIR, INFOEXTENTION);
      unlink(filename.str_show());
      is_text = 1;
   }
   /* first delete all the old help files */
   else if (!STRCASECMP(the_builder->get_areaname(), _("help")))
   {
      filename.sprintf("%s/%s*%s", the_config.basedir.str_show(), HELPDIR, 
                                                               HELPEXTENTION);
      unlink(filename.str_show());
      is_text = 1;
   }
   /* first delete all the old banner files */
   else if (!STRCASECMP(the_builder->get_areaname(), _("banner")))
   {
      filename.sprintf("%s/%s*%s", the_config.basedir.str_show(), BANNERDIR, 
                                                              BANNEREXTENTION);
      unlink(filename.str_show());
      is_text = 1;
   }
   else if (!STRCASECMP(the_builder->get_areaname(), _("bulletins")))
   {
      filename.sprintf("%s/%s*%s", the_config.basedir.str_show(), 
                                                    BULLETINDIR, BULLEXTENTION);
      unlink(filename.str_show());
      is_bulletin = 1;
   }
   else
   {
      /* open the file to write to */
      if ((!is_action) && (!is_ability) && (!is_quest) && (!is_level) &&
          (!is_race) && (!is_mask) && (!is_inclination) && (!is_talent) &&
          (!is_chatline))
         filename.sprintf("%s/%s/%s.area", the_config.basedir.str_show(), 
                                       AREADIR, the_builder->get_areaname());
      else if (is_ability)
         filename.sprintf("%s/%s/abilities.dat",  
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_quest)
         filename.sprintf("%s/%s/quests.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_level)
         filename.sprintf("%s/%s/levels.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_race)
         filename.sprintf("%s/%s/races.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_inclination)
         filename.sprintf("%s/%s/inclinations.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_talent)
         filename.sprintf("%s/%s/talents.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_mask)
         filename.sprintf("%s/%s/masks.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else if (is_chatline)
         filename.sprintf("%s/%s/chatlines.dat",
                                     the_config.basedir.str_show(), DATA_DIR);
      else
         filename.sprintf("%s/%s/actions.dat",
                                     the_config.basedir.str_show(), DATA_DIR);

      if ((write_file = xfopen(filename.str_show(), "w", 
                                              "write_area_file")) == NULL)
      {
         the_builder->send_bldr(_("Error opening write file, could not save.\n"));
         xfclose(write_file, "write_area_file");
         return;
      }

   }

   the_builder->send_bldr(_("Committing data to file.\n"));
   tmp_dbase->write_area(write_file, 0);

   the_builder->send_bldr(_("Saved.\n"));

   /* delete the builder saved file */

   if ((!is_text) && (!is_bulletin))
   {
      xfclose(write_file, "write_area_file");
      cmnd.sprintf("%s/%s/%s.bldr", the_config.basedir.str_show(), 
                                        BLDRDIR, the_builder->get_areaname());
      unlink(cmnd.str_show());
   }

   /* remove from memory the area file */
   the_builder->delete_area();   

   the_parsed = NULL;        
}


/***********************************************************************
 **  bldrnewcom - creates a mudobject
 **
 ***********************************************************************/

void bldrnewcom(Parse *the_parsed, MudObject *the_user){
   Strings  tmp_word;
   Builder  *the_builder;
   Area_Dbase *the_dbase;
   Timespan the_time;
   int      text_area = 0;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if (the_parsed->get_target1() == NULL)
   {
      the_builder->send_bldr(_("format: new <objecttype> <name>\n"));
      return;
   }

   if ((the_builder->get_area() == NULL) && 
        STRNCASECMP(the_parsed->get_target1(), _("area"), 
                                strlen(the_parsed->get_target1())))
   {
      the_builder->send_bldr(_("You need to load an area first.\n"));
      return;
   }

   if (!isalpha(*(the_parsed->get_speech())))
   {
      the_builder->send_bldr(_("A new name must begin with a letter.\n"));
      return;
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("abilities"))))
   {
      Strings holder;
      Area_Dbase *the_list;

      the_list = the_builder->get_area();

      if (!STRNCASECMP(the_parsed->get_target1(), _("spell"), 
                                strlen(the_parsed->get_target1())))
      {
         Spell *new_spell;

	 if (the_parsed->get_speech() == NULL)
	 {
            the_builder->
               send_bldr(_("You need to supply a name for the new spell.\n"));
            return;
         }

         holder.assign_word(the_parsed->get_speech(), 1);
         new_spell = new_Spell(holder.str_show());

         if (the_list->add_areaobject((Entity *)new_spell) == -1)
	 {
            the_builder->send_bldr(_("A spell already exists by that name.\n"));
            delete_Spell(new_spell);
            return;
         }
         the_builder->modify_object(new_spell->get_name());
         the_builder->send_bldr(_("Spell created with name %s\n"), 
                                                  new_spell->get_name());
        
         return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("skill"), 
                                strlen(the_parsed->get_target1())))
      {
         Skill *new_skill;

	 if (the_parsed->get_speech() == NULL)
	 {
            the_builder->
               send_bldr(_("You need to supply a name for the new skill.\n"));
            return;
         }

         holder.assign_word(the_parsed->get_speech(), 1);
         new_skill = new_Skill(holder.str_show());

         if (the_list->add_areaobject((Entity *)new_skill) == -1)
	 {
            the_builder->send_bldr(_("A skill already exists by that name.\n"));
            delete_Skill(new_skill);
            return;
         }
         the_builder->modify_object(new_skill->get_name());
         the_builder->send_bldr(_("Skill created with name %s\n"), 
                                                  new_skill->get_name());
        
         return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("dependency"),
                               strlen(the_parsed->get_target1())))
      {  Ability   *the_ability;
         Entity    *the_modified; 
         Strings   holder;
         int       results = 0;

         if ((the_modified = the_builder->get_modify()) == NULL)
         {
            the_builder->send_bldr(_("You need to set an ability to modify first.\n"));
            return; 
         }
        if ((the_modified->get_type() != OBJ_TYPE_SPELL) &&
	     (the_modified->get_type() != OBJ_TYPE_SKILL))
         {
            the_builder->send_bldr(_("You can only do this with abilities.\n"));
            return;
         }
        the_ability = (Ability *) the_modified;
     
        if (!STRCASECMP(the_parsed->get_speech(), "musthaveitem"))
	   results = the_ability->add_dependency(the_parsed->get_speech(), "");
        else
	   results = the_ability->add_dependency(the_parsed->get_speech(), 0);

	if (results == 0)
	{
	  the_builder->send_bldr("That dependency already exists.\n");
	  return;
	}
        else if (results != 1)
         {
  	   int i = 0;
	   int line_len = 0;
	   the_builder->
	    send_bldr(_("That is not a valid type.  Possible types are:\n"));
	   while (depend_name[i] != NULL)
	   {
	     line_len += strlen(depend_name[i]) + 1;
	     holder.str_cat(depend_name[i]);
	     if (line_len > 70)
	     {
	       holder.str_cat("\n");
	       line_len = 0;
	       the_builder->send_bldr(holder.str_show());
	       holder.truncate(0);
	     }
	     else
	        holder.str_cat(" ");
	     i++;
	   }
	   holder.str_cat("\n");
	   the_builder->send_bldr(holder.str_show());
   
           return;
         }

         the_builder->send_bldr(_("Dependency &+M%s&* created.\n"), 
                                 the_parsed->get_speech());
	 return;
      }
      else
      {
         the_builder->
           send_bldr(_("You can only create spells or skills in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("quests"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("quests"), 
                                strlen(the_parsed->get_target1())))
      {
        Quest *new_quest;
        Strings holder;
        Area_Dbase *the_list;

        if (!the_config.conf_flags->get_flag(CF_USEQUESTS))
	{
           the_builder->send_bldr(_("Quests have been turned off.\n"));
           return;
	}

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new quest.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_quest = new_Quest(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_quest) == -1)
	{
           the_builder->send_bldr(_("A quest already exists by that name.\n"));
           delete_Quest(new_quest);
           return;
        }
        the_builder->modify_object(new_quest->get_name());
        the_builder->send_bldr(_("Quest created with name %s\n"), 
                                                  new_quest->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new quests in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("races"))))
   {

      if (!STRNCASECMP(the_parsed->get_target1(), _("races"), 
                                strlen(the_parsed->get_target1())))
      {
        Race *new_race;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new race.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_race = new_Race(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_race) == -1)
	{
           the_builder->send_bldr(_("A race already exists by that name.\n"));
           delete_Race(new_race);
           return;
        }
        the_builder->modify_object(new_race->get_name());
        the_builder->send_bldr(_("Race created with name %s\n"), 
                                                  new_race->get_name());
        
        return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("tutorial"),
                               strlen(the_parsed->get_target1())))
      { 
         Entity *the_modified;
         Race *the_race;
         int results;

         if ((the_modified = the_builder->get_modify()) == NULL)
         {
            the_builder->
                    send_bldr(_("You need to set an object to modify first.\n"));
            return;
         }

         if (the_modified->get_type() != OBJ_TYPE_RACE)
         {
            the_builder->
                    send_bldr(_("You can only assign tutorials to races.\n"));
            return;
         }

         the_race = (Race *) the_modified;
      
         if (the_parsed->get_speech() == NULL)
         {
	   the_builder->send_bldr(_("You must give it a unique name.\n"
                                "format: new tutorial <name>\n"));
            return;
         }

         if ((results = the_race->add_tutorial(the_parsed->get_speech())) == -2)
         {
            the_builder->send_bldr(_("An tutorial under that name already exists.\n"));
            return;
         }

         the_builder->send_bldr(_("Tutorial added with name: %s\n"), 
                                                  the_parsed->get_speech());
         return;
      }
      else
      {
         the_builder->
           send_bldr(_("You can only create new races or tutorials in this area.\n"));
         return;
      }
   }


   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("masks"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("masks"), 
                                strlen(the_parsed->get_target1())))
      {
        Mask *new_mask;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new mask.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_mask = new_Mask(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_mask) == -1)
	{
           the_builder->send_bldr(_("A mask already exists by that name.\n"));
           delete_Mask(new_mask);
           return;
        }
        the_builder->modify_object(new_mask->get_name());
        the_builder->send_bldr(_("Mask created with name %s\n"), 
                                                  new_mask->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new masks in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("chatlines"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("chatlines"), 
                                strlen(the_parsed->get_target1())))
      {
        ChatLine *new_chatline;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new chatline.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_chatline = new_ChatLine(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_chatline) == -1)
	{
           the_builder->send_bldr(_("A chatline already exists by that name.\n"));
           delete_ChatLine(new_chatline);
           return;
        }
        the_builder->modify_object(new_chatline->get_name());
        the_builder->send_bldr(_("ChatLine created with name %s\n"), 
                                                  new_chatline->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new chatlines in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("inclinations"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("inclinations"), 
                                strlen(the_parsed->get_target1())))
      {
        Inclination *new_inclination;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new inclination.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_inclination = new_Inclination(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_inclination) == -1)
	{
           the_builder->send_bldr(_("A inclination already exists by that name.\n"));
           delete_Inclination(new_inclination);
           return;
        }
        the_builder->modify_object(new_inclination->get_name());
        the_builder->send_bldr(_("Inclination created with name %s\n"), 
                                                  new_inclination->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new inclinations in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("talents"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("talents"), 
                                strlen(the_parsed->get_target1())))
      {
        Talent *new_talent;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new talent.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_talent = new_Talent(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_talent) == -1)
	{
           the_builder->send_bldr(_("A talent already exists by that name.\n"));
           delete_Talent(new_talent);
           return;
        }
        the_builder->modify_object(new_talent->get_name());
        the_builder->send_bldr(_("Talent created with name %s\n"), 
                                                  new_talent->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new talents in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("levels"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("levels"), 
                                strlen(the_parsed->get_target1())))
      {
        Level *new_level;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new level.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_level = new Level(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_level) == -1)
	{
           the_builder->send_bldr(_("An object already exists by that name.\n"));
           delete new_level;
           return;
        }
        the_builder->modify_object(new_level->get_name());
        the_builder->send_bldr(_("Level created with name %s\n"), 
                                                  new_level->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new levels in this area.\n"));
         return;
      }
   }

   if ((the_builder->get_area() != NULL) && 
       (!STRCASECMP((the_builder->get_area())->get_areaname(), _("actions"))))
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("action"), 
                                strlen(the_parsed->get_target1())))
      {
        Action *new_action;
        Strings holder;
        Area_Dbase *the_list;

	if (the_parsed->get_speech() == NULL)
	{
           the_builder->
               send_bldr(_("You need to supply a name for the new action.\n"));
           return;
        }

        the_list = the_builder->get_area();
        holder.assign_word(the_parsed->get_speech(), 1);
        new_action = new_Action(holder.str_show());

        if (the_list->add_areaobject((Entity *)new_action) == -1)
	{
           the_builder->send_bldr(_("An action already exists by that name.\n"));
           delete_Action(new_action);
           return;
        }
        the_builder->modify_object(new_action->get_name());
        the_builder->send_bldr(_("Action created with name %s\n"), 
                                                  new_action->get_name());
        
        return;
      }
      else
      {
         the_builder->
               send_bldr(_("You can only create new actions in this area.\n"));
         return;
      }
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("area"), 
                                strlen(the_parsed->get_target1())))
   {
      Strings    area_filename;
      Strings    cmnd;
      FILE       *tmp_file = NULL;
      Area_Dbase *tmp_area;

      tmp_word.assign_word(the_parsed->get_speech(), 1);
      if (tmp_word.str_show() == NULL)
      {
         the_builder->send_bldr(_("You must assign a name to a new area.\n"));
         return;
      }

      tmp_word.lowercase();
      area_filename.sprintf("%s/%s/%s.area", the_config.basedir.str_show(), 
                                                 AREADIR, tmp_word.str_show());
      if ((tmp_file = xfopen(area_filename.str_show(), "r", 
                                                 "read_area_file")) != NULL)
      {
         the_builder->send_bldr(_("The area '%s' area already exists!\n"), 
                                                   tmp_word.str_show());
         xfclose(tmp_file, "read_area_file");
         return;
      }

      area_filename.sprintf("%s/%s/%s.bldr", the_config.basedir.str_show(), 
                                                  BLDRDIR, tmp_word.str_show());
      if ((tmp_file = xfopen(area_filename.str_show(), "r", 
                                                 "read_area_file")) != NULL)
      {
         the_builder->send_bldr(_("The area '%s' area already exists!\n"), 
                                              the_parsed->get_speech());
         xfclose(tmp_file, "read_area_file");
         return;
      }

      /* create the builder file and write it to file */
      if ((tmp_file = xfopen(area_filename.str_show(), "w", 
                                          "write_area_file")) == NULL)
      {
         mainstruct->log_error(_("Error opening newly created area"), 
                                                  "bldrnewcom");
         return;
      }
      tmp_area = new_Area_Dbase(tmp_word.str_show());
      tmp_area->write_area(tmp_file, 1);
      xfclose(tmp_file, "write_area_file");
      delete_Area_Dbase(tmp_area);

      /* now load the newly created empty area */
      if ((tmp_file = xfopen(area_filename.str_show(), "r", 
                                          "read_area_file")) == NULL)
      {
         mainstruct->log_error(_("Error opening newly created area"), 
                                                  "bldrnewcom");
         return;
      }

      if (the_builder->get_area() != NULL)
         the_builder->unload_area();

      if (the_builder->load_area(tmp_word.str_show(), tmp_file,
                                             mainstruct->get_log(), 1) < 0)
      {
         mainstruct->log_error(_("Error loading newly created areafile"), 
                                                   "bldrnewcom");
         xfclose(tmp_file, "read_area_file");
         return;
      }

      the_builder->send_bldr(_("Successfully created area '%s'\n"), 
                                                  tmp_word.str_show());
      return;
      
   }

   tmp_word.assign_word(the_parsed->get_speech(), 1);
   if (tmp_word.str_show() == NULL)
   {
      the_builder->send_bldr(_("format: new <type> <name>\n"));
      return;
   }

   the_dbase = the_builder->get_area();
 

   if ((!STRCASECMP(the_dbase->get_areaname(), _("info"))) ||
       (!STRCASECMP(the_dbase->get_areaname(), _("help"))) || 
       (!STRCASECMP(the_dbase->get_areaname(), _("banner"))))
   {
      text_area = 1;

   }

   if  ((text_area) &&
        (STRCASECMP(the_parsed->get_target1(), _("info"))) &&
        (STRCASECMP(the_parsed->get_target1(), _("help"))) &&
        (STRCASECMP(the_parsed->get_target1(), _("banner"))) &&
        (STRCASECMP(the_parsed->get_target1(), _("text"))))
   {
      the_builder->
        send_bldr(_("You can't create an object of that type in this area.\n"));
      return;
   }

   if ((!STRCASECMP(the_dbase->get_areaname(), _("bulletins"))) &&
       (STRCASECMP(the_parsed->get_target1(), _("bulletin"))))
   {
      the_builder->
        send_bldr(_("You can only create bulletins in the bulletin area.\n"));
      return;
   }
  
   if ((the_parsed->get_target1() == NULL) ||
      (!STRNCASECMP(the_parsed->get_target1(), _("location"), 
                               strlen(the_parsed->get_target1()))))
   {
      Location *new_loc;

      new_loc = new_Location(tmp_word.str_show(), the_builder->get_areaname());
      new_loc->set_modified(1);
      new_loc->set_title(_("New Location"));
      new_loc->set_desc(0, _("No Description."));
      new_loc->set_location(_("none"));
      
      if (the_dbase->add_areaobject((MudObject *) new_loc) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_loc->get_name()); 
         delete_Location(new_loc);
         return;
      }
      the_builder->modify_object(new_loc->get_name());
      the_builder->send_bldr(_("New location '%s' created.\n"), 
                                                  new_loc->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("marker"), 
                               strlen(the_parsed->get_target1())))
   {
      Marker *new_mark;

      new_mark = new_Marker(tmp_word.str_show(), the_builder->get_areaname());
      new_mark->set_modified(1);
      new_mark->set_title(_("noname"));
      new_mark->set_desc(_("No Description."));
      new_mark->set_location(_("None"));
      
      if (the_dbase->add_areaobject((MudObject *) new_mark) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_mark->get_name()); 
         delete_Marker(new_mark);
         return;
      }
      the_builder->modify_object(new_mark->get_name());
      the_builder->send_bldr(_("New marker '%s' created.\n"), 
                                                  new_mark->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("door"), 
                               strlen(the_parsed->get_target1())))
   {  Door       *new_door;

      new_door = new_Door(tmp_word.str_show(),the_builder->get_areaname());
 
      new_door->set_door_state(0);
      new_door->set_title(_("NoName"));
      new_door->set_desc(_("No description"));
      
      if (the_dbase->add_areaobject((MudObject *) new_door) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_door->get_name()); 
         delete_Door(new_door);
         return;
      }
      
      the_builder->modify_object(new_door->get_name());
      the_builder->send_bldr(_("New door '%s' created.\n"),
                             new_door->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("moveable"), 
                               strlen(the_parsed->get_target1())))
   {  Moveable   *new_moveable;

      new_moveable = new_Moveable(tmp_word.str_show(),the_builder->get_areaname());
 
      new_moveable->set_size(0);
      new_moveable->set_weight(0);
      new_moveable->set_title(_("NoName"));
      new_moveable->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_moveable) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_moveable->get_name()); 
         delete_Moveable(new_moveable);
         return;
      }
      
      the_builder->modify_object(new_moveable->get_name());
      the_builder->send_bldr(_("New moveable object '%s' created.\n"),
                             new_moveable->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("wearable"), 
                               strlen(the_parsed->get_target1())))
   {  Wearable   *new_wearable;

      new_wearable = new_Wearable(tmp_word.str_show(), 
                                          the_builder->get_areaname());
 
      new_wearable->set_size(0);
      new_wearable->set_weight(0);
      new_wearable->set_title(_("NoName"));
      new_wearable->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_wearable) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_wearable->get_name()); 
         delete_Wearable(new_wearable);
         return;
      }
      
      the_builder->modify_object(new_wearable->get_name());
      the_builder->send_bldr(_("New wearable object '%s' created.\n"),
                             new_wearable->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("weapon"), 
                               strlen(the_parsed->get_target1())))
   {  Weapon   *new_weapon;

      new_weapon = new_Weapon(tmp_word.str_show(), 
                                              the_builder->get_areaname());
 
      new_weapon->set_size(0);
      new_weapon->set_weight(0);
      new_weapon->set_title(_("NoName"));
      new_weapon->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_weapon) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_weapon->get_name()); 
         delete_Weapon(new_weapon);
         return;
      }
      
      the_builder->modify_object(new_weapon->get_name());
      the_builder->send_bldr(_("New weapon object '%s' created.\n"),
                             new_weapon->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("merger"), 
                               strlen(the_parsed->get_target1())))
   {  Merger   *new_merger;

      new_merger = new_Merger(tmp_word.str_show(),the_builder->get_areaname());
 
      new_merger->set_size(0);
      new_merger->set_weight(0);
      new_merger->set_number_of(0);
      new_merger->set_title(_("NoName"));
      new_merger->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_merger) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_merger->get_name()); 
         delete_Merger(new_merger);
         return;
      }
      
      the_builder->modify_object(new_merger->get_name());
      the_builder->send_bldr(_("New merger object '%s' created.\n"),
                             new_merger->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("money"), 
                               strlen(the_parsed->get_target1())))
   {  Money   *new_money;

      new_money = new_Money(tmp_word.str_show(),the_builder->get_areaname());
 
      new_money->set_size(0);
      new_money->set_weight(0);
      new_money->set_number_of(0);
      new_money->set_money_index(0);
      new_money->set_title(_("NoName"));
      new_money->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject  *) new_money) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_money->get_name()); 
         delete_Money(new_money);
         return;
      }
      
      the_builder->modify_object(new_money->get_name());
      the_builder->send_bldr(_("New money object '%s' created.\n"),
                             new_money->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("food"), 
                               strlen(the_parsed->get_target1())))
   {  Food   *new_food;

      new_food = new_Food(tmp_word.str_show(),the_builder->get_areaname());
 
      new_food->set_size(0);
      new_food->set_weight(0);
      new_food->set_number_of(0);
      new_food->set_energy(0);
      new_food->set_title(_("NoName"));
      new_food->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject  *) new_food) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_food->get_name()); 
         delete_Food(new_food);
         return;
      }
      
      the_builder->modify_object(new_food->get_name());
      the_builder->send_bldr(_("New food object '%s' created.\n"),
                             new_food->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("key"), 
                               strlen(the_parsed->get_target1())))
   {  Key   *new_key;

      new_key = new_Key(tmp_word.str_show(),the_builder->get_areaname());
 
      new_key->set_size(0);
      new_key->set_weight(0);
      new_key->set_title(_("NoName"));
      new_key->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_key) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_key->get_name()); 
         delete_Key(new_key);
         return;
      }
      
      the_builder->modify_object(new_key->get_name());
      the_builder->send_bldr(_("New key object '%s' created.\n"),
                             new_key->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("boat"), 
                               strlen(the_parsed->get_target1())))
   {  Boat  *new_boat;

      new_boat = new_Boat(tmp_word.str_show(),the_builder->get_areaname());
 
      new_boat->set_size(0);
      new_boat->set_weight(0);
      new_boat->set_title(_("NoName"));
      new_boat->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_boat) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_boat->get_name()); 
         delete_Boat(new_boat);
         return;
      }
      
      the_builder->modify_object(new_boat->get_name());
      the_builder->send_bldr(_("New boat object '%s' created.\n"),
                             new_boat->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("book"), 
                               strlen(the_parsed->get_target1())))
   {  Book   *new_book;

      new_book = new_Book(tmp_word.str_show(),the_builder->get_areaname());
 
      new_book->set_title(_("NoName"));
      new_book->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_book) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_book->get_name()); 
         delete_Book(new_book);
         return;
      }
      
      the_builder->modify_object(new_book->get_name());
      the_builder->send_bldr(_("New book object '%s' created.\n"),
                             new_book->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("rope"), 
                               strlen(the_parsed->get_target1())))
   {  Rope   *new_rope;

      new_rope = new_Rope(tmp_word.str_show(),the_builder->get_areaname());
 
      new_rope->set_title(_("NoName"));
      new_rope->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_rope) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_rope->get_name()); 
         delete_Rope(new_rope);
         return;
      }
      
      the_builder->modify_object(new_rope->get_name());
      the_builder->send_bldr(_("New rope object '%s' created.\n"),
                             new_rope->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("mobile"), 
                               strlen(the_parsed->get_target1())))
   {  Mobile   *new_mob;

      new_mob = new_Mobile(tmp_word.str_show(),the_builder->get_areaname());
 
      new_mob->set_title(_("NoName"));
      new_mob->set_desc(_("No description"));

      if (the_dbase->add_areaobject((MudObject *) new_mob) <= 0)
      {
         the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_mob->get_name()); 
         delete_Mobile(new_mob);
         return;
      }
      
      the_builder->modify_object(new_mob->get_name());
      the_builder->send_bldr(_("New mobile object '%s' created.\n"),
                             new_mob->get_name());
   }
   else if ((!STRNCASECMP(the_parsed->get_target1(), _("text"), 
                               strlen(the_parsed->get_target1()))) ||
            (!STRNCASECMP(the_parsed->get_target1(), _("info"), 
                               strlen(the_parsed->get_target1()))) ||
            (!STRNCASECMP(the_parsed->get_target1(), _("help"), 
                               strlen(the_parsed->get_target1()))) ||
            (!STRNCASECMP(the_parsed->get_target1(), _("banner"), 
                               strlen(the_parsed->get_target1()))))
   {  
      Text   *new_text;
      int    text_type = 0;

      if (!text_area)
      {
        the_builder->
         send_bldr(_("You can't create an object of that type in this area.\n"));
        return;
      }

      if (!STRCASECMP(the_dbase->get_areaname(), _("info")))
         text_type = TEXT_TYPE_INFO;
      else if (!STRCASECMP(the_dbase->get_areaname(), _("help")))
         text_type = TEXT_TYPE_HELP;
      else if (!STRCASECMP(the_dbase->get_areaname(), _("banner")))
         text_type = TEXT_TYPE_BANNER;

      tmp_word.lowercase();
      new_text = new_Text(tmp_word.str_show(), text_type);
 
      new_text->set_title(_("New Text"));
      new_text->set_body(_("No body"));

      if (the_dbase->add_areaobject((Entity *) new_text) <= 0)
      {
         the_builder->send_bldr(_("The text '%s' already exists!\n"), 
                                                  new_text->get_name()); 
         delete_Text(new_text);
         return;
      }

      the_builder->modify_object(new_text->get_name());      
      the_builder->send_bldr(_("New text object '%s' created.\n"),
                             new_text->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("bulletin"), 
                               strlen(the_parsed->get_target1())))
   {  
      Bulletin *new_bull;

      new_bull = new_Bulletin(tmp_word.str_show(), 1);
 
      new_bull->set_location(_("none"));

      if (the_dbase->add_areaobject((Entity *) new_bull) <= 0)
      {
         the_builder->send_bldr(_("The bulletin '%s' already exists!\n"), 
                                                  new_bull->get_name()); 
         delete_Bulletin(new_bull);
         return;
      }

      the_builder->modify_object(new_bull->get_name());      
      the_builder->send_bldr(_("New bulletin object '%s' created.\n"),
                                                   new_bull->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("reply"),
                               strlen(the_parsed->get_target1())))
   {  Mobile    *the_mob;
      Entity    *the_modified;      
      Com_List  *new_list; 
      int       reply_num;

      if ((the_modified = the_builder->get_modify()) == NULL)
      {
         the_builder->send_bldr(_("You need to set a mobile to modify first.\n"));
         return;
      }
      if (the_modified->get_type() != OBJ_TYPE_MOBILE)
      {
         the_builder->send_bldr(_("You can only do this with mobiles.\n"));
         return;
      }
      the_mob = (Mobile *) the_modified;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("You must give it one or more keywords.\n"
                                "format: new reply <keywords>\n"));
         return;
      }

      new_list = new_Com_List();
      new_list->next_com = NULL;
      new_list->the_keywords = the_parsed->get_speech();
      new_list->the_reply = _("REPLY PENDING");
      reply_num = the_mob->add_comlist(new_list);
      the_builder->send_bldr(_("Tell Reply &+M%d&* created with keywords: %s\n"), 
                              reply_num, new_list->the_keywords.str_show());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("special"), 
                               strlen(the_parsed->get_target1())))
   {  Specials *new_special;

      new_special = new_Specials(tmp_word.str_show(),the_builder->get_areaname()); 
      new_special->set_trigger(_("None"));

      if (the_dbase->add_areaobject((Entity *) new_special) <= 0)
      {
         the_builder->send_bldr(_("The special '%s' already exists!\n"), 
                                                  new_special->get_name()); 
         delete_Specials(new_special);
         return;
      }
      
      the_builder->modify_object(new_special->get_name());
      the_builder->send_bldr(_("New special object '%s' created.\n"),
                             new_special->get_name());
   }
   else if (!STRNCASECMP(the_parsed->get_target1(), _("shop"),
                               strlen(the_parsed->get_target1())))
   { 
      Entity *the_modified;
      Shop *new_shop;
      Mobile *the_mobile;

      if ((the_modified = the_builder->get_modify()) == NULL)
      {
         the_builder->
                    send_bldr(_("You need to set an object to modify first.\n"));
         return;
      }

      if (the_modified->get_type() != OBJ_TYPE_MOBILE)
      {
         the_builder->
                    send_bldr(_("You can only assign shops to mobiles.\n"));
         return;
      }

      the_mobile = (Mobile *) the_modified;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("You must give it a name.\n"
                                "format: new shop <name>\n"));
         return;
      }

      new_shop = new_Shop();
      new_shop->set_name(the_parsed->get_speech());
      the_mobile->set_shop(new_shop);
      the_builder->send_bldr(_("Shop created with name: %s\n"), 
                                                    new_shop->get_name());
   }

   else if (!STRNCASECMP(the_parsed->get_target1(), _("item"),
                               strlen(the_parsed->get_target1())))
   { 
      Entity *the_modified;
      Shop *the_shop;
      Mobile *the_mobile;

      if ((the_modified = the_builder->get_modify()) == NULL)
      {
         the_builder->
                    send_bldr(_("You need to set an object to modify first.\n"));
         return;
      }

      if (the_modified->get_type() != OBJ_TYPE_MOBILE)
      {
         the_builder->
                    send_bldr(_("You can only assign items to mobiles.\n"));
         return;
      }

      the_mobile = (Mobile *) the_modified;
      
      if ((the_shop = the_mobile->get_shop()) == NULL)
      {
         the_builder->send_bldr(_("You need to create a shop first.\n"));
         return;
      }

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("You must give it a unique alias.\n"
                                "format: new item <alias>\n"));
         return;
      }

      if ((the_shop->add_shop_item(_("noobj"), the_parsed->get_speech(), 
                                                  1)) == -2)
      {
         the_builder->send_bldr(_("An item under that alias already exists.\n"));
         return;
      }

      the_builder->send_bldr(_("Shop item added with alias: %s\n"), 
                                                  the_parsed->get_speech());
   }

   else
   {
      the_builder->send_bldr(_("The object type '%s' is not a valid type.\n"), 
                                                the_parsed->get_target1());
      return;
   }
}


/***********************************************************************
 **  bldrdeletecom - deletes a mudobject from the area
 **
 ***********************************************************************/

void bldrdeletecom(Parse *the_parsed, MudObject *the_user){
   Strings    tmp_word;
   Builder    *the_builder;
   Entity     *the_object;
   Inp_Handler *this_handler;
   Area_Dbase  *the_dbase;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;
   this_handler = the_builder->get_input_handler();

   if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(the_parsed->get_target1(), _("area"))))
   {
      Strings    del_area;
      Strings    filename;
      int        only_builder = 0;
      Builder    *builder_list;

      del_area.assign_word(the_parsed->get_speech(), 1);
      if (del_area.str_show() == NULL)
      {
         if (the_builder->get_area() == NULL)
	 {
            the_builder->send_bldr(_("Delete which area?\n"));
            return;
	 }

         del_area = (the_builder->get_area())->get_areaname();
      }

      filename.sprintf("%s/%s/%s.area", 
             the_config.basedir.str_show(), AREADIR, del_area.str_show());
      if (!file_exists(filename.str_show()))
      {
         filename.sprintf("%s/%s/%s.bldr", 
              the_config.basedir.str_show(), BLDRDIR, del_area.str_show());
         if (!file_exists(filename.str_show()))
         {
            the_builder->send_bldr(_("Could not find area '%s' to delete.\n"),
                                                del_area.str_show());
               return;
         }
         only_builder = 1;
      }


      /* first go through and see if anyone has it loaded and if so, don't
         let them delete it */
      builder_list = mainstruct->get_first_builder();
      while (builder_list != NULL)
      {
         if ((builder_list->get_areaname() != NULL) && 
                                 (!STRCASECMP(del_area.str_show(), 
                                            builder_list->get_areaname())))
         {
            if (STRCASECMP(the_builder->get_name(), builder_list->get_name()))
	    {
               the_builder->send_bldr(_("%s has the area loaded. "
                                      "You can't delete it.\n"), 
                                      the_builder->get_title());
               return;
            }
         }
         builder_list = builder_list->get_next_builder();
      }
      
      /* now unload the area if they have it loaded */
      if ((the_builder->get_areaname() != NULL) && 
          (!STRCASECMP(the_builder->get_areaname(), del_area.str_show())))
      {
         the_builder->unload_area();        
      }

      if (!only_builder)
      {
         filename.sprintf("%s/%s/%s.area", 
                the_config.basedir.str_show(), AREADIR, del_area.str_show());
         unlink(filename.str_show());
      }

      filename.sprintf("%s/%s/%s.bldr", 
                the_config.basedir.str_show(), BLDRDIR, del_area.str_show());
      unlink(filename.str_show());
      the_builder->send_bldr(_("Area file '%s' deleted.\n"), 
                                                        del_area.str_show());
      
      return;

   }

   if ((the_object = the_builder->get_modify()) == NULL)
   {
      the_builder->send_bldr(_("You have not set an object to modify yet.\n"));
      return;
   }

   if ((the_dbase = the_builder->get_area()) == NULL)
   {
      the_builder->send_bldr(_("Unexpected Error encountered. Failed.\n"));
      return;
   }

   if (the_parsed->get_target1() != NULL)
   {
      if (!STRNCASECMP(the_parsed->get_target1(), _("reply"), 
                                         strlen(the_parsed->get_target1())))
      {  Mobile    *the_mob; 
         int       the_num;

         if (the_object->get_type() != OBJ_TYPE_MOBILE)
         {
            the_builder->send_bldr(_("You can only do this with mobiles.\n"));
            return;
         }

         the_mob = (Mobile *) the_object;

         if ((the_parsed->get_speech() == NULL) || 
             (!isdigit(*(the_parsed->get_speech()))))
         {
           the_builder->send_bldr(_("You need to indicate which one to delete.\n"
                                  "format: delete reply <number>\n"));
           return;
         }

         the_num = atoi(the_parsed->get_speech());
         if (the_mob->del_comlist(the_num) == -1)
	 {
            the_builder->send_bldr(_("That reply does not exist.\n"));
            return;
         }
         the_builder->send_bldr(_("Reply successfully deleted.\n"));
         return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("item"), 
                                         strlen(the_parsed->get_target1())))
      {  Mobile    *the_mob; 
         Shop      *the_shop;
         Strings   the_alias;

         if (the_object->get_type() != OBJ_TYPE_MOBILE)
         {
            the_builder->send_bldr(_("You can only do this with mobile shops.\n"));
            return;
         }

         the_mob = (Mobile *) the_object;

         if ((the_shop = the_mob->get_shop()) == NULL)
	 {
            the_builder->send_bldr(_("This mobile does not have a shop.\n"));
            return;
	 }

         the_alias.assign_word(the_parsed->get_speech(), 1);
         if (the_alias.str_show() == NULL)
         {
           the_builder->send_bldr(_("You need to indicate which one to delete.\n"
                                  "format: delete item <alias>\n"));
           return;
         }

         if (the_shop->del_shop_item(the_alias.str_show()) == -1)
	 {
            the_builder->send_bldr(_("That shop item does not exist.\n"));
            return;
         }
         the_builder->send_bldr(_("Shop item successfully deleted.\n"));
         return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("tutorial"), 
                                         strlen(the_parsed->get_target1())))
      {  
         Race      *the_race;
         Strings   the_name;

         if (the_object->get_type() != OBJ_TYPE_RACE)
         {
            the_builder->send_bldr(_("You can only do this with races.\n"));
            return;
         }

         the_race = (Race *) the_object;

         the_name.assign_word(the_parsed->get_speech(), 1);
         if (the_name.str_show() == NULL)
         {
           the_builder->send_bldr(_("You need to indicate which one to delete.\n"
                                  "format: delete tutorial <name>\n"));
           return;
         }

         if (the_race->del_tutorial(the_name.str_show()) == -1)
	 {
            the_builder->send_bldr(_("That tutorial does not exist.\n"));
            return;
         }
         the_builder->send_bldr(_("Tutorial successfully deleted.\n"));
         return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("shop"), 
                                         strlen(the_parsed->get_target1())))
      {  Mobile    *the_mob; 

         if (the_object->get_type() != OBJ_TYPE_MOBILE)
         {
            the_builder->send_bldr(_("You can only do this with mobile shops.\n"));
            return;
         }

         the_mob = (Mobile *) the_object;

         if (the_mob->get_shop() == NULL)
	 {
            the_builder->send_bldr(_("This mobile does not have a shop.\n"));
            return;
	 }

         the_mob->delete_shop();
         the_builder->send_bldr(_("Shop item successfully deleted.\n"));
         return;
      }
      else if (!STRNCASECMP(the_parsed->get_target1(), _("dependency"), 
                                         strlen(the_parsed->get_target1())))
      {  
	 Ability    *the_abil;
	 Strings    the_type;

         if ((the_object->get_type() != OBJ_TYPE_SPELL) &&
	     (the_object->get_type() != OBJ_TYPE_SKILL)) 
         { 
            the_builder->
	        send_bldr(_("You can only do this with ability dependencies.\n"));
            return;
         }

         the_abil = (Ability *) the_object;

         the_type.assign_word(the_parsed->get_speech(), 1);
         if (the_type.str_show() == NULL)
         {
           the_builder->send_bldr(_("You need to indicate which one to delete.\n"
                                  "format: delete dependency <type>\n"));
           return;
         }

	 if (the_abil->del_dependency(the_type.str_show()) == 0)
	 {
           the_builder->send_bldr(_("Could not find that dependency\n"));
           return;
	 }

         the_builder->send_bldr(_("Dependency %s successfully deleted.\n"), 
				                      the_type.str_show());
         return;
      }
   }

   tmp_word.sprintf(_("Deleting %s. Are you sure you want to do this? "),
                                                    the_object->get_name());
   this_handler->push_input_handler(delete_yes_no, tmp_word.str_show(), 0);
                                
}


/***********************************************************************
 **  bldrunload - unloads a zone from memory, or if someone else has it
 **               loaded still, just doesnt point this player to it
 **
 ***********************************************************************/

void bldrunloadcom(Parse *the_parsed, MudObject *the_user) {
   Builder *the_builder;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if (the_builder->get_area() == NULL)
   {
      the_builder->send_bldr(_("You don't have an area loaded yet.\n"));
      return;
   }      

   the_builder->unload_area();

   the_builder->send_bldr(_("Successfully unloaded area\n"));
   the_parsed = NULL;
}


/***********************************************************************
 **  bldrcopycom - copies a mudobject, giving it a different name
 **
 ***********************************************************************/

void bldrcopycom(Parse *the_parsed, MudObject *the_user){
   Strings    tmp_word;
   Builder    *the_builder;
   Entity     *the_object;
   Area_Dbase *the_dbase;
   Entity  *new_object;
   Timespan   the_time;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if ((the_object = the_builder->get_modify()) == NULL)
   {
      the_builder->send_bldr(_("You have not set an object to modify yet.\n"));
      return;
   }

   if ((the_dbase = the_builder->get_area()) == NULL)
   {
      the_builder->send_bldr(_("Unexpected Error encountered. Failed.\n"));
      return;
   }

   if (the_parsed->get_target1() == NULL)
   {
      the_builder->
               send_bldr(_("You need to specify a name for the new object.\n"));
      return;
   }

   if (!STRCASECMP(the_object->get_name(), the_parsed->get_target1()))
   {
      the_builder->
        send_bldr(_("The new object can't have the same name as its parent.\n"));
      return;
   }

   new_object = the_object->copy_obj(the_parsed->get_target1());

   if (the_dbase->add_areaobject(new_object) <= 0)
   {
      the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_object->get_name()); 
      mainstruct->delete_entity(new_object);
      return;
   }
   the_builder->modify_object(new_object->get_name());
   the_builder->send_bldr(_("New MudObject '%s' created.\n"), 
                                                  new_object->get_name());
                              
   the_parsed = NULL;                                  
}


/***********************************************************************
 **  bldrrenamecom - copies a mudobject, giving it a different name, then
 **                  deletes the original mudobject
 **
 ***********************************************************************/

void bldrrenamecom(Parse *the_parsed, MudObject *the_user){
   Strings    tmp_word;
   Strings    syscom;
   Builder    *the_builder;
   Entity     *the_object;
   Area_Dbase *the_dbase;
   Entity     *new_object;
   Timespan   the_time;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;

   the_builder = (Builder *) the_user;

   if ((the_dbase = the_builder->get_area()) == NULL)
   {
      the_builder->send_bldr(_("You do not have an area loaded.\n"));
      return;
   }

   if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(the_parsed->get_target1(), _("area"))))
   {
     Strings fromfile;
     Strings tofile;
      tmp_word.assign_word(the_parsed->get_speech(), 1);
      if (tmp_word.str_show() == NULL)
      {
         the_builder->
            send_bldr(_("What would you like to change the area name to?\n"));
         return;
      }

      tofile.sprintf("%s/%s/%s.area", the_config.basedir.str_show(), AREADIR, 
                                                          tmp_word.str_show());
      fromfile.sprintf("%s/%s/%s.area", the_config.basedir.str_show(), AREADIR, 
		                                the_dbase->get_areaname());
      
      if (file_exists(tofile.str_show()))
      {
	     the_builder->send_bldr("An area by that name already exists.\n");
	     return;
      }

#ifdef WIN32
	  rename(fromfile.str_show(), tofile.str_show());
#else
      link(fromfile.str_show(), tofile.str_show());
      unlink(fromfile.str_show());
#endif
      the_dbase->set_areaname(tmp_word.str_show());
      the_builder->set_areaname(tmp_word.str_show());
      the_builder->set_prompt();
      the_builder->send_bldr(_("Area name changed to '%s'\n"), 
                                                 the_dbase->get_areaname());
      return;     
   }

   if ((the_object = the_builder->get_modify()) == NULL)
   {
      the_builder->send_bldr(_("You have not set an object to modify yet.\n"));
      return;
   }

   if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(the_parsed->get_target1(), _("item"))))
   {
      Shop    *the_shop;
      Mobile  *the_mob;
      int     results;
      Strings new_alias;

      if (the_object->get_type() != OBJ_TYPE_MOBILE)
      {
         the_builder->send_bldr(_("You can only rename shop items on a Mobile.\n"));
         return;
      }  

      the_mob = (Mobile *) the_object;

      if ((the_shop = the_mob->get_shop()) == NULL)
      {
         the_builder->send_bldr(_("This mobile does not have a shop attached.\n"));
         return;
      } 

      tmp_word.assign_word(the_parsed->get_speech(), 1);
      if (tmp_word.str_show() == NULL)
      {
	the_builder->send_bldr(_("You must specify an alias to rename.\n"
                                "format: rename item <Alias> <NewAlias>\n"));
         return;
      }      

      new_alias.assign_word(the_parsed->get_speech(), 2);
      if (new_alias.str_show() == NULL)
      {
         the_builder->send_bldr(_("You must specify a new name for the item.\n"
                                "format: rename item <Alias> <NewAlias>\n"));
         return;
      }
 
      results = the_shop->rename_shop_item(tmp_word.str_show(), 
                                           new_alias.str_show());
      if (results == -1)
      {
         the_builder->send_bldr(_("The alias '%s' does not exist.\n"), 
                                              tmp_word.str_show());
         return;
      }
      if (results == -2)
      {
         the_builder->send_bldr(_("The new alias '%s' already exists.\n"), 
                                             new_alias.str_show());
         return;
      }

      the_builder->send_bldr(_("Alias '%s' for shop item renamed to '%s'.\n"), 
                       tmp_word.str_show(), new_alias.str_show());
      return;
   }
   if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(the_parsed->get_target1(), _("tutorial"))))
   {
      Race    *the_race;
      int     results;
      Strings tmp_word;
      Strings new_name;

      if (the_object->get_type() != OBJ_TYPE_RACE)
      {
         the_builder->send_bldr(_("You can only rename tutorials on a Race.\n"));
         return;
      }  

      the_race = (Race *) the_object;

      tmp_word.assign_word(the_parsed->get_speech(), 1);
      if (tmp_word.str_show() == NULL)
      {
         the_builder->send_bldr(_("You must specify a tutorial by name to rename.\n"
                                "format: rename tutorial <Name> <NewName>\n"));
         return;
      }      

      new_name.assign_word(the_parsed->get_speech(), 2);
      if (new_name.str_show() == NULL)
      {
         the_builder->send_bldr(_("You must specify a new name for the tutorial.\n"
                                "format: rename tutorial <Name> <NewName>\n"));
         return;
      }
 
      results = the_race->set_tutorial_name(tmp_word.str_show(), 
                                           new_name.str_show());
      if (results == -2)
      {
         the_builder->send_bldr(_("The tutorial '%s' does not exist.\n"), 
                                              tmp_word.str_show());
         return;
      }
      if (results == -3)
      {
         the_builder->send_bldr(_("The tutorial '%s' already exists.\n"), 
                                             new_name.str_show());
         return;
      }

      the_builder->send_bldr(_("Name '%s' for tutorial renamed to '%s'.\n"), 
                       tmp_word.str_show(), new_name.str_show());
      return;
   }
   else if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(the_parsed->get_target1(), _("reply"))))
   {
      Strings tmp_word;
      int     the_num;   
      Mobile  *the_mob;

      if (the_object->get_type() != OBJ_TYPE_MOBILE)
      {
         the_builder->send_bldr(_("You can only rename replies on a Mobile.\n"));
         return;
      }  

      the_mob = (Mobile *) the_object;

      tmp_word.assign_word(the_parsed->get_speech(), 1);

      if ((tmp_word.str_show() == NULL) || 
          (!isdigit(*(tmp_word.str_show()))))
      {
        the_builder->send_bldr(_("You need to indicate which one to rename.\n"
                              "format: rename reply <number> <NewKeywords>\n"));
           return;
      }

      the_num = atoi(tmp_word.str_show());

      tmp_word.assign_phrase(the_parsed->get_speech(), 1);

      if (tmp_word.str_show() == NULL)
      {
         the_builder->send_bldr(_("You must provide keywords to rename to.\n"
                                "format: rename reply <number> <NewKeywords>\n"));
         return;
      }

      if (the_mob->rename_comlist(the_num, tmp_word.str_show()) <= 0)
      {
         the_builder->send_bldr(_("Could not find indicated reply.\n"));
         return;
      }

      the_builder->send_bldr(_("Keywords for reply %d renamed to '%s'.\n"), 
                                             the_num, tmp_word.str_show());
      return;
   }
   else if ((the_parsed->get_target1() != NULL) && 
       (!STRCASECMP(the_parsed->get_target1(), _("shop"))))
   {
      Mobile  *the_mob;
      Shop    *the_shop;

      if (the_object->get_type() != OBJ_TYPE_MOBILE)
      {
         the_builder->send_bldr(_("You can only rename Shops on a Mobile.\n"));
         return;
      }  

      the_mob = (Mobile *) the_object;

      if ((the_shop = the_mob->get_shop()) == NULL)
      {
         the_builder->send_bldr(_("This mobile does not have a shop attached.\n"));
         return;
      }

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("You must provide a title to rename to.\n"
                                "format: rename shop <newname>\n"));
         return;
      }
 
      the_shop->set_name(the_parsed->get_speech());

      the_builder->send_bldr(_("Title for shop set to '%s'.\n"), 
                                            the_shop->get_name());
      return;
   }

   if (the_parsed->get_target1() == NULL)
   {
      the_builder->
               send_bldr(_("You need to specify a name for the new object.\n"));
      return;
   }

   if (!STRCASECMP(the_object->get_name(), the_parsed->get_target1()))
   {
      the_builder->
        send_bldr(_("You can't rename using the same name.\n"));
      return;
   }

   new_object = the_object->copy_obj(the_parsed->get_target1());

   if (the_dbase->add_areaobject(new_object) <= 0)
   {
      the_builder->send_bldr(_("The object '%s' already exists!\n"), 
                                                  new_object->get_name()); 
      mainstruct->delete_entity(new_object);
      return;
   }
   the_builder->modify_object(new_object->get_name());

   if (the_dbase->delete_areaobject(the_object) <= 0)
   {
      the_builder->
             send_bldr(_("Error, could not delete old object for rename.\n"));
      return;
   }

   the_builder->send_bldr(_("Object renamed to '%s'.\n"), 
                                                  new_object->get_name());
                              
   the_parsed = NULL;                                  
}


/***********************************************************************
 **  autoassociate - toggles autoassociate on and off
 **
 ***********************************************************************/

void autoassociate(Parse *the_parsed, MudObject *the_user) {
   Builder  *the_builder;
   Flags    *admin_flags;

   if (the_user->get_type() == OBJ_TYPE_BUILDER)
      the_builder = (Builder *) the_user;
   else
      return;

   admin_flags = the_builder->get_adminflags();

   if (admin_flags->get_flag(ADMINFLAG_AUTOASSOCIATE))
   {
      admin_flags->clr_flag(ADMINFLAG_AUTOASSOCIATE);
      the_builder->send_bldr(_("Autoassociate has been turned off.\n"));
   }
   else
   {
      admin_flags->set_flag(ADMINFLAG_AUTOASSOCIATE);
      the_builder->send_bldr(_("Autoassociate has been turned on.\n"));
   }
   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  bldrsocketcom - lists the current sockets open and who is populating
 **                  them along with their hosts, made for the builder
 **                  port
 **
 ***********************************************************************/

void bldrsocketcom(Parse *the_parsed, MudObject *the_user) {
   Player  *the_list;
   Builder *bldr_list;
   int     count = 0;    /* the number of players online */
   Builder *the_builder;
   struct  tm *time_struct;
   char    buff[30];
   time_t  idle_time;
   Connection *the_conn;
   int     is_bldr = 1;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return;
   the_builder = (Builder *) the_user;


   if (the_parsed->get_target1() != NULL)
   {
      if (!STRNCASECMP(_("builder"), the_parsed->get_target1(), 
                                       strlen(the_parsed->get_target1())))
      {
         is_bldr = 1;
      }
      else if (!STRNCASECMP(_("game"), the_parsed->get_target1(), 
                                       strlen(the_parsed->get_target1())))
      {
         is_bldr = 0;
      }
      else
      {
         the_builder->send_bldr(_("The '%s' port does not exist.\n"), 
                                           the_parsed->get_target1());
         return;
      }
   }

   /* display the header */
   the_builder->send_bldr(_("&+WSocket List for %s&*\n\n"), is_bldr ? 
                                            _("Builder Port") : _("Game Port"));

   the_builder->send_bldr(_("Socket      Name            Idle      Host&*\n"));
   the_builder->send_bldr(
      "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=&*\n");


   if (!is_bldr)
   {   
      the_list = mainstruct->get_first_player();
      while (the_list != NULL)
      {
         idle_time = the_list->get_idle();
         time_struct = (struct tm *) gmtime((const time_t *) &idle_time);

         /* prepare idle string */
         BZERO(buff, 30);
         strftime (buff, 10, "%H:%M:%S", time_struct);

         the_conn = the_list->get_connection();
         the_builder->send_bldr("  &+W%-8d  &+M%-13s &+W%10s  &+m%s\n", 
            the_conn->get_socknum(), the_list->get_title(), buff, 
            (the_conn->get_host_addr() == NULL) ? the_conn->get_ip_addr() : 
                                               the_conn->get_host_addr());
            count++;

         the_list = the_list->get_next_player();
      }
   } else {
      bldr_list = mainstruct->get_first_builder();
      while (bldr_list != NULL)
      {
         idle_time = bldr_list->get_idle();
         time_struct = (struct tm *) gmtime((const time_t *) &idle_time);

         /* prepare idle string */
         BZERO(buff, 30);
         strftime (buff, 10, "%H:%M:%S", time_struct);

         the_conn = bldr_list->get_connection();
         the_builder->send_bldr("  &+W%-8d  &+M%-13s &+W%10s  &+m%s\n", 
            the_conn->get_socknum(), bldr_list->get_title(), buff, 
            (the_conn->get_host_addr() == NULL) ? the_conn->get_ip_addr() : 
                                               the_conn->get_host_addr());
            count++;

         bldr_list = bldr_list->get_next_builder();
      }

   }

   the_builder->send_bldr(
       "&+B-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=&*\n");

   the_builder->send_bldr(_("There %s &+W%d&* %s open.\n"),
                           ((count == 1) ? _("is") : _("are")), count,
                           ((count == 1) ? _("socket") : _("sockets")));
   the_parsed = NULL;
}


/***********************************************************************
 **  nospectatecom - toggles the ability of other players to spectate you
 **                  on and off
 **
 ***********************************************************************/

void nospectatecom(Parse *the_parsed, MudObject *the_user) {
   Builder      *the_builder;
   Flags        *admin_flags;

   if (the_user->get_type() == OBJ_TYPE_BUILDER)
      the_builder = (Builder *) the_user;
   else
      return;

   admin_flags = the_builder->get_adminflags();

   if (admin_flags->get_flag(ADMINFLAG_NOSPECTATE))
   {
      admin_flags->clr_flag(ADMINFLAG_NOSPECTATE);
      the_builder->send_bldr(_("NoSpectate has been turned off.\n"));
   }
   else
   {
      admin_flags->set_flag(ADMINFLAG_NOSPECTATE);
      the_builder->send_bldr(_("NoSpectate has been turned on.\n"));
      (the_builder->get_connection())->remove_all_snoop(SNOOP_SPECTATE);

   }

   return;
   the_parsed = NULL;
}


/***********************************************************************
 **  spectatecom - starts spectating somebody
 **
 ***********************************************************************/

void spectatecom(Parse *the_parsed, MudObject *the_user)
{
   Builder      *the_builder;
   Flags        *admin_flags;
   Builder      *tmp_builder;

   if (the_user->get_type() == OBJ_TYPE_BUILDER)
      the_builder = (Builder *) the_user;
   else
      return;

   /* if they don't say who to spectate, remove their spectate */
   if (the_parsed->get_target1() == NULL)
   {
      tmp_builder = mainstruct->get_first_builder();
      while (tmp_builder != NULL)
      {
         if ((tmp_builder->get_connection())->
               has_snoop(the_builder->get_connection()))
         {
            (tmp_builder->get_connection())->
                                 remove_snoop(the_builder->get_connection());
            the_builder->send_bldr(_("Stopped spectating.\n"));
            return;
         }
         tmp_builder = tmp_builder->get_next_builder();
      }
      the_builder->send_bldr(_("You are not spectating anyone.\n"));
      return;
   }

   if ((tmp_builder = mainstruct->get_builder(the_parsed->get_target1()))
                                                                     == NULL)
   {
      the_builder->send_bldr(_("I don't know of the builder '%s'\n"), 
                                       the_parsed->get_target1());
      return;
   }

   admin_flags = tmp_builder->get_adminflags();
   if (admin_flags->get_flag(ADMINFLAG_NOSPECTATE))
   {
      the_builder->send_bldr(_("%s does not allow spectators.\n"), 
                                                      tmp_builder->get_title());
      return;
   }   

   if ((tmp_builder->get_connection())->add_snoop(the_builder->get_connection())
                                                                 == -3)
   {
      the_builder->send_bldr(_("You can't spectate on yourself!\n"));
      return;
   }

   the_builder->send_bldr(_("Started spectating %s.\n"), tmp_builder->get_title());
   tmp_builder->send_bldr(_("%s has started spectating on you.\n"),
                                            the_builder->get_title());
                   
   return;

}

#endif

