/**********************************************************************
 ** Bulletin class: Maintains a bulletin board that can either be accessed
 **                 from anywhere or attached to a specific object
 **
 **
 **
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef BULLETIN_C
#define BULLETIN_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "bulletin.h"
#include "pager.h"
#include "global.h"
#include "utils.h"
#include "lexer.h"
#include "gameflags.h"
#include "newfuncts.h"
#include "memchk.h"
#include "objtype.h"
#include "bullflags.h"
#include "adminflags.h"
#include "flags.h"

/***********************************************************************
 ** Bulletin (constructor) - Creates the bulletin and fills it with info
 **
 ** Parameters: bull_name - the name to give the bulletin board
 **             build_port - are we creating this for the builder port?
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
Bulletin::Bulletin(char *bull_name, int build_port)
{
   Strings the_name;

   for_builder = build_port;

   the_name.assign_word(bull_name, 1);

   if (the_name.str_len() > MAXNAMELEN)
      the_name.truncate(MAXNAMELEN);
   set_name(the_name.str_show());

   obj_type = OBJ_TYPE_BULLETIN;

   the_name.lowercase();

   bullfilename.sprintf("%s/%s%s%s", the_config.basedir.str_show(),
                       BULLETINDIR,  the_name.str_show(), BULLEXTENTION); 

   loaded_bull = NULL;
   build_ptr = NULL;

   {
      char *tmp_char;
      tmp_char = new char[10];
   }

   bull_flags = new_Flags(1);
}


/***********************************************************************
 ** ~Bulletin (destructor) - closes the bulletin
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Bulletin::~Bulletin()
{ 
   delete_bull();
   delete_Flags(bull_flags);
}


/***********************************************************************
 ** set_location - sets the location string that this bulletin board will
 **                be attached to (any object, not just locations)
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void Bulletin::set_location(char *the_str)
{
   location = the_str;
}


/***********************************************************************
 ** get_location - gets the location string that this bulletin board will
 **                be attached to (any object, not just locations)
 **
 ***********************************************************************/

char *Bulletin::get_location()
{
   return location.str_show();
}

/***********************************************************************
 ** get_bullflags - gets the bulletin board flags object
 **
 ***********************************************************************/

Flags *Bulletin::get_bullflags()
{
   return bull_flags;
}


/***********************************************************************
 ** set_title - sets the title string for this bulletin board
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

int Bulletin::set_title(char *the_str)
{
   title = the_str;
   return 1;
}


/***********************************************************************
 ** get_title - sets the title string for this bulletin board
 **
 ***********************************************************************/

char *Bulletin::get_title()
{
   return title.str_show();
}

/***********************************************************************
 ** set_authorized - sets the list of authorized users for this board
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void Bulletin::set_authorized(char *the_str)
{
   authorized = the_str;
}


/***********************************************************************
 ** get_authorized - sets the list of authorized users for this board
 **
 ***********************************************************************/

char *Bulletin::get_authorized()
{
   return authorized.str_show();
}

/***********************************************************************
 ** player_authorized - is the player passed in authorized to use this board
 **
 ** Parameters: the_player - the player to look for
 **
 ** Returns: 1 for authorized, -1 for not
 **
 ***********************************************************************/

int Bulletin::player_authorized(Player *the_player)
{
   int counter = 1;
   Strings the_name;
   
   the_name.assign_word(authorized.str_show(), counter);
   while (the_name.str_show() != NULL)
   {
      if (!STRCASECMP(the_player->get_name(), the_name.str_show()))
         return 1;
      counter++;
      the_name.assign_word(authorized.str_show(), counter);
   }

   return 0;
}

/***********************************************************************
 ** load_bulletin - loads in the bulletin along with all entries into memory
 **
 ** Parameters: None
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Bulletin::load_bulletin(ErrLog *error_log)
{
   Strings      holder;
   token_record *the_token;
   BullEntry    *new_bull;
   BullEntry    *prev_bull = NULL;

   /* if the bulletin does not have a data file, return error */
   if ((the_bullfile = xfopen(bullfilename.str_show(), "r", 
                                                   "bull_file")) == NULL)
   {

      return -1;
   }

   /* read in the header information */

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_bullfile, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "bulletins", "title", 
				    "load_bulletin");
     return -1;
   }

   /* get the title of this bulletin */
   the_token = get_token(the_bullfile, '^');
   set_title(the_token->the_string);   

   /* get the location of this bulletin */
   the_token = get_token(the_bullfile, '\0');
   location = the_token->the_string;   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_bullfile, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_attr_format(get_name(), "bulletins", "authorized", 
				    "load_bulletin");
     return -1;
   }

   /* get the authorized individuals for this bulletin */
   the_token = get_token(the_bullfile, '^');
   set_authorized(the_token->the_string);

   if (bull_flags->read_flags(the_bullfile, error_log) == -1)
   {
     error_log->invalid_attr_format(get_name(), "bulletins", "bullflags", 
				    "load_bulletin");
     return -1;
   }

   /* if this is the builder port, we don't read in the entries but we
      grab the entries loaded on the game port */
   if (for_builder)
   {
      Object_List *tmp_dbase;
      Area_Dbase  *bull_area;
      Bulletin    *tmp_bull;
      Entity      *tmp_obj;
      
      tmp_dbase = mainstruct->get_dbase();
 
      if ((bull_area = tmp_dbase->get_area(_("bulletins"))) == NULL)
      {
         holder.sprintf(_("Could not access gameport bulletin board '%s'."),
                                                               get_name());
         error_log->log_err(holder.str_show(), "load_bulletin");
         return -1;
      }

      /* if it doesn't exist on the game port it must be a new one that
         hasn't been loaded on the game port.  Make them do it */
      if ((tmp_obj = bull_area->get_areaobject(get_name())) == NULL)
      {
         holder.sprintf(_("Bulletin board %s must be loaded on the game "
                        "port first to load to the builder port."), get_name());
         error_log->log_err(holder.str_show(), "load_bulletin");
         return -1;
      }

      if (tmp_obj->get_type() != OBJ_TYPE_BULLETIN)
      {
         holder.sprintf(_("Bulletin board '%s' on game port not actual "
                        "bulletin object"), get_name());
         error_log->log_err(holder.str_show(), "load_bulletin");
         return -1;
      }

      tmp_bull = (Bulletin *) tmp_obj;
      build_ptr = tmp_bull->get_loaded_bull();
      return 1;      
   }


   /* read in each bulletin entry */
   the_token = get_token(the_bullfile, '\0');
   while ((the_token->token_type > 0) && 
          (the_token->token_type != T_POUND))
   {
      new_bull = new_BullEntry(this);
      if (new_bull->read_bull(the_bullfile, error_log) == -1)
      {
         delete_BullEntry(new_bull);
         the_token = get_token(the_bullfile, '\0');
         continue;
      }

      if (prev_bull == NULL)
      {
         loaded_bull = new_bull;
      }
      else
      {
         prev_bull->set_next_bull(new_bull);
      }
      prev_bull = new_bull;
      the_token = get_token(the_bullfile, '\0');
   }

   xfclose(the_bullfile, "bull_file");
   return 1;
}

/***********************************************************************
 ** delete_bull - deletes all the bulletin entries in the linked list
 **
 ** Parameters: None
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Bulletin::delete_bull()
{
   BullEntry *tmp_bull;

   tmp_bull = loaded_bull;
   while (loaded_bull != NULL)
   {
      tmp_bull = loaded_bull;
      loaded_bull = loaded_bull->get_next_bull();
      delete_BullEntry(tmp_bull);
   }
   return 1;
}


/***********************************************************************
 ** write_object - writes the bulletin and all entries to disk
 **
 ** Parameters: None
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

void Bulletin::write_object(FILE *the_file, int build_format)
{
   the_file = NULL;
   build_format = 0;

   write_bull();
}


/***********************************************************************
 ** write_bull - writes the bulletin and all entries to disk
 **
 ** Parameters: None
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Bulletin::write_bull()
{
   Strings      holder;
   BullEntry    *tmp_bull;

   /* open the bulletin file for writing */
   if ((the_bullfile = xfopen(bullfilename.str_show(), "w", 
                                                     "bull_file")) == NULL)
   {
      holder.sprintf(_("Could not open bulletin '%s' data file."), get_name());
      mainstruct->log_error(holder.str_show(), "write_bull");
      return 0;
   }

   /* write the header information */
   /* no header information yet */
   fprintf(the_bullfile, "^%s^\n", (get_title() == NULL) ? _("none") : 
                                                            get_title());
   fprintf(the_bullfile, "%s\n", (get_location() == NULL) ? _("none") : 
                                                            get_location());
   fprintf(the_bullfile, "^%s^\n", (get_authorized() == NULL) ? "" : 
                                                            get_authorized());
   bull_flags->write_flag(the_bullfile);

   /* write the bulletin messages */

   if (for_builder)
   {
      if (build_ptr != NULL)
         tmp_bull = *build_ptr;
      else
         tmp_bull = NULL;
   }
   else
      tmp_bull = loaded_bull;

   while (tmp_bull != NULL)
   {
      tmp_bull->write_bull(the_bullfile);
      tmp_bull = tmp_bull->get_next_bull();
   }
   fprintf(the_bullfile, "#\n");
   
   xfclose(the_bullfile, "bull_file");
   return 1;
}


/***********************************************************************
 ** display_help - displays help for the bulletin to the player
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

void Bulletin::display_help(Player *the_player)
{
   the_player->send_plr(_("\n&+WCommands for Bulletin:&*\n"));
   the_player->send_plr("&+B---------------------------&*\n");
   the_player->send_plr(_("&+Gquit &+W- &*quit from the bulletin\n"));
   the_player->send_plr(_("&+Ghelp &+W- &*this help display\n"));
   the_player->send_plr(_("&+Glist &+W- &*list all bulletin entries\n"));
   the_player->send_plr(_("&+Gread &+W- &*read a bulletin message\n"));
   the_player->send_plr(_("&+Gpost &+W- &*post a bulletin message\n"));
   the_player->send_plr(_("&+Gdelete &+W- &*delete a bulletin message\n"));

   the_player->send_plr("\n");
}


/***********************************************************************
 ** add_entry - adds a bulletin entry to the board
 **
 ** Parameters: the_entry - the entry to add to the bulletin
 **
 ***********************************************************************/

int Bulletin::add_entry(BullEntry *the_entry)
{
   BullEntry *tmp_bull;
   int    counter = 0;

   if (the_entry == NULL)
      return 0;

   tmp_bull = loaded_bull;

   if (tmp_bull == NULL)
   {
      loaded_bull = the_entry;
      return 1;
   }

   while (tmp_bull->get_next_bull() != NULL)
   {
      counter++;
      tmp_bull = tmp_bull->get_next_bull();
   }
   
   tmp_bull->set_next_bull(the_entry);

   return counter;
}


/***********************************************************************
 ** delete_entry - deletes a bulletin entry from the board
 **
 ** Parameters: the_num - the number of the entry to delete
 **
 ** Returns: 1 for deleted, 0 for not found, -1 for failure
 **
 ***********************************************************************/

int Bulletin::delete_entry(int the_num)
{
   BullEntry *tmp_bull;
   BullEntry *prev_bull = NULL;
   int    counter = 1;

   if (the_num <= 0)
      return -1;

   tmp_bull = loaded_bull;
   while ((tmp_bull != NULL) && (counter < the_num))
   {
      prev_bull = tmp_bull;
      tmp_bull = tmp_bull->get_next_bull();
      counter++;
   }

   if (tmp_bull == NULL)
      return 0;

   if (prev_bull == NULL)
   {
      loaded_bull = loaded_bull->get_next_bull();
   }
   else
   {
      prev_bull->set_next_bull(tmp_bull->get_next_bull());
   }

   delete_BullEntry(tmp_bull);

   return 1;
}


/***********************************************************************
 ** list_entries - lists the current entries in the bulletin board
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

int Bulletin::list_entries(Player *the_player)
{
   BullEntry *the_bull;
   int    counter = 1;
   Strings holder;
   char *the_year;
   Strings the_date;

   the_player->send_plr(
      _("&+BCurrent Entries for '&+M%s&+B' Bulletin Board:&*\n\n"), get_name());
   the_player->send_plr(
      _("&+W No.   Author          Date Posted               Subject&*\n"));
   the_player->send_plr(
       "&+B-----------------------------------------------------------&*\n");

   the_bull = loaded_bull;
   while (the_bull != NULL)
   {
      the_year = the_bull->get_date();
      the_year += 20;
      the_date = the_bull->get_date();
      the_date.truncate(12);
      the_date.str_cat(the_year);
      the_player->send_plr("&+g[&*%3.3d&+g]  %-14s %s  %s&*\n", counter, 
                             the_bull->get_author(), 
                             the_date.str_show(), the_bull->get_subject());
      counter++;
      the_bull = the_bull->get_next_bull();

   }
   the_player->send_plr("\n");
   return counter;
}


/***********************************************************************
 ** get_entry - gets a bulletin board entry based on the number submitted
 **
 ** Parameters: the_num - the number of the entry to get
 **
 ***********************************************************************/

BullEntry *Bulletin::get_entry(int the_num)
{
   BullEntry *tmp_bull;
   int    counter = 1;

   if (the_num <= 0)
      return NULL;

   tmp_bull = loaded_bull;
   while ((tmp_bull != NULL) && (counter < the_num))
   {
      tmp_bull = tmp_bull->get_next_bull();
      counter++;
   }

   return tmp_bull;
}


/***********************************************************************
 ** describe - describes the bulletin board to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Bulletin::describe(Builder *the_builder)
{
   the_builder->send_bldr(_("\n&+GBulletin: \t&+M%s&*\n"), get_name());
   the_builder->send_bldr(_("&+GTitle:&* \t\t%s\n"), get_title());
   if (!bull_flags->get_flag(BULLFLAG_GLOBAL))
   {
      the_builder->send_bldr(_("&+GLocation: \t&+m%s&*\n"), get_location());
   }
   if (bull_flags->get_flag(BULLFLAG_LIMITED))
   {
      the_builder->send_bldr(_("&+GAuthorized: \t&+m%s&*\n"), get_authorized());
   }

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the bulletin board to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Bulletin::describe(Player *the_player)
{
   the_player->send_plr(_("\n&+GBulletin: \t&+M%s&*\n"), get_name());
   the_player->send_plr(_("&+GTitle:&* \t%s\n"), get_title());
   if (!bull_flags->get_flag(BULLFLAG_GLOBAL))
   {
      the_player->send_plr(_("&+GLocation: \t&+m%s&*\n"), get_location());
   }
   if (bull_flags->get_flag(BULLFLAG_LIMITED))
   {
      the_player->send_plr(_("&+GAuthorized: \t&+m%s&*\n"), get_authorized());
   }
   the_player->send_plr("\n");
}

/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Bulletin::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr(_("You can set the following attributes on a bulletin.\n"
               "   location, title, authorized and bullflags\n"));
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("location"), 
                               strlen(the_parsed->get_target1())))
   {
      set_location(the_parsed->get_speech());
      the_builder->send_bldr(_("Location on bulletin %s set to: %s\n"), 
                                           get_name(), get_location());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("title"), 
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("You need to specify a title to set to.\n"));
         return -1;
      }
      set_title(the_parsed->get_speech());
      the_builder->send_bldr(_("Title on bulletin %s set to: %s\n"), 
                                          get_name(), title.str_show());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("authorized"), 
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
          send_bldr(_("You need to specify authorized users to set to.\n"));
         return -1;
      }
      set_authorized(the_parsed->get_speech());
      the_builder->send_bldr(_("Authorized users on bulletin %s set to: %s\n"), 
                                          get_name(), get_authorized());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("bullflags"),
                               strlen(the_parsed->get_target1())))
   {
      int flagnum;
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("Set which bulletin flag?\n"));
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
     
      if ((flagnum = 
          bull_flags->get_by_name(holder.str_show(), bullflagnames)) == -1)
      {
         the_builder->send_bldr(_("That is not a bulletin flag.\n"));
         return -1;
      }

      holder.assign_word(the_parsed->get_speech(), 2);
      if (holder.str_show() == NULL)
      {
         the_builder->send_bldr(_("Set that flag to what?\n"
                                "Valid choices are: On or Off\n"));
         return -1;
      }

      if (holder.str_n_cmp(_("On"), holder.str_len()))
         bull_flags->set_flag(flagnum);
      else if (holder.str_n_cmp(_("Off"), holder.str_len()))
         bull_flags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr(_("That is not a valid setting.\n"
                                "Valid choices are: On or Off\n"));
         return -1;
      }
      the_builder->send_bldr(_("Flag &+M%s&* has been set to: %s\n"),
      bullflagnames[flagnum], (bull_flags->get_flag(flagnum)) ? 
			                       _("&+GOn&*") : _("&+ROff&*"));
      return 1;
   }

   the_builder->send_bldr(_("The attribute '%s' is not a bulletin attribute.\n"),
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** copy_object - copies the bulletin to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Bulletin::copy_object(Entity *copy_obj)
{
   Bulletin *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_BULLETIN)
      return 0;

   copy_from = (Bulletin *) copy_obj;

   /****** set up bulletin attributes ******/
   set_title(copy_from->get_title());
   set_location(copy_from->get_location());
   set_authorized(copy_from->get_authorized());

   return 1;
}

/***********************************************************************
 ** is_build_port - returns 1 if loaded for the builder port, 0 otherwise
 **
 ***********************************************************************/

int Bulletin::is_build_port()
{
   return for_builder;
}


/***********************************************************************
 ** is_build_port - returns 1 if loaded for the builder port, 0 otherwise
 **
 ***********************************************************************/

BullEntry **Bulletin::get_loaded_bull()
{
   return &loaded_bull;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this bulletin is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Bulletin::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Bulletin::get_mem_size_dynamic()
{
   int  size = 0;

   size += location.get_mem_size_dynamic();
   size += title.get_mem_size_dynamic();
   size += authorized.get_mem_size_dynamic();

   size += get_mem_size_entity();

   return size;
}


/***********************************************************************
 **               BullEntry methods
 ***********************************************************************/


/***********************************************************************
 ** BullEntry (constructor) - Creates the bulletin entry
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

BullEntry::BullEntry(Bulletin *assign_board)
{
   the_board = assign_board;
   next_bull = NULL;
}


/***********************************************************************
 ** ~BullEntry (destructor) - deletes the bulletin entry
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

BullEntry::~BullEntry()
{ 
}


/***********************************************************************
 ** set_author - sets a name in the author field
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void BullEntry::set_author(char *the_str)
{
   author = the_str;
}


/***********************************************************************
 ** get_date - gets the date string
 **
 ***********************************************************************/

char *BullEntry::get_date()
{
   return date.str_show();
}


/***********************************************************************
 ** get_author - gets the author string
 **
 ***********************************************************************/

char *BullEntry::get_author()
{
   return author.str_show();
}


/***********************************************************************
 ** get_subject - gets the subject string
 **
 ***********************************************************************/

char *BullEntry::get_subject()
{
   return subject.str_show();
}


/***********************************************************************
 ** get_body - gets the body string
 **
 ***********************************************************************/

char *BullEntry::get_body()
{
   return body.str_show();
}

/***********************************************************************
 ** set_subject - sets a subject for this bulletin entry
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void BullEntry::set_subject(char *the_str)
{
   subject = the_str;
}


/***********************************************************************
 ** set_body - sets the body for this bulletin entry
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void BullEntry::set_body(char *the_str)
{
   body = the_str;
}


/***********************************************************************
 ** edit_body - edits the body of the bulletin entry
 **
 ** Parameters: the_player - the player to start editing with
 **
 ***********************************************************************/

int BullEntry::edit_body(Player *the_player)
{
   Editor      *input_editor;
   Flags       *tmp_gameflags;
   Inp_Handler *tmp_handler;
   int         results;

   tmp_handler = the_player->get_input_handler();

   input_editor = new Editor();

   tmp_gameflags = the_player->get_gameflags();

   if (!tmp_gameflags->get_flag(GAMEFLAG_FULLEDIT))
      input_editor->set_simple_mode();


   tmp_handler->add_pop_function(post_on_pop, the_player->get_name(), 
                                                       NULL, NULL, NULL); 

   the_player->send_plr(_("&+gMessage body:&*\n"));
   results = input_editor->start_editor(the_player, &body);

   tmp_handler = the_player->get_input_handler();

   tmp_handler->add_pop_function(pop_on_pop, the_player->get_name(), 
                                                       NULL, NULL, NULL);

   return results;
}


/***********************************************************************
 ** get_next_bull - gets the next bulletin entry attached to this one
 **
 **
 ***********************************************************************/

BullEntry *BullEntry::get_next_bull()
{
   return next_bull;
}


/***********************************************************************
 ** set_next_bull - sets the next bulletin entry attached to this one
 **
 ** Parameters: new_ltr - the letter to add as next
 **
 ***********************************************************************/

void BullEntry::set_next_bull(BullEntry *new_bull)
{
   next_bull = new_bull;
}


/***********************************************************************
 ** copy_ltr - copies the letter to another letter
 **
 ** Parameters: new_ltr - the letter to add as next
 **
 ***********************************************************************/

void BullEntry::copy_bull(BullEntry *copy_from)
{
   date = copy_from->get_date();
   author = copy_from->get_author();
   subject = copy_from->get_subject();
   body = copy_from->get_body();
}


/***********************************************************************
 ** write_bull - writes the bulletin entry to a file
 **
 ** Parameters: the_file - the file to write the entry to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int BullEntry::write_bull(FILE *the_file)
{
   fprintf(the_file, "+\n");
   fprintf(the_file, "^%s^\n", date.str_show());
   fprintf(the_file, "^%s^\n", author.str_show());
   fprintf(the_file, "^%s^\n", subject.str_show());
   fprintf(the_file, "^%s^\n\n", body.str_show());
   return 1;
}


/***********************************************************************
 ** read_bull - reads the bulletin entry from a file
 **
 ** Parameters: the_file - the file to read the letter from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int BullEntry::read_bull(FILE *the_file, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;
   char         *temp_body;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err(_("Invalid format in bulletin file"), "read_bull");
      return -1;
   }

   /* get the date of this bulletin entry */
   the_token = get_token(the_file, '^');
   date = the_token->the_string;   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err(_("Invalid format in bulletin file"), "read_bull");
      return -1;
   }

   /* get the author of this letter */
   the_token = get_token(the_file, '^');
   author = the_token->the_string;   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err(_("Invalid format in bulletin file"), "read_bull");
      return -1;
   }

   /* get the subject of this letter */
   the_token = get_token(the_file, '^');
   subject = the_token->the_string;   


   temp_body = read_desc_type(the_file, error_log, NULL);
   
   if (temp_body == NULL)
      return -1;
   body = temp_body;
   delete temp_body;


   return 1;
}


/***********************************************************************
 ** display_bull - displays the bulletin entry to the screen
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

void BullEntry::display_bull(Player *the_player)
{
   the_player->send_plr(_("\n&+gAuthor:&* %s\n"), author.str_show());
   the_player->send_plr(_("&+gDate:&* %s\n"), date.str_show());
   the_player->send_plr(_("&+gSubject:&* %s\n"), subject.str_show());
   the_player->send_plr(_("&+gBody:&* \n%s\n"), body.str_show());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_board - sets the bulletin board this entry is a part of to this
 **             pointer
 **
 ** Parameters: new_board - the new bulletin board to set this entry to
 **
 ***********************************************************************/

void BullEntry::set_board(Bulletin *new_board)
{
   the_board = new_board;
}

/***********************************************************************
 ** get_board - gets the bulletin board pointer that this entry exists in
 **
 ***********************************************************************/

Bulletin *BullEntry::get_board()
{
   return the_board;
}

/***********************************************************************
 ** set_date - sets the date to the current time
 **
 ***********************************************************************/

void BullEntry::set_date()
{
   date = get_time_str();
}



/**************************************************************************
 **    Bulletin input handlers
 **************************************************************************/

/**************************************************************************
 ** bulletin_command_handler - Main input handler for the bulletin board
 **
 ** This input handler is the one the bulletin board should start in.. 
 ** It handles the input of the bulletin when this is in command mode. 
 **
 ** Parameters - the_obj : the player using the bulletir
 **              the_input : the command
 **
 ** returns 1 on success, -1 on error
 ** 
 **************************************************************************/

int bulletin_command_handler(MudObject *the_obj, char *the_input)
{
   Player      *the_player;
   Inp_Handler *tmp_handler;
   Strings     the_inpstr;
   Strings     the_cmd;
   Bulletin    *the_board;
 
   the_player = (Player *) the_obj;

   tmp_handler = the_player->get_input_handler();
           
   the_board = (Bulletin *)tmp_handler->get_data();

   /* get the input that was passed in and remove the newline */
   the_inpstr = the_input;
   the_inpstr.remove_newline();


   /* get the first word of the input, it is all we care about */
   the_cmd.assign_word(the_inpstr.str_show(),1);

   if (!STRNCASECMP(the_cmd.str_show(), _("quit"), the_cmd.str_len()))
   {
      the_player->send_plr(_("&+B[&+WQuitting Bulletin Board&+b]&*\n"));
      the_board->write_bull();
      tmp_handler->pop_input_handler();
   }
   else if ((!STRNCASECMP(the_cmd.str_show(), _("help"), the_cmd.str_len())) ||
            ((*(the_cmd.str_show())) == '?'))
   {
      the_board->display_help(the_player);
   }
   else if ((!STRNCASECMP(the_cmd.str_show(), _("list"), the_cmd.str_len())) ||
            ((*(the_cmd.str_show())) == '?'))
   {
      the_board->list_entries(the_player);
   }
   else if ((!STRNCASECMP(the_cmd.str_show(), _("post"), the_cmd.str_len())) ||
            ((*(the_cmd.str_show())) == '?'))
   {
      BullEntry *new_bull;
      Strings tmp_to;
      Strings *prompt;

      new_bull = new_BullEntry((Bulletin *)tmp_handler->get_data());
      new_bull->set_author(the_player->get_title());
      new_bull->set_date();
      the_player->send_plr(_("&+gAuthor: &*%s\n"), new_bull->get_author());

      prompt = new_Strings(_("&+gSubject:&* "));

      /* get the subject from the user */
      tmp_handler->push_input_handler(bulletin_get_subject, 
                                  prompt, 0, HANDLER_DATA_BULLENTRY, 0);
      tmp_handler->set_data(new_bull);
   }

   else if (!STRNCASECMP(the_cmd.str_show(), _("read"), the_cmd.str_len()))
   {
      BullEntry *the_bull;
      Strings the_num;

      the_num.assign_word(the_inpstr.str_show(), 2);

      if (the_num.str_show() == NULL)
      {
         the_player->send_plr(
                         _("You need to specify a message number to read.\n"));
         return 0;
      }

      if (!isdigit(*(the_num.str_show())))
      {
         the_player->send_plr(_("Format: read <messagenumber>.\n"));
         return 0;
      }

      /* need a read function here */
      the_bull = the_board->get_entry(atoi(the_num.str_show()));

      if (the_bull == NULL)
      {
         the_player->send_plr(_("That entry number does not exist.\n"));
         return 0;
      }

      the_bull->display_bull(the_player);
   }

   else if (!STRNCASECMP(the_cmd.str_show(), _("delete"), the_cmd.str_len()))
   {
      Strings   the_num;
      BullEntry *the_entry;
      Flags     *tmp_adminflags;

      the_num.assign_word(the_inpstr.str_show(), 2);

      if (the_num.str_show() == NULL)
      {
         the_player->send_plr(
                    _("Format: delete <entrynumber>.\n"));
         return 0;
      }

      if (!isdigit(*(the_num.str_show())))
      {
         the_player->send_plr(
                    _("Format: delete <entrynumber>.\n"));
         return 0;

      }


      tmp_adminflags = the_player->get_admflags();

      /* need the get_bull thing here too */
      if ((the_entry = the_board->get_entry(atoi(the_num.str_show()))) == NULL)
      {
         the_player->send_plr(_("That entry number does not exist.\n"));
         return 0;
      }

      if ((STRCASECMP(the_entry->get_author(), the_player->get_name())) &&
          (!tmp_adminflags->get_flag(ADMINFLAG_BOARDADMIN)))
      {
         the_player->send_plr(_("You are not the author of that post "
                              "and therefore can't delete it.\n"));
         return 0;
      }

      the_board->delete_entry(atoi(the_num.str_show()));

      the_player->send_plr(_("Entry number &+W%s&* deleted.\n"),
                                                        the_num.str_show());
   }

   else
   {
      the_player->send_plr(_("Unknown command.\n"));
      return -1;
   }
   return 1;
}


/**************************************************************************
 ** bulletin_get_subject - the input handler to get the subject from the user
 **
 **
 ** Parameters - the_obj : the player using the bulletir
 **              the_input : the command
 **
 ** returns 1 on success, -1 on error
 ** 
 **************************************************************************/

int bulletin_get_subject(MudObject *the_obj, char *the_input)
{
   Player      *the_player;
   Inp_Handler *tmp_handler;
   Strings     the_inpstr;
   Strings     the_cmd;
   BullEntry   *the_bull;
 
   the_player = (Player *) the_obj;

   tmp_handler = the_player->get_input_handler();
           
   the_bull = (BullEntry *)tmp_handler->get_data();

   /* get the input that was passed in and remove the newline */
   the_inpstr = the_input;
   the_inpstr.remove_newline();
  
   the_bull->set_subject(the_inpstr.str_show());

   return the_bull->edit_body(the_player);
}


/***********************************************************************
 ** pop_on_pop - when the body has been filled and the editor input
 **              handler is popped, pop the lower input handler too
 **
 ** Parameters: notused1 - should be set to null
 **             notused2 - should be set to null
 **             notused3 - should be set to null
 **             notused4 - should be set to null
 **             
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int pop_on_pop(Inp_Handler *the_handler, char *notused1, char *notused2, 
                                char *notused3, Strings *notused4)
{
   the_handler->pop_input_handler();

   /* get rid of compile errors */
   notused1 = notused2 = notused3 = NULL;
   notused4 = NULL;
 
   return 1;
}

/***********************************************************************
 ** post_on_pop - when the entry has been finished, post it on the board
 **
 ** Parameters: notused1 - should be set to null
 **             notused2 - should be set to null
 **             notused3 - should be set to null
 **             notused4 - should be set to null
 **             
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int post_on_pop(Inp_Handler *the_handler, char *notused1, char *notused2, 
                char *notused3, Strings *notused4)
{
   Bulletin    *the_board;
   BullEntry   *the_entry;

   the_entry = (BullEntry *) the_handler->get_pop_data();

   the_board = the_entry->get_board();
   
   the_board->add_entry(the_entry);

   /* get rid of compile errors */
   notused1 = notused2 = notused3 = NULL;
   notused4 = NULL;

   return 1;
}


#endif








