

/**********************************************************************
 ** Door class: A door is an object that can act as an exit between
 **             rooms. When the door is closed the exit is denied, when
 **             it is open players can pass between rooms
 **
 **
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef DOOR_C
#define DOOR_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "objtype.h"
#include "door.h"
#include "utils.h"
#include "flags.h"
#include "item.h"
#include "memchk.h"
#include "global.h"
#include "newfuncts.h"

/***********************************************************************
 ** Door (constructor) - creates the door
 **
 ** Parameters: the_name - the name of the door
 **             the_area - the area the door belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Door::Door(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_DOOR;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   doorstate = 0;
   distance = 1;
}

/***********************************************************************
 ** ~Door (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Door::~Door()
{
}

/***********************************************************************
 ** set_outside_loc - sets the outside location for this door
 **
 ** Parameters: thelocation - the location we are setting it to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_outside_loc(char *thelocation)
{
   if (thelocation == NULL)
      return -1;

   outside_loc = thelocation;
   return 1;
}

/***********************************************************************
 ** get_outside_loc - Gets the outside location name.
 **
 ** Parameters: none.
 **
 ** Returns:  NULL if there is no outside location available, or if
 **           something went wrong.
 **           a string with the location name when successful
 **
 ***********************************************************************/

char *Door::get_outside_loc(void)
{ 
   return outside_loc.str_show();
}


/***********************************************************************
 ** set_rope_name - sets the rope name that is tied to this
 **
 ** Parameters: the_name - the name we are setting it to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_rope_name(char *the_name)
{
   if (the_name == NULL)
      return -1;
 
   rope_name = the_name;
   return 1;   
}


/***********************************************************************
 ** get_rope_name - gets the rope name that is tied to this
 **
 ** Parameters: None
 **
 ** Returns:  pointer to the name
 **
 ***********************************************************************/

char *Door::get_rope_name(void)
{
   return rope_name.str_show();   
}


/***********************************************************************
 ** set_outside_desc - sets the outside desc for this door
 **
 ** Parameters: thedesc - the desc we are setting it to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_outside_desc(char *thedesc)
{
   if (thedesc == NULL)
      return -1;
 
   outside_desc = thedesc;
   return 1;   
}


/***********************************************************************
 ** get_outside_desc - gets the outside desc for this door
 **
 ** Parameters: None
 **
 ** Returns:  pointer to the desc
 **
 ***********************************************************************/

char *Door::get_outside_desc(void)
{
   return outside_desc.str_show();   
}


/***********************************************************************
 ** get_inside_desc - gets the inside desc for this door
 **
 ** Parameters: None
 **
 ** Returns:  pointer to the desc
 **
 ***********************************************************************/

char *Door::get_inside_desc(void)
{
   return get_desc();   
}


/***********************************************************************
 ** get_door_desc - gets the description of the door based on your location
 **
 ** Parameters: locationname - the name of the location you are in.
 **
 ** Returns:  NULL if there is no description available, or if
 **           something went wrong.
 **           a string with the description when successful
 **
 ***********************************************************************/

char *Door::get_door_desc(Location *the_loc)
{ 
  char *tmpstr = NULL;
  Strings holder;
  Strings holder2;
  int i;

  if (the_loc == NULL)
     return NULL;

  if (doorstate < 0 || doorstate > DOOR_MAXSTATE) 
     return NULL;

  holder = outside_loc.str_show();
  holder2.sprintf("%s@%s", the_loc->get_name(), the_loc->get_area());

  for (i=0; i<2; i++)
  {
     if (!holder.num_char('@'))
     {
        make_full_area(&holder, get_area());
     }

     if (holder.str_cmp(holder2.str_show()))
     {
        if (i==0)
           tmpstr = get_outside_desc();
        else
           tmpstr = get_inside_desc();
        return tmpstr;
     }
     if (i==0)
        holder = get_location();
  }
  return NULL;
}


/***********************************************************************
 ** set_door_state - sets the state of the door
 **
 ** Parameters: newstats - the state to set the door to
 **                        DOOR_OPEN for open
 **                        DOOR_CLOSED for closed
 **                        DOOR_LOCKED for locked
 **                        DOOR_MLOCKED for magically locked
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_door_state(int newstate)
{
   if (newstate < 0 || newstate > DOOR_MAXSTATE)
      return -1;
   doorstate = newstate;
   return 1;
}

/***********************************************************************
 ** get_door_state - gets the state of the door
 **
 ** Parameters: none
 **
 ** Returns: the state of the door, you can use the macros
 **          DOOR_OPEN, DOOR_CLOSED etc to see what state it is in.
 ***********************************************************************/

int Door::get_door_state(void)
{
   return doorstate;
}



/***********************************************************************
 ** set_distance - sets the distance between the two door sides, used
 **                only for rope doors (doors used to open connections
 **                when you tie a rope to it)
 **
 ** Parameters: newval - the new distance value
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_distance(int newval)
{
   if (newval <= 0)
      return -1;
   distance = newval;
   return 1;
}

/***********************************************************************
 ** get_distance - gets the distance for this door
 **
 ** Parameters: none
 **
 ** Returns: the distance
 ***********************************************************************/

int Door::get_distance(void)
{
   return distance;
}


/***********************************************************************
 ** set_inside_brief - sets the inside brief descriptions to a certain
 **                    string
 **
 ** Parameters: descnr - the description state
 **             thebriefdesc - the string for the brief description
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_inside_brief(int descnr, char *thebriefdesc)
{
  if ((descnr < 0) || (descnr > DOOR_MAXSTATE) || (thebriefdesc == NULL))
      return -1;
  
  inside_brief[descnr] = thebriefdesc;
  return 1;
}


/***********************************************************************
 ** set_outside_brief - sets the outside brief descriptions to a certain
 **                    string
 **
 ** Parameters: descnr - the description state
 **             thebriefdesc - the string for the brief description
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_outside_brief(int descnr, char *thebriefdesc)
{
  if ((descnr < 0) || (descnr > DOOR_MAXSTATE) || (thebriefdesc == NULL))
      return -1;
  
  outside_brief[descnr] = thebriefdesc;
  return 1;
}

/***********************************************************************
 ** get_brief - Gets the brief description which belongs to the state
 **             the door is in viewed from a specific room.
 **
 ** Parameters: locationname - the name of the location you are in.
 **
 ** Returns:  NULL if there is no brief description available, or if
 **           something went wrong.
 **           a string with the description when successful
 **
 ***********************************************************************/

char *Door::get_brief(Location *the_loc)
{ 
  char *tmpstr = NULL;
  Strings holder;
  Strings holder2;
  int i;

  if (the_loc == NULL)
     return NULL;

  if (doorstate < 0 || doorstate > DOOR_MAXSTATE) 
     return NULL;

  holder = outside_loc.str_show();
  holder2.sprintf("%s@%s", the_loc->get_name(), the_loc->get_area());

  for (i=0; i<2; i++)
  {
     if (!holder.num_char('@'))
     {
        make_full_area(&holder, get_area());
     }

     if (holder.str_cmp(holder2.str_show()))
     {
        if (i==0)
           tmpstr = get_outside_brief(doorstate);
        else
           tmpstr = get_inside_brief(doorstate);
        return tmpstr;
     }
     if (i==0)
        holder = get_location();
  }
  return NULL;
}

/***********************************************************************
 ** get_inside_brief - Gets the brief description which belongs to the 
 **                    state the door is in viewed from the inside of 
 **                    the door.
 **
 ** Parameters: descnr - the state the door is in.
 **
 ** Returns:  NULL if there is no brief description available, or if
 **           something went wrong.
 **           a string with the description when successful
 **
 ***********************************************************************/

char *Door::get_inside_brief(int descnr)
{
  if (descnr < 0 || descnr > DOOR_MAXSTATE) 
     return NULL;

  return inside_brief[descnr].str_show();
}

/***********************************************************************
 ** get_outside_brief - Gets the brief description which belongs to the 
 **                     state the door is in viewed from the outside of 
 **                     the door.
 **
 ** Parameters: descnr - the state the door is in.
 **
 ** Returns:  NULL if there is no brief description available, or if
 **           something went wrong.
 **           a string with the description when successful
 **
 ***********************************************************************/

char *Door::get_outside_brief(int descnr)
{ 
  if (descnr < 0 || descnr > DOOR_MAXSTATE) 
     return NULL;

  return outside_brief[descnr].str_show();
}


/***********************************************************************
 ** write_object - writes the location to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Door::write_object(FILE *the_file, int build_format)
{
   int i;

   fprintf(the_file, "\ndoor %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);

   fprintf(the_file, "^%s^\n", (get_key_names() == NULL) ? "" : 
                                                            get_key_names());
   fprintf(the_file, "%d\n", get_door_state());
   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      fprintf(the_file, "^%s^\n", (get_inside_brief(i) == NULL) ? "" : 
                                                        get_inside_brief(i));
   }
   fprintf(the_file, "%s\n", (get_outside_loc() == NULL) ? "needloc" : 
                                                        get_outside_loc());
   fprintf(the_file, "^%s^\n", (get_outside_desc() == NULL) ? "" : 
                                                        get_outside_desc());
   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      fprintf(the_file, "^%s^\n", (get_outside_brief(i) == NULL) ? "" : 
                                                      get_outside_brief(i));
   }

   fprintf(the_file, "%d\n", get_distance());
   fprintf(the_file, "%s\n", (get_rope_name() == NULL) ? "none" : 
                                                          get_rope_name());
}

/***********************************************************************
 ** describe - describes the door to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Door::describe(Builder *the_builder)
{
   int  i;
 
   the_builder->send_bldr("\n&+GDoor: \t\t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t\t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GKeynames: \t\t&+g%s&*\n", get_key_names());
   the_builder->send_bldr("&+GClones: \t\t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t\t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t\t&+g%s&*\n", get_guards());

   if (itemflags->get_flag(ITEMFLAG_ROPETIE))
      the_builder->send_bldr("&+GInitialState: \t\t&+W%s&*\n", 
                                       ropestatenames[get_door_state()]);
   else
      the_builder->send_bldr("&+GInitialState: \t\t&+W%s&*\n", 
                                         statenames[get_door_state()]);

   the_builder->send_bldr("&+GInsideLocation: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GInsideDesc:&*\n%s\n", get_inside_desc());
   for (i=0; i <= DOOR_MAXSTATE; i++)
   {
      the_builder->send_bldr("&+GInsideBrief[%d]:&*\n%s\n",i, 
                                               get_inside_brief(i)); 
   }
   the_builder->send_bldr("&+GOutsideLocation: \t&+M%s&*\n", 
                                                          get_outside_loc());
   the_builder->send_bldr("&+GOutsideDesc:&*\n%s\n", get_outside_desc());
   for (i=0; i <= DOOR_MAXSTATE; i++)
   { 
       the_builder->send_bldr("&+GOutsidebrief[%d]:&*\n%s\n",i,
                              get_outside_brief(i));
   }

   if (itemflags->get_flag(ITEMFLAG_ROPETIE))
   {
      the_builder->send_bldr("&+GDistance: \t\t&+W%d&*\n", get_distance());
      if (get_door_state() == DOOR_OPEN)
         the_builder->send_bldr("&+GRopeName: \t\t&+g%s&*\n", 
                                                          get_rope_name());
   }

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the door to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Door::describe(Player *the_player)
{
   int  i;
   MudObject *tmp_container;
   Strings   container_name; 

   the_player->send_plr("\n&+GDoor: \t\t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t\t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GKeynames: \t\t&+g%s&*\n", get_key_names());
   the_player->send_plr("&+GClones: \t\t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t\t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t\t&+g%s&*\n", get_guards());

   if (itemflags->get_flag(ITEMFLAG_ROPETIE))
      the_player->send_plr("&+GInitialState: \t\t&+W%s&*\n", 
                                       ropestatenames[get_door_state()]);
   else
      the_player->send_plr("&+GInitialState: \t\t&+W%s&*\n", 
                                         statenames[get_door_state()]);

   the_player->send_plr("&+GInsideLocation: \t&+M%s&*\n", get_location());
   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr("&+GCurrentLoc: \t\t&+M%s&*\n", 
                                                 container_name.str_show());

   the_player->send_plr("&+GInsideDesc:&*\n%s\n", get_inside_desc());
   for (i=0; i <= DOOR_MAXSTATE; i++)
   {
      the_player->send_plr("&+GInsideBrief[%d]:&*\n%s\n",i, 
                                               get_inside_brief(i)); 
   }
   the_player->send_plr("&+GOutsideLocation: \t&+M%s&*\n", 
                                                          get_outside_loc());
   the_player->send_plr("&+GOutsideDesc:&*\n%s\n", get_outside_desc());
   for (i=0; i <= DOOR_MAXSTATE; i++)
   { 
       the_player->send_plr("&+GOutsidebrief[%d]:&*\n%s\n",i,
                              get_outside_brief(i));
   }

   if (itemflags->get_flag(ITEMFLAG_ROPETIE))
   {
      the_player->send_plr("&+GDistance: \t\t&+W%d&*\n", get_distance());
      if (get_door_state() == DOOR_OPEN)
         the_player->send_plr("&+GRopeName: \t\t&+g%s&*\n", 
                                                          get_rope_name());
   }

   list_specials(the_player);

   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Door::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->send_bldr("You can set the following attributes "
             "on a door.\n"
             "   title, altnames, itemflags, keynames, clones, desc,\n"
             "   insidedesc, outsidedesc, insidebrief, outsidebrief\n"
             "   insideloc, outsideloc, guards, specials, ropename,\n"
             "   distance and initialstate\n");
       return -1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "keynames",
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify keynames to set to.\n");
         return -1;
      }
      set_key_names(the_parsed->get_speech());
      the_builder->send_bldr("Keynames on %s set to: %s\n", get_name(), 
                                                        get_key_names());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "insidedesc", 
                               strlen(the_parsed->get_target1())))
   {
      Strings tmp_str;

      if (the_builder->get_long_input(&desc) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "insidelocation", 
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a location to set to.\n");
         return -1;
      }
      set_location(the_parsed->get_speech());
      the_builder->send_bldr("Location on %s set to: %s\n", get_name(), 
                                                        get_location());
      set_modified(1);
      return 1;
   }
   
   if (!STRNCASECMP(the_parsed->get_target1(), "outsidelocation",
                    strlen(the_parsed->get_target1())))
   {  if (the_parsed->get_speech() == NULL)
      {   
         the_builder->send_bldr("You need to specify a location to set to.\n");
         return -1;
      }
      set_outside_loc(the_parsed->get_speech());
      the_builder->send_bldr("Outside location set to %s\n",get_outside_loc());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initialstate",
                    strlen(the_parsed->get_target1())))
   {
      int i;

      if (the_parsed->get_speech() == NULL)
      {   
         the_builder->send_bldr("You need to specify a state to set to.\n");
         return -1;
      }
      
      for (i=0; i<= DOOR_MAXSTATE; i++)
      {
         if ((!itemflags->get_flag(ITEMFLAG_ROPETIE)) && 
                (!STRNCASECMP(statenames[i], the_parsed->get_speech(), 
                                    strlen(the_parsed->get_speech()))))
	 {
            doorstate = i;
            set_modified(1);
            the_builder->send_bldr("InitialState is now '%s'\n", 
                                                          statenames[i]);
            set_modified(1);
            return 1;

         }
         else if ((itemflags->get_flag(ITEMFLAG_ROPETIE)) && 
                (!STRNCASECMP(ropestatenames[i], the_parsed->get_speech(), 
                                    strlen(the_parsed->get_speech()))))
	 {
            if ((i == DOOR_LOCKED) || (i == DOOR_MLOCKED))
	    {
               the_builder->send_bldr("A rope door can only be "
                                      "set to 'tied' or 'untied'.\n");
               return -1; 
	    }
            doorstate = i;
            set_modified(1);
            the_builder->send_bldr("InitialState is now '%s'\n", 
                                                   ropestatenames[i]);
            set_modified(1);
            return 1;

         }
      }
      the_builder->send_bldr("That is not a valid state.\n"
                 "Valid states are: Open, Close, Locked, Magically Locked\n");
      return -1;
   }  
   if (!STRNCASECMP(the_parsed->get_target1(), "outsidedesc",
                    strlen(the_parsed->get_target1())))
   { 
      if (the_builder->get_long_input(&outside_desc) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }

      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "insidebrief",
                    strlen(the_parsed->get_target1())))
   {  Strings tmp_str;
      Strings tmp_word;
      int     statenr;

      tmp_str = the_parsed->get_speech();

      tmp_word.assign_word(tmp_str.str_show(),1);

      if (tmp_word.str_show() == NULL)
      {
         the_builder->send_bldr("Usage: set insidebrief <nr>\n");
         return -1;
      }

      if (isdigit(*(tmp_word.str_show())))
      {  
         statenr = atoi(tmp_word.str_show());
      }
      else
      {  the_builder->send_bldr("Usage: set insidebrief <nr>\n");
         return -1;
      }

      if (statenr < 0 || statenr > DOOR_MAXSTATE)
      {   the_builder->send_bldr("The state needs to be between 0 and %d.\n",
                                 DOOR_MAXSTATE);
          return -1;
      }

      if (the_builder->get_long_input(&inside_brief[statenr]) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
  
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "outsidebrief",
                    strlen(the_parsed->get_target1())))
   {  Strings tmp_str;
      Strings tmp_word;
      int     statenr;

      tmp_str = the_parsed->get_speech();

      tmp_word.assign_word(tmp_str.str_show(),1);

      if ((tmp_word.str_show() != NULL) && 
          (tmp_word.str_len() > 0) &&
          (isdigit(*(tmp_word.str_show()))))
      {  
         statenr = atoi(tmp_word.str_show());
      }
      else
      {  the_builder->send_bldr("Usage: set outsidebrief <nr>\n");
         return -1;
      }

      if (statenr < 0 || statenr > DOOR_MAXSTATE)
      {   the_builder->send_bldr("The state needs to be between 0 and %d.\n",
                                 DOOR_MAXSTATE);
          return -1;
      }

      if (the_builder->get_long_input(&outside_brief[statenr]) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
  
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "ropename", 
                               strlen(the_parsed->get_target1())))
   {
      if (!itemflags->get_flag(ITEMFLAG_ROPETIE))
      {
         the_builder->send_bldr("You must set the RopeTie itemflag "
                                "first to set this attribute.\n");
         return -1;
      }
      if (get_door_state() != DOOR_OPEN)
      {
         the_builder->send_bldr("Door initialstate must be set to "
                                "Tied first.\n");
         return -1;
      }
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a RopeName to set to.\n");
         return -1;
      }
      set_rope_name(the_parsed->get_speech());
      the_builder->send_bldr("RopeName on %s set to: %s\n", get_name(), 
                                                        get_rope_name());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "distance", 
                               strlen(the_parsed->get_target1())))
   {
      if (!itemflags->get_flag(ITEMFLAG_ROPETIE))
      {
         the_builder->send_bldr("You must set the RopeTie itemflag "
                                "first to set this attribute.\n");
         return -1;
      }

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as distance.\n");
         return -1;
      }

      set_distance(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("Distance set to %d on door object %s.\n",
                                               get_distance(), get_name());
      return 1;
   }

   the_builder->send_bldr("The attribute '%s' is not a door attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}

/***********************************************************************
 ** get_link - gets the corresponding location given the opposite
 **
 ** Parameters: the_location - the location we are at
 **             return_val - the reason for a NULL value
 **
 ** Returns:  pointer to the new location
 **           NULL if failed
 **
 ** return_val values: 1 if a null location was passed in
 **                    2 if the door is closed
 **                    3 if there was an error getting the exit
 **                    4 if the door is not tied (rope door)
 **
 ***********************************************************************/
   
char *Door::get_link(Location *the_location, int *return_val){
   Strings holder;
   Strings holder2;

   if (the_location == NULL)
   {
      *return_val = 1; 
      return NULL;
   }

   if ((doorstate != DOOR_OPEN) && (itemflags->get_flag(ITEMFLAG_ROPETIE)))
   {

      /* if the rope is too short */
      if (doorstate == DOOR_LOCKED)
         *return_val = 5;

      /* no rope tied here */
      else
         *return_val = 4;

      return NULL;
   }
   else if (doorstate != DOOR_OPEN)
   {
      *return_val = 2;
      return NULL;
   }

   /* we check the first area first */
   holder = get_outside_loc();  /* the outside location */
   make_full_area(&holder, get_area());
   holder2 = the_location->get_name();
   make_full_area(&holder2, the_location->get_area());
   if (!STRCASECMP(holder.str_show(), holder2.str_show()))
   {
      *return_val = 0;
      return get_location();
   }

   holder = get_location();  /* the inside location */
   make_full_area(&holder, get_area());
   if (!STRCASECMP(holder.str_show(), holder2.str_show()))
   {
      *return_val = 0;
      return get_outside_loc();
   }

   *return_val = 3;
   return NULL;
}


/***********************************************************************
 ** find_key - checks to see if a keyname is in the list of keynames
 **
 ** Parameters: keyname - the name of the key we are looking for
 **
 ** Returns:  1 if found
 **           0 if not found
 **
 **
 ***********************************************************************/
   
int Door::find_key(char *keyname){
   return key_names.find_in_str(keyname);
}


/***********************************************************************
 ** set_key_names - sets the key names string of keys that can unlock this
 **
 ** Parameters: thenames - the names we are setting it to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Door::set_key_names(char *thenames)
{
   if (thenames == NULL)
      return -1;
 
   key_names = thenames;
   return 1;   
}


/***********************************************************************
 ** get_key_names - returns the list of keynames
 **
 ** Parameters: None
 **
 ** Returns:  A pointer to the keynames string
 **
 ***********************************************************************/

char *Door::get_key_names(void)
{
   return key_names.str_show();
}


/***********************************************************************
 ** toggle_lock - checks the key to see if it works in this door, and if
 **               so, toggles the lock between locked and unlocked
 **
 ** Parameters: keyname - the keyname of the key they are using
 **
 ** Returns:  1 for it worked, 0 for didnt work, -1 for it failed
 **
 ***********************************************************************/

int Door::toggle_lock(char *keyname)
{
   char *tmp_str;

   /* first we check if this key is in the same area as the door, if so
      then we dont need to send the areaname so chop it */
   tmp_str = keyname;
   while ((*tmp_str) && (*tmp_str != '@'))
      tmp_str++;

   if ((*(tmp_str+1)) && (!STRCASECMP((tmp_str+1), this->get_area())))
   {
      *tmp_str = '\0';
   }

   if ((keyname == NULL) || (!key_names.find_in_str(keyname)))
      return 0;

   if (doorstate == 1)
   {
      state = 2;
      return 1;
   }
   else if (doorstate == 2)
   {
      doorstate = 1;
      return 1;
   }
   else
      return -1;
}


/***********************************************************************
 ** copy_object - copies the door to another object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Door::copy_object(Entity *copy_obj)
{
   int i;
   Door *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_DOOR)
      return 0;

   copy_from = (Door *) copy_obj;


   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the door attributes ******/
   set_outside_loc(copy_from->get_outside_loc());
   set_outside_desc(copy_from->get_outside_desc());
   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      set_outside_brief(i, copy_from->get_outside_brief(i));
      set_inside_brief(i, copy_from->get_inside_brief(i));
   }
   set_door_state(copy_from->get_door_state());   
   set_key_names(copy_from->get_key_names());


   return 1;
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Door *Door::operator = (Door *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** read_door_attrib - reads in door attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Door::read_door_attrib(FILE *read_file, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;
   int          i;
   char         *temp_desc;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(), 
                                                                 get_area());
      error_log->log_err(holder.str_show(), "read_door_attrib");
      return -1;
   }

   /* get the keynames */
   the_token = get_token(read_file, '^');
   set_key_names(the_token->the_string);

   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(),
                                                                 get_area());
      error_log->log_err(holder.str_show(), "read_door_attrib");
      return -1;
   }
   set_door_state(atoi(the_token->the_string));

   for (i=0; i<= DOOR_MAXSTATE; i++)
   {
      /* set the inside briefs */
      temp_desc = 
              read_desc_type(read_file, error_log, (MudObject *) this); 
      if (temp_desc == NULL)
         return -1;
      set_inside_brief(i, temp_desc);
      delete temp_desc;
   }

   /* get the object outside location */
   the_token = get_token(read_file, '\0');
   set_outside_loc(the_token->the_string);

   /* set the outside desc */
   temp_desc = read_desc_type(read_file, error_log, (MudObject *) this); 
   if (temp_desc == NULL)
      return -1;
   set_outside_desc(temp_desc);
   delete temp_desc;

   for (i=0; i<= DOOR_MAXSTATE; i++)
   {
      /* set the outside briefs */
      temp_desc = read_desc_type(read_file, error_log, (MudObject *) this); 
      if (temp_desc == NULL)
         return -1;
      set_outside_brief(i, temp_desc);
      delete temp_desc;
   }

   /* Set distance */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
         sprintf("Invalid format for attribute distance in door %s, area %s", 
                                             get_name(), get_area());

      error_log->log_err(holder.str_show(), "read_door_attrib");
      return -1;
   }
   set_distance(atoi(the_token->the_string));

   /* get the rope name */
   the_token = get_token(read_file, '\0');
   if (!STRCASECMP("none", the_token->the_string))
      rope_name.truncate(0);
   else
      set_rope_name(the_token->the_string);
   
   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Door::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Door::get_mem_size_dynamic()
{
   int  size = 0;
   int  i;   

   size += outside_loc.get_mem_size_dynamic();

   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      size += inside_brief[i].get_mem_size_dynamic();
   }

   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      size += outside_brief[i].get_mem_size_dynamic();
   }

   size += key_names.get_mem_size_dynamic();
   size += rope_name.get_mem_size_dynamic();
 
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}

#endif

