/**********************************************************************
 ** AberPdirt - functions to convert an abermud pdirt area to the
 **             aime format
 **  
 **
 **    
 **
 **********************************************************************/

#ifndef ABERPDIRT_C
#define ABERPDIRT_C

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "aberpdirt.h"
#include "../../include/strings.h"
#include "../../include/lexer.h"
#include "convconfig.h"
#include "shared.h"
#include "../../include/sysdep.h"
#include "../../include/utils.h"
#include "../../include/objtype.h"

#ifdef WIN32
#define NEWLINE "\r\n"
#else
#define NEWLINE "\n"
#endif

#ifdef WIN32
#include <afxwin.h>
#include "../../win32/resource.h"
#include "../../win32/ChooseObjtype.h"
#endif

temp_loc a_loc;
temp_mob a_mob;
temp_obj an_obj;



/* left side is lflag equivalent to the right side locflag. */
/* So Pdirt Light is equivalent to AIME Lit */

#define NUM_LFLAGS 34

flag_name lflags[NUM_LFLAGS] = {
  {"Light",        "Lit"},
  {"Outdoors",     "Outside"},
  { NULL,          "litatday"},
  {"NoMobiles",    "NoMobiles"},
  {"Peaceful",     "Peaceful"},
  {"OnWater",      "Aquatic"},
  { NULL,          "Shore"},
  {"Dark",         NULL},
  {"Real",         NULL},
  {"TempOrdinary", NULL},
  {"Hot",          NULL},
  {"Cold",         NULL},
  {"TempReal",     NULL},
  {"Death",        NULL},
  {"CantSummon",   NULL},
  {"NoSummon",     NULL},
  {"NoQuit",       NULL},
  {"NoSnoop",      NULL},
  {"NoMagic",      NULL},
  {"Soundproof",   NULL},
  {"OnePerson",    NULL},
  {"Party",        NULL},
  {"Private",      NULL},
  {"UnderWater",   NULL},
  {"TempExtreme",  NULL},
  {"NegRegen",     NULL},
  {"RealTime",     NULL},
  {"Maze",         NULL},
  {"WizOnly",      NULL},
  {"AwizOnly",     NULL},
  {"DGodOnly",     NULL},
  {"GodOnly",      NULL},
  {"Jail",         NULL},
  {"Airless",      NULL}};


/* left side is pflag equivalent to the right side indflag. */
/* So Pdirt Female is equivalent to AIME Male, albeit opposite */

#define NUM_PFLAGS 14

flag_name pflags[NUM_PFLAGS] = {
  {NULL,          "Male"},
  {NULL,          "Fighting"},
  {NULL,          "AttackPlayer"},
  {"NoExorcise",  NULL},
  {"NoSnoop",     NULL},
  {"NoHassle",    NULL},
  {"NoAlias",     NULL},
  {"NoZap",       NULL},
  {"NoSummon",    NULL},
  {"NoTrace",     NULL},
  {"NoSteal",     NULL},
  {"NoMagic",     NULL},
  {"NoForce",     NULL},
  {"NoTimeout",   NULL}};


/* left side is sflag equivalent to the right side indflag. */
/* So Pdirt Female is equivalent to AIME Male, albeit opposite */

#define NUM_SFLAGS 4

flag_name sflags[NUM_SFLAGS] = {
  {"Female",      "Male"},
  {NULL,          "Fighting"},
  {NULL,          "AttackPlayer"},
  {"Glowing",     NULL}};


/* left side is mflag equivalent to the right side indflag. */
/* So Pdirt Female is equivalent to AIME Male, albeit opposite */

#define NUM_MFLAGS 38

flag_name mflags[NUM_MFLAGS] = {
  {NULL,          "Male"},
  {NULL,          "Fighting"},
  {NULL,          "AttackPlayer"},
  {"NoSteal",     "NoSteal"},
  {NULL,          "Sitting"},
  {NULL,          "Sleeping"},
  {"PickStuff",   "GetStuff"},
  {"CanFireball", NULL},
  {"CanMissile",  NULL},
  {"CanFrost",    NULL},
  {"CanShock",    NULL},
  {"NegFireball", NULL},
  {"NegMissile",  NULL},
  {"NegFrost",    NULL},
  {"NegShock",    NULL},
  {"NoHeat",      NULL},
  {"Thief",       NULL},
  {"StealWeapon", NULL},
  {"StealWorn",   NULL},
  {"PitIt",       NULL},
  {"DrainLev",    NULL},
  {"DrainScr",    NULL},
  {"BarNorth",    NULL},
  {"BarSouth",    NULL},
  {"BarEast",     NULL},
  {"BarWest",     NULL},
  {"BarUp",       NULL},
  {"BarDown",     NULL},
  {"QFood",       NULL},
  {"Blind",       NULL},
  {"Deaf",        NULL},
  {"Dumb",        NULL},
  {"NoGrab",      NULL},
  {"GrabHostile", NULL},
  {"Block",       NULL},
  {"Cross",       NULL},
  {"NoTalk",      NULL},
  {"NoCorpse",    NULL}};


/* left side is oflag equivalent to the right side itemflag. */

#define NUM_OFLAGS_ITEM 40

flag_name oflags_item[NUM_OFLAGS_ITEM] = {
  {"Container",   "Container"},
  {"Lightable",   "Lightable"},
  {"Lit",         "Lit"},
  {"MagicLit",    "Glowing"},
  {"Booze",       "Booze"},
  {"NoGet",       NULL},
  {"Openable",    NULL},
  {"Lockable",    NULL},
  {"Pushable",    NULL},
  {"PushToggle",  NULL},
  {"Food",        NULL},
  {"Armor",       NULL},
  {"Unlocateable", NULL},
  {"Destroyed",   NULL},
  {"Weapon",      NULL},
  {"Extinguish",  NULL},
  {"AKey",        NULL},
  {"GetFlips",    NULL},
  {"Lit",         NULL},
  {"Boat",        NULL},
  {"Newbie",      NULL},
  {"Poison",      NULL},
  {"Wearable",    NULL},
  {"ExcludeWear", NULL},
  {"WearOnFace",  NULL},
  {"WearOnHead",  NULL},
  {"WearOnNeck",  NULL},
  {"WearOnBody",  NULL},
  {"WearOnBack",  NULL},
  {"WearOnArms",  NULL},
  {"WearOnHands", NULL},
  {"WearOnLegs",  NULL},
  {"WearOnFeet",  NULL},
  {"NoMobGet",    NULL},
  {"Rope",        NULL},
  {"EnhanceMagic", NULL},
  {"BlockMagic",  NULL},
  {"WearBlMagic", NULL},
  {"Donated",     NULL},
  {"CantPick",    NULL}};


/* left side is oflag equivalent to the right side itemflag. */

#define NUM_OFLAGS_WEAR 42

flag_name oflags_wear[NUM_OFLAGS_WEAR] = {
  {"WearOnHead",  "OnHead"},
  {"WearOnFace",  "OnFace"},
  {"WearOnNeck",  "OnNeck"},
  {"WearOnBody",  "OnChest"},
  {"WearOnBack",  "OnBack"},
  {"WearOnArms",  "OnArms"},
  {"WearOnHands", "OnHands"},
  {NULL,          "OnWaist"},
  {"WearOnLegs",  "OnLegs"},
  {"WearOnFeet",  "OnFeet"},
  {NULL,          "Wieldable"},
  {"ExcludeWear", "TopLayer"},
  {"Destroyed",   NULL},
  {"NoGet",       NULL},
  {"Container",   NULL},
  {"Lightable",   NULL},
  {"Lit",         NULL},
  {"MagicLit",    NULL},
  {"Openable",    NULL},
  {"Lockable",    NULL},
  {"Pushable",    NULL},
  {"PushToggle",  NULL},
  {"Food",        NULL},
  {"Armor",       NULL},
  {"Unlocateable", NULL},
  {"Weapon",      NULL},
  {"Extinguish",  NULL},
  {"AKey",        NULL},
  {"GetFlips",    NULL},
  {"Lit",         NULL},
  {"Boat",        NULL},
  {"Newbie",      NULL},
  {"Poison",      NULL},
  {"Booze",       NULL},
  {"Wearable",    NULL},
  {"NoMobGet",    NULL},
  {"Rope",        NULL},
  {"EnhanceMagic", NULL},
  {"BlockMagic",  NULL},
  {"WearBlMagic", NULL},
  {"Donated",     NULL},
  {"CantPick",    NULL}};


/***********************************************************************
 ** ap_convert - the main conversion procedure
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the pseudo-data file we write to
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_convert(FILE *read_file, FILE *write_file, char *filename, float version)
{
   token_record *the_token; 

   version = 0.0;  // eliminate compile warnings
 
   filename = NULL;

   /* reset the read file descriptor back to the beginning */
   fseek(read_file, 0, SEEK_SET); 

   /* write the AIME pseudo-data area file header */
   fprintf(write_file, "StartHeader\n");
   fprintf(write_file, "Flag1 %d\n", DEF_AREA_FLAGS1);
   fprintf(write_file, "Flag2 %d\n", DEF_AREA_FLAGS1);
   fprintf(write_file, "ReloadSecs %d\n", DEF_RELOAD_INT);
   fprintf(write_file, "EndHeader\n\n");

   /* get to the good stuff in the pdirt area file */
   the_token = get_token(read_file, '\0');
   while (the_token->token_type > 0)
   {
      /* ignore comments in the file */
      if (the_token->token_type == T_DIVIDE)
      {
         the_token = get_token(read_file, '\0');
         if (the_token->token_type == T_MULT)
		 {
            the_token = get_token(read_file, '/');
		 }
         the_token = get_token(read_file, '\0');
      }

      /* it is an include, ignore it */
      if (the_token->token_type == T_POUND)
      {
         next_line(read_file);
      }
      else if (the_token->token_type == T_IDENTIFIER)
      {
         if (!STRCASECMP(the_token->the_string, "Zone"))
		 {
            ap_read_header(read_file);
         }
         else if (!STRCASECMP(the_token->the_string, "Location"))
		 {
            ap_read_location(read_file, write_file);
         }
         else if (!STRCASECMP(the_token->the_string, "Mobile"))
		 {
            ap_read_mobile(read_file, write_file);
         }
         else if (!STRCASECMP(the_token->the_string, "Object"))
		 {
            ap_read_object(read_file, write_file);
         }
         else
		 {
            printf("Unexpected element '%s' found in Aber Pdirt file.\n", 
                               the_token->the_string);
            return -1;            
         }
      }
      else
      {
         printf("Unexpected element '%s' found in Aber Pdirt file.\n", 
                               the_token->the_string);
         return -1;
      }
      the_token = get_token(read_file, '\0');
   }

   return 1;
}


/***********************************************************************
 ** ap_read_header - reads in the pdirt zone header, converting if
 **                  necessary
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_read_header(FILE *read_file)
{
   token_record *the_token;

   /* we currently just blow right through this */
   the_token = get_token(read_file, '\0');
   while ((the_token->token_type > 0) && 
          (STRCASECMP(the_token->the_string, "endzone")))
      the_token = get_token(read_file, '\0');
   next_line(read_file);


   return 1;
}


/***********************************************************************
 ** ap_get_loc_element - reads in an element from a location
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **             the_token - the token for this element
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_get_loc_element(FILE *read_file, token_record **the_token)
{
   int  i;
   int  found = 0;
   char *dir[10] = {"North", "South", "East", "West", "Up", "Down",  
	 	    "Northeast", "Northwest", "Southeast", "Southwest"};
   char the_char;
   int  char_int;   
   char *tmp_desc;
 
   for (i=0; i<10; i++)
   {
      if (!STRCASECMP((*the_token)->the_string, dir[i]))
      {
         *the_token = get_token(read_file, '\0');
         *the_token = get_token(read_file, '\0');
         if ((*the_token)->token_type == T_CARROT)
            *the_token = get_token(read_file, '\0');
         a_loc.dir[i] = (*the_token)->the_string;
         *the_token = get_token(read_file, '\0');
         return 1;
      }
   }

   if (!STRCASECMP((*the_token)->the_string, "Lflags"))
   {
      *the_token = get_token(read_file, '\0');
      do
      {
         *the_token = get_token(read_file, '\0');
         for (i=0; i<NUM_LFLAGS; i++)
         {
            if ((lflags[i].aberflag != NULL) && 
                (!STRCASECMP(lflags[i].aberflag, (*the_token)->the_string)))
	    {
               if (lflags[i].aimeflag != NULL)
	       {
                  a_loc.lflags->set_flag(i);
               }
               i = NUM_LFLAGS;
               found = 1;
            }
         }
         if (!found)
         {
            printf("unrecognized Lflag '%s', ignoring.\n", 
                                               (*the_token)->the_string);
         }
         *the_token = get_token(read_file, '\0');
      } while ((*the_token)->token_type == T_COMMA);
   }
   else if (!STRCASECMP((*the_token)->the_string, "Title"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Title.\n");
         return -1;
      }
      *the_token = get_token(read_file, the_char);
      a_loc.title = (*the_token)->the_string;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Desc"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Title.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      a_loc.desc = tmp_desc;
      if (tmp_desc != NULL)
         delete tmp_desc;
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Trap"))
   {
      while ((((*the_token)->token_type > 0) || 
              ((*the_token)->token_type == -3)) && 
             (STRCASECMP((*the_token)->the_string, "EndTrap")))
         *the_token = get_token(read_file, '\0');

      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "UseTrap"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      if ((*the_token)->token_type == T_COLON)
      {
         *the_token = get_token(read_file, '\0');
         *the_token = get_token(read_file, '\0');
      }
   }
   else
   {
      printf("Invalid element '%s'\n", (*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** ap_get_mob_element - reads in an element from a mobile
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **             the_token - the token for this element
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_get_mob_element(FILE *read_file, token_record **the_token)
{
   int  i;
   int  found = 0;
   char the_char;
   int  char_int;   
   char *tmp_desc;
   char *tmp_holder;

   if (!STRCASECMP((*the_token)->the_string, "Pflags"))
   {
      *the_token = get_token(read_file, '\0');
      do
      {
         *the_token = get_token(read_file, '\0');
         for (i=0; i<NUM_PFLAGS; i++)
         {
            if ((pflags[i].aberflag != NULL) && 
                (!STRCASECMP(pflags[i].aberflag, (*the_token)->the_string)))
	    {
               if (pflags[i].aimeflag != NULL)
	       {
                  a_mob.indflags->set_flag(i);
               }
               i = NUM_PFLAGS;
               found = 1;
            }
         }
         if (!found)
         {
            printf("unrecognized Pflag '%s', ignoring.\n", 
                                               (*the_token)->the_string);
         }
         *the_token = get_token(read_file, '\0');
      } while ((*the_token)->token_type == T_COMMA);
   }
   else if (!STRCASECMP((*the_token)->the_string, "Sflags"))
   {
      *the_token = get_token(read_file, '\0');
      do
      {
         *the_token = get_token(read_file, '\0');
         for (i=0; i<NUM_SFLAGS; i++)
         {
            if ((sflags[i].aberflag != NULL) && 
                (!STRCASECMP(sflags[i].aberflag, (*the_token)->the_string)))
	    {
               if (sflags[i].aimeflag != NULL)
	       {
                  if (!STRCASECMP(sflags[i].aberflag, "Female"))
                     a_mob.indflags->clr_flag(i);
                  else
                     a_mob.indflags->set_flag(i);
               }
               i = NUM_SFLAGS;
               found = 1;
            }
         }
         if (!found)
         {
            printf("unrecognized Sflag '%s', ignoring.\n", 
                                               (*the_token)->the_string);
         }
         *the_token = get_token(read_file, '\0');
      } while ((*the_token)->token_type == T_COMMA);
   }
   else if (!STRCASECMP((*the_token)->the_string, "Mflags"))
   {
      *the_token = get_token(read_file, '\0');
      do
      {
         *the_token = get_token(read_file, '\0');
         for (i=0; i<NUM_MFLAGS; i++)
         {
            if ((mflags[i].aberflag != NULL) && 
                (!STRCASECMP(mflags[i].aberflag, (*the_token)->the_string)))
	    {

               a_mob.indflags->set_flag(i);
               i = NUM_MFLAGS;
               found = 1;
            }
         }
         if (!found)
         {
            printf("unrecognized Mflag '%s', ignoring.\n", 
                                               (*the_token)->the_string);
         }
         *the_token = get_token(read_file, '\0');
      } while ((*the_token)->token_type == T_COMMA);
   }
   else if (!STRCASECMP((*the_token)->the_string, "Name"))
   {
      char *title_ptr;
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Title.\n");
         return -1;
      }
      *the_token = get_token(read_file, the_char);

      title_ptr = (*the_token)->the_string;
      /* get rid of "the" as in "the piper", it should just be piper */
      if (!STRNCASECMP((*the_token)->the_string, "the", 3))
      {
         title_ptr += 4;
      }
      a_mob.title = title_ptr;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Location"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if (!STRCASECMP((*the_token)->the_string, "limbo@limbo"))
         a_mob.location = "none";
      else
         a_mob.location = (*the_token)->the_string;
      *the_token = get_token(read_file, '\0');
   }

   else if (!STRCASECMP((*the_token)->the_string, "Strength"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Strength element");
         return -1;
      }
      a_mob.str = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Damage"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Damage element");
         return -1;
      }
      a_mob.damage = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Speed"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Speed element");
         return -1;
      }
      a_mob.speed = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Aggression"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Aggression element");
         return -1;
      }
      a_mob.aggression = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Armor"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Armor element");
         return -1;
      }
      a_mob.armor = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Desc"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Desc.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      tmp_holder = tmp_desc;
      while ((*tmp_desc) && (*tmp_desc == ' ') && (*tmp_desc == '\n') &&
             (*tmp_desc == '\r'))
         tmp_desc++;
      a_mob.desc = tmp_desc;
      if (tmp_holder != NULL)
         delete tmp_holder;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Examine"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Examine.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      a_mob.examine = tmp_desc;
      if (tmp_desc != NULL)
         delete tmp_desc;
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "UseTrap"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      if ((*the_token)->token_type == T_COLON)
      {
         *the_token = get_token(read_file, '\0');
         *the_token = get_token(read_file, '\0');
      }
   }
   else if (!STRCASECMP((*the_token)->the_string, "Trap"))
   {
      while ((((*the_token)->token_type > 0) || 
              ((*the_token)->token_type == -3)) && 
             (STRCASECMP((*the_token)->the_string, "EndTrap")))
         *the_token = get_token(read_file, '\0');

      *the_token = get_token(read_file, '\0');
   }
   else
   {
      printf("Invalid element '%s'\n", (*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** ap_get_obj_element - reads in an element from an object
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **             the_token - the token for this element
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_get_obj_element(FILE *read_file, token_record **the_token)
{
   int  i;
   int  found = 0;
   char the_char;
   int  char_int;   
   char *tmp_desc;
   char *tmp_holder;

   if (!STRCASECMP((*the_token)->the_string, "Oflags"))
   {
      *the_token = get_token(read_file, '\0');
      do
      {
         *the_token = get_token(read_file, '\0');
        
         /* lets look to see if we can determine the obj type by this
            flag */
         if ((!STRCASECMP((*the_token)->the_string, "Food")) ||
             (!STRCASECMP((*the_token)->the_string, "Booze")))
            an_obj.poss_type.str_cat("This might be food.\n");

         if ((!STRCASECMP((*the_token)->the_string, "Armor")) ||
             (!STRCASECMP((*the_token)->the_string, "Wearable")))
            an_obj.poss_type.str_cat("This might be a wearable\n");

         if (!STRCASECMP((*the_token)->the_string, "AKey"))
            an_obj.poss_type.str_cat("This might be a key\n");

         if (!STRCASECMP((*the_token)->the_string, "Weapon"))
            an_obj.poss_type.str_cat("This might be a weapon\n");

         if (!STRCASECMP((*the_token)->the_string, "NoGet"))
            an_obj.poss_type.str_cat("This might be a marker\n");

         /* first check the itemflags for a match */
         for (i=0; i<NUM_OFLAGS_ITEM; i++)
         {
            if ((oflags_item[i].aberflag != NULL) && 
                (!STRCASECMP(oflags_item[i].aberflag, 
                                               (*the_token)->the_string)))
	    {
               if (oflags_item[i].aimeflag != NULL)
	       {
                  found = 1;
                  an_obj.itemflags->set_flag(i);
               }
               i = NUM_OFLAGS_ITEM;
            }
         }

	 /* if it wasnt found in the itemflags, check wearflags */
         if (!found)
	 {
            for (i=0; i<NUM_OFLAGS_WEAR; i++)
            {
               if ((oflags_wear[i].aberflag != NULL) && 
                   (!STRCASECMP(oflags_wear[i].aberflag, 
                                               (*the_token)->the_string)))
	       {
                  if (oflags_wear[i].aimeflag != NULL)
	          {
                     an_obj.wearflags->set_flag(i);
                     an_obj.poss_type.str_cat("This might be a wearable\n");
                  }
                  i = NUM_OFLAGS_ITEM;
                  found = 1;
               }
            }
         }

         /* it wasnt found at all */
         if (!found)
         {
            printf("unrecognized Oflag '%s', ignoring.\n", 
                                               (*the_token)->the_string);
         }
         *the_token = get_token(read_file, '\0');
      } while ((*the_token)->token_type == T_COMMA);
   }
   else if (!STRCASECMP((*the_token)->the_string, "Name"))
   {
      char *title_ptr;
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Title.\n");
         return -1;
      }
      *the_token = get_token(read_file, the_char);

      title_ptr = (*the_token)->the_string;
      /* get rid of "the" as in "the stick", it should just be stick */
      if (!STRNCASECMP((*the_token)->the_string, "the", 3))
      {
         title_ptr += 4;
      }
      an_obj.title = title_ptr;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Altname"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Altname.\n");
         return -1;
      }
      *the_token = get_token(read_file, the_char);

      an_obj.altname = (*the_token)->the_string;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Location"))
   {
      Strings tmp_name;
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      tmp_name = (*the_token)->the_string;
      *the_token = get_token(read_file, '\0');
      if ((*the_token)->token_type == T_COLON)
      {
         *the_token = get_token(read_file, '\0');
         tmp_name = (*the_token)->the_string;
         *the_token = get_token(read_file, '\0');
      }

      if (!STRCASECMP(tmp_name.str_show(), "limbo@limbo"))
         an_obj.location = "none";
      else
         an_obj.location = tmp_name.str_show();
   }
   else if (!STRCASECMP((*the_token)->the_string, "UseTrap"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      if ((*the_token)->token_type == T_COLON)
      {
         *the_token = get_token(read_file, '\0');
         *the_token = get_token(read_file, '\0');
      }
   }
   else if (!STRCASECMP((*the_token)->the_string, "Linked"))
   {
      Strings tmp_name;
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      an_obj.linked = (*the_token)->the_string;
      *the_token = get_token(read_file, '\0');

      an_obj.poss_type.str_cat("This might be a door\n");
   }
   else if (!STRCASECMP((*the_token)->the_string, "State"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in State element");
         return -1;
      }
      an_obj.state = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "MaxState"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in MaxState element");
         return -1;
      }
      an_obj.maxstate = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "BaseValue"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in BaseValue element");
         return -1;
      }
      an_obj.basevalue = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Size"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Size element");
         return -1;
      }
      an_obj.size = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Damage"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Damage element");
         return -1;
      }
      an_obj.damage = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
      an_obj.poss_type.str_cat("This might be a weapon\n");
   }
   else if (!STRCASECMP((*the_token)->the_string, "Armor"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Armor element");
         return -1;
      }
      an_obj.armor = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
      an_obj.poss_type.str_cat("This might be a wearable\n");
   }
   else if (!STRCASECMP((*the_token)->the_string, "Weight"))
   {
 
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');

      if ((*the_token)->token_type != T_NUMERICAL)
      {
         printf("Invalid format in Weight element");
         return -1;
      }
      an_obj.weight = atoi((*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Desc0"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Desc0.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      tmp_holder = tmp_desc;
      while ((*tmp_desc) && (*tmp_desc == ' ') && (*tmp_desc == '\n') &&
             (*tmp_desc == '\r'))
         tmp_desc++;
      an_obj.desc0 = tmp_desc;
      if (tmp_holder != NULL)
         delete tmp_holder;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Desc1"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Desc0.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      tmp_holder = tmp_desc;
      while ((*tmp_desc) && (*tmp_desc == ' ') && (*tmp_desc == '\n') &&
             (*tmp_desc == '\r'))
         tmp_desc++;
      an_obj.desc1 = tmp_desc;
      if (tmp_holder != NULL)
         delete tmp_holder;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Desc2"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Desc2.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      tmp_holder = tmp_desc;
      while ((*tmp_desc) && (*tmp_desc == ' ') && (*tmp_desc == '\n') &&
             (*tmp_desc == '\r'))
         tmp_desc++;
      an_obj.desc2 = tmp_desc;
      if (tmp_holder != NULL)
         delete tmp_holder;
      *the_token = get_token(read_file, '\0');
   } 
   else if (!STRCASECMP((*the_token)->the_string, "Examine"))
   {
      *the_token = get_token(read_file, '\0');
      *the_token = get_token(read_file, '\0');
      char_int = (*the_token)->token_type;
      if (char_int == T_QUOTE)
         the_char = '"';         
      else if (char_int == T_APOST)
         the_char = '\'';
      else if (char_int == T_CARROT)
         the_char = '^';
      else
      {
         printf("Invalid format in Examine.\n");
         return -1;
      }
      tmp_desc = read_mod_desc_type(read_file, char_int, the_char);
      an_obj.examine = tmp_desc;
      if (tmp_desc != NULL)
         delete tmp_desc;
      *the_token = get_token(read_file, '\0');
   }
   else if (!STRCASECMP((*the_token)->the_string, "Trap"))
   {
      while ((((*the_token)->token_type > 0) || 
              ((*the_token)->token_type == -3)) && 
             (STRCASECMP((*the_token)->the_string, "EndTrap")))
         *the_token = get_token(read_file, '\0');

      *the_token = get_token(read_file, '\0');
   }
   else
   {
      printf("Invalid element '%s'\n", (*the_token)->the_string);
      *the_token = get_token(read_file, '\0');
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** ap_read_object - reads in an object and converts it to AIME format
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/

int ap_read_object(FILE *read_file, FILE *write_file)
{
   token_record *the_token;

   an_obj.title.truncate(0);
   an_obj.itemflags = new Flags(2);
   an_obj.wearflags = new Flags(1);
   an_obj.altname.truncate(0);
   an_obj.linked.truncate(0);

   an_obj.location = "none";
   an_obj.damage = 0;
   an_obj.state = 0;
   an_obj.maxstate = 0;
   an_obj.basevalue = 0;
   an_obj.size = 0;
   an_obj.weight = 0;
   an_obj.desc0 = "No Desc";
   an_obj.desc1 = "No Desc";
   an_obj.desc2 = "No Desc";
   an_obj.examine = "No Examine";
   an_obj.poss_type.truncate(0);

   /* get the object name */
   the_token = get_token(read_file, '\0');
   the_token = get_token(read_file, '\0');
   if (the_token->token_type == T_QUOTE)
      the_token = get_token(read_file, '\0');

   if (the_token->token_type != T_IDENTIFIER)
   {
      printf("Error reading name.\n");
      return -1;
   }
   an_obj.name = the_token->the_string;

   the_token = get_token(read_file, '\0'); 
   if (the_token->token_type == T_QUOTE)
      the_token = get_token(read_file, '\0');

   printf("Converting object '%s'\n", an_obj.name.str_show());
   while (the_token->token_type > 0)
   {
      if (ap_get_obj_element(read_file, &the_token) < 0)
         return -1;

      if (!STRCASECMP(the_token->the_string, "EndObject"))
      {
         next_line(read_file);
         break;
      }
   }

   /* we have all the information, now we write it to file */

   ap_write_obj(write_file);

   return 1;
}


/***********************************************************************
 ** ap_read_mobile - reads in a mobile and converts it to AIME format
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **
 ** Returns:  1 for success
 **          -1 for failure
 **
 ***********************************************************************/

int ap_read_mobile(FILE *read_file, FILE *write_file)
{
   token_record *the_token;
   flag_long_int *tmp_flags;

   a_mob.title.truncate(0);
   a_mob.indflags = new Flags(2);
   a_mob.indflags->set_flag(0);  // Set the male flag for now

   a_mob.location = "none";
   a_mob.str = 10;
   a_mob.damage = 15;
   a_mob.speed = 5;
   a_mob.aggression = 0;
   a_mob.armor = 0;
   a_mob.desc = "No Desc";
   a_mob.examine = "No Examine";

   /* get the mobile name */
   the_token = get_token(read_file, '\0');
   the_token = get_token(read_file, '\0');
   if (the_token->token_type == T_QUOTE)
      the_token = get_token(read_file, '\0');

   if (the_token->token_type != T_IDENTIFIER)
   {
      printf("Error reading name\n");
      return -1;
   }
   a_mob.name = the_token->the_string;

   the_token = get_token(read_file, '\0'); 
   if (the_token->token_type == T_QUOTE)
      the_token = get_token(read_file, '\0');

   printf("Converting mobile '%s'\n", a_mob.name.str_show());
   while (the_token->token_type > 0)
   {
      if (ap_get_mob_element(read_file, &the_token) < 0)
         return -1;

      if (!STRCASECMP(the_token->the_string, "EndMobile"))
      {
         next_line(read_file);
         break;
      }
   }

   /* we have all the information, now we write it to file */
   fprintf(write_file, "StartMobile\n");
   fprintf(write_file, "Name ^%s^\n", a_mob.name.str_show());
   fprintf(write_file, "Title ^%s^\n", (a_mob.title.str_show() == NULL) ? 
                            a_mob.name.str_show() : a_mob.title.str_show());
   fprintf(write_file, "Location ^%s^\n", a_mob.location.str_show());
   fprintf(write_file, "Desc ^\n%s^\n", (a_mob.examine.str_show() == NULL) ? 
                                         "" : a_mob.examine.str_show());
   fprintf(write_file, "Brief ^%s^\n", (a_mob.desc.str_show() == NULL) ? 
                                         "" : a_mob.desc.str_show());
   fprintf(write_file, "Strength %d\n", (a_mob.damage * 4)); 

   fprintf(write_file, "Dexterity %d\n", DEF_DEXTERITY);
   fprintf(write_file, "Constitution %d\n", (a_mob.str / 10));
   fprintf(write_file, "Intel %d\n", DEF_INTEL);
   tmp_flags = (a_mob.indflags->get_flagstruct());
   fprintf(write_file, "MobFlag1 %ld\n", tmp_flags->flags);
   tmp_flags = tmp_flags->next_flag;
   fprintf(write_file, "MobFlag2 %ld\n", tmp_flags->flags);
   fprintf(write_file, "Speed %d\n", a_mob.speed);
   fprintf(write_file, "Aggression %d\n", a_mob.aggression);
   fprintf(write_file, "EndMobile\n\n");
   if (a_mob.indflags != NULL)
      delete a_mob.indflags;
   a_mob.indflags = NULL;

   return 1;
}


/***********************************************************************
 ** ap_read_location - reads in a location and converts it to AIME format
 **
 ** Parameters: read_file - the aber file we read from
 **             write_file - the AIME file we write to
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_read_location(FILE *read_file, FILE *write_file)
{
   flag_long_int *tmp_flags;
   token_record *the_token;
   int          i;
   char *dir[10] = {"North", "South", "East", "West", "Up", "Down", "Northeast", "Northwest", 
	   "Southeast", "Southwest"};

   for (i=0; i<10; i++)
      (a_loc.dir[i]).truncate(0);
   a_loc.title = "No Title";
   a_loc.lflags = new Flags(2);
   a_loc.desc = "No Desc";

   /* get the location name */
   the_token = get_token(read_file, '\0');
   the_token = get_token(read_file, '\0');
   if (the_token->token_type == T_QUOTE)
      the_token = get_token(read_file, '\0');

   if (the_token->token_type != T_IDENTIFIER)
   {
      printf("Error reading name\n");
      return -1;
   }
   a_loc.name = the_token->the_string;

   the_token = get_token(read_file, '\0'); 
   if (the_token->token_type == T_QUOTE)
      the_token = get_token(read_file, '\0');

   printf("Converting location '%s'\n", a_loc.name.str_show());
   while (the_token->token_type > 0)
   {
      if (ap_get_loc_element(read_file, &the_token) < 0)
         return -1;

      if (!STRCASECMP(the_token->the_string, "EndLocation"))
      {
         next_line(read_file);
         break;
      }
   }

   /* we have all the information, now we write it to file */
   fprintf(write_file, "StartLocation\n");
   fprintf(write_file, "Name ^%s^\n", a_loc.name.str_show());
   fprintf(write_file, "Title ^%s^\n", a_loc.title.str_show() == NULL ? "" : 
																a_loc.title.str_show());
   fprintf(write_file, "Desc ^\n%s^\n", a_loc.desc.str_show() == NULL ? "" : 
																a_loc.desc.str_show());
   for (i=0; i<10; i++)
   {
      fprintf(write_file, "%s ^%s^\n", dir[i], a_loc.dir[i].str_show() == NULL ? "" : 
																a_loc.dir[i].str_show());
   }

   tmp_flags = (a_loc.lflags->get_flagstruct());
   fprintf(write_file, "LocFlag1 %ld\n", tmp_flags->flags);
   tmp_flags = tmp_flags->next_flag;
   fprintf(write_file, "LocFlag2 %ld\n", tmp_flags->flags);
   fprintf(write_file, "EndLocation\n\n");

   return 1;
}


/***********************************************************************
 ** ap_write_obj - asks the user what kind of object they want to make
 **                this and writes it to file accordingly
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int ap_write_obj(FILE *write_file)
{
#ifndef WIN32
   char user_input[50];
#endif
   int  got_answer = 0;
   int  convert_type;

#ifndef WIN32
   ap_display_obj();
   if ((an_obj.poss_type.str_show() == NULL) ||
       (an_obj.poss_type.str_len() <= 0))
      printf("Unable to determine object type.\n");
   else
      printf("%s\n", an_obj.poss_type.str_show());
#else
   CChooseObjtype dlg;
   Strings holder;

   ap_display_obj(&holder);
   dlg.m_objdata = holder.str_show();
   if (dlg.DoModal() != IDOK)
   {
		return -1;
   }
   
   if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Book"))
   {
		convert_type = OBJ_TYPE_BOOK;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Weapon"))
   {
		convert_type = OBJ_TYPE_WEAPON;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Wearable"))
   {
		convert_type = OBJ_TYPE_WEARABLE;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Key"))
   {
		convert_type = OBJ_TYPE_KEY;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Moveable"))
   {
		convert_type = OBJ_TYPE_MOVEABLE;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Marker"))
   {
		convert_type = OBJ_TYPE_MARKER;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Door"))
   {
		convert_type = OBJ_TYPE_DOOR;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Food"))
   {
		convert_type = OBJ_TYPE_FOOD;
   }
   else if (!STRCASECMP(LPCTSTR(dlg.m_selecttype), "Money"))
   {
		convert_type = OBJ_TYPE_MONEY;
   }
   else
   {
		convert_type = 9999;
   }
#endif

#ifndef WIN32
   while (!got_answer)
   {
      printf("What would you like to convert this to? (? for list) ");
      fflush(stdout);

      fgets(user_input, 50, stdin);
      if (user_input[0] == '?')
      {
         ap_display_options();
      }
      else if (!STRNCASECMP(user_input, "book", 4))
      {
		 convert_type = OBJ_TYPE_BOOK;
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "weapon", 4))
      {
		 convert_type = OBJ_TYPE_WEAPON;
         ap_write_as_weapon(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "wearable", 4))
      {
		 convert_type = OBJ_TYPE_WEARABLE;
         ap_write_as_wearable(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "key", 3))
      {
		 convert_type = OBJ_TYPE_KEY;
         ap_write_as_key(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "moveable", 4))
      {
		 convert_type = OBJ_TYPE_MOVEABLE;
         ap_write_as_moveable(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "marker", 4))
      {
		 convert_type = OBJ_TYPE_MARKER;
         ap_write_as_marker(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "door", 4))
      {
		 convert_type = OBJ_TYPE_DOOR;
         ap_write_as_door(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "food", 4))
      {
		 convert_type = OBJ_TYPE_FOOD;
         ap_write_as_food(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "money", 4))
      {
		 convert_type = OBJ_TYPE_MONEY;
         ap_write_as_money(write_file);
         got_answer = 1;
      }
      else if (!STRNCASECMP(user_input, "none", 4))
      {
	 /* we ignore it, it doesn't get written */
		 convert_type = 9999;
         got_answer = 1;
      }
      else
      {
         printf("That is not a valid option.\n\n");
      }
   } 
#endif

   switch(convert_type)
   {
   case OBJ_TYPE_BOOK:
         ap_write_as_book(write_file);
		 break;

   case OBJ_TYPE_WEAPON:
         ap_write_as_weapon(write_file);
		 break;

   case OBJ_TYPE_WEARABLE:
         ap_write_as_wearable(write_file);
		 break;

   case OBJ_TYPE_KEY:
         ap_write_as_key(write_file);
		 break;

   case OBJ_TYPE_MOVEABLE:
         ap_write_as_moveable(write_file);
		 break;

   case OBJ_TYPE_MARKER:
         ap_write_as_marker(write_file);
		 break;

   case OBJ_TYPE_DOOR:
         ap_write_as_door(write_file);
		 break;

   case OBJ_TYPE_FOOD:
         ap_write_as_food(write_file);
		 break;

   case OBJ_TYPE_MONEY:
         ap_write_as_money(write_file);
		 break;

   default:
	   break;
   }

   return 1;
}


/***********************************************************************
 ** ap_display_options - displays the options that they have to convert
 **                      the object to
 **
 ** Parameters: None
 **
 ** Returns: void
 **
 ***********************************************************************/

void ap_display_options()
{
   printf(
      "\nValid options are:\n"
      "   Book, Door, Key, Marker, Mobile, Moveable, Weapon,\n"
      "   Merger, Money, Wearable, Food, None (does not convert it)\n\n");
}


/***********************************************************************
 ** ap_display_obj - displays the object and it's attributes
 **
 ** Parameters: None
 **
 ** Returns: void
 **
 ***********************************************************************/

void ap_display_obj()
{
   Strings holder;

   ap_display_obj(&holder);
   printf("%s\n", holder.str_show());
}

void ap_display_obj(Strings *holder)
{
	Strings tmp_line;

	tmp_line.sprintf("Object %s%s", an_obj.name.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Title: %s%s", an_obj.title.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Altname: %s%s", an_obj.altname.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Location: %s%s", an_obj.location.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Linked: %s%s", an_obj.linked.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   State: %d%s", an_obj.state, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   MaxState: %d%s", an_obj.maxstate, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   BaseValue: %d%s", an_obj.basevalue, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Damage: %d%s", an_obj.damage, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Armor: %d%s", an_obj.armor, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Size: %d%s", an_obj.size, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Weight: %d%s", an_obj.weight, NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Desc0: %s%s", an_obj.desc0.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Desc1: %s%s", an_obj.desc1.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Desc2: %s%s", an_obj.desc2.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
	tmp_line.sprintf("   Examine: %s%s", an_obj.examine.str_show(), NEWLINE);
	holder->str_cat(tmp_line.str_show());
}


/***********************************************************************
 ** ap_write_mudobj_attrib - writes all the mudobject attributes to file
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_mudobj_attrib(FILE *write_file)
{
   fprintf(write_file, "Title ^%s^\n", (an_obj.title.str_show() == NULL) ? 
                          an_obj.name.str_show() : an_obj.title.str_show());
   fprintf(write_file, "Altname ^%s^\n", (an_obj.altname.str_show() == NULL) ? 
                                                  "" : an_obj.altname.str_show());
   fprintf(write_file, "Location ^%s^\n", (an_obj.location.str_show() == NULL) ? 
                                                  "" : an_obj.location.str_show());
   fprintf(write_file, "Desc ^%s^\n", (an_obj.examine.str_show() == NULL) ? 
                                                  "" : an_obj.examine.str_show());
}


/***********************************************************************
 ** ap_write_item_attrib - writes all the item attributes to file
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_item_attrib(FILE *write_file)
{
   flag_long_int *tmp_flags;

   tmp_flags = (an_obj.itemflags->get_flagstruct());
   fprintf(write_file, "ItemFlag1 %ld\n", tmp_flags->flags);
   tmp_flags = tmp_flags->next_flag;
   fprintf(write_file, "ItemFlag2 %ld\n", tmp_flags->flags);
}

/***********************************************************************
 ** ap_write_moveable_attrib - writes all the moveable attributes to file
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_moveable_attrib(FILE *write_file)
{
   fprintf(write_file, "Brief1 ^%s^\n", (an_obj.desc1.str_show() == NULL) ? 
                   (an_obj.desc0.str_show()) : (an_obj.desc1.str_show()));
   fprintf(write_file, "Brief2 ^%s^\n", (an_obj.desc0.str_show() == NULL) ? 
                   (an_obj.desc1.str_show()) : (an_obj.desc0.str_show()));
   fprintf(write_file, "Size %d\nWeight %d\n", an_obj.size, an_obj.weight);
}


/***********************************************************************
 ** ap_write_merger_attrib - writes all the marker attributes to file
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_merger_attrib(FILE *write_file)
{
}


/***********************************************************************
 ** ap_write_as_weapon - writes this object to file as a weapon obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_weapon(FILE *write_file)
{

   fprintf(write_file, "StartWeapon\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());
   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);

   fprintf(write_file, "Damage %d\n", an_obj.damage);
   fprintf(write_file, "EndWeapon\n\n");   
}


/***********************************************************************
 ** ap_write_as_book - writes this object to file as a book obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_book(FILE *write_file)
{
   fprintf(write_file, "StartBook\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());

   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);

   fprintf(write_file, "Subject ^NoSubject^\n");
   fprintf(write_file, "EndBook\n\n");
}


/***********************************************************************
 ** ap_write_as_wearable - writes this object to file as a wearable obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_wearable(FILE *write_file)
{
   flag_long_int *tmp_flags;

   fprintf(write_file, "StartWearable\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());

   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);

   fprintf(write_file, "Armor %d\n", an_obj.armor);

   tmp_flags = (an_obj.itemflags->get_flagstruct());
   fprintf(write_file, "WearFlag1 %ld\n", tmp_flags->flags);

   fprintf(write_file, "EndWearable\n\n");
   
}


/***********************************************************************
 ** ap_write_as_moveable - writes this object to file as a moveable obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_moveable(FILE *write_file)
{
   fprintf(write_file, "StartMoveable\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());
   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);
   fprintf(write_file, "EndMoveable\n\n");
}


/***********************************************************************
 ** ap_write_as_marker - writes this object to file as a marker obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_marker(FILE *write_file)
{
   fprintf(write_file, "StartMarker\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());

   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   fprintf(write_file, "EndMarker\n\n");
}

/***********************************************************************
 ** ap_write_as_door - writes this object to file as a door obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_door(FILE *write_file)
{
   fprintf(write_file, "StartDoor\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());

   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);

   fprintf(write_file, "Brief1 ^%s^\n", (an_obj.desc1.str_show() == NULL) ? "" : 
												an_obj.desc1.str_show());
   fprintf(write_file, "Brief2 ^%s^\n", (an_obj.desc0.str_show() == NULL) ? "" : 
												an_obj.desc0.str_show());
   fprintf(write_file, "EndDoor\n\n");
}


/***********************************************************************
 ** ap_write_as_food - writes this object to file as a food obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_food(FILE *write_file)
{
   fprintf(write_file, "StartFood\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());

   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);
   ap_write_merger_attrib(write_file);
   fprintf(write_file, "EndFood\n\n");

}



/***********************************************************************
 ** ap_write_as_key - writes this object to file as a key obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_key(FILE *write_file)
{
   fprintf(write_file, "StartKey\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());

   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);
   fprintf(write_file, "EndKey\n\n");
}


/***********************************************************************
 ** ap_write_as_money - writes this object to file as a money obj
 **
 ** Parameters: write_file - the AIME file we write to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void ap_write_as_money(FILE *write_file)
{
   fprintf(write_file, "StartMoney\n");
   fprintf(write_file, "Name ^%s^\n", an_obj.name.str_show());
   ap_write_mudobj_attrib(write_file);
   ap_write_item_attrib(write_file);
   ap_write_moveable_attrib(write_file);
   ap_write_merger_attrib(write_file);
   fprintf(write_file, "EndMoney\n\n");
}

#endif





