#
# Copyright 2001 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#
# 
# Form functionality tests.
#
# $Id: pagemodule.py,v 1.2 2004/01/15 05:31:02 andrewm Exp $

import sys
import os
import unittest
import albatross
import re

class DummyRequest:
    def __init__(self):
        self.__content = []

    def has_field(self, name):
        pass

    def field_names(self):
        return []

    def write_header(self, name, value):
        pass

    def end_headers(self):
        pass

    def write_content(self, data):
        self.__content.append(data)

    def get_content(self):
        return '\n'.join(self.__content)

    def set_status(self, status):
        pass

    def return_code(self):
        pass

class TestContext(albatross.AppContext,
                  albatross.NameRecorderMixin,
                  albatross.SessionBase):

    def __init__(self, app):
        albatross.AppContext.__init__(self, app)
        albatross.NameRecorderMixin.__init__(self)
        albatross.SessionBase.__init__(self)

    def load_session(self):
        pass

    def save_session(self):
        pass

    def remove_session(self):
        pass

class TestApp(albatross.ModularApp):
    def __init__(self, mod_name):
        base_dir = os.path.dirname(sys.modules[__name__].__file__)
        module_dir = os.path.join(base_dir, 'modules')
        albatross.ModularApp.__init__(self, 'foo.py', module_dir,
                                      module_dir, mod_name, 'secret')

    def create_context(self):
        self.__ctx = TestContext(self)
        return self.__ctx

    def get_test_ctx(self):
        return self.__ctx

class PageModuleCase(unittest.TestCase):
    def _test_mod(self, mod_name):
        app = TestApp(mod_name)
        req = DummyRequest()
        app.run(req)
        return req.get_content()

    def _mod_result(self, mod_name, want):
        result = self._test_mod(mod_name)
        self.assertEqual(result, want, 
                         'wanted %r, got:\n%s' % (want, result))

    def _mod_regexp(self, mod_name, pattern):
        result = self._test_mod(mod_name)
        self.failUnless(re.search(pattern, result),
                        'Did not find "%s" in:\n%s' % (pattern, result))

    def missing_module(self):
        self._mod_regexp('nonexistent_module', 
                         'ApplicationError: No module named')

    def simple_page_module(self):
        self._mod_result('simple_module', '')

    def missing_methods(self):
        self._mod_regexp('missing_methods', 
                         'ApplicationError: .*does not define one of')

    def run_template(self):
        self._mod_result('run_template', 'The value of x is "Foo"\n')

    def module_hierarchy(self):
        self._mod_result('submod.module_hierarchy', 'The value of x is "Bah"\n')
        self._mod_result('submod', 'The value of x is "silkiest"\n')

    def module_decode(self):
        """
        pickle and unpickle a session containing an instance of an
        object defined in the page module
        """
        app = TestApp('submod.module_decode')
        req = DummyRequest()
        app.run(req)
        ctx = app.get_test_ctx()
        session = ctx.encode_session()
        # force re-import
        del sys.modules[app.mod_holder_name]
        del sys.modules[app.mod_holder_name + '.submod.module_decode']
        app = TestApp('submod.module_decode')
        # to break
        # app._PageModuleMixin__base_dir = 'poo'
        ctx = app.create_context()
        ctx.decode_session(session)

class PageModuleSuite(unittest.TestSuite):
    test_list = (
        'missing_module',
        'simple_page_module',
        'missing_methods',
        'run_template',
        'module_hierarchy',
        'module_decode',
    )
    def __init__(self):
        unittest.TestSuite.__init__(self, map(PageModuleCase, self.test_list))

def suite():
    return PageModuleSuite()

if __name__ == '__main__':
    unittest.main(defaultTest='suite')
