// ---------------------------------------------------------------------------
// - Socket.cpp                                                              -
// - aleph:net library - socket class implementation                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Socket.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"
#include "cnet.hpp"
#include "csio.hpp"
#include "cerr.hpp"

namespace aleph {

  // the socket eval quarks
  static const long QUARK_BROADCAST = String::intern ("BROADCAST");
  static const long QUARK_DONTROUTE = String::intern ("DONT-ROUTE");
  static const long QUARK_KEEPALIVE = String::intern ("KEEP-ALIVE");
  static const long QUARK_LINGER    = String::intern ("LINGER");
  static const long QUARK_RCVSIZE   = String::intern ("RCV-SIZE");
  static const long QUARK_SNDSIZE   = String::intern ("SND-SIZE");
  static const long QUARK_HOPLIMIT  = String::intern ("HOP-LIMIT");
  static const long QUARK_MCASTLOOP = String::intern ("MULTICAST-LOOPBACK");
  static const long QUARK_MCASTHOP  = String::intern ("MULTICAST-HOP-LIMIT");
  static const long QUARK_MAXSEG    = String::intern ("MAX-SEGMENT-SIZE");
  static const long QUARK_NODELAY   = String::intern ("NO-DELAY");

  // the socket supported quarks
  static const long QUARK_BIND      = String::intern ("bind");
  static const long QUARK_EOFP      = String::intern ("eof-p");
  static const long QUARK_READ      = String::intern ("read");
  static const long QUARK_PUSHB     = String::intern ("pushback");
  static const long QUARK_CLOSE     = String::intern ("close");
  static const long QUARK_WRITE     = String::intern ("write");
  static const long QUARK_SETOPT    = String::intern ("set-option");
  static const long QUARK_IPV6P     = String::intern ("ipv6-p");
  static const long QUARK_CONNECT   = String::intern ("connect");
  static const long QUARK_READLN    = String::intern ("readln");
  static const long QUARK_BUFLEN    = String::intern ("get-buffer-length");
  static const long QUARK_VALIDP    = String::intern ("valid-p");
  static const long QUARK_WRITELN   = String::intern ("writeln");
  static const long QUARK_NEWLINE   = String::intern ("newline");
  static const long QUARK_SHUTDOWN  = String::intern ("shutdown");
  static const long QUARK_SOCKADDR  = String::intern ("get-socket-address");
  static const long QUARK_SOCKPORT  = String::intern ("get-socket-port");
  static const long QUARK_PEERADDR  = String::intern ("get-peer-address");
  static const long QUARK_PEERPORT  = String::intern ("get-peer-port");
  static const long QUARK_SETOPTION = String::intern ("set-option");

  // create a default socket

  Socket::Socket (void) {
    d_sid = -1;
  }

  // create a socket by id

  Socket::Socket (const int sid) {
    d_sid = sid;
  }

  // destroy this socket

  Socket::~Socket (void) {
    close ();
  }

  // return the class name

  String Socket::repr (void) const {
    return "Socket";
  }

  // return true if we have an ipv6 socket

  bool Socket::isipv6 (void) const {
    return c_isipv6 (d_sid);
  }

  // set a socket option

  bool Socket::setopt (t_option opt, bool flag) {
    wrlock ();
    bool result = false;
    switch (opt) {
    case SOCK_BROADCAST:
      result = c_ipsetopt (d_sid, SOCKET_BROADCAST, flag, 0);
      break;
    case SOCK_DONTROUTE:
      result = c_ipsetopt (d_sid, SOCKET_DONTROUTE, flag, 0);
      break;
    case SOCK_KEEPALIVE:
      result = c_ipsetopt (d_sid, SOCKET_KEEPALIVE, flag, 0);
      break;
    case SOCK_MCASTLOOP:
      result = c_ipsetopt (d_sid, SOCKET_MCASTLOOP, flag, 0);
      break;
    case SOCK_NODELAY:
      result = c_ipsetopt (d_sid, SOCKET_NODELAY,   flag, 0);
      break;
    default:
      break;
    }
    unlock ();
    return result;
  }

  // set a socket option with a value

  bool Socket::setopt (t_option opt, bool flag, long val) {
    wrlock ();
    bool result = false;
    switch (opt) {
    case SOCK_LINGER:
      result = c_ipsetopt (d_sid, SOCKET_LINGER,   flag, val);
      break;
    case SOCK_RCVSIZE:
      result = c_ipsetopt (d_sid, SOCKET_RCVSIZE,  flag, val);
      break;
    case SOCK_SNDSIZE:
      result = c_ipsetopt (d_sid, SOCKET_SNDSIZE,  flag, val);
      break;
    case SOCK_HOPLIMIT:
      result = c_ipsetopt (d_sid, SOCKET_HOPLIMIT, flag, val);
      break;
    case SOCK_MCASTHOP:
      result = c_ipsetopt (d_sid, SOCKET_MCASTHOP, flag, val);
      break;
    case SOCK_MAXSEG:
      result = c_ipsetopt (d_sid, SOCKET_MAXSEG,   flag, val);
      break;
    default:
      break;
    }
    unlock ();
    return result;
  }

  // connect this socket by port and address

  bool Socket::connect (t_word port, const Address& addr) {
    return c_ipconnect (d_sid, port, addr.p_addr);
  }

  // bind a socket with a port

  bool Socket::bind (t_word port) {
    return c_ipbind (d_sid, port);
  }

  // bind a socket with a port and an address

  bool Socket::bind (t_word port, const Address& addr) {
    return c_ipbind (d_sid, port, addr.p_addr);
  }

  // close this socket 
  
  bool Socket::close (void) {
    wrlock ();
    if ((d_sid == -1) || (Object::uref (this) == false)) {
      unlock ();
      return true;
    }
    if (c_close (d_sid) == false) {
      unlock ();
      return false;
    }
    d_sid = -1;
    unlock ();
    return true;
  }

  // shutdown this socket

  bool Socket::shutdown (const bool mode) {
    wrlock ();
    bool result = false;
    if (mode == false) result = c_ipshut (d_sid, SOCKET_SHUT_RECV);
    if (mode == true)  result = c_ipshut (d_sid, SOCKET_SHUT_SEND);
    unlock ();
    return result;
  }

  // return the socket address

  Address* Socket::getsockaddr (void) const {
    rdlock ();
    t_byte* addr = c_ipsockaddr (d_sid);
    unlock ();
    if (addr == nilp) return nilp;
    return new Address (addr);
  }

  // return the socket port

  t_word Socket::getsockport (void) const {
    rdlock ();
    t_word result = c_ipsockport (d_sid);
    unlock ();
    return result;
  }

  // return the peer address

  Address* Socket::getpeeraddr (void) const {
    rdlock ();
    t_byte* addr = c_ippeeraddr (d_sid);
    unlock ();
    if (addr == nilp) return nilp;
    return new Address (addr);
  }

  // return the peer port

  t_word Socket::getpeerport (void) const {
    rdlock ();
    t_word result = c_ippeerport (d_sid);
    unlock ();
    return result;
  }

  // evaluate a quark statically

  Object* Socket::meval (Runnable* robj, Nameset* nset, const long quark) {
    if (quark == QUARK_BROADCAST) return new Integer (SOCK_BROADCAST);
    if (quark == QUARK_DONTROUTE) return new Integer (SOCK_DONTROUTE);
    if (quark == QUARK_KEEPALIVE) return new Integer (SOCK_KEEPALIVE);
    if (quark == QUARK_LINGER)    return new Integer (SOCK_LINGER);
    if (quark == QUARK_RCVSIZE)   return new Integer (SOCK_RCVSIZE);
    if (quark == QUARK_SNDSIZE)   return new Integer (SOCK_SNDSIZE);
    if (quark == QUARK_HOPLIMIT)  return new Integer (SOCK_HOPLIMIT);
    if (quark == QUARK_MCASTLOOP) return new Integer (SOCK_MCASTLOOP);
    if (quark == QUARK_MCASTHOP)  return new Integer (SOCK_MCASTHOP);
    if (quark == QUARK_MAXSEG)    return new Integer (SOCK_MAXSEG);
    if (quark == QUARK_NODELAY)   return new Integer (SOCK_NODELAY);
    throw Exception ("eval-error", "cannot evaluate member",
		     String::qmap (quark));
  }

  // apply this socket with a set of arguments and a quark

  Object* Socket::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {

    // dispatch derived arguments
    if (quark == QUARK_READ)    return Input::apply  (robj, nset, quark, argv);
    if (quark == QUARK_EOFP)    return Input::apply  (robj, nset, quark, argv);
    if (quark == QUARK_PUSHB)   return Input::apply  (robj, nset, quark, argv);
    if (quark == QUARK_READLN)  return Input::apply  (robj, nset, quark, argv);
    if (quark == QUARK_VALIDP)  return Input::apply  (robj, nset, quark, argv);
    if (quark == QUARK_BUFLEN)  return Input::apply  (robj, nset, quark, argv);

    if (quark == QUARK_WRITE)   return Output::apply (robj, nset, quark, argv);
    if (quark == QUARK_WRITELN) return Output::apply (robj, nset, quark, argv);
    if (quark == QUARK_NEWLINE) return Output::apply (robj, nset, quark, argv);

    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_IPV6P)    return new Boolean (isipv6 ());
      if (quark == QUARK_SOCKADDR) return getsockaddr ();
      if (quark == QUARK_PEERADDR) return getpeeraddr ();
      if (quark == QUARK_SOCKPORT) return new Integer (getsockport ());
      if (quark == QUARK_PEERPORT) return new Integer (getpeerport ());
      if (quark == QUARK_CLOSE)    return new Boolean (close       ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_BIND) {
	long port = argv->getint (0);
	bind (port);
	return nilp;
      }
      if (quark == QUARK_SHUTDOWN) {
	bool mode = argv->getbool (0);
	return new Boolean (shutdown (mode));
      }
    }

    // dispatch 2 argument
    if (argc == 2) {
      if (quark == QUARK_BIND) {
	long port = argv->getint (0);
	Address* addr = dynamic_cast <Address*> (argv->get (1));
	if (addr == nilp) 
	  throw Exception ("argument-error", "address expected with bind");
	bind (port, *addr);
	return nilp;
      }
      if (quark == QUARK_CONNECT) {
	long port = argv->getint (0);
	Address* addr = dynamic_cast <Address*> (argv->get (1));
	if (addr == nilp) 
	  throw Exception ("argument-error", "address expected with connect");
	connect (port, *addr);
	return nilp;
      }
      if (quark == QUARK_SETOPTION) {
	long opt = argv->getint (0);
	Object* obj = argv->get (1);
	Boolean* bobj = dynamic_cast <Boolean*> (obj);
	if (bobj != nilp) {
	  bool flg = bobj->toboolean ();
	  return new Boolean (setopt ((t_option) opt, flg));
	}
	Integer* iobj = dynamic_cast <Integer*> (obj);
	if (iobj != nilp) {
	  long val = iobj->tointeger (); 
	  return new Boolean (setopt ((t_option) opt, true, val));
	}
	throw Exception ("argument-error", "invalid argument with set-option");
      }
    }

    // dispatch 3 argument
    if (argc == 3) {
      if (quark == QUARK_SETOPTION) {
	long opt = argv->getint  (0);
	bool flg = argv->getbool (1);
	long val = argv->getint  (2);
	return new Boolean (setopt ((t_option) opt, flg, val));
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
