// ---------------------------------------------------------------------------
// - CgiQuery.cpp                                                            -
// - aleph:www library - cgi query class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Buffer.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "CgiQuery.hpp"
#include "Exception.hpp"

namespace aleph {
  // the cgi query supported quarks
  static const long QUARK_GET      = String::intern ("get");
  static const long QUARK_PARSE    = String::intern ("parse");
  static const long QUARK_EXISTS   = String::intern ("exists-p");
  static const long QUARK_LENGTH   = String::intern ("length");
  static const long QUARK_LOOKUP   = String::intern ("lookup");
  static const long QUARK_GETNAME  = String::intern ("get-name");
  static const long QUARK_GETVALUE = String::intern ("get-value");
  static const long QUARK_GETQUERY = String::intern ("get-query");

  // this procedure map an hexadecimal character to its value
  static inline t_byte htob (const char c) {
    if ((c >= '0') && (c <= '9')) return (c - '0');
    if ((c >= 'a') && (c <= 'f')) return (c - 'a' + 10);
    if ((c >= 'A') && (c <= 'F')) return (c - 'A' + 10);
    throw Exception ("cgi query-error", "invalid escape character", c);
  }

  // this procedure fix a query name by converting the escape characters
  static String qtohs (const String& name) {
    long len = name.length ();
    if (len == 0) return name;
    // create a buffer and loop
    Buffer buf;
    for (long i = 0; i < len; i++) {
      char c = name[i];
      if (c == '+') {
	buf.add (' ');
	continue;
      }
      if (c == '%') {
	t_byte val = 0;
	if (++i < len) val  = htob (name[i]) << 4;
	if (++i < len) val += htob (name[i]);
	if (i >= len) 
	  throw Exception ("cgi query-error", "unterminated escape sequence");
	c = (char) val;
      }
      buf.add (c);
    }
    return buf.tostring ();
  }

  // create a default query

  CgiQuery::CgiQuery (void) {
    reset ();
  }

  // create a cgi query by name

  CgiQuery::CgiQuery (const String& query) {
    reset ();
    parse (query);
  }

  // return the class name

  String CgiQuery::repr (void) const {
    return "CgiQuery";
  }

  // reset an cgi query information

  void CgiQuery::reset (void) {
    wrlock ();
    d_query = "";
    d_names.reset ();
    d_value.reset ();
    unlock ();
  }

  // parse an cgi query and update the cgi query data structure

  void CgiQuery::parse (const String& query) {
    // check for nothing first
    long len = query.length ();
    if (len == 0) {
      reset ();
      return;
    }
    // ok let's deal with it
    wrlock ();
    reset  ();
    try {
      // get the name/pair value
      Strvec npvec = Strvec::split (query, "&");
      // now split each pair
      long nplen = npvec.length ();
      for (long i = 0; i < nplen; i++) {
	String pair = npvec.get (i);
	Strvec data = Strvec::split (pair, "=");
	long len = data.length ();
	if (len == 0) continue;
	if (len == 1) {
	  d_names.add (qtohs (data.get (0)));
	  d_value.add ("");
	  continue;
	}
	if (len == 2) {
	  d_names.add (qtohs (data.get (0)));
	  d_value.add (qtohs (data.get (1)));
	  continue;
	}
	throw Exception ("query-error", "invalid query string", query);
      }
      // check that both length are equal
      if (d_names.length () != d_value.length ())
	throw Exception ("query-error", "cannot parse query string", query);
      d_query = query;
      unlock ();
    } catch (...) {
      reset  ();
      unlock ();
      throw;
    }
  }

  // return the number of name-pair values

  long CgiQuery::length (void) const {
    rdlock ();
    long result = d_names.length ();
    unlock ();
    return result;
  }

  // return true if a name exists

  bool CgiQuery::exists (const String& name) const {
    rdlock ();
    bool result = d_names.exists (name);
    unlock ();
    return result;
  }

  // return the name index by name

  long CgiQuery::lookup (const String& name) const {
    rdlock ();
    try {
      long result = d_names.index (name);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a value by name

  String CgiQuery::get (const String& name) const {
    rdlock ();
    long index = d_names.lookup (name);
    if (index == -1) {
      unlock ();
      return "";
    }
    String result = d_value.get (index);
    unlock ();
    return result;
  }


  // return a name by index

  String CgiQuery::getname (const long index) const {
    rdlock ();
    try {
      String result = d_names.get (index);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return a value by index

  String CgiQuery::getvalue (const long index) const {
    rdlock ();
    try {
      String result = d_value.get (index);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // create a new cgi query object in a generic way

  Object* CgiQuery::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new CgiQuery;

    // check for one argument
    if (argc == 1) {
      String query = argv->getstring (0);
      return new CgiQuery (query);
    }
    
    // wrong arguments
    throw Exception ("argument-error", "invalid arguments with cgi query");
  }

  // apply this cgi query object with a set of arguments and a quark

  Object* CgiQuery::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH)   return new Integer (length ());
      if (quark == QUARK_GETQUERY) return new String  (d_query);
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GET) 
	return new String (get (argv->getstring (0)));
      if (quark == QUARK_EXISTS)
	return new Boolean (exists (argv->getstring (0)));
      if (quark == QUARK_LOOKUP) 
	return new Integer (lookup (argv->getstring (0)));
      if (quark == QUARK_GETNAME) 
	return new String (getname (argv->getint (0)));
      if (quark == QUARK_GETVALUE) 
	return new String (getvalue (argv->getint (0)));
      if (quark == QUARK_PARSE) {
	String query = argv->getstring (0);
	parse (query);
	return nilp;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
