// ---------------------------------------------------------------------------
// - cerr.cxx                                                                -
// - standard system library - c error system function implementation        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cerr.hpp"
#include "csys.hpp"
#include "csio.hpp"
#include "cstr.hpp"
#include "cerr.hxx"

namespace aleph {
  // the static array of message
  static const char* aleph_err_list[] = {
    "invalid arguments",          // ALEPH_ERR_IARG
    "interrupt received",         // ALEPH_ERR_INTR
    "privilege violations",       // ALEPH_ERR_PRIV
    "name is too long",           // ALEPH_ERR_LONG
    "connection refused",         // ALEPH_ERR_CREF
    "network is unreachable",     // ALEPH_ERR_UNRE
    "resource has timed-out",     // ALEPH_ERR_TOUT
    "resource already in use",    // ALEPH_ERR_IUSE
    "cannot find protocol",       // ALEPH_ERR_POTO
    "invalid inet address",       // ALEPH_ERR_ADDR
    "end point is not connected"  // ALEPH_ERR_UCON
  };

  // return a string representation of the error

  const char* c_errmsg (const long eid) {
    if ((eid <= ALEPH_ERR_EMAX) || (eid >= 0)) return nilp;
    return aleph_err_list[-eid - 1];
  }

  // this function print an bort message in case of unexpected exception
  static void abort_unexpected (void) {
    int eid         = c_stderr ();
    const char* msg = "fatal: unexpected exception - aborting\n";
    c_write (eid, msg, c_strlen (msg));
    c_abort ();
  }

  // set an unexpected handler

  void c_errsetexpt (t_errhandler func) {
    if (func == nilp)
      std::set_unexpected (abort_unexpected);
    else
      std::set_unexpected (func);
  }
}

#ifdef ALEPH_HAVE_ERRNO
namespace aleph {
  // map the unix errno to aleph error
  long c_errmap (const long val) {
    long result = ALEPH_ERR_IARG;
    switch (val) {
    case EINTR:
      result = ALEPH_ERR_INTR;
      break;
    case EACCES:
      result = ALEPH_ERR_PRIV;
      break;
    case ENAMETOOLONG:
      result = ALEPH_ERR_LONG;
      break;
    case ECONNREFUSED:
      result = ALEPH_ERR_CREF;
      break;
    case ENETUNREACH:
      result = ALEPH_ERR_UNRE;
      break;
    case ETIMEDOUT:
      result = ALEPH_ERR_TOUT;
      break;
    case EADDRINUSE:
      result = ALEPH_ERR_IUSE;
      break;
    case ENOTCONN:
      result = ALEPH_ERR_UCON;
      break;
    default:
      break;
    }
    return result;
  }
}
#else
namespace aleph {
  long c_errmap (const long val) {
    return ALEPH_ERR_IARG;
  }
}
#endif
