// ---------------------------------------------------------------------------
// - csys.cxx                                                                -
// - standard system library - c system function implementation              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "csys.hpp"
#include "cstr.hpp"
#include "cthr.hpp"
#include "csys.hxx"

namespace aleph {

  // mutex for the temporary name
  static void* mtx = nilp;
  static char* pgm = nilp;

  // this function destroy the mutex
  static void mtx_destroy (void) {
    c_mtxdestroy (mtx);
  }
  // this function destroy the program name
  static void pgm_destroy (void) {
    delete [] pgm;
  }

  // set the program name

  void c_setpgm (const char* name) {
    if (pgm == nilp) c_atexit (pgm_destroy);
    if (pgm != nilp) delete [] pgm;
    pgm = c_strdup (name);
  }

  // return the program name
  
  char* c_getpgm (void) {
    return c_strdup (pgm);
  }

  // return a temporary name

  char* c_tmpname (void) {
    static int tcnt = 0;
    // get the mutex if needed
    if (mtx == nilp) {
      mtx = c_mtxcreate ();
      c_atexit (mtx_destroy);
    }
    c_mtxlock (mtx);
    char buffer[512];
    sprintf (buffer, "aleph-tmp-%ld-%d", c_getpid (), tcnt++);
    c_mtxunlock (mtx);
    return c_strdup (buffer);
  }

  // return the os name
  const char* c_osname (void) {
    return ALEPH_PLATFORM_NAME;
  }

  // return the os type

  const char* c_ostype (void) {
    return ALEPH_PLATFORM_TYPE;
  }

  // return the native option separator

  const char c_optsep (void) {
    return ALEPH_PLATFORM_OSEP;
  }

  // exit without executing the register code

  void c_abort (void) {
    _exit (1);
  }

  // exit unconditionnaly with a status code

  void c_exit (int status) {
    exit (status);
  }

  // register a function to execute at exit

  void c_atexit (void (*func) (void)) {
    atexit (func);
  }

  // return the process id

  long c_getpid (void) {
    return getpid ();
  }

  // return an environment variable value

  const char* c_getenv (const char* name) {
    if (c_strlen (name) == 0) return nilp;
    return getenv (name);
  }
}

#ifdef ALEPH_HAVE_HOSTNAME
namespace aleph {
  // return the host name
  char* c_hostname (void) {
    char name[1024];
    if (gethostname (name,1024) != 0) return nilp;
    return c_strdup (name);
  }
}
#endif

#ifdef ALEPH_HAVE_SYSINFO
namespace aleph {
  // return the host name
  char* c_hostname (void) {
    char name[1024];
    if (sysinfo (SI_HOSTNAME, name,1024) == -1) return nilp;
    return c_strdup (name);
  }
}
#endif

#ifdef ALEPH_HAVE_UID
namespace aleph {
  // return the user name
  char* c_username (void) {
    // get the user id
    uid_t uid = getuid ();
    // get the password line entry
    struct passwd* data = getpwuid (uid);
    if (data == nilp) return c_strdup (c_getenv ("USER"));
    return c_strdup (data->pw_name);
  }
}
#endif

#ifdef ALEPH_HAVE_BACKTRACE
#ifdef  __GNUC__
#define GET_CALLER_FRAME __builtin_frame_address(1);
#else
#define GET_CALLER_FRAME 0
#endif

namespace aleph {
  // the begining of the frame
  struct s_frame {
    s_frame*    p_prev;
    void*       p_pc;
  };

  struct s_finfo {
    void*       p_pc;
    int         d_fidx;
    const char* p_name;
    s_finfo*    p_prev;
    // simple constructor
    s_finfo (void) {
      p_pc   = 0;
      p_name = 0;
      p_prev = 0;
      d_fidx = 0;
    }
    // simple destructor
    ~s_finfo (void) {
      delete p_name;
      delete p_prev;
    } 
  };

  // this function resolve the pc name from its address
  static inline const char* get_pc_name (void* pc) {
    if (pc == 0) return 0;
    Dl_info dlinfo;
    if (dladdr (pc, &dlinfo) == 0) return 0;
    return c_strdup (dlinfo.dli_sname);
  }

  // return the backtrace of the calling function
  void* c_backtrace (void) {
    // get the caller frame
    s_frame* frame = (s_frame*) GET_CALLER_FRAME;
    s_finfo* finfo = 0;
    int      index = 0;

    // loop in the frame
    while (frame != 0) {
      s_finfo* data = new s_finfo;
      data->p_pc   = frame->p_pc;
      data->p_name = get_pc_name (data->p_pc);
      data->p_prev = finfo;
      data->d_fidx = index++;
      finfo = data;
      frame = frame->p_prev;
      if (data->p_name == 0) break;
    }
    return finfo;
  }

  // print the stack trace
  void c_printtrace (void* bptr) {
    if (bptr == 0) return;
    // get to the first frame
    s_finfo* finfo = (s_finfo*) bptr;
    if (finfo->p_prev != 0) c_printtrace (finfo->p_prev);
    // print frame info
    int fidx = finfo->d_fidx;
    if (fidx == 0)
      fprintf (stderr, "\n\t%d: %s\n", fidx, finfo->p_name);
    else
      fprintf (stderr, "\t%d: %s\n", fidx, finfo->p_name);
  }

  // destroy a stack trace
  void c_destroytrace (void* bptr) {
    if (bptr == 0) return;
    s_finfo* finfo = (s_finfo*) bptr;
    delete finfo;
  }
}
#else
namespace aleph {
  // return the stack trace
  void* c_backtrace (void) {
    return 0;
  }

  // print the stack trace
  void c_printtrace (void*) {
    return;
  }

  // destroy a stack trace
  void c_destroytrace (void*) {
    return;
  }
}
#endif
