// ---------------------------------------------------------------------------
// - Character.cpp                                                           -
// - standard object library - character class implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "Output.hpp"
#include "Vector.hpp"
#include "Recycle.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // the character supported quarks
  static const long QUARK_OPP      = String::intern ("++");
  static const long QUARK_OMM      = String::intern ("--");
  static const long QUARK_ADD      = String::intern ("+");
  static const long QUARK_SUB      = String::intern ("-");
  static const long QUARK_EQL      = String::intern ("==");
  static const long QUARK_NEQ      = String::intern ("!=");
  static const long QUARK_LTH      = String::intern ("<");
  static const long QUARK_LEQ      = String::intern ("<=");
  static const long QUARK_GTH      = String::intern (">");
  static const long QUARK_GEQ      = String::intern (">=");
  static const long QUARK_AEQ      = String::intern ("+=");
  static const long QUARK_SEQ      = String::intern ("-=");
  static const long QUARK_EOLP     = String::intern ("eol-p");
  static const long QUARK_EOFP     = String::intern ("eof-p");
  static const long QUARK_NILP     = String::intern ("nil-p");
  static const long QUARK_TOINT    = String::intern ("to-integer");
  static const long QUARK_ALPHAP   = String::intern ("alpha-p");
  static const long QUARK_DIGITP   = String::intern ("digit-p");
  static const long QUARK_BLANKP   = String::intern ("blank-p");

  // the character recycler
  static Recycle recycler;

  // allocate a new character
  void* Character::operator new (const t_size size) {
    return recycler.pop (size);
  }

  // delete a character
  void Character::operator delete (void* handle) {
    recycler.push (handle);
  }

  // create a new default character

  Character::Character (void) {
    d_value = nilc;
  }

  // create a new character from a native character

  Character::Character (const char value) {
    d_value = value;
  }
  
  // create a new character from a string representation

  Character::Character (const String& value) {
    if (value.length () == 1) {
      d_value = value[0];
    } else if ((value.length () == 3) && (value[0] == '\'') && 
	       (value[2] == '\'')) {
      d_value = value[1];
    } else
      throw Exception ("format-error",
		       "illegal character representation",value);
  }

  // copy constructor for this character class

  Character::Character (const Character& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Character::repr (void) const {
    return "Character";
  }

  // return a literal representation of this character

  String Character::toliteral (void) const {
    String quote = "'";
    return (quote + d_value + quote);
  }

  // return a string representation of this character

  String Character::tostring (void) const {
    String result = d_value;
    return result;
  }

  // return a clone of this character

  Object* Character::clone (void) const {
    return new Character (*this);
  }

  // return the character serial code

  t_byte Character::serialid (void) const {
    return SERIAL_CHAR_ID;
  }

  // serialize this character

  void Character::wrstream (Output& os) const {
    rdlock ();
    os.write (d_value);
    unlock ();
  }

  // deserialize this character

  void Character::rdstream (Input& is) {
    wrlock ();
    d_value = is.read ();
    unlock ();
  }

  // return a native character

  char Character::tochar (void) const {
    return d_value;
  }

  // set an character with a native value

  Character& Character::operator = (const char value) {
    d_value = value;
    return *this;
  }

  // set an character with a character

  Character& Character::operator = (const Character& value) {
    d_value = value.d_value;
    return *this;
  }

  // compare an character with a native value
  
  bool Character::operator == (const char value) const {
    return (d_value == value);
  }
  
  bool Character::operator != (const char value) const {
    return (d_value != value);
  }
  
  // compare two character characters
  
  bool Character::operator == (const Character& value) const {
    return (d_value == value.d_value);
  }
  
  bool Character::operator != (const Character& value) const {
    return (d_value != value.d_value);
  }
  
  bool Character::operator < (const Character& value) const {
    return (d_value < value.d_value);
  }
  
  bool Character::operator <= (const Character& value) const {
    return (d_value <= value.d_value);
  }
  
  bool Character::operator > (const Character& value) const {
    return (d_value > value.d_value);
  }
  
  bool Character::operator >= (const Character& value) const {
    return (d_value >= value.d_value);
  }
  
  // return true if the character is alpha

  bool Character::isalpha (void) const {
    bool result = ((d_value > 'a') && (d_value < 'z')) ||
      ((d_value > 'A') && (d_value < 'Z'));
    return result;
  }

  // return true if the character is a digit

  bool Character::isdigit (void) const {
    return ((d_value >= '0') && (d_value <= '9'));
  }

  // return true if the character is a blank or tab
  
  bool Character::isblank (void) const {
    return ((d_value == ' ') || (d_value == '\t'));
  }

  // return true if the character is an eol

  bool Character::iseol (void) const {
    return (d_value == eolc);
  }
  
  // return true if the character is an eof

  bool Character::iseof (void) const {
    return (d_value == eofc);
  }

  // return true if the character is nil

  bool Character::isnil (void) const {
    return (d_value == nilc);
  }

  // evaluate an object to a character value

  char Character::evalto (Runnable* robj, Nameset* nset, Object* object) {
    Object*    obj = (object == nilp) ? nilp : object->eval (robj, nset);
    Character* val = dynamic_cast <Character*> (obj);
    if (val == nilp) throw Exception ("type-error", "nil object to evaluate");
    return val->tochar ();
  }

  // create a new character in a generic way

  Object* Character::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Character;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with character constructor");
    // try to map the character argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Character;

    // try an integer object
    Integer* ival = dynamic_cast <Integer*> (obj);
    if (ival != nilp) return new Character (ival->tointeger ());

    // try a character object
    Character* cval = dynamic_cast <Character*> (obj);
    if (cval != nilp) return new Character (*cval);

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Character (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with character constructor",
		     obj->repr ());
  }

  // operate this character with another object

  Object* Character::oper (Runnable* robj, t_oper type, Object* object) {
    Integer*   iobj = dynamic_cast <Integer*>   (object);
    Character* cobj = dynamic_cast <Character*> (object);
    switch (type) {
    case Object::ADD:
      if (iobj != nilp) return new Character (d_value + iobj->tointeger ());
      break;
    case Object::SUB:
      if (iobj != nilp) return new Character (d_value - iobj->tointeger ());
      break;
    case Object::EQL:
      if (cobj != nilp) return new Boolean (d_value == cobj->d_value);
      break;
    case Object::NEQ:
      if (cobj != nilp) return new Boolean (d_value != cobj->d_value);
      break;
    case Object::GEQ:
      if (cobj != nilp) return new Boolean (d_value >= cobj->d_value);
      break;
    case Object::GTH:
      if (cobj != nilp) return new Boolean (d_value > cobj->d_value);
      break;
    case Object::LEQ:
      if (cobj != nilp) return new Boolean (d_value <= cobj->d_value);
      break;
    case Object::LTH:
      if (cobj != nilp) return new Boolean (d_value < cobj->d_value);
      break;
    default:
      throw Exception ("operator-error", "unsupported character operator");
    }
    throw Exception ("type-error", "invalid operand with character",
		     Object::repr (object));
  }

  // set an object to this character

  Object* Character::vdef (Runnable* robj, Nameset* nset, Object* object) {
    Character* cobj = dynamic_cast <Character*> (object);
    if (cobj != nilp) {
      d_value = cobj->d_value;
      return this;
    }
    throw Exception ("type-error", "invalid object with character vdef",
		     Object::repr (object));
  }

  // apply this character with a set of arguments and a quark

  Object* Character::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_TOINT) return new Integer (tochar ());
      
      if (quark == QUARK_OPP) {
	d_value++;
	return this;
      }
      if (quark == QUARK_OMM) {
	d_value--;
	return this;
      }
      if (quark == QUARK_ALPHAP) return new Boolean (isalpha ());
      if (quark == QUARK_DIGITP) return new Boolean (isdigit ());
      if (quark == QUARK_BLANKP) return new Boolean (isblank ());
      if (quark == QUARK_EOLP)   return new Boolean (iseol ());
      if (quark == QUARK_EOFP)   return new Boolean (iseof ());
      if (quark == QUARK_NILP)   return new Boolean (isnil ());
    }

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_ADD) return oper (robj, Object::ADD, argv->get (0));
      if (quark == QUARK_SUB) return oper (robj, Object::SUB, argv->get (0));
      if (quark == QUARK_EQL) return oper (robj, Object::EQL, argv->get (0));
      if (quark == QUARK_NEQ) return oper (robj, Object::NEQ, argv->get (0));
      if (quark == QUARK_LTH) return oper (robj, Object::LTH, argv->get (0));
      if (quark == QUARK_LEQ) return oper (robj, Object::LEQ, argv->get (0));
      if (quark == QUARK_GTH) return oper (robj, Object::GTH, argv->get (0));
      if (quark == QUARK_GEQ) return oper (robj, Object::GEQ, argv->get (0));

      if (quark == QUARK_AEQ) {
	t_long val = argv->getint (0);
	d_value += val;
	return this;
      }
      if (quark == QUARK_SEQ) {
	t_long val = argv->getint (0);
	d_value -= val;
	return this;
      }
    }

    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
