// ---------------------------------------------------------------------------
// - OutputString.hpp                                                        -
// - standard object library - output string stream class definition         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_OUTPUTSTRING_HPP
#define  ALEPH_OUTPUTSTRING_HPP

#ifndef  ALEPH_OUTPUT_HPP
#include "Output.hpp"
#endif

#ifndef  ALEPH_BUFFER_HPP
#include "Buffer.hpp"
#endif

namespace aleph {

  /// The OutputString class offers a facility to treat a string buffer like
  /// a string. Each call to the write method accumulates characters in the 
  /// string buffer. When the tostring method is called, the accumulated string
  /// is returned and the buffer is flushed. The OutputString class is very
  /// usefull to perform stream based testing.
  /// @author amaury darsch

  class OutputString: public Output {
  private:
    /// the string buffer
    Buffer d_buffer;

  public:
    /// create a new string stream. 
    OutputString (void);

    /// create a new string stream with a value
    /// @param value the string initial value
    OutputString (const String& value);

    /// @return the class name
    String repr (void) const;

    /// write one character on the string stream.
    /// @param value the character to write
    void write (const char value);

    /// write a character string to the output stream
    /// @param value the character string to write  
    void write (const char* value);

    /// @return the length of this output string buffer
    long length (void) const;

    /// @return a string representation of this buffer and flush
    String tostring (void);

    /// flush this string buffer
    void flush (void);

    /// generate a new output string
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this output string with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the copy constructor private
    OutputString (const OutputString&);
    // make the assignment operator private
    OutputString& operator = (const OutputString&);
  };
}

#endif
