// ---------------------------------------------------------------------------
// - Stack.hpp                                                               -
// - standard object library - object stack class definition                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_STACK_HPP
#define  ALEPH_STACK_HPP

#ifndef  ALEPH_OBJECT_HPP
#include "Object.hpp"
#endif

namespace aleph {

  /// The Stack class is a simle stack for the aleph standard object. The
  /// stack is dynamically resized as needed when objects are pushed onto it.
  /// The push and pop methods are the standard methods to manipulate the
  /// stack. The unwind method pop all objects from the stack.
  /// @author amaury darsch

  class Stack : public virtual Object {
  private:
    /// the stack size in bytes
    long d_size;
    /// the base of the stack
    Object** p_base;
    /// the top of the stack
    Object** p_top;
    /// the current stack pointer
    Object** p_sp;
    /// the current frame pointer
    Object** p_fp;

  public:
    /// create a default stack
    Stack (void);

    /// destroy this stack
    ~Stack (void);

    /// @return the class name
    String repr (void) const;

    /// push an object on this stack
    /// @param object the object to push
    void push (Object* object);

    /// @return the object on top of the stack
    Object* pop (void);

    /// @return the integer from a stack object
    long popint (void);

    /// @return true if the stack is empty
    bool empty (void) const;

    /// @return the current stack pointer
    Object** getsp (void) const {
      return p_sp;
    }

    /// @return the current frame pointer
    Object** getfp (void) const {
      return p_fp;
    }
 
    /// set the frame pointer
    /// @param the new frame pointer
    void setfp (Object** sp);

    /// unwind this stack completly
    void unwind (void);
 
    /// unwind this stack upto the old sp and restore the fp
    /// @param sp the stack pointer ti unwind
    /// @param fp the frame pointer to restore
    void unwind (Object** sp, Object** fp);

    /// @return an object in reference to the fp
    Object* get (const long index) const;
 
    /// set a stack value by index
    /// @param index the stack index
    /// @param object the object to set
    void set (const long index, Object* object);

  private:
    // make the copy constructor private
    Stack (const Stack&);
    // make the assignement operator private
    Stack& operator = (const Stack&);
    // resize the stack
    void resize (const long size);
  };
}

#endif
