// ---------------------------------------------------------------------------
// - String.cpp                                                              -
// - standard object library - string class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "String.hpp"
#include "Strvec.hpp"
#include "Vector.hpp"
#include "Buffer.hpp"
#include "Reactor.hpp"
#include "Monitor.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "csys.hpp"
#include "cmem.hpp"
#include "cstr.hpp"
#include "ccnv.hpp"

namespace aleph {
  // the unique reactor for quark mapping
  static Reactor* p_reactor = nilp;
  static Monitor* p_monitor = nilp;

  // the deallocator for the reactor
  static void dealloc_reactor (void) {
    delete p_reactor;
    delete p_monitor;
    p_reactor = nilp;
    p_monitor = nilp;
  }

  // the reactor allocator
  static inline void alloc_reactor (void) {
    if (p_reactor == nilp) {
      p_monitor = new Monitor;
      p_reactor = new Reactor;
      c_gcleanup (dealloc_reactor);
    }
  }

  // the supported quark for this string class
  static const long QUARK_ADD       = String::intern ("+");
  static const long QUARK_EQL       = String::intern ("==");
  static const long QUARK_NEQ       = String::intern ("!=");
  static const long QUARK_AEQ       = String::intern ("+=");
  static const long QUARK_GET       = String::intern ("get");
  static const long QUARK_SPLIT     = String::intern ("split");
  static const long QUARK_STRIP     = String::intern ("strip");
  static const long QUARK_LENGTH    = String::intern ("length");
  static const long QUARK_HASHID    = String::intern ("hashid");
  static const long QUARK_STRIPL    = String::intern ("strip-left");
  static const long QUARK_STRIPR    = String::intern ("strip-right");
  static const long QUARK_SUBSTR    = String::intern ("substr");
  static const long QUARK_EXTRACT   = String::intern ("extract");
  static const long QUARK_TOUPPER   = String::intern ("to-upper");
  static const long QUARK_TOLOWER   = String::intern ("to-lower");
  static const long QUARK_SUBLEFT   = String::intern ("sub-left");
  static const long QUARK_SUBRIGHT  = String::intern ("sub-right");
  static const long QUARK_FILLLEFT  = String::intern ("fill-left");
  static const long QUARK_FILLRIGHT = String::intern ("fill-right");

  // the string data structure is private to this unit
  struct s_string {
    // string holder
    char* p_buffer;
    // reference count
    long  d_rcount;
    
    // simple constructor
    s_string (void) {
      p_buffer = nilp;
      d_rcount = 1;
    }
  };

  // compute the length of a string - so much used we do it here
  static inline long lenstr (const char* s) {
    if (s == nilp) return 0;
    long result = 0;
    while (*s++ != nilc) result++;
    return result;
  }
  
  // compare two strings - again local inline for performance
  static inline bool cmpstr (const char* s1, const char* s2) {
    // nilp case first
    const char* ss1 = (s1 == nilp) ? "\0" : s1;
    const char* ss2 = (s2 == nilp) ? "\0" : s2;
    
    // check first character for fast compare
    if (*ss1 != *ss2) return false;
    
    // normal compare now
    while (*ss1 != nilc) {
      if (*ss2 == nilc) break;
      if (*ss1++ != *ss2++) return false;
    }
    return (*ss1 == *ss2) ? true : false;
  }
  
  // create an empty string
  
  String::String (void) {
    p_string = new s_string;
  }
  
  // Create a string from a c-string
  
  String::String (const char* s) {
    // create a new string holder
    p_string = new s_string;
    // copy the string
    p_string->p_buffer = c_strdup (s);
  }
  
  // create a string from a character
  
  String::String (const char c) {
    // create a new string holder
    p_string = new s_string;
    // copy the string
    p_string->p_buffer = c_strmak (c);
  }
  
  // copy constructor
  
  String::String (const String& that) {
    that.p_string->d_rcount++;
    p_string = that.p_string;
  }
  
  // destroy this string
  
  String::~String (void) {
    if (--p_string->d_rcount == 0) {
      delete [] p_string->p_buffer;
      delete p_string;
    }
  }

  // return the class name

  String String::repr (void) const {
    return "String";
  }

  // return a literal representation of this string
  
  String String::toliteral (void) const {
    String result = "\"";
    result = result + p_string->p_buffer + "\"";
    return result;
  }

  // return a printable representation of this string
  
  String String::tostring (void) const {
    return *this;
  }

  // return a clone of this string

  Object* String::clone (void) const {
    return new String (*this);
  }

  // return the string serial code

  t_byte String::serialid (void) const {
    return SERIAL_STRG_ID;
  }

  // serialize this string

  void String::wrstream (Output& os) const {
    rdlock ();
    long len = length ();
    if (len > 0) os.write (p_string->p_buffer, len);
    os.write (nilc);
    unlock ();
  }

  // deserialize this string

  void String::rdstream (Input& is) {
    wrlock ();
    Buffer buffer;
    char c = nilc;
    while ((c = is.read ()) != nilp) buffer.add (c);
    *this = buffer.tostring ();
    unlock ();
  }

  // return a quark from this string

  long String::toquark (void) const {
    alloc_reactor ();
    p_monitor->enter ();
    long result = p_reactor->intern (*this);
    p_monitor->leave ();
    return result;
  }

  // intern a cstring and return a quark

  long String::intern (const char* name) {
    return String(name).toquark();
  }

  // remap a string from a quark

  const String& String::qmap (const long quark) {
    alloc_reactor ();
    p_monitor->enter ();
    const String& result = p_reactor->qmap (quark);
    p_monitor->leave ();
    return result;
  }

  // return true if the string is nil

  bool String::isnil (void) const {
    return (p_string->p_buffer == nilp);
  }

  // assign a c-string to this string
  
  String& String::operator = (const char* s) {
    // disconnect the string or free
    if (p_string->d_rcount > 1) {
      p_string->d_rcount--;
      p_string = new s_string;
    } else {
      delete [] p_string->p_buffer;
    }
    
    // copy localy the string
    p_string->p_buffer = c_strdup (s);
    return *this;
  }
  
  // assign a character to this string
  
  String& String::operator = (const char c) {
    // disconnect the string or free
    if (p_string->d_rcount > 1) {
      p_string->d_rcount--;
      p_string = new s_string;
    } else {
      delete [] p_string->p_buffer;
    }
    
    // copy localy the string
    p_string->p_buffer = c_strmak (c);
    return *this;
  }
  
  // assign a string to this string
  
  String& String::operator = (const String& s) {
    // protect the case s = s
    s.p_string->d_rcount++;
    
    // disconnect the string
    if (--p_string->d_rcount == 0) {
      delete [] p_string->p_buffer;
      delete p_string;
    }
    
    // copy localy the string
    p_string = s.p_string;
    return *this;
  }

  // get a character at a certain position

  char String::operator [] (const long index) const {
    // check for good position first
    if ((index < 0) || (index >= this->length ()))
      throw Exception ("bound-error","in string operator []");
    // everything is fine
    return p_string->p_buffer[index];
  }

  // add a string to the current one. return a newly created string
  
  String String::operator + (const String& s) const {
    // create a temporary buffer to hold the string
    long size = this->length () + s.length () + 1;
    char* buffer = new char [size];
    
    // copy and concat the string
    c_strcpy (buffer,p_string->p_buffer);
    c_strcat (buffer,s.p_string->p_buffer);
    
    // create a new string and clean buffer
    String result = buffer;
    delete [] buffer;
    return result;
  }
  
  // add a character to the current string. Return a newly created string
  
  String String::operator + (const char c) const {
    // create a temporary buffer to hold the string
    long  size = this->length () + 2;
    char* buffer = new char [size];
    
    // copy the string and set character
    c_strcpy (buffer,p_string->p_buffer);
    buffer[size - 2] = c;
    buffer[size - 1] = nilc;
    
    // create a new string and clean buffer
    String result = buffer;
    delete [] buffer;
    return result;
  }

  // add an integer to the current string. Return a newly created string
  
  String String::operator + (const long value) const {
    // create a temporary buffer to hold the string
    char* cvalue = c_ltoa (value);
    
    // create a temporary buffer to hold the string
    long  size     = this->length () + lenstr (cvalue) + 1;
    char* buffer = new char [size];
    
    // copy and concat the string
    c_strcpy (buffer,p_string->p_buffer);
    c_strcat (buffer,cvalue);
    
    // create a new string and clean buffer
    String result = buffer;
    delete [] buffer;
    delete [] cvalue;
    return result;
  }
  
  // compare a string with another one
  
  bool String::operator == (const String& s) const {
    return cmpstr (p_string->p_buffer,s.p_string->p_buffer);
  }

  // compare a string with a c-string
  
  bool String::operator == (const char* s) const {
    return cmpstr (p_string->p_buffer,s);
  }
  
  // compare a string with another one
  
  bool String::operator != (const String& s) const {
    bool result = cmpstr (p_string->p_buffer,s.p_string->p_buffer);
    return result ? false : true;
  }
  
  // compare a string with a c-string
  
  bool String::operator != (const char* s) const {
    bool result = cmpstr (p_string->p_buffer,s);
    return result ? false : true;
  }

  // return the length of this string
  
  long String::length (void) const {
    return lenstr (p_string->p_buffer);
  }
  
  // return the length of a c-string
  
  long String::length (const char* s) {
    return lenstr (s);
  }

  // return a c-string representation or nilp for this string

  char* String::tochar (void) const {
    return c_strdup (p_string->p_buffer);
  }
  
  // remove leading blank from this string
  
  String String::stripl (void) const {
    char* buffer  = c_rmlead (p_string->p_buffer);
    String result = buffer;
    delete [] buffer;
    return result;
  }
  
  // remove trailing blank from this string
  
  String String::stripr (void) const {
    char*  buffer  = c_rmtrail (p_string->p_buffer);
    String result  = buffer;
    delete [] buffer;
    return result;
  }
  
  // remove leading and trailing blank from this string
  
  String String::strip (void) const {
    char*  lbuffer = c_rmlead  (p_string->p_buffer);
    char*  rbuffer = c_rmtrail (lbuffer);
    String result  = rbuffer;
    delete [] lbuffer;
    delete [] rbuffer;
    return result;
  }

  // convert the string to upper case
  
  String String::toupper (void) const {
    char* buffer  = c_toupper (p_string->p_buffer);
    String result = buffer;
    delete [] buffer;
    return result;
  }

  // convert the string to lower case
  
  String String::tolower (void) const {
    char* buffer  = c_tolower (p_string->p_buffer);
    String result = buffer;
    delete [] buffer;
    return result;
  }

  // return the hashid for this string
  
  long String::hashid (void) const {
    if (p_string->p_buffer == nilp) return 0;
    
    // each letter is shifted from 0 to 23 bits, like 17, 10, 3, etc ...
    char  c;
    long  result = 0;
    long  cshift = 17;
    char* cptr   = p_string->p_buffer;
    
    while ((c = *cptr++) != nilc) {
      result = result ^ (((long) c) << cshift);
      if ((cshift = cshift - 7) < 0) cshift += 24;
    }
    return (result > 0) ? result : -result;
  }

  // return the right substring starting at a certin index

  String String::rsubstr (const long index) const {
    String result;
    long len = length ();
    if ((len == 0) || (index >= len-1)) return result;
    char* sptr = p_string->p_buffer;
    return String (sptr + index);
  }

  // return the left substring of a string

  String String::lsubstr (const long index) const {
    String result;
    long len = length ();
    if ((len == 0) || (index > len)) return result;
    char* sptr = c_strdup (p_string->p_buffer);
    sptr[index] = nilc;
    result = sptr;
    delete [] sptr;
    return result;
  }

  // return a substring of a string

  String String::substr (const long lidx, const long ridx) const {
    long len = length ();
    if ((lidx >= ridx) || (lidx < 0) || (lidx >= len) || (ridx <0) ||
	(ridx >  len)  || (len == 0)) 
      throw Exception ("index-error", "invalid index for sub string");
    // create th result string
    String result;
    char* buf = c_strdup (p_string->p_buffer);
    char* sub = buf + lidx;
    buf[ridx] = nilc;
    result = sub;
    delete [] buf;
    return result;
  }

  // fill this string with a character until a given size is reached
  
  String String::lfill (const char c, const long size) const {
    // do nothing if size exceed
    long len = size - length ();
    if (len <= 0) return *this;
    // fill the string first
    String result;
    for (long i = 0; i < len; i++) result = result + c;
    return result + *this;
  }

  // fill this string with a character until a given size is reached
  
  String String::rfill (const char c, const long size) const {
    // do nothing if size exceed
    long len = size - length ();
    if (len <= 0) return *this;
    // fill the string first
    String result = *this;
    for (long i = 0; i < len; i++) result = result + c;
    return result;
  }

  // split a string with blank and tab

  Vector* String::split (void) const {
    return split ("");
  }

  // split a string according to a string break

  Vector* String::split (const String& sbrk) const {
    rdlock ();
    Vector* result = new Vector;
    try {
      Strvec vec = Strvec::split (*this, sbrk);
      long len = vec.length ();
      for (long i = 0; i < len; i++) {
	String data = vec.get (i);
	result->append (new String (data));
      }
    } catch (...) {
      delete result;
      unlock ();
      throw;
    }
    return result;
  }

  // accumulate strings between a control character

  Vector* String::extract (const char cbrk) const {
    rdlock ();
    Vector* result = new Vector;
    long len = length ();
    for (long i = 0; i < len; i++) {
      char c = p_string->p_buffer[i];
      if (c == cbrk) {
	i++;
	Buffer buf;
	while ((c = p_string->p_buffer[i]) != cbrk) {
	  buf.add (c);
	  i++;
	  if (i == len) {
	    delete result;
	    unlock ();
	    throw Exception ("extract-error", "unterminated string", *this);
	  }
	}
	result->append (new String (buf.tostring ()));
      }
    }
    unlock ();
    return result;
  }

  // create a new string in a generic way

  Object* String::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new String;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with string constructor");
    // try to map the string argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new String;

    // try a literal object
    Literal* lval = dynamic_cast <Literal*> (obj);
    if (lval != nilp) return new String (lval->tostring ());

    // illegal object
    throw Exception ("type-error", "illegal object with string constructor",
		     obj->repr ());
  }

  // operate this string with another object

  Object* String::oper (Runnable* robj, t_oper type, Object* object) {
    Literal* lobj = dynamic_cast <Literal*> (object);
    String*  sobj = dynamic_cast <String*>    (object);

    switch (type) {
    case Object::ADD:
      if (lobj != nilp) return new String (*this + lobj->tostring ());
      break;  
    case Object::EQL:
      if (sobj != nilp) return new Boolean (*this == *sobj);
      break;
    case Object::NEQ:
      if (sobj != nilp) return new Boolean (*this != *sobj);
      break;
    default:
      throw Exception ("operator-error", "unsupported string operator");
    }
    throw Exception ("type-error", "invalid operand with string",
		     Object::repr (object));
  }

  // set an object to this string

  Object* String::vdef (Runnable* robj, Nameset* nset, Object* object) {
    Literal* lobj = dynamic_cast <Literal*> (object);
    if (lobj != nilp) {
      *this = lobj->tostring ();
      return this;
    }
    throw Exception ("type-error", "invalid object with string vdef",
		     Object::repr (object));
  }

  // apply this string with a set of arguments and a quark

  Object* String::apply (Runnable* robj, Nameset* nset, long quark,
			 Vector* argv) {
    // get the arguments length
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_SPLIT)   return split ();
      if (quark == QUARK_LENGTH)  return new Integer (length  ());
      if (quark == QUARK_STRIPL)  return new String  (stripl  ());
      if (quark == QUARK_STRIPR)  return new String  (stripr  ());
      if (quark == QUARK_STRIP)   return new String  (strip   ());
      if (quark == QUARK_TOUPPER) return new String  (toupper ());
      if (quark == QUARK_TOLOWER) return new String  (tolower ());
      if (quark == QUARK_HASHID)  return new Integer (hashid  ());
    }

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_SPLIT) return split (argv->getstring (0));
      if (quark == QUARK_ADD) return oper (robj, Object::ADD, argv->get (0));
      if (quark == QUARK_EQL) return oper (robj, Object::EQL, argv->get (0));
      if (quark == QUARK_NEQ) return oper (robj, Object::NEQ, argv->get (0));

      if (quark == QUARK_AEQ) {
	String val = argv->getstring (0);
	*this = *this + val;
	return this;
      }
      if (quark == QUARK_GET) {
	t_long val = argv->getint (0);
	char c = (*this)[val];
	return new Character (c);
      }
      if (quark == QUARK_EXTRACT) {
	char cbrk = argv->getchar (0);
	return extract (cbrk);
      }
      if (quark == QUARK_SUBRIGHT) {
	t_long val = argv->getint (0);
	String result = rsubstr (val);
	return new String (result);
      }
      if (quark == QUARK_SUBLEFT) {
	t_long val = argv->getint (0);
	String result = lsubstr (val);
	return new String (result);
      }
    }

    // dispatch two arguments
    if (argc == 2) {
      if (quark == QUARK_FILLLEFT) {
	char   c    = argv->getchar (0);
	t_long size = argv->getint  (1);
	String result = lfill (c, size);
	return new String (result);
      }
      if (quark == QUARK_FILLRIGHT) {
	char   c    = argv->getchar (0);
	t_long size = argv->getint  (1);
	String result = rfill (c, size);
	return new String (result);
      }
      if (quark == QUARK_SUBSTR) {
	t_long lidx = argv->getint (0);
	t_long ridx = argv->getint (1);
	String result = substr (lidx, ridx);
	return new String (result);
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
