// ---------------------------------------------------------------------------
// - Strvec.cpp                                                              -
// - standard object library - string vector class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Strvec.hpp"
#include "Buffer.hpp"
#include "Exception.hpp"

namespace aleph {

  // check that a character is in the c-string. 
  static bool match_break_sequence (char c, const char* str) {
    long size = String::length (str);
    // loop and compare
    for (long i = 0; i < size; i++)
      if (c == str[i]) return true;
    return false;
  }

  // create an empty string vector

  Strvec::Strvec (void) {
    d_size   = 0;
    d_length = 0;
    p_vector = nilp;
  }

  // create a string vector with an original size

  Strvec::Strvec (const long size) {
    if (size < 0) throw Exception ("size-error","in strvec constructor");
    d_size   = size;
    d_length = 0;
    p_vector = new String[d_size];
  }

  // copy constructor for this string vector

  Strvec::Strvec (const Strvec& that) {
    that.rdlock ();
    // copy arguments
    d_size   = that.d_length;
    d_length = that.d_length;
    p_vector = nilp;
    // create a new vector of strings and copy them
    if ((d_length > 0) && (that.p_vector != nilp)) {
      p_vector = new String[d_length];
      for (long i = 0; i < d_length; i++) p_vector[i] = that.p_vector[i];
    }
    that.unlock ();
  }

  // destroy this string vector
  
  Strvec::~Strvec (void) {
    delete [] p_vector;
  }

  // return the class name

  String Strvec::repr (void) const {
    return "Strvec";
  }

  // assign a string vector to this one
  
  Strvec& Strvec::operator = (const Strvec& that) {
    // check againt equal equal
    if (this == &that) return *this;
    // lock everything
    wrlock ();
    that.rdlock ();
    // delete old value
    delete [] p_vector;
    // copy arguments
    d_size   = that.d_length;
    d_length = that.d_length;
    p_vector = nilp;
    // create a new string vector of strings and copy them
    if ((d_length > 0) && (that.p_vector != nilp)) {
      p_vector = new String[d_size];
      for (long i = 0; i < d_length; i++) p_vector[i] = that.p_vector[i];
    }
    that.unlock ();
    unlock ();
    return *this;
  }

  // add a new element in this string vector
  
  void Strvec::add (const String& str) {
    wrlock ();
    // check if we have to resize the Strvec
    if (d_length + 1 >= d_size) {
      long size = (d_size <= 0) ? 1 : d_size * 2;
      String* vector = new String[size];
      for (long i = 0; i < d_length; i++) vector[i] = p_vector[i];
      delete [] p_vector;
      d_size   = size;
      p_vector = vector;
    }
    // set the string in this Strvec
    p_vector[d_length++] = str;
    unlock ();
  }

  // set an string at a certain position in this vector
  // the old string is destroyed

  void Strvec::set (const long index, const String& str) {
    // check that we are bounded
    if (index >= d_length) 
      throw Exception ("index-error","in string vector set");
    wrlock ();
    p_vector[index] = str;
    unlock ();
  }
  
  // get an string at a certain position

  const String& Strvec::get (const long index) const {
    // check that we are bounded
    if (index >= d_length) 
      throw Exception ("index-error","in strvec set");
    rdlock ();
    const String& result = p_vector[index];
    unlock ();
    return result;
  }

  // return the first string in this vector

  String Strvec::first (void) const {
    rdlock ();
    try {
      String result = get (0);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the last string in this vector

  String Strvec::last (void) const {
    rdlock ();
    try {
      String result = get (d_length-1);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the number of element in this string vector

  long Strvec::length (void) const {
    rdlock ();
    long result = d_length;
    unlock ();
    return result;
  }

  // check that a string exists in this vector

  bool Strvec::exists (const String& name) const {
    rdlock ();
    if (d_length == 0) {
      unlock ();
      return false;
    }
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == name) {
	unlock ();
	return true;
      }
    }
    unlock ();
    return false;
  }

  // return the index of a key in this string vector

  long Strvec::index (const String& key) const {
    rdlock ();
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == key) {
	unlock ();
	return i;
      }
    }
    unlock ();
    throw Exception ("key-error","key not found",key);
  }
  
  // return the index of a key or -1

  long Strvec::lookup (const String& key) const {
    rdlock ();
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == key) {
	unlock ();
	return i;
      }
    }
    unlock ();
    return -1;
  }

  // reset this vector

  void Strvec::reset (void) {
    wrlock ();
    delete [] p_vector;
    d_size   = 0;
    d_length = 0;
    unlock ();
  }

  // split this string with a sequence of characters

  Strvec Strvec::split (const String& name, const String& sbrk) {
    Strvec result;
    Buffer buffer;
    
    // first thing first - do we have a nil string
    if (name.length () == 0) return result;
    
    // get a c-string representation
    char* data = name.tochar ();
    char* cptr = data;
    
    // fix the break sequence in case it is nil
    const char* cbrk = (sbrk.length () == 0) ? " \t\n" : sbrk.tochar ();
    
    // loop and accumulate - if a character match the break sequence
    // the buffer is inserted into the vector
    
    char c;
    buffer.reset ();
    while ((c = *data++) != nilp) {
      if (match_break_sequence (c,cbrk) == true) {
	result.add (buffer.tostring());
	buffer.reset ();
	continue;
      }
      buffer.add (c);
    }
    
    // check if the buffer is not empty
    if (buffer.length () != 0) result.add (buffer.tostring());
    
    // clean the break sequence and return
    if (sbrk.length () != 0) delete [] cbrk;
    delete [] cptr;
    return result;
  }

  // return the maximum string length in this vector

  long Strvec::maxlen (void) const {
    rdlock ();
    long result = 0;
    for (long i = 0; i < d_length; i++) {
      long len = p_vector[i].length ();
      if (len > result) result = len;
    }
    unlock ();
    return result;
  }

  // return the minimum string length in this vector

  long Strvec::minlen (void) const {
    rdlock ();
    long result = 0;
    for (long i = 0; i < d_length; i++) {
      long len = p_vector[i].length ();
      if (len < result) result = len;
    }
    unlock ();
    return result;
  }

  // return an array of quarks for this vector

  long* Strvec::toquarks (void) const {
    rdlock ();
    if (d_length == 0) {
      unlock ();
      return nilp;
    }
    long* result = new long[d_length];
    for (long i = 0; i < d_length; i++) result[i] = p_vector[i].toquark ();
    unlock ();
    return result;
  }
}
