/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <boost/smart_ptr.hpp>
#include <PlatformSupport/XalanOutputStreamPrintWriter.hpp>
#include <PlatformSupport/XalanStdOutputStream.hpp>
#include <XMLSupport/FormatterToXML.hpp>
#include <XMLSupport/FormatterToHTML.hpp>
#include <XMLSupport/FormatterToText.hpp>
#include <XPath/XString.hpp>
#include "include.hpp"
#include "strX.hpp"
#include "xslCompiler.hpp"

using namespace boost;
using namespace std;

XSLCompiler::XSLCompiler( const string &_xslScriptFileName,
                          FormatterListener::eFormat _format,
                          const string &_outputEncoding,
                          const string &_schemaURI,
                          const string &_emptyResult,
                          XSLParseErrorHandlerPtr _xslParseErrorHandler )
  throw (Error):
  xslScriptFileName( _xslScriptFileName ), format( _format ),
  compiledStyleSheet( NULL ), outputEncoding( _outputEncoding ),
  schemaURI( _schemaURI ), emptyResult(_emptyResult),
  xslParseErrorHandler(_xslParseErrorHandler)
{
  int theResult = theXalanTransformer.
    compileStylesheet( xslScriptFileName.c_str(), compiledStyleSheet );
#ifndef NDEBUG
  cerr << "Compiled XSL stylesheet";
  if ( theResult != 0 )
    cerr << " (error: " << (const char *)theXalanTransformer.getLastError()
         << ")";
  cerr << '.' << endl;
#endif
  ERRORMACRO( theResult == 0, Error, ,
              "Error while loading XSL script \"" << _xslScriptFileName
              << "\": " << (const char *)theXalanTransformer.getLastError() );
}

XSLCompiler::~XSLCompiler(void)
{
  theXalanTransformer.destroyStylesheet( compiledStyleSheet );
}

void XSLCompiler::translate( istream &inputStream,
                             ostream &outputStream ) const
  throw (Error)
{
  XalanStdOutputStream xalanOutputStream( outputStream );
  XalanOutputStreamPrintWriter writer( xalanOutputStream );

  shared_ptr< FormatterListener > formatterListener;
  
  // XSLTResultTarget::setEncoding doesn't have any effect!
  switch ( format ) {
  case FormatterListener::OUTPUT_METHOD_XML:
    // Using XalanDOMString-constructor, which is present in Xalan-C version
    // 1.9 and in earlier versions as well!
    formatterListener = shared_ptr< FormatterListener >
      ( new FormatterToXML ( writer, XalanDOMString( "1.0" ),
                             true, FormatterToXML::eDefaultIndentAmount,
                             XalanDOMString( outputEncoding.c_str() ) ) );
    break;
  case FormatterListener::OUTPUT_METHOD_HTML:
    formatterListener = shared_ptr< FormatterListener >
      ( new FormatterToHTML( writer,
                             XalanDOMString( outputEncoding.c_str() ) ) );
    break;
  case FormatterListener::OUTPUT_METHOD_TEXT:
    formatterListener = shared_ptr< FormatterListener >
      ( new FormatterToText( writer,
                             XalanDOMString( outputEncoding.c_str() ) ) );
    break;
  default:
    ERRORMACRO( false, Error, ,
                "Unknown output-format number " << format
                << " for XSL compiler." );
  };

  XSLTResultTarget target( *formatterListener );
  
  XalanTransformer transformer;
  if ( !schemaURI.empty() ) {
    if ( schemaURI.find( ' ' ) < schemaURI.size() )
      transformer.setExternalSchemaLocation
        ( StrX<XMLCh,char>( schemaURI.c_str() ).getBuffer() );
    else
      transformer.setExternalNoNamespaceSchemaLocation
        ( StrX<XMLCh,char>( schemaURI.c_str() ).getBuffer() );
    transformer.setUseValidation( true );
  };

  // Pass global parameters to stylesheet.
  for ( map< string, string >::const_iterator i=params.begin();
        i != params.end(); i++ )
    transformer.setStylesheetParam( i->first.c_str(), i->second.c_str() );

  int retVal = transformer.transform( &inputStream, compiledStyleSheet,
                                      target );
  if ( retVal != 0 ) {
    XSLParseError e( xslScriptFileName, transformer.getLastError() );
    xslParseErrorHandler->error( e );
    // Write empty result in case error-handler chooses to ignore error. 
    outputStream << emptyResult;
  };
}

void XSLCompiler::setParam( const string &name, const string &value )
{
  params[ name ] = value;
}

bool XSLCompiler::clearParam( const string &name )
{
  return params.erase( name ) != 0;
}
