/*
 *  apcnet.c  -- network parts for apcupsd package
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *		 BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *	      -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

 /*
  * Major rewrite following same logic 11 Jan 2000, Kern Sibbald
  */

#include "apc.h"

/* Local variables */
static int socketfd;
static int newsocketfd;
static struct sockaddr_in my_adr;
static int masterlen;
static struct sockaddr_in master_adr;
static struct in_addr master_h_addr;
static struct netdata get_data;

/* 
 * Possible state for a slave (also partially for a master)
 * RMT_NOTCONNECTED    not yet connected, except for RECONNECT, this is the
 *		       only time when the slave sends back a message in response
 *		       to the packet we send.  He sends back his usermagic
 *		       or security id string.
 * RMT_CONNECTED       All is OK
 * RMT_RECONNECT       Must redo initial packet swap, but this time we
 *		       verify his usermagic rather than just store it.
 *		       This occurs after a connect() error.
 * RMT_ERROR	       Set when some packet error occurs, presumably the
 *		       next call should work.
 * RMT_DOWN	       This occurs when we detect a security violation (e.g.
 *		       unauthorized slave. We mark him down and no longer
 *		       talk to him.
 */

/********************************************************************** 
 *
 * Called by the master to connect to the slaves. 
 */
int prepare_master(UPSINFO *ups)
{
    int i;

    for (i=0; i<slave_count; i++) {
	slaves[i].remote_state = RMT_NOTCONNECTED;
	send_to_each(ups, i);
    }
    return 0;			  /* OK */
}

/*********************************************************************/
void send_to_slaves(UPSINFO *ups)
{
    int i;
    int stat;

    for (i=0; i<slave_count; i++) {
       stat = send_to_each(ups, i);
       /* Print error message once */
       if (slaves[i].error == 0) {
	  switch(stat) {
	  case 6:
              log_event(ups, LOG_WARNING, "Cannot resolve slave name %s\n", slaves[i].name);
	      break;
	  case 5:
              log_event(ups, LOG_WARNING,"Got slave shutdown from %s", slaves[i].name);
	      break;	      
	  case 4:
              log_event(ups, LOG_WARNING,"Cannot write to slave %s", slaves[i].name);
	      break;
	  case 3:
              log_event(ups, LOG_WARNING,"Cannot read magic from slave %s", slaves[i].name);
	      break;
	  case 2:
              log_event(ups, LOG_WARNING,"Connect to slave %s failed", slaves[i].name);
	      break;
	  case 1:
              log_event(ups, LOG_WARNING,"Cannot open stream socket");
	      break;
	  case 0:
	      break;
	  default:
              log_event(ups, LOG_WARNING,"Unknown Error (send_to_slaves)");
	      break;
	  }
       }
       slaves[i].error = stat;
       if (stat != 0)
	   slaves[i].errorcnt++;
    }
}

/********************************************************************* 
 * Called from master to send data to a specific slave (who).
 * Returns: 0 if OK
 *	    non-zero, see send_to_slaves();
 *
 * Note, be careful not to change RMT_NOTCONNECTED or 
 * RMT_RECONNECT until we have completed the connection, even if an
 * error occurs.
 */
int send_to_each(UPSINFO *ups, int who)
{
    struct hostent *slavent;
    struct netdata read_data;
    struct netdata send_data;
    int stat = 0;
    long l;

    switch (slaves[who].remote_state) {
    case RMT_NOTCONNECTED:
	if ((slavent = gethostbyname(slaves[who].name)) == NULL) {
	    slaves[who].remote_state = RMT_DOWN;
	    return 6;
	}
	/* memset is more portable than bzero */
	memset((void *) &slaves[who].addr, 0, sizeof(struct sockaddr_in));
	slaves[who].addr.sin_family = AF_INET;
	memcpy((void *)&slaves[who].addr.sin_addr,
		 (void *)slavent->h_addr,  sizeof(struct in_addr));
	slaves[who].addr.sin_port = htons(ups->NetUpsPort);
	slaves[who].usermagic[0] = 0;
	break;
    case RMT_DOWN:
	return 0;
    default:
	break;
    }

    if ((slaves[who].socket = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	slaves[who].remote_state = RMT_DOWN;
	return 1;
    }

    if ((connect(slaves[who].socket, (struct sockaddr *) &slaves[who].addr,
		 sizeof(slaves[who].addr))) == -1) {
	if (slaves[who].remote_state == RMT_CONNECTED)
	    slaves[who].remote_state = RMT_RECONNECT;
	close(slaves[who].socket);
	return 2;
    }
    /* Extra complexity is because of compiler
     * problems with htonl(ups->OnBatt);
     */
    l = ups->OnBatt;
    send_data.OnBatt	    = htonl(l);
    l = ups->BattLow;
    send_data.BattLow	    = htonl(l);
    l = ups->BattChg;
    send_data.BattChg	    = htonl(l);
    l = ups->ShutDown;
    send_data.ShutDown	    = htonl(l);
    l = ups->nettime;
    send_data.nettime	    = htonl(l);
    l = ups->TimeLeft;
    send_data.TimeLeft	    = htonl(l);
    l = ups->ChangeBatt;
    send_data.ChangeBatt    = htonl(l);
    l = ups->load;
    send_data.load	    = htonl(l);
    l = ups->timedout;
    send_data.timedout	    = htonl(l);
    l = ups->timelout;
    send_data.timelout	    = htonl(l);
    l = ups->emergencydown;
    send_data.emergencydown = htonl(l);
    l = ups->UPS_Cap[CI_BATTLEV];
    send_data.cap_battlev   = htonl(l);
    l = ups->UPS_Cap[CI_RUNTIM];
    send_data.cap_runtim    = htonl(l);

    send_data.remote_state  = htonl(slaves[who].remote_state);
    strcpy(send_data.apcmagic, APC_MAGIC);
    strcpy(send_data.usermagic, slaves[who].usermagic);

    /* Send our data to Slave */
    if ((write(slaves[who].socket, &send_data,
			  sizeof(send_data))) != sizeof(send_data)) {
	if (slaves[who].remote_state == RMT_CONNECTED)
	    slaves[who].remote_state = RMT_ERROR;
	stat = 4;	      /* write error */

    /*
     * If slave not yet connected, he will respond with his
     *  two "magic" values.
     */
    } else if (slaves[who].remote_state == RMT_NOTCONNECTED ||
		slaves[who].remote_state == RMT_RECONNECT) {
	fd_set rfds;
	struct timeval tv;

	read_data.apcmagic[0] = 0;
	read_data.usermagic[0] = 0;

	FD_ZERO(&rfds);
	FD_SET(slaves[who].socket, &rfds);
	tv.tv_sec = 10; 	      /* wait 10 seconds for response */
	tv.tv_usec = 0;
	   
	errno = 0;
	switch (select(slaves[who].socket+1, &rfds, NULL, NULL, &tv)) {
	case 0: 	     /* No chars available in 10 seconds. */
	case -1:	     /* error */
	    slaves[who].remote_state = RMT_RECONNECT;
	    close(slaves[who].socket);
	    slaves[who].socket = -1;
	    return 2;
	default:
	    break;
	}
	read(slaves[who].socket, &read_data, sizeof(read_data));
	if (strcmp(APC_MAGIC, read_data.apcmagic) == 0) { 
	    if (slaves[who].remote_state == RMT_NOTCONNECTED) {
		strcpy(slaves[who].usermagic, read_data.usermagic);
	    } else {
		if (strcmp(slaves[who].usermagic, read_data.usermagic) != 0)
		    stat = 3;	    /* bad magic */
	    }
	} else {
	    stat = 3;	      /* bad magic */
	}
	if (stat == 0) {
            log_event(ups, LOG_WARNING,"Connect to slave %s succeeded", slaves[who].name);
	    slaves[who].remote_state = RMT_CONNECTED;
	} 
    }
    close(slaves[who].socket);
    return stat;
}


/*********************************************************************/

/* slaves
 *  Note, here we store certain information about the master in
 *  the slave[0] packet.
 */


/********************************************************************* 
 *
 * Called by a slave to open a socket and listen for the master 
 * Returns: 1 on error
 *	    0 OK
 */
int prepare_slave(UPSINFO *ups)
{
    int i, bound;
    struct hostent *mastent;
    int turnon = 1;

    slaves[0].remote_state = RMT_DOWN;
    if ((mastent = gethostbyname(ups->master_name)) == NULL) {
        log_event(ups, LOG_ERR,"Can't resolve master name %s", ups->master_name);
	return 1;
    }
    /* save host address.  
     * Note, this is necessary because on Windows, mastent->h_addr
     * is not valid later when the connection is made.
     */
    memcpy((void *)&master_h_addr, (void *)mastent->h_addr, sizeof(struct in_addr));
    /* Open socket for network communication */
    if ((socketfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        log_event(ups, LOG_ERR, "Cannot open stream socket");
	return 1;
    }

   /*
    * Reuse old sockets 
    */
   if (setsockopt(socketfd, SOL_SOCKET, SO_REUSEADDR, &turnon, sizeof(turnon)) < 0) {
      log_event(ups, LOG_WARNING, "Cannot set SO_REUSEADDR on socket: %s\n" , strerror(errno));
   }

    /* memset is more portable than bzero */
    memset((char *) &my_adr, 0, sizeof(struct sockaddr_in));
    my_adr.sin_family = AF_INET;
    my_adr.sin_addr.s_addr = htonl(INADDR_ANY);
    my_adr.sin_port = htons(ups->NetUpsPort);

    bound = FALSE;
    for (i=0; i<30; i++) {	  /* retry every 30 seconds for 15 minutes */
	if (bind(socketfd, (struct sockaddr *) &my_adr, sizeof(my_adr)) < 0) {
            log_event(ups, LOG_WARNING, "Cannot bind port %d, retrying ...", ups->NetUpsPort);
	    sleep(30);
	} else {
	    bound = TRUE;
	    break;
	}
    }
    if (!bound) {
        log_event(ups, LOG_ERR, "Cannot bind port %d, probably already in use", ups->NetUpsPort);
	close(socketfd);
	return 1;
    }

    if (listen(socketfd, 1) == -1) {
        log_event(ups, LOG_ERR, "Listen Failure");
	close(socketfd);
	return 1;
    }

    slaves[0].remote_state = RMT_NOTCONNECTED;
    strcpy(slaves[0].name, ups->master_name);
    return 0;
}

/********************************************************************
 * 
 * Get data from master, we hang on accept() until the master decides
 * that it is time to send us something. We must be patient.
 *
 * Returns: 0 OK 
 *	    non-zero is error, see get_from_master();
 */
static int get_data_from_master(UPSINFO *ups) 
 {
    int stat;
    int ShutDown;

    masterlen = sizeof(master_adr);
    if ((newsocketfd = accept(socketfd, (struct sockaddr *) &master_adr,
			      &masterlen)) < 0) {
	slaves[0].remote_state = RMT_DOWN;
	return 1;
    }
#ifdef HAVE_LIBWRAP
    /*
     * This function checks the incoming client and if it's not
     * allowed closes the connection.
     */
    if (check_wrappers(argvalue, newsocketfd) == FAILURE) {
	slaves[0].remote_state = RMT_DOWN;
	close(newsocketfd);
	return 2;
    }
#endif


    /*
     *  Let's add some basic security
     */
    if (memcmp((void *) &master_adr.sin_addr, (void *)&master_h_addr,
	       sizeof(struct in_addr)) != 0) {
	slaves[0].remote_state = RMT_DOWN;
	close(newsocketfd);
	return 2;
    }

    if ((read(newsocketfd, &get_data, sizeof(get_data))) == -1) {
	close(newsocketfd);
	slaves[0].remote_state = RMT_ERROR;
	return 3;
    }

    if (strcmp(APC_MAGIC, get_data.apcmagic) != 0) { 
	close(newsocketfd);
	slaves[0].remote_state = RMT_ERROR;
	return 4;
    }
	  
    stat = ntohl(get_data.remote_state);
    /* If not connected, send him our user magic */
    if (stat == RMT_NOTCONNECTED || stat == RMT_RECONNECT) {
	strcpy(get_data.apcmagic, APC_MAGIC);
	strcpy(get_data.usermagic, ups->usermagic);
	write(newsocketfd, &get_data, sizeof(get_data));
        log_event(ups, LOG_WARNING, "Connect from master %s succeeded", slaves[0].name);
	time(&ups->last_master_connect_time);
    }

    if (strcmp(ups->usermagic, get_data.usermagic) == 0) {
	ups->OnBatt	   = ntohl(get_data.OnBatt);
	ups->BattLow	   = ntohl(get_data.BattLow);
	ups->BattChg	   = ntohl(get_data.BattChg);
	ShutDown	   = ntohl(get_data.ShutDown);
	ups->nettime	   = ntohl(get_data.nettime);
	ups->TimeLeft	   = ntohl(get_data.TimeLeft);
/*
 * Setting ChangeBatt triggers false alarms if the master goes
 * down and comes back up, so remove it for now.  KES 27Feb01
 *
 *	ups->ChangeBatt    = ntohl(get_data.ChangeBatt);   
 */
	ups->load	   = ntohl(get_data.load);
	ups->timedout	   = ntohl(get_data.timedout);
	ups->timelout	   = ntohl(get_data.timelout);
	ups->emergencydown = ntohl(get_data.emergencydown);
	ups->remote_state  = ntohl(get_data.remote_state);
	ups->UPS_Cap[CI_BATTLEV] = ntohl(get_data.cap_battlev);
	ups->UPS_Cap[CI_RUNTIM] = ntohl(get_data.cap_runtim);
    } else {
	slaves[0].remote_state = RMT_ERROR;
	close(newsocketfd);
	return 5;
    }

    if (ShutDown)		    /* if master has shutdown */
	ups->remotedown = TRUE; /* we go down too */

    close(newsocketfd);
    slaves[0].remote_state = RMT_CONNECTED;
    time(&ups->last_master_connect_time);
    return 0;
}

/*********************************************************************
 *
 * Called from slave to get data from the master 
 * Returns: 0 on error
 *	    1 OK
 */
static int get_from_master(UPSINFO *ups)
{
    int stat = get_data_from_master(ups);
    /* Print error message once */
    if (slaves[0].error == 0) {
       switch(stat) {
	  case 0:
	     break;
	  case 1:
             log_event(ups, LOG_ERR, "Socket accept error");
	     break;
	  case 2:
             log_event(ups, LOG_ERR, "Unauthorised attempt from master %s",
			inet_ntoa(master_adr.sin_addr));
	     break;
	  case 3:
             log_event(ups, LOG_ERR, "Read failure from socket");
	     break;
	  case 4:
             log_event(ups, LOG_ERR, "Bad APC magic from master: %s", get_data.apcmagic);
	     break;
	  case 5:
             log_event(ups, LOG_ERR, "Bad user magic from master: %s", get_data.usermagic);
	     break;
	  default:
             log_event(ups, LOG_ERR, "Unknown Error in get_from_master");
	     break;
       }
    }
    slaves[0].error = stat;
    if (stat != 0)
       slaves[0].errorcnt++;
    return 0;
}


/*
 * XXX
 * 
 * Why ?
 *
 * This function seem to do nothing except for a special case ... and
 * it do only logging.
 *
 */
void kill_net(UPSINFO *ups) {
    log_event(ups, LOG_WARNING,"%s: Waiting For ShareUPS Master to shutdown.",
	    argvalue);
    sleep(60);
    log_event(ups, LOG_WARNING,"%s: Great Mains Returned, Trying a REBOOT",
	    argvalue);
    log_event(ups, LOG_WARNING, "%s: Drat!! Failed to have power killed by Master",
	    argvalue);
    log_event(ups, LOG_WARNING,"%s: Perform CPU-reset or power-off",
	    argvalue);
}

/********************************************************************* 
 *
 * Each slave executes this code, reading the status from the master
 * and doing the actions necessary (do_action).
 * Note, that the slave hangs on an accept() in get_from_master()
 * waiting for the master to send it data. 
 *
 * When time permits, we should implement an alert() to interrupt
 * the accept() periodically and check when the last time the master
 * gave us data.  He should be talking to us roughly every nettime
 * seconds. If not (for example, he is absent more than 2*nettime),
 * then we should issue a message saying that he is down.
 */
void do_net(UPSINFO *ups)
{
    init_thread_signals();

#ifdef HAVE_CYGWIN     /* needed for NT */
    attach_ipc(ups, 0);
#endif

    while (1) {
	/* Note, we do not lock shm so that apcaccess can
	 * read it.  We are the only one allowed to update 
	 * it.
	 */
	read_shmarea(ups, 0);
	get_from_master(ups);
	write_shmarea(ups);
	do_action(ups, 0);
    }
}


/********************************************************************* 
 *
 * The master executes this code, periodically sending the status
 * to the slaves.
 *
 * When time permits, we should periodically (once every 10 minutes)
 * mark all slaves that are RMT_DOWN as RMT_NOTCONNECTED and
 * clear their error count so that the code will try again 
 * to connect them. We should also mark any
 * slave with too many errors (more than 100) as RMT_DOWN. This
 * prevents us from being flooded with errors. Note the code
 * already ensures that only one message is printed when there
 * is an error condition.
 */
void do_slaves(UPSINFO *ups)
{
    time_t now;
    time_t last_time_net = 0;
      
    init_thread_signals();

#ifdef HAVE_CYGWIN     /* needed for NT */
    attach_ipc(ups, 0);
#endif

    while (1) {
	time(&now);

        /* This read is necessary to update "ups" */
	read_shmarea(ups, 0);
	if (((now - last_time_net) > ups->nettime) || ups->FastPoll) {
	    send_to_slaves(ups);
	    time(&last_time_net);
	}
	if (ups->FastPoll) {
	    sleep(1);		      /* urgent send data faster */
	} else {
	    sleep(TIMER_SLAVES);      /* wait 10 seconds */
	}
    }
}
