// matchers.cc
//
//  Copyright 2000,2001 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Grammer for the condition language (probably not very correctly done, but
// close enough :) ):
//
//  CONDITION := CONDITION-LIST
//  CONDITION-LIST := CONDITION-AND-GROUP '|' CONDITION-LIST
//                 := CONDITION-AND-GROUP
//  CONDITION-AND-GROUP := CONDITION-ATOM CONDITION-AND-GROUP
//                      := CONDITION-ATOM
//  CONDITION-ATOM := '(' CONDITION-LIST ')'
//                 := '!' CONDITION-ATOM
//                 := '~'field-id [string]
//                 := string

#include "matchers.h"
#include "apt.h"
#include "tasks.h"

#include "../aptitude.h"

#include <apt-pkg/error.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/version.h>

#include <ctype.h>
#include <stdio.h>

using namespace std;

inline const char *stristr(const char *haystack, const char *needle)
  // There are no basic_string methods to efficiently compare two strings
  // case-insensitively, and no string functions to do the same for
  // const char * objects, so I'm writing my own.
{
  int lenh=strlen(haystack), lenn=strlen(needle);
  if(lenn==0 || lenh==0)
    return NULL;
  for(int i=0; i<lenh-lenn+1; i++)
    {
      bool found=true;
      for(int j=0; j<lenn; j++)
	if(tolower(haystack[i+j])!=tolower(needle[j]))
	  {
	    found=false;
	    break;
	  }
      if(found)
	return haystack+i;
    }
  return NULL;
}

inline bool string_matches(string pattern, string s, bool match_all)
  // Altering this to support globs or regexps is all that's needed to add
  // support for those to the program (if that's done, the inline should be
  // removed :) )
  //
  //  If match_all is true, the string must be an exact match for the pattern;
  // otherwise, any match is allowed.
{
  if(match_all)
    return pattern==s;
  else
    return stristr(s.c_str(), pattern.c_str())!=NULL;
}

typedef bool (*mfunc)(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all);

template<mfunc f>
class pkg_string_matcher:public pkg_matcher
{
  string s;
  bool match_all;
public:
  pkg_string_matcher(string _s, bool _match_all):s(_s), match_all(_match_all) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return f(s, pkg, ver, match_all);
  }
};

bool name_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  return string_matches(s, pkg.Name(), match_all);
}
typedef pkg_string_matcher<name_matches> pkg_name_matcher;

bool description_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  return !ver.end() && string_matches(s, apt_package_records->Lookup(ver.FileList()).LongDesc(), match_all);
}
typedef pkg_string_matcher<description_matches> pkg_description_matcher;

bool maintainer_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  return !ver.end() && string_matches(s, apt_package_records->Lookup(ver.FileList()).Maintainer(), match_all);
}
typedef pkg_string_matcher<maintainer_matches> pkg_maintainer_matcher;

bool priority_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  return !ver.end() && ver.PriorityType() &&
    string_matches(s, ver.PriorityType(), match_all);
}
typedef pkg_string_matcher<priority_matches> pkg_priority_matcher;

bool section_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  return !ver.end() && ver.Section() && string_matches(s, ver.Section(), match_all);
}
typedef pkg_string_matcher<section_matches> pkg_section_matcher;

bool version_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  return !ver.end() && ver.VerStr() && string_matches(s, ver.VerStr(), match_all);
}
typedef pkg_string_matcher<version_matches> pkg_version_matcher;

bool task_matches(string s, pkgCache::PkgIterator pkg, pkgCache::VerIterator ver, bool match_all)
{
  list<string> *l=get_tasks(pkg);

  if(!l)
    return false;

  for(list<string>::iterator i=l->begin();
      i!=l->end();
      ++i)
    if(string_matches(s, *i, match_all))
      return true;

  return false;
}
typedef pkg_string_matcher<task_matches> pkg_task_matcher;

//  Package-file info matchers.  Match a package if any of its
// available files (for all versions) match the given criteria.
//
//  Should I use templates?
bool origin_matches(string s,
		    pkgCache::PkgIterator pkg,
		    pkgCache::VerIterator ver, bool match_all)
{
  for(pkgCache::VerIterator v=pkg.VersionList(); !v.end(); ++v)
    for(pkgCache::VerFileIterator f=v.FileList(); !f.end(); ++f)
      {
	pkgCache::PkgFileIterator cur=f.File();

	if(!cur.end() && cur.Origin() && string_matches(s,
							cur.Origin(),
							match_all))
	  return true;
      }

  return false;
}
typedef pkg_string_matcher<origin_matches> pkg_origin_matcher;

bool archive_matches(string s,
		     pkgCache::PkgIterator pkg,
		     pkgCache::VerIterator ver, bool match_all)
{
  if(ver.end() || ver.FileList().end())
    return false;

  for(pkgCache::VerIterator v=pkg.VersionList(); !v.end(); ++v)
    for(pkgCache::VerFileIterator f=v.FileList(); !f.end(); ++f)
      {
	pkgCache::PkgFileIterator cur=f.File();

	if(!cur.end() && cur.Archive() && string_matches(s,
							 cur.Archive(),
							 match_all))
	  return true;
      }

  return false;
}
typedef pkg_string_matcher<archive_matches> pkg_archive_matcher;

class pkg_auto_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return
      (!pkg.CurrentVer().end() || (*apt_cache_file)[pkg].Install()) &&
      ((*apt_cache_file)->get_ext_state(pkg).installremove_reason!=aptitudeDepCache::manual);
  }
};

class pkg_broken_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
	aptitudeDepCache::StateCache &state=(*apt_cache_file)[pkg];
	return state.NowBroken() || state.InstBroken();
      }
  }
};

// Matches packages with unmet dependencies of a particular type.
class pkg_broken_type_matcher:public pkg_matcher
{
  pkgCache::Dep::DepType type; // Which type to match
public:
  pkg_broken_type_matcher(pkgCache::Dep::DepType _type)
    :type(_type) {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
	pkgCache::DepIterator dep=ver.DependsList();

	while(!dep.end())
	  {
	    // Skip to the end of the Or group to check GInstall
	    while(dep->CompareOp & pkgDepCache::Dep::Or)
	      ++dep;

	    if(dep->Type==type &&
	       !((*apt_cache_file)[dep]&pkgDepCache::DepGInstall))
	      // Oops, it's broken..
	      return true;

	    ++dep;
	  }
      }
    return false;
  }
};

// This matches packages based on the action that will be taken with them.
//
// It will treat a request for a non-auto type as also being a request
// for the auto type.
class pkg_action_matcher:public pkg_matcher
{
  pkg_action_state type;
  bool require_purge;
public:
  pkg_action_matcher(pkg_action_state _type, bool _require_purge)
    :type(_type), require_purge(_require_purge)
  {
  }

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(require_purge &&
       ((*apt_cache_file)[pkg].iFlags & pkgDepCache::Purge) == 0)
      return false;
    else
      {
	pkg_action_state thetype=find_pkg_state(pkg);

	switch(type)
	  {
	  case pkg_install:
	    return thetype==pkg_install || thetype==pkg_auto_install;
	  case pkg_hold:
	    return !pkg.CurrentVer().end() && ((*apt_cache_file)[pkg].Held() || (*apt_cache_file)->get_ext_state(pkg).selection_state==pkgCache::State::Hold);
	  case pkg_remove:
	    return thetype==pkg_remove || thetype==pkg_auto_remove ||
	      thetype==pkg_unused_remove;
	  default:
	    return thetype==type;
	  }
      }
  }
};

class pkg_virtual_matcher:public pkg_matcher
// Matches *pure* virtual packages -- ie, those that /have/ no versions (and
// therefore will always pass in an 'end' version)
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return ver.end();
  }
};

class pkg_installed_matcher:public pkg_matcher
// Matches packages which are installed..slightly quirky in that it matches
// *any* version of an installed package.
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return !pkg.CurrentVer().end();
  }
};

class pkg_essential_matcher:public pkg_matcher
// Matches essential packages
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return
      (pkg->Flags&pkgCache::Flag::Essential)==pkgCache::Flag::Essential ||
      (pkg->Flags&pkgCache::Flag::Important)==pkgCache::Flag::Important;
  }
};

class pkg_configfiles_matcher:public pkg_matcher
// Matches a package which was removed but has config files remaining
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return pkg->CurrentState==pkgCache::State::ConfigFiles;
  }
};

// Matches packages with a dependency on the given pattern.
// FIXME: make this more general (for recommends, eg?)
class pkg_dep_matcher:public pkg_matcher
{
public:
  enum match_mode {depends, conflict};

private:
  pkg_matcher *pattern;
  match_mode mode;

public:
  pkg_dep_matcher(pkg_matcher *_pattern,
		  match_mode _mode)
  {
    pattern=_pattern;
    mode=_mode;
  }
  ~pkg_dep_matcher() {delete pattern;}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    assert(!pkg.end());
    if(ver.end())
      return false;

    for(pkgCache::DepIterator dep=ver.DependsList(); !dep.end(); ++dep)
      {
	if( (mode==depends && (dep->Type==pkgCache::Dep::Depends ||
			       dep->Type==pkgCache::Dep::PreDepends)) ||
	    (mode==conflict && (dep->Type==pkgCache::Dep::Conflicts)))
	  {
	    for(pkgCache::VerIterator i=dep.TargetPkg().VersionList(); !i.end(); i++)
	      if(_system->VS->CheckDep(i.VerStr(), dep->CompareOp, dep.TargetVer()))
		if(pattern->matches(dep.TargetPkg(), i))
		  return true;
	  }
      }

    return false;
  }
};

class pkg_or_matcher:public pkg_matcher
{
  pkg_matcher *left,*right;
public:
  pkg_or_matcher(pkg_matcher *_left, pkg_matcher *_right):left(_left),right(_right) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return left->matches(pkg, ver) || right->matches(pkg, ver);
  }
  ~pkg_or_matcher() {delete left; delete right;}
};

class pkg_and_matcher:public pkg_matcher
{
  pkg_matcher *left,*right;
public:
  pkg_and_matcher(pkg_matcher *_left, pkg_matcher *_right):left(_left),right(_right) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return left->matches(pkg, ver) && right->matches(pkg, ver);
  }
  ~pkg_and_matcher() {delete left; delete right;}
};

class pkg_not_matcher:public pkg_matcher
{
  pkg_matcher *child;
public:
  pkg_not_matcher(pkg_matcher *_child):child(_child) {}
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return !child->matches(pkg, ver);
  }
  ~pkg_not_matcher() {delete child;}
};

// Matches packages that were garbage-collected.
class pkg_garbage_matcher:public pkg_matcher
{
public:
  pkg_garbage_matcher() {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      return (*apt_cache_file)->get_ext_state(pkg).garbage;
  }
};

//  Now back from the dead..it seems some people were actually using it ;-)
//
// Matches (non-virtual) packages which no installed package declares
// an "important" dependency of the given type on.
//
// Note that the notion of "importantness" is affected by the current
// settings!
class pkg_norevdep_matcher:public pkg_matcher
{
public:
  pkg_norevdep_matcher() {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
        pkgCache::DepIterator dep=pkg.RevDependsList();

        while(!dep.end())
          {
            if((*apt_cache_file)->GetPolicy().IsImportantDep(dep) &&
               !dep.ParentVer().ParentPkg().CurrentVer().end())
              return false;

            ++dep;
          }

        return true;
      }
  }
};

// Matches (non-virtual) packages which no installed package declares
// a dependency of the given type on.
class pkg_norevdep_type_matcher:public pkg_matcher
{
  pkgCache::Dep::DepType type; // Which type to match
public:
  pkg_norevdep_type_matcher(pkgCache::Dep::DepType _type)
    :type(_type) {}

  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    if(ver.end())
      return false;
    else
      {
	pkgCache::DepIterator dep=pkg.RevDependsList();

	while(!dep.end())
	  {
	    // Return false if the depender is installed.
	    if(dep->Type==type &&
	       !dep.ParentVer().ParentPkg().CurrentVer().end())
	      return false;

	    ++dep;
	  }
      }
    return true;
  }
};

class pkg_new_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    // Don't match virtual packages.
    if(pkg.VersionList().end())
      return false;
    else
      return (*apt_cache_file)->get_ext_state(pkg).new_package;
  }
};

class pkg_upgradable_matcher:public pkg_matcher
{
public:
  bool matches(pkgCache::PkgIterator pkg, pkgCache::VerIterator ver)
  {
    return !pkg.CurrentVer().end() && (*apt_cache_file)[pkg].Upgradable();
  }
};

// Parses a dependency type.  Returns (ick) -1 if the type is not
// recognized.
pkgCache::Dep::DepType parse_deptype(string s)
{
  if(!strcasecmp(s.c_str(), "depends"))
    return pkgCache::Dep::Depends;
  if(!strcasecmp(s.c_str(), "predepends"))
    return pkgCache::Dep::PreDepends;
  if(!strcasecmp(s.c_str(), "recommends"))
    return pkgCache::Dep::Recommends;
  else if(!strcasecmp(s.c_str(), "suggests"))
    return pkgCache::Dep::Suggests;
  else if(!strcasecmp(s.c_str(), "conflicts"))
    return pkgCache::Dep::Conflicts;
  else // ewww.
    return (pkgCache::Dep::DepType) -1;
}

pkg_matcher *parse_condition_list(string s, unsigned int &loc,
				  bool &exact_match, bool flag_errors);

pkg_matcher *parse_atom(string s, unsigned int &loc,
			bool &exact_match, bool flag_errors)
{
  while(loc<s.size() && isspace(s[loc]))
    loc++;

  while(loc<s.size() && s[loc]!='|' && s[loc]!=')')
    {
      if(s[loc]=='!')
	{
	  loc++;
	  pkg_matcher *atom=parse_atom(s, loc, exact_match, flag_errors);
	  if(!atom)
	    return NULL;
	  return new pkg_not_matcher(atom);
	}
      else if(s[loc]=='(')
	// Recurse into the list
	{
	  loc++;
	  pkg_matcher *lst=parse_condition_list(s, loc,
						exact_match, flag_errors);
	  if(!lst)
	    return NULL;

	  if(!(loc<s.size() && s[loc]==')'))
	    {
	      if(flag_errors)
		_error->Error(_("Unmatched '('"));
	      delete lst;
	      return NULL;
	    }
	  else
	    {
	      loc++;
	      return lst;
	    }
	}
      else if(s[loc]=='~')
	{
	  if(loc+1==s.size())
	    {
	      loc++;
	      return new pkg_name_matcher("~", false);
	    }
	  else
	    {
	      unsigned int prevloc=loc;
	      loc+=2;
	      switch(s[prevloc+1])
		// Nested switch statements, mmmm...
		// Ok, there really is a reason here.  For all of the match
		// types that need a string argument, some prefix code (see
		// below) is needed to find the string's end.  But this would
		// be worse than unnecessary for the others.  So I have this
		// double check -- first test for anything that doesn't need
		// the prefix code, then work out which of the other forms
		// we have.
		{
		case 'v':
		  return new pkg_virtual_matcher;
		case 'b':
		  return new pkg_broken_matcher;
		case 'g':
		  return new pkg_garbage_matcher;
		case 'r':
		  return new pkg_norevdep_matcher;
		case 'c':
		  return new pkg_configfiles_matcher;
		case 'i':
		  return new pkg_installed_matcher;
		case 'E':
		  return new pkg_essential_matcher;
		case 'e':
		  exact_match=!exact_match;
		  break;
		case 'M':
		  return new pkg_auto_matcher;
		case 'N':
		  return new pkg_new_matcher;
		case 'U':
		  return new pkg_upgradable_matcher;
		case 'D':
		case 'C':
		  {
		    pkg_matcher *m=parse_atom(s, loc, exact_match, flag_errors);

		    if(!m)
		      return NULL;

		    if(s[prevloc+1]=='D')
		      return new pkg_dep_matcher(m, pkg_dep_matcher::depends);
		    else
		      return new pkg_dep_matcher(m, pkg_dep_matcher::conflict);
		  }
		default:
		  while(loc<s.size() &&
			s[loc]!='(' &&
			s[loc]!=')' &&
			s[loc]!='!' &&
			s[loc]!=')' &&
			s[loc]!='~' &&
			s[loc]!='|')
		    loc++;
		  switch(s[prevloc+1])
		    {
		    case '~':
		      return new pkg_name_matcher(s.substr(prevloc+1, loc-prevloc-1), exact_match);
		    case 'a':
		      {
			string type=s.substr(prevloc+2, loc-prevloc-2);

			// Match packages to be installed
			if(!strcasecmp(type.c_str(), "install"))
			  return new pkg_action_matcher(pkg_install, false);

			// Match packages to be upgraded
			else if(!strcasecmp(type.c_str(), "upgrade"))
			  return new pkg_action_matcher(pkg_upgrade, false);

			// Match packages to be removed OR purged
			else if(!strcasecmp(type.c_str(), "remove"))
			  return new pkg_action_matcher(pkg_remove, false);

			// Match packages to be purged
			else if(!strcasecmp(type.c_str(), "purge"))
			  return new pkg_action_matcher(pkg_remove, true);

			// Match packages to be reinstalled
			else if(!strcasecmp(type.c_str(), "reinstall"))
			  return new pkg_action_matcher(pkg_reinstall, false);

			// Match held packages
			else if(!strcasecmp(type.c_str(), "hold"))
			  return new pkg_action_matcher(pkg_hold, false);

			else
			  {
			    char buf[512];
			    snprintf(buf, 512, _("Unknown action type: %s"),
				     type.c_str());
			    if(flag_errors)
			      _error->Error("%s", buf);
			    return NULL;
			  }
		      }
		    case 'A':
		      return new pkg_archive_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 'B':
		    case 'R':
		      {
			string type=s.substr(prevloc+2, loc-prevloc-2);
			pkgCache::Dep::DepType ptype=parse_deptype(type);

			if(ptype!=-1)
			  {
			    switch(s[prevloc+1])
			      {
			      case 'B':
				return new pkg_broken_type_matcher(ptype);
			      case 'R':
				return new pkg_norevdep_type_matcher(ptype);
			      }
			  }
			else
			  {
			    char buf[512];
			    snprintf(buf, 512, _("Unknown dependency type: %s"),
				     type.c_str());
			    if(flag_errors)
			      _error->Error("%s", buf);
			    return NULL;
			  }
		      }
		    case 'n':
		      return new pkg_name_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 'd':
		      return new pkg_description_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 'm':
		      return new pkg_maintainer_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 'O':
		      return new pkg_origin_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 'p':
		      return new pkg_priority_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 's':
		      return new pkg_section_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 'V':
		      return new pkg_version_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    case 't':
		      return new pkg_task_matcher(s.substr(prevloc+2, loc-prevloc-2), exact_match);
		    default:
		      {
			char buf[512];
			snprintf(buf, 512, _("Unknown pattern type: %c"), s[1]);
			if(flag_errors)
			  _error->Error("%s", buf);
			return NULL;
		      }
		    }
		}
	    }
	}
      else
	{
	  unsigned int prevloc=loc;
	  while(loc<s.size() && s[loc]!='(' && s[loc]!=')' && s[loc]!='!' && s[loc]!=')' && s[loc]!='~' && s[loc]!='|')
	    loc++;
	  return new pkg_name_matcher(s.substr(prevloc, loc-prevloc), exact_match);
	}
    }

  // If we get here, the string was empty.
  return NULL;
}

pkg_matcher *parse_and_group(string s, unsigned int &loc,
			     bool &exact_match, bool flag_errors)
{
  pkg_matcher *rval=NULL;
  while(loc<s.size() && isspace(s[loc]))
    loc++;

  while(loc<s.size() && s[loc]!='|' && s[loc]!=')')
    {
      pkg_matcher *atom=parse_atom(s, loc, exact_match, flag_errors);
      if(!atom)
	{
	  delete rval;
	  return NULL;
	}

      if(rval==NULL)
	rval=atom;
      else
	rval=new pkg_and_matcher(rval, atom);

      while(loc<s.size() && isspace(s[loc]))
	loc++;
    }
  if(rval==NULL && flag_errors)
    _error->Error(_("Unexpected empty expression"));
  return rval;
}

pkg_matcher *parse_condition_list(string s, unsigned int &loc,
				  bool &exact_match, bool flag_errors)
  // This (and the grammer) should be optimized to avoid deep recursion
  // what does the above line mean? -- DNB 11/21/01
{
  pkg_matcher *grp=parse_and_group(s, loc, exact_match, flag_errors);
  if(!grp)
    return NULL;

  while(loc<s.size() && isspace(s[loc]))
    loc++;

  while(loc<s.size() && s[loc]!=')')
    {
      if(loc<s.size() && s[loc]=='|')
	{
	  loc++;
	  pkg_matcher *grp2=parse_condition_list(s, loc,
						 exact_match, flag_errors);
	  if(!grp2)
	    {
	      delete grp;
	      return NULL;
	    }
	  return new pkg_or_matcher(grp, grp2);
	}
      else
	// We should never come here in a well-formed expression
	{
	  delete grp;
	  if(flag_errors)
	    _error->Error(_("Badly formed expression"));
	  return NULL;
	}
      // Or here.
      while(loc<s.size() && isspace(s[loc]))
	loc++;
    }
  // Nothing else?  Well..
  return grp;
}

pkg_matcher *parse_pattern(string s, bool flag_errors)
{
  unsigned int loc=0;
  bool exact_match=false;

  // Just filter blank strings out immediately.
  while(loc!=s.size() && isspace(s[loc]))
    loc++;

  if(loc==s.size())
    return NULL;

  pkg_matcher *rval=parse_condition_list(s, loc, exact_match, flag_errors);

  if(rval==NULL)
    return NULL;
  else if(loc!=s.size())
    {
      if(flag_errors)
	_error->Error(_("Unexpected ')'"));
      delete rval;
      return NULL;
    }
  else
    return rval;
}
