// cmine.cc
//
//  Copyright 2000 Daniel Burrows

#include "cmine.h"

#include <aptitude.h>
#include <ui.h>

#include <sigc++/bind.h>

#include <vscreen/config/keybindings.h>
#include <vscreen/config/colors.h>

#include <vscreen/vscreen_widget.h>
#include <vscreen/vscreen.h>

#include <vscreen/vs_button.h>
#include <vscreen/vs_center.h>
#include <vscreen/vs_editline.h>
#include <vscreen/vs_frame.h>
#include <vscreen/vs_label.h>
#include <vscreen/vs_radiogroup.h>
#include <vscreen/vs_table.h>
#include <vscreen/vs_togglebutton.h>
#include <vscreen/vs_util.h>

#include <string>
#include <fstream>

#ifndef DONT_USE_FANCYBOXES
// Some systems (*cough* Solaris xterms *cough*) don't like the fancy ASCII
// graphics provided by libcurses
#define MINE_ULCORNER ACS_ULCORNER
#define MINE_URCORNER ACS_URCORNER
#define MINE_HLINE ACS_HLINE
#define MINE_VLINE ACS_VLINE
#define MINE_LLCORNER ACS_LLCORNER
#define MINE_LRCORNER ACS_LRCORNER
#else
#define MINE_ULCORNER '+'
#define MINE_URCORNER '+'
#define MINE_HLINE '-'
#define MINE_VLINE '|'
#define MINE_LLCORNER '+'
#define MINE_LRCORNER '+'
#endif

using namespace std;

keybindings *cmine::bindings;

vs_editline::history_list cmine::load_history, cmine::save_history;

size cmine::size_request()
{
  if(board)
    return size(board->get_width(), 1+board->get_height());
  else
    return size(0, 1);
}

void cmine::update_header()
{
  timeout_num=vscreen_addtimeout(slot(this, &cmine::update_header), 500);

  vscreen_update();
};

void cmine::paint_header()
  // Shows the header with its extra info
{
  if(board)
    {
      int width,height;
      getmaxyx(height,width);

      string header="Minesweeper";
      char buf[200];

      if(board->get_state()==mine_board::playing)
	snprintf(buf,
		 sizeof(buf),
		 "%i/%i mines  %d %s",
		 board->get_nummines()-board->get_numflags(),
		 board->get_nummines(),
		 (int) board->get_duration(),
		 board->get_duration()==1?"second":"seconds");
      else
	snprintf(buf,
		 sizeof(buf),
		 "Minesweeper    %s in %d %s",
		 board->get_state()==mine_board::won?"Won":"Lost",
		 (int) board->get_duration(),
		 board->get_duration()==1?"second":"seconds");

      while(header.size()+strlen(buf)<(unsigned) width)
	header+=' ';

      unsigned int loc=0;
      while(header.size()<(unsigned) width)
	{
	  assert(loc<strlen(buf));
	  header+=buf[loc++];
	}

      display_header(header.c_str(), get_color("ScreenHeaderColor"));
    }
  else
    display_header("Minesweeper", get_color("ScreenHeaderColor"));
}

void cmine::do_load_game(string s)
{
  if(s!="")
    {
      ifstream in(s.c_str());
      if(!in)
	{
	  char buf[512];

	  snprintf(buf, 512, _("Could not open file \"%s\""), s.c_str());

	  popup_widget(vs_dialog_ok(buf, NULL, get_color("Error")));
	}
      else
	{
	  mine_board *brd=new mine_board;
	  // The advantage of creating a new board instead of loading into
	  // the current one is that we don't lose the current game if the
	  // user tries to load a game from, say, /etc/passwd.
	  if(!brd->load(in))
	    {
	      char buf[512];

	      snprintf(buf, 512, _("Could not load game from %s"), s.c_str());

	      popup_widget(vs_dialog_ok(buf, NULL, get_color("Error")));
	      delete brd;
	    }
	  else
	    {
	      set_board(brd);
	      vscreen_update();
	    }
	}
    }
}

void cmine::do_save_game(string s)
{
  if(s!="")
    {
      ofstream out(s.c_str());
      if(!out)
	{
	  char buf[512];

	  snprintf(buf, 512, "Could not open file \"%s\"", s.c_str());

	  popup_widget(vs_dialog_ok(buf, NULL, get_color("Error")));
	}
      else
	{
	  board->save(out);
	  vscreen_update();
	}
    }
}

void cmine::do_start_custom_game(vscreen_widget *w,
				 vs_editline *heightedit,
				 vs_editline *widthedit,
				 vs_editline *minesedit)
{
  string s=heightedit->get_text();

  char *end=const_cast<char *>(s.c_str());

  long height=strtol(s.c_str(), &end, 0);

  if(s.c_str()[0]=='\0' || *end!='\0')
    {
      popup_widget(vs_dialog_ok(_("The board height must be a number"),
				NULL,
				get_color("Error")));
      return;
    }

  s=widthedit->get_text();
  end=const_cast<char *>(s.c_str());
  long width=strtol(s.c_str(), &end, 0);

  if(s.c_str()[0]=='\0' || *end!='\0')
    {
      popup_widget(vs_dialog_ok(_("The board width must be a number"),
				NULL,
				get_color("Error")));
      return;
    }

  s=minesedit->get_text();
  end=const_cast<char *>(s.c_str());
  long mines=strtol(s.c_str(), &end, 0);

  if(s.c_str()[0]=='\0' || *end!='\0')
    {
      // awkward wording
      popup_widget(vs_dialog_ok(_("The number of mines must be a number"),
				NULL,
				get_color("Error")));
      return;
    }

  w->destroy();

  set_board(new mine_board(width, height, mines));
}

void cmine::do_custom_game()
{
  int attr=get_color("DefaultWidgetBackground")|A_REVERSE;

  vs_center *center=new vs_center;

  vs_table *table=new vs_table;
  table->set_bg(attr);

  vs_label *overalllabel=new vs_label(_("Setup custom game"),
				      attr);

  vs_label *heightlabel=new vs_label(_("Height of board: "),
				     attr);
  vs_editline *heightedit=new vs_editline("");

  vs_label *widthlabel=new vs_label(_("Width of board: "),
				    attr);
  vs_editline *widthedit=new vs_editline("");

  vs_label *mineslabel=new vs_label(_("Number of mines: "),
				    attr);
  vs_editline *minesedit=new vs_editline("");

  vs_button *okbutton=new vs_button("Ok");
  okbutton->set_bg(attr);
  vs_button *cancelbutton=new vs_button("Cancel");
  cancelbutton->set_bg(attr);

  table->connect_key("Confirm", &global_bindings, okbutton->pressed.slot());

  okbutton->pressed.connect(bind(bind(slot(this, &cmine::do_start_custom_game),
				      widthedit,
				      minesedit),
				 center,
				 heightedit));
  cancelbutton->pressed.connect(slot(center, &vscreen_widget::destroy));

  table->connect_key("Cancel", &global_bindings, cancelbutton->pressed.slot());

  vs_center *cyes=new vs_center(okbutton);
  vs_center *cno=new vs_center(cancelbutton);

  table->add_widget(overalllabel, 0, 0, 1, 2, true, false);

  table->add_widget(heightlabel, 1, 0, 1, 1, true, false);
  table->add_widget(heightedit, 1, 1, 1, 1, true, false);

  table->add_widget(widthlabel, 2, 0, 1, 1, true, false);
  table->add_widget(widthedit, 2, 1, 1, 1, true, false);

  table->add_widget(mineslabel, 3, 0, 1, 1, true, false);
  table->add_widget(minesedit, 3, 1, 1, 1, true, false);

  table->add_widget(cyes, 4, 0, 1, 1, false, false);
  table->add_widget(cno, 4, 1, 1, 1, false, false);

  overalllabel->show();
  heightlabel->show();
  heightedit->show();
  widthlabel->show();
  widthedit->show();
  mineslabel->show();
  minesedit->show();
  okbutton->show();
  cancelbutton->show();
  cyes->show();
  cno->show();

  vs_frame *frame=new vs_frame(table);
  frame->set_bg(attr);
  center->add_widget(frame);

  popup_widget(center);
}

void cmine::do_new_game()
{
  int attr=get_color("DefaultWidgetBackground")|A_REVERSE;

  vs_center *center=new vs_center;

  vs_table *table=new vs_table;
  table->set_bg(attr);

  vs_label *overalllabel=new vs_label(_("Choose difficulty level"), attr);

  vs_radiobutton *easybutton=new vs_radiobutton(_("Easy"), true);
  vs_radiobutton *mediumbutton=new vs_radiobutton(_("Medium"), false);
  vs_radiobutton *hardbutton=new vs_radiobutton(_("Hard"), false);
  vs_radiobutton *custombutton=new vs_radiobutton(_("Custom"), false);

  easybutton->set_bg(attr);
  mediumbutton->set_bg(attr);
  hardbutton->set_bg(attr);
  custombutton->set_bg(attr);

  vs_button *okbutton=new vs_button("Ok");
  okbutton->set_bg(attr);
  vs_button *cancelbutton=new vs_button("Cancel");
  cancelbutton->set_bg(attr);

  table->connect_key("Confirm", &global_bindings, okbutton->pressed.slot());

  vs_radiogroup *grp=new vs_radiogroup;
  grp->add_button(easybutton, 0);
  grp->add_button(mediumbutton, 1);
  grp->add_button(hardbutton, 2);
  grp->add_button(custombutton, 3);

  okbutton->pressed.connect(bind(bind(slot(this, &cmine::do_continue_new_game),
				      grp),
				 true,
				 center));
  cancelbutton->pressed.connect(bind(bind(slot(this, &cmine::do_continue_new_game),
					  grp),
				     false,
				     center));

  vs_center *cok=new vs_center(okbutton);
  vs_center *ccancel=new vs_center(cancelbutton);

  table->add_widget(overalllabel, 0, 0, 1, 2, true, false);
  table->add_widget(easybutton,   1, 0, 1, 2, true, false);
  table->add_widget(mediumbutton, 3, 0, 1, 2, true, false);
  table->add_widget(hardbutton,   4, 0, 1, 2, true, false);
  table->add_widget(custombutton, 5, 0, 1, 2, true, false);
  table->add_widget(cok,          6, 0, 1, 1, false, false);
  table->add_widget(ccancel,      6, 1, 1, 1, false, false);

  vs_frame *frame=new vs_frame(table);
  frame->set_bg(attr);
  center->add_widget(frame);

  popup_widget(center);
}

void cmine::do_continue_new_game(bool start,
				 vscreen_widget *w,
				 vs_radiogroup *grp)
{
  if(start)
    switch(grp->get_selected())
      {
      case 0:
	set_board(easy_game());
	break;
      case 1:
	set_board(intermediate_game());
	break;
      case 2:
	set_board(hard_game());
	break;
      case 3:
	do_custom_game();
	break;
      default:
	popup_widget(vs_dialog_ok("Internal error: execution reached an impossible point",
				  NULL,
				  get_color("Error")));
	break;
      }

  delete grp;
  w->destroy();
};

cmine::cmine():board(NULL)
{
  set_board(easy_game());
  vscreen_addtimeout(slot(this, &cmine::update_header), 500);
  //set_status(_("n - New Game  Cursor keys - move cursor  f - flag  enter - check  ? - help"));
}

void cmine::checkend()
  // Prints out silly messages when the player wins or loses
{
  if(board->get_state()==mine_board::won)
    // I should continue the Nethack theme, but I've never won so I don't
    // know what the message should be :)
    popup_widget(vs_dialog_ok(_("You have won.")));
  else if(board->get_state()==mine_board::lost)
    {
      popup_widget(vs_dialog_ok(_("You lose!")));
#if 0
      // (messages in reverse order because the minibuf is a stack by default..
      // I could use the special feature of sticking them at the end, but I
      // want them to override whatever's there (probably nothing :) )
      add_status_widget(new vs_label(_("You die...  --More--"),
				     retr_status_color()));
      switch(rand()/(RAND_MAX/8))
	{
	case 0:
	case 1:
	case 2:
	case 3:
	  if(rand()<(RAND_MAX/2))
	    {
	      if(rand()<(RAND_MAX/3))
		{
		  if(rand()<(RAND_MAX/2))
		    add_status_widget(new vs_label(_("The spikes were poisoned!  The poison was deadly..  --More--"),
						   retr_status_color()));

		  add_status_widget(new vs_label(_("You land on a set of sharp iron spikes!  --More--"),
						 retr_status_color()));
		}
	      add_status_widget(new vs_label(_("You fall into a pit!  --More--"),
					     retr_status_color()));
	    }
	  add_status_widget(new vs_label(_("KABOOM!  You step on a land mine.  --More--"),
					 retr_status_color()));
	  break;
	case 4:
	  if(rand()<RAND_MAX/2)
	    add_status_widget(new vs_label(_("The dart was poisoned!  The poison was deadly...  --More--"),
					   retr_status_color()));
	  add_status_widget(new vs_label(_("A little dart shoots out at you!  You are hit by a little dart!  --More--"),
					 retr_status_color()));
	  break;
	case 5:
	  add_status_widget(new vs_label(_("You turn to stone... --More--"),
					 retr_status_color()));
	  add_status_widget(new vs_label(_("Touching the cockatrice corpse was a fatal mistake.  --More--"),
					 retr_status_color()));
	  add_status_widget(new vs_label(_("You feel here a cockatrice corpse.  --More--"),
					 retr_status_color()));
	  break;
	case 6:
	  add_status_widget(new vs_label(_("Click!  You trigger a rolling boulder trap!  You are hit by a boulder! --More--"),
					 retr_status_color()));
	  break;
	case 7:
	  if(rand()<(RAND_MAX/2))
	    {
	      string type;
	      switch(rand()/(RAND_MAX/8))
		{
		case 0:
		  type="sleep";
		  break;
		case 1:
		  type="striking";
		  break;
		case 2:
		  type="death";
		  break;
		case 3:
		  type="polymorph";
		  break;
		case 4:
		  type="magic missile";
		  break;
		case 5:
		  type="secret door detection";
		  break;
		case 6:
		  type="invisibility";
		  break;
		case 7:
		  type="cold";
		  break;
		}

	      char buf[512];

	      snprintf(buf, 512, _("Your wand of %s breaks apart and explodes!  --More--"));

	      add_status_widget(new vs_label(buf,
					     retr_status_color()));
	    }

	  add_status_widget(new vs_label(_("You are jolted by a surge of electricity!  --More--"),
					 retr_status_color()));
	  break;
	}
#endif
    }
}

void cmine::set_board(mine_board *_board)
{
  int width, height;
  getmaxyx(height, width);

  delete board;
  board=_board;

  curx=_board->get_width()/2;
  cury=_board->get_height()/2;

  basex=(width-_board->get_width()-2)/2;
  basey=(height-_board->get_height()-4)/2;

  vscreen_update();
}

bool cmine::handle_char(chtype ch)
  // Input-handling routine.
{
  int width,height;
  getmaxyx(height,width);
  // Not all branches need it but it's cheap..

  if(bindings->key_matches(ch, "MineUncoverSweepSquare"))
    {
      if(board->get_state()==mine_board::playing)
	{
	  if(!board->get_square(curx, cury).uncovered)
	    board->uncover(curx, cury);
	  else
	    board->sweep(curx, cury);

	  checkend();
	}
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "MineUncoverSquare"))
    {
      board->uncover(curx, cury);
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "MineSweepSquare"))
    {
      board->sweep(curx, cury);
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "MineFlagSquare"))
    {
      board->toggle_flag(curx, cury);
      // FIXME: handle errors?
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "Up"))
    {
      if(cury>0)
	{
	  cury--;
	  while(basey+cury+1<1)
	    basey++;
	  vscreen_update();
	}
    }
  else if(bindings->key_matches(ch, "Down"))
    {
      if(cury<board->get_height()-1)
	{
	  cury++;
	  while(basey+cury+1>=height-3)
	    basey--;
	  vscreen_update();
	}
    }
  else if(bindings->key_matches(ch, "Left"))
    {
      if(curx>0)
	{
	  curx--;
	  while(basex+curx+1<1)
	    basex++;
	  vscreen_update();
	}
    }
  else if(bindings->key_matches(ch, "Right"))
    {
      if(curx<board->get_width()-1)
	{
	  curx++;
	  while(basex+curx+1>=width-1)
	    basex--;
	  vscreen_update();
	}
    }
  else if(bindings->key_matches(ch, "MineNewGame"))
    do_new_game();
  else if(bindings->key_matches(ch, "MineLoadGame"))
    prompt_string(_("Enter the filename to load: "),
		  "",
		  slot(this, &cmine::do_load_game),
		  NULL,
		  NULL,
		  &load_history);
  else if(bindings->key_matches(ch, "MineSaveGame"))
    prompt_string(_("Enter the filename to save: "),
		  "",
		  slot(this, &cmine::do_save_game),
		  NULL,
		  NULL,
		  &save_history);
  else if(bindings->key_matches(ch, "Help"))
    {
      char buf[512];

      snprintf(buf, 512, HELPDIR "/%s", _("mine-help.txt"));

      vscreen_widget *w=vs_dialog_fileview(buf);
      w->show_all();

      popup_widget(w);
    }
  else
    return false;

  return true;
}

void cmine::paint_square(int x, int y)
  // Displays a single square.
{
  int width,height;
  getmaxyx(height, width);

  assert(x>=0 && x<board->get_width());
  assert(y>=0 && y<board->get_height());

  int screenx=basex+1+x,screeny=(basey+1)+1+y;

  if(screenx>=0 && screenx<width && screeny>=0 && screeny<height-1)
    {
      chtype ch;
      const mine_board::board_entry &entry=board->get_square(x, y);
      // We want to handle the case of 'game-over' differently from
      // the case of 'still playing' -- when the game is over, all
      // the mines should be revealed.
      if(board->get_state()==mine_board::playing)
	{
	  if(!entry.uncovered)
	    {
	      if(entry.flagged)
		ch='F'|get_color("MineFlagColor");
	      else
		ch=' '|get_color("ScreenBackgroundColor");
	    }
	  else if(entry.contains_mine)
	    ch='^'|get_color("MineBombColor");
	  else if(entry.adjacent_mines==0)
	    ch='.'|get_color("ScreenBackgroundColor");
	  else
	    ch=('0'+entry.adjacent_mines)|get_color("ScreenBackgroundColor");
	}
      else
	{
	  if(entry.contains_mine)
	    {
	      if(board->get_state()==mine_board::lost &&
		 x==board->get_minex() && y==board->get_miney())
		ch='*'|get_color("MineDetonatedColor");
	      else
		ch='^'|get_color("MineBombColor");
	    }
	  else if(entry.uncovered)
	    {
	      if(entry.adjacent_mines==0)
		ch='.'|get_color("ScreenBackgroundColor");
	      else
		ch=('0'+entry.adjacent_mines)|get_color("ScreenBackgroundColor");
	    }
	  else
	    ch=' '|get_color("ScreenBackgroundColor");
	}
      if(board->get_state()==mine_board::playing && x==curx && y==cury)
	ch|=A_REVERSE;
      mvaddch(screeny, screenx, ch);
    }
}

void cmine::paint()
{
  if(get_win())
    {
      int width,height;
      getmaxyx(height, width);

      if(height!=prevheight || width!=prevwidth)
	// If the window size has changed (or we just got a window for the
	// first time) we need to reset the boundaries.  Really, this ought to
	// be done by a callback (aka virtual function) that gets called for
	// the first assignment of a cwindow to the vscreen or when the
	// screen resizes.
	{
	  prevwidth=width;
	  prevheight=height;

	  curx=board->get_width()/2;
	  cury=board->get_height()/2;

	  basex=(width-board->get_width()-2)/2;
	  basey=(height-board->get_height()-4)/2;
	}

      paint_header();

      if(board)
	{
	  attrset(get_color("ScreenBackgroundColor"));

	  int right=basex+board->get_width()+1, down=basey+1+board->get_height()+1;
	  // x and y coordinates, respectively, of the right and lower board
	  // edges.
	  int line_start_x=basex>0?basex+1:1, line_start_y=basey>=0?basey+2:1;
	  // The starting coordinates of the horizontal and vertical lines,
	  // respectively.

	  int horiz_line_width, vert_line_height;
	  if(basex+board->get_width()>=width)
	    horiz_line_width=width-line_start_x;
	  else
	    horiz_line_width=basex+1+board->get_width()-line_start_x;

	  if(basey+board->get_height()>=height-2)
	    vert_line_height=height-1-line_start_y;
	  else
	    vert_line_height=basey+2+board->get_height()-line_start_y;
	  // Calculate the widths of the sides of the board.  This might look
	  // a little like black voodoo magic.  It is.  You wouldn't believe
	  // how many chickens I had to..er, nevermind :)
	  // Probably this could be cleaned up; there seems to be a lot of
	  // confusion resulting from the value of basey.  I think now that
	  // basey+y should go from 1 to height instead of 0 to height-1 -- it
	  // might make things more confusing elsewhere but would simplify
	  // the drawing routines tremendously.

	  // The reason that we only use 'basey+1' below is that we have to
	  // draw the board inside the 'main' window provided by the
	  // minibuf_win.
	  //
	  //  A better long-term solution is to have the minibuf_win divide
	  // itself into three subwindows and hand us one.  This may happen
	  // eventually, but maybe after I hand the assignment in :)

	  int minx=(basex<-1?-basex:0),miny=(basey<-1?-1-basey:0);
	  // The /board coordinates/ of the minimal x and y values visible
	  int maxx,maxy;
	  if(basex+1+board->get_width()>width)
	    maxx=minx+(basex==0?width-1:width);
	  // The visible area is effectively one square thinner when basex==0.
	  else
	    maxx=board->get_width();

	  if(basey+1+board->get_height()>height-2)
	    maxy=miny+(basey==0?height-3:height-2);
	  else
	    maxy=board->get_height();
	  // A hairy expression for /one plus/ the maximum visible coordinates
	  // (the one plus is to make the loop below slightly simpler)

	  /////////////////////////////////////////////////
	  // Start drawing; first the border:
	  if(basey>=0)
	    {
	      if(basex>=0)
		mvaddch(basey+1, basex, MINE_ULCORNER);

	      mvhline(basey+1, line_start_x, MINE_HLINE, horiz_line_width);
	    }

	  if(basex>=0)
	    mvvline(line_start_y, basex, MINE_VLINE, vert_line_height);

	  if(right<width)
	    {
	      if(basey>=0)
		mvaddch(basey+1, right, MINE_URCORNER);
	      mvvline(line_start_y, right, MINE_VLINE, vert_line_height);
	      if(down<height-1)
		mvaddch(down, right, MINE_LRCORNER);
	    }

	  if(down<height-1)
	    {
	      if(basex>=0)
		mvaddch(down, basex, MINE_LLCORNER);
	      mvhline(down, line_start_x, MINE_HLINE, horiz_line_width);
	    }

	  // Now the squares:
	  for(int y=miny; y<maxy; y++)
	    for(int x=minx; x<maxx; x++)
	      paint_square(x, y);
	}
    }
}

void cmine::init_bindings()
{
  srand(time(0));

  set_color("MineFlagColor", COLOR_RED, COLOR_BLACK, A_BOLD);
  set_color("MineBombColor", COLOR_RED, COLOR_BLACK, A_BOLD);
  set_color("MineDetonatedColor", COLOR_CYAN, COLOR_BLACK, 0);

  global_bindings.set("MineUncoverSweepSquare", KEY_ENTER);
  global_bindings.set("MineFlagSquare", 'f');
  global_bindings.set("MineNewGame", 'n');
  global_bindings.set("MineSaveGame", 's');
  global_bindings.set("MineLoadGame", 'l');

  bindings=new keybindings(&global_bindings);
}
