// vscreen.h                          -*-c++-*-
//
//  Copyright 2000 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  This file declares general routines and interfaces for the vscreen
// code.  Formerly this declared a vscreen class, but with the coming
// of the Great Widget Revolution, things changed..a LOT.
//
//  Basically, you can set the toplevel widget here and run a main loop.
//  The toplevel widget is responsible for almost everything else.

#ifndef VSCREEN_H
#define VSCREEN_H

#include <sigc++/basic_signal.h>

class vscreen_widget;

void vscreen_init();
// Performs initialization tasks (including calling init_curses())

void vscreen_settoplevel(vscreen_widget *widget);
// Changes the toplevel widget

void vscreen_queuelayout();
// Queues a re-layout of all widgets (needed when something has to be resized)

void vscreen_layoutnow();
// Lays out all widgets immediately.

//   Main loop handlers:

void vscreen_mainloop(int synch=0);
// Enters a loop, calling getch() over and over and over again..
// A valid vscreen must be currently displayed.

bool vscreen_poll();
//  Checks for pending input and dispatches any it finds and calls any needed
// timeouts.  Can be called from within a input-handling or timeout routine
// called by vscreen_main(); there (should!) be no reentrance issues.
//
//  Returns true if pending input was found.

void vscreen_exitmain();
// Exits the main loop.

void vscreen_redraw();
// Redraws the screen completely from scratch

void vscreen_update();
// Redraws the screen; like display(); refresh(); except that it will handle
// overlapped widgets properly

void vscreen_tryupdate();
// Handles any pending draws/redraws.

void vscreen_updatecursor();
// Just updates the cursor status

void vscreen_suspend();
// Hides all virtual screens and suspends Curses operation until vscreen_resume
// is called.
void vscreen_resume();

int vscreen_addtimeout(const SigC::Slot0<void> &slot, int msecs);
// Adds a timer which will expire in the given number of milliseconds; when it
// does, to is called.  Returns the timeout's ID, used to delete it.
void vscreen_deltimeout(int id);

void vscreen_handleresize();
// Does anything needed to handle a window resize event.
// FIXME: I --think-- that this is now redundant

//  Thread safety routines
//  BE VERY CAREFUL WHEN USING THESE!
//
//  The threading system used by vscreen is very simple: there's one lock which
// is acquired before any vscreen-using activity takes place.  Normally
// programs never have to worry about this, as the main loop always acquires
// the lock before executing any callback code.  HOWEVER, if you wish to
// write a thread running in parallel to the main one which uses vscreens, it
// is imperative that you use vscreen_acquirelock() and vscreen_releaselock()
// INTELLIGENTLY!  Note also that you should be VERY CAREFUL about deleting
// vscreens from within threads other than the main loop (read: DON'T DO IT)
// as the main loop may hold a reference to the vscreen that it's reading input
// from!
//
//  One more note: treat vscreen_poll() as you would any other function; that
// is, BE SURE that you hold the global lock before entering it!

void vscreen_acquirelock();
// Acquire the global lock

void vscreen_releaselock();
// Release it.

extern SigC::Signal0<void> main_hook;
// Called right after we finish handling input in the mainloop.  Can be used
// (eg) to insert extra actions to be performed after all user-input (aptitude
// uses this to check for apt errors and pop up a message about them)

#endif
