/*
 * Copyright 1999, 2001 Silicon Graphics, Inc.
 * Copyright 2001 Ralf Baechle
 *           2001,2002 Guido Guenther <agx@sigxcpu.org>
 */
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <types.h>

#include <arc.h>
#include <elf.h>

#include <sys/types.h>
#include <linux/ext2_fs.h>
#include <ext2fs.h>

#include <asm/addrspace.h>
#include "arcboot.h"

#define ANSI_CLEAR	"\033[2J"
#define CONF_FILE	"/etc/arcboot.conf"

#define PAGE_SIZE	4096
#define STACK_PAGES	16

/*
 *  Reserve this memory for loading kernel
 *  Don't put loader structures there because they would be overwritten
 *  If you change this, make sure you edit the linker script too
 */
ULONG reserve_base = 0x88002000;
ULONG reserve_size = 0x030f000;

CHAR *OSLoadPartition = NULL;
CHAR *OSLoadFilename = NULL;
CHAR *OSLoadOptions = "";

static void Wait(const char *prompt)
{
	int ch;

	if (prompt != NULL)
		puts(prompt);

	do {
		ch = getchar();
	} while ((ch != EOF) && (((char) ch) != ' '));
}


void Fatal(const CHAR * message, ...)
{
	va_list ap;

	if (message != NULL) {
		printf("FATAL ERROR:  ");
		va_start(ap, message);
		vprintf(message, ap);
		va_end(ap);
	}

	Wait("\n\r--- Press <spacebar> to enter ARC interactive mode ---");
	ArcEnterInteractiveMode();
}


void InitMalloc(void)
{
	MEMORYDESCRIPTOR *current = NULL;
	ULONG stack = (ULONG) & current;
#ifdef DEBUG
	printf("stack starts at: 0x%x\n\r", stack);
#endif

	current = ArcGetMemoryDescriptor(current);
	if(! current ) {
		Fatal("Can't find any valid memory descriptors!\n\r");
	}
	while (current != NULL) {
		/*
		 *  The spec says we should have an adjacent FreeContiguous
		 *  memory area that includes our stack.  It would be much
		 *  easier to just look for that and give it to malloc, but
		 *  the Indy only shows FreeMemory areas, no FreeContiguous.
		 *  Oh well.
		 */
		if (current->Type == FreeMemory) {
			ULONG start = KSEG0ADDR(current->BasePage * PAGE_SIZE);
			ULONG end =
			    start + (current->PageCount * PAGE_SIZE);
#if DEBUG
			printf("Free Memory(%u) segment found at (0x%x,0x%x).\n\r",
					current->Type, start, end); 
#endif

			/* Leave some space for our stack */
			if ((stack >= start) && (stack < end))
				end =
				    (stack -
				     (STACK_PAGES *
				      PAGE_SIZE)) & ~(PAGE_SIZE - 1);
			/* Don't use memory from reserved region */
			if ((start >= reserve_base)
			    && (start < (reserve_base + reserve_size)))
				start = reserve_base + reserve_size;
			if ((end > reserve_base)
			    && (end <=
				(reserve_base + reserve_size))) end =
				    reserve_base;
			if (end > start) {
#ifdef DEBUG
				printf("Adding %u bytes at 0x%x to the list of available memory\n\r", 
						end-start, start);
#endif
				arclib_malloc_add(start, end - start);
			}
		}
		current = ArcGetMemoryDescriptor(current);
	}
}


void ProcessArguments(LONG argc, CHAR * argv[])
{
	LONG arg;
	CHAR *equals;
	size_t len;

	for (arg = 1; arg < argc; arg++) {
		equals = strchr(argv[arg], '=');
		if (equals != NULL) {
			len = equals - argv[arg];
			if (strncmp(argv[arg], "OSLoadPartition", len) == 0) 
				OSLoadPartition = equals + 1;
			if (strncmp(argv[arg], "OSLoadFilename", len) == 0)
				OSLoadFilename = equals + 1;
			if (strncmp(argv[arg], "OSLoadOptions", len) == 0) {
				/* Copy options to local memory to avoid overwrite later */
				OSLoadOptions = strdup(equals + 1);
				if (OSLoadOptions == NULL)
					Fatal ("Cannot allocate memory for options string\n\r");
			}
		}
	}
}


void LoadProgramSegments(ext2_file_t file, Elf32_Ehdr * header)
{
	int idx;
	Boolean loaded = False;
	Elf32_Phdr *segment, *segments;
	size_t size = header->e_phentsize * header->e_phnum;
	errcode_t status;

	if (size <= 0)
		Fatal("No program segments\n\r");

	segments = malloc(size);
	if (segments == NULL)
		Fatal("Cannot allocate memory for segment headers\n\r");

	status =
	    ext2fs_file_lseek(file, header->e_phoff, EXT2_SEEK_SET, NULL);
	if (status != 0) {
		print_ext2fs_error(status);
		Fatal("Cannot seek to program segment headers\n\r");
	}

	status = ext2fs_file_read(file, segments, size, NULL);
	if (status != 0) {
		print_ext2fs_error(status);
		Fatal("Cannot read program segment headers\n\r");
	}

	segment = segments;
	for (idx = 0; idx < header->e_phnum; idx++) {
		if (segment->p_type == PT_LOAD) {
			printf
			    ("Loading program segment %u at 0x%x, size = 0x%x\n\r",
			     idx + 1, KSEG0ADDR(segment->p_vaddr),
			     segment->p_filesz);

			status =
			    ext2fs_file_lseek(file, segment->p_offset,
					      EXT2_SEEK_SET, NULL);
			if (status != 0) {
				print_ext2fs_error(status);
				Fatal("Cannot seek to program segment\n\r");
			}

			status = ext2fs_file_read(file,
						  (void *) (KSEG0ADDR(
						            segment->p_vaddr)),
						  segment->p_filesz, NULL);
			if (status != 0) {
				print_ext2fs_error(status);
				Fatal("Cannot read program segment\n\r");
			}

			size = segment->p_memsz - segment->p_filesz;
			if (size > 0) {
				printf
				    ("Zeroing memory at 0x%x, size = 0x%x\n\r",
				     (KSEG0ADDR(segment->p_vaddr +
				      segment->p_filesz)), size);
				memset((void *)
				       (KSEG0ADDR(segment->
					 p_vaddr + segment->p_filesz)), 0, size);
			}

			loaded = True;
		}

		segment =
		    (Elf32_Phdr *) (((char *) segment) +
				    header->e_phentsize);
	}

	if (!loaded)
		Fatal("No loadable program segments found\n\r");

	free(segments);
}


ULONG LoadKernelFile(ext2_file_t file)
{
	Elf32_Ehdr header;
	errcode_t status;

	status =
	    ext2fs_file_read(file, (void *) &header, sizeof(header), NULL);
	if (status != 0) {
		print_ext2fs_error(status);
		Fatal("Can't read file header\n\r");
	}

	if (memcmp(&(header.e_ident[EI_MAG0]), ELFMAG, SELFMAG) != 0)
		Fatal("Not an ELF file\n\r");

	if (header.e_ident[EI_CLASS] != ELFCLASS32)
		Fatal("Not a 32-bit file\n\r");

	if (header.e_ident[EI_DATA] != ELFDATA2MSB)
		Fatal("Not a big-endian file\n\r");

	if (header.e_ident[EI_VERSION] != EV_CURRENT)
		Fatal("Wrong ELF version\n\r");

	if (header.e_type != ET_EXEC)
		Fatal("Not an executable file\n\r");

	if (header.e_machine != EM_MIPS)
		Fatal("Unsupported machine type\n\r");

	if (header.e_version != EV_CURRENT)
		Fatal("Wrong ELF version\n\r");

	LoadProgramSegments(file, &header);

	printf("Starting kernel; entry point = 0x%x\n\r",
	       ((ULONG) KSEG0ADDR(header.e_entry)));
	return KSEG0ADDR(header.e_entry);
}


Boolean OpenFile(const char *partition, const char *filename, ext2_file_t* file)
{
	extern io_manager arc_io_manager;
	ext2_filsys fs;
	ext2_ino_t file_inode;
	errcode_t status;

	initialize_ext2_error_table();

	status = ext2fs_open(partition, 0, 0, 0, arc_io_manager, &fs);
	if (status != 0) {
		print_ext2fs_error(status);
		return False;
	}

	status = ext2fs_namei_follow
	    (fs, EXT2_ROOT_INO, EXT2_ROOT_INO, filename, &file_inode);
	if (status != 0) {
		print_ext2fs_error(status);
		return False;
	}

	status = ext2fs_file_open(fs, file_inode, 0, file);
	if (status != 0) {
		print_ext2fs_error(status);
		return False;
	}
	return True;
}

ULONG LoadKernel(const char *partition, const char *filename)
{
	ext2_file_t file;

	if(!OpenFile( partition, filename, &file ))
		Fatal("Can't load kernel!\n\r");
	return LoadKernelFile(file);
}


void _start(LONG argc, CHAR * argv[], CHAR * envp[])
{
	CHAR** nargv;
	CHAR** params;
	int nargc;
	void (*kernel_entry)(int argc, CHAR * argv[], CHAR * envp[]);

	/* Print identification */
	printf(ANSI_CLEAR "\n\rarcsboot: ARCS Linux ext2fs loader 0.3.4\n\n\r");

	InitMalloc();

	ProcessArguments(argc, argv);

#if DEBUG
	printf("Command line: \n\r");
	for(i = 0; i < argc; i++ )
		printf("%u: %s\n\r", i, argv[i]);
#endif

	if (OSLoadPartition == NULL)
		Fatal("Invalid load parition\n\r");
	if (OSLoadFilename == NULL)
		Fatal("Invalid load filename\n\r");
#if DEBUG
	printf("OSLoadPartition: %s\n\r", OSLoadPartition);
	printf("OSLoadFilename: %s\n\r", OSLoadFilename);
#endif
	/* 
	 * XXX: let's play stupid for now: assume /etc/arcboot.conf
	 * is on the OSLoadPartition
	 *
	 * if OSLoadFilename starts with "/" it is a kernel path otherwise an image name
	 */
	if( OSLoadFilename[0] != '/' ) {
		if( !(params = ReadConfFile(OSLoadPartition, CONF_FILE, OSLoadFilename))) {
			printf("Failed to read configuration file %s - using PROM defaults\n\r.", CONF_FILE);
			nargc = argc;
			nargv = argv;
		} else {
			OSLoadFilename = params[1];
			nargv = &params[1]; 		/* nargv[0] is the labels name */
			for( nargc=0; nargv[nargc]; nargc++);	/* count nargv argumnts */
			if(OSLoadOptions != 0x0) {		/* append OSLoadOptions if present */
				nargv[nargc] = OSLoadOptions;
				nargc++;
			}
		}
	}
#if DEBUG
	printf("Command line after config file: \n\r");
	for(i = 0; i < nargc; i++ )
		printf("%u: %s\n\r", i, nargv[i]);
#endif
	printf("Loading %s...\n\r",(params) ? params[0] : OSLoadFilename);
	kernel_entry = LoadKernel(OSLoadPartition, OSLoadFilename);
	/* XXX: we should remove already processed arguments, especially OSLoadPartition */
	ArcFlushAllCaches();
	if( kernel_entry )
		(*kernel_entry)(nargc ,nargv, envp);
	else
		printf("Invalid kernel entry NULL\n\r");

	/* Not likely to get back here in a functional state, but what the heck */
	Wait("\n\r--- Press <spacebar> to restart ---");
	ArcRestart();
}
