# inventory.sh: package inventory tool
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
#       --debug
# 
debug=

if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --debug)  shift
		debug=--debug
		set -x
		;;

      --version|-V) exec larch --version
                    ;;


      --help|-h)
	        printf "inventory a source tree\\n"
                printf "usage: inventory [options] [--] [dir]*\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -s --source                   list source files\\n"
                printf " -p --precious                 list precious files\\n"
                printf " -b --backups                  list backup files\\n"
                printf " -j --junk                     list junk files\\n"
                printf " -u --unrecognized             list unrecognized files\\n"
                printf " -t --trees                    list roots of nested trees\\n"
                printf "\\n"
                printf " -r --report                   print a report\\n"
                printf "\\n"
                printf " -d --directories              list only directories\\n"
                printf " -f --files                    list only non-directories\\n"
                printf " -B --both                     list both dirs and files\\n"
                printf " --kind                        show file kinds\\n"
                printf "    --all                      include arch control files\\n"
                printf " --nested                      include nested trees\\n"
                printf "\\n"
                printf "  --tags                       list with id tags\\n"
                printf "                               (only applies to source)\\n"
                printf "  --untagged                   include untagged files\\n"
                printf "\\n"
		printf " --explicit                    require an explicit file tag\\n"
		printf " --implicit                    use the implicit tagging method\\n"
		printf " --names                       use the name-based tagging method\\n"
		printf "\\n"
                printf "With no arguments, print a human-readable inventory report.\\n"
                printf "\\n"
                printf "With -r or --report, category options (--source etc)\\n"
		printf "limit the report to just those files.  With no other options,\\n"
		printf "the report includes printf all sections and files.\\n"
                printf "\\n"
                printf "The options -d, -f, and -b cancel each other.  Thus \"inventory -d -f\"\\n"
 		printf "is the same as \"inventory -f\" and \"inventory -f -d\" is the same as\\n"
 		printf "\"inventory -d\", etc.\\n"
                printf "\\n"
                printf "If a directory is precious, junk, or unrecognized, only the\\n"
                printf "directory name itself is printed -- its contents are not\\n"
                printf "searched.\\n"
                printf "\\n"
                exit 0
                ;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 

report=0
all_flag=

source=0
precious=0
backups=0
junk=0
unrecognized=0
trees=0

source_opt=
precious_opt=
backups_opt=
junk_opt=
unrecognized_opt=
trees_opt=

type_opt=
kind_opt=

tags=
untagged=

method=
nested=

if test $# -ne 0 ; then

  while test $# -ne 0 ; do
    opt="$1"
    case $opt in
	
      --nested)			shift
				nested=--nested
				;;

      --explicit)		shift
  	  			method=--explicit
				;;

      --implicit)		shift
    				method=--implicit
				;;

      --names)			shift	
	    			method=--names
				;;

      -d|--directories)
		type_opt=--directories
		shift
		;;
      -f|--files)
		type_opt=--files
		shift
		;;
      -B|--both)
		type_opt=--both
		shift
		;;

      --kind)
		kind_opt=--kind
		shift
		;;

      -r|--report)
	    	report=1
		shift
		;;
      --all)
		all_flag=--all
		shift
		;;

      -s|--source)
		source=1
		source_opt=--source
		shift
		;;
      -p|--precious)
		precious=1
		precious_opt=--precious
		shift
		;;
      -b|--backups)
		backups=1
		backups_opt=--backups
		shift
		;;
      -j|--junk)
		junk=1
		junk_opt=--junk
		shift
		;;
      -u|--unrecognized)
		unrecognized=1
		unrecognized_opt=--unrecognized
		shift
		;;
      -t|--trees)
		trees=1
		trees_opt=--trees
		shift
		;;

      --tags)	shift
		tags=--tags
		;;

      --untagged)	shift
			untagged=--untagged
			;;

      --)	shift
	        break
		;;

      --*)
		printf "inventory: unrecognized option (%s)\\n" "$opt" 1>&2
		printf "try --help\\n" 1>&2
		exit 1
		;;

      *)        break
      		;;
    esac
  done
fi

################################################################
# Ordinary Arguments
# 


################################################################
# Sanity Check and Process Defaults
# 

selections=$(($source + $precious + $backups + $junk + $unrecognized + $trees))


if test $selections -eq 0 ; then
  report=1
  source=1
  source_opt=--source
  precious=1
  precious_opt=--precious
  backups=1
  backups_opt=--backups
  junk=1
  junk_opt=--junk
  unrecognized=1
  unrecognized_opt=--unrecognized
  trees=1
  trees_opt=--trees
fi


################################################################
# Do It
# 

if test $report -eq 0 ; then

  if test $# -ne 0 ; then
    inventory --control-dir "{arch}" \
	      $kind_opt $source_opt $precious_opt $backups_opt $junk_opt $unrecognized_opt $trees_opt $type_opt $all_flag $nested $tags $untagged $method "$@"
  else
    inventory --control-dir "{arch}" \
	      $kind_opt $source_opt $precious_opt $backups_opt $junk_opt $unrecognized_opt $trees_opt $type_opt $all_flag $nested $tags $untagged $method .
  fi

else

  # make a report

  # check if column is in our path
  #
  if column < /dev/null > /dev/null 2>&1 ; then
    column="column -x"
  else
    column="cat"
  fi

  common_flags="$debug $nested $typeflag $all_flag"

  if test $source -eq 1 ; then
    printf "\\n"
    printf "Source Files: "
    printf "\\n"
    if test $# -ne 0 ; then
      larch inventory  --source $common_flags $tags "$@" | $column
    else
      larch inventory  --source $common_flags $tags . | $column
    fi
    printf "\\n"
  fi

  if test $precious -eq 1 ; then
    printf "\\n"
    printf "Precious Files: "
    printf "\\n"
    if test $# -ne 0 ; then
      larch inventory  --precious $common_flags "$@" | $column
    else
      larch inventory  --precious $common_flags . | $column
    fi
    printf "\\n"
  fi

  if test $backups -eq 1 ; then
    printf "\\n"
    printf "Backup Files: "
    printf "\\n"
    if test $# -ne 0 ; then
      larch inventory  --backups $common_flags "$@" | $column
    else
      larch inventory  --backups $common_flags . | $column
    fi
    printf "\\n"
  fi

  if test $junk -eq 1 ; then
    printf "\\n"
    printf "Junk Files: "
    printf "\\n"
    if test $# -ne 0 ; then
      larch inventory  --junk $common_flags "$@" | $column
    else
      larch inventory  --junk $common_flags . | $column
    fi
    printf "\\n"
  fi

  if test $unrecognized -eq 1 ; then
    printf "\\n"
    printf "Unrecognized Files: "
    printf "\\n"
    if test $# -ne 0 ; then
      larch inventory  --unrecognized $common_flags "$@" | $column
    else
      larch inventory  --unrecognized $common_flags . | $column
    fi
    printf "\\n"
  fi

  if test $trees -eq 1 ; then
    printf "\\n"
    printf "Nested Project Trees: "
    printf "\\n"
     if test $# -ne 0 ; then
      larch inventory  $common_flags --trees "$@" | $column
    else
      larch inventory  $common_flags --trees . | $column
    fi
    printf "\\n"
  fi

fi

# tag: package inventory tool
#
