# library-file.sh: find a file in a revision library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "find a file in a revision library\\n"
                printf "usage: library-file [options] file [archive/]revision\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " --tag                         interpret FILE as an inventory tag\\n"
                printf " --this                        find the tag of FILE relative to\\n"
                printf "                                 the current directory, then find\\n"
                printf "                                 the corresponding file in the library\\n"
                printf "\\n"
                printf " --silent                      don't print an error message\\n"
                printf " -e --errname NAME             use NAME in error messages\\n"
                printf "\\n"
		printf "Find FILE in REVISION in your revision library and print\\n"
		printf "its path. If the revision or file is not present, print\\n"
		printf "an error message (unless \"--silent\" is specified) and\\n"
		printf "exit with non-0 status.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

tag=
this=

silent=
errname=library-file


while test $# -ne 0 ; do

  case "$1" in 

    --tag)		shift
    			tag=--tag
			this=
			;;

    --this)		shift
    			tag=
			this=--this
			;;

    -e|--errname)		shift
				if test $# -eq 0 ; then
				  printf "library-file: -e and --errname require an argument\\n" 1>&2
				  printf "try --help\\n" 1>&2
				  exit 1
				fi
				errname="$1"
				shift
				;;

    -s|--silent)	shift
    			silent=--silent
			;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-file: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-file: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -*)                 printf "library-file: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -lt 1 -o $# -gt 2 ; then
  printf "usage: library-file [options] file [[archive/]revision]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

file="$1"
shift

if test $# -ne 0 ; then
  rvnspec="$1"
  shift
else
  rvnspec=
fi

################################################################
# Sanity Check and Process Defaults
# 

root="`larch library-find -R \"$archroot\" -A \"$archive\" \"$rvnspec\"`"

if test ! -z "$this" ; then
  file="`larch file-tag \"$file\"`"
  tag=--tag
  this=
fi

if test -z "$tag" ; then
  if printf "%s\\n" "$file" | grep "^/" ; then
    printf "\\n" 1>&2
    printf "library-file: absolute file names not allowed\\n" 1>&2
    printf "  file: %s\\n" "$file" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi
fi

################################################################
# Do It
# 

cd "$root"

if test ! -z "$tag" ; then

  path="`printf '%s\n' \"$file\" | join -o 2.1 -1 1 -2 2 - ,,index | sed -e 's/^\.\///`"

  if test -z "$path" ; then 
    printf "\\n" 1>&2
    printf "library-file: tag not found in revision\\n" 1>&2
    printf "  tag: %s\\n" "$file" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi

  file="$path"

fi


if test -e "$file" ; then
  printf "%s\\n" "`pwd`/$file"
else
  printf "\\n" 1>&2
  printf "library-file: file not found in revision\\n" 1>&2
  printf "  file: %s\\n" "$file" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

# tag: Tom Lord Sun Jan 20 02:39:39 2002 (library/library-file.sh)
#
