# 
# file-list.sh: format a file list as part of an outline body
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "format a file list as part of an outline body\\n"
		printf "usage: file-list [options]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --sub                         indent for a sub-level\\n"
		printf " --html                        generate html instead of outline\\n"
		printf " --root DIR                    specify root dir for hyper-link targets\\n"
		printf " --suffix SFX                  specify filename suffix for link targets\\n"
		printf " --fill                        fill the list (HTML only)\\n"
		printf " --target TARGET               specify a hyperlink target (HTML only)\\n"
		printf "\\n"
		printf "Format a list of filenames as an outline body.  In html mode,\\n"
		printf "if either --root or --suffix is provided, each list item is a\\n"
		printf "link to a real file.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

sublvl=
sub=
html=
fill=
target=
root=
suffix=

while test $# -ne 0 ; do

  case "$1" in 

    --root)		shift
    			if test $# -eq 0 ; then
			  printf "file-list: --root requires an argument\\n" 1>&2
			  printf "\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			root="$1"
			shift
			;;

    --suffix)		shift
    			if test $# -eq 0 ; then
			  printf "file-list: --suffix requires an argument\\n" 1>&2
			  printf "\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			suffix="$1"
			shift
			;;

    --sub)		shift
    			sublvl="$sublvl*"
			sub="$sub --sub"
			;;

    --html)		shift
    			html=--html
			;;

    --fill)		shift
    			fill=--fill
			;;

    --target)		shift
    			if test $# -eq 0 ; then
			  printf "file-list: --target requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			target="$1"
			shift
			;;

    --)			shift
    			break
			;;

    -*)			printf "file-list: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Do It
# 

if test -z "$html" ; then

  awk '{
         print $0;
	 non_empty = 1;
       }
       END {
	     if (!non_empty)
	       print "<none>";
	   }' \
  | larch body-indent $sub

else

  sed -e '{
	    s/&/\&amp;/g
	    s/</\&lt;/g
	    s/>/\&gt;/g
	    s/"/\&quot;/g
	  }' \
  | awk -v root="$root" -v suffix="$suffix" \
        -v fill="$fill" \
        -v target="$target" \
      'function some()
       {
	 if (!non_empty)
	   {
	     if (fill == "")
	       print "<ul>";
	     else
	       print "<p>";
	   }
         non_empty = 1;
       }
       BEGIN {
	        if (target != "")
		  target = "target=\"" target "\"";
	        in_body = 0;
		if (fill == "")
		  li = "<li>";
		else
		  li = "";
	     }
       {
         some();
	 if (match($0, "^[ \t]"))
	   {
	     if (!in_body)
	       {
	         print "<pre>";
		 in_body = 1;
	       }
	     print $0;
	   }
         else
	   {
	     if (in_body)
	       {
	         print "</pre>";
		 in_body = 0;
	       }
  	     if ( "" == root suffix )
	       {
	         print li " <code>" $0 "</code>";
	       }
	     else
	       {
	         print li " <a " target " type=\"text/plain\" href=\"" root "/" $1 suffix "\"><code>" $0 "</code></a>";
	       }
	   }
       }
       END {
	     if (in_body)
	       print "</pre>";
	     if (non_empty)
	       {
	         if (fill == "")
	           print "</ul>";
		 else
		   print "</p>";
	       }
	     else
	       print "<pre>&lt;none&gt;</pre>"
	   }'

fi

# tag: Tom Lord Thu Dec 13 03:51:40 2001 (output/file-list.sh)
#
