# 
# cat-log.sh - spew a project tree log file
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "print the contents of a project tree log entry\\n"
                printf "usage: cat-log [options] archive/revision \\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
                printf "\\n"
                printf "Print the contents of the log entry for REVISION in the working\\n"
                printf "directory containing DIR (or the current directory)\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
dir=.
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --debug)		shift
    			debug_opt=--debug
			printf "\n" 1>&2
			printf "cat-log: DEBUGGING ACTIVATED\n" 1>&2
			printf "\n" 1>&2
			set -x
			;;
			
    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "cat-log: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "cat-log: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "cat-log: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;
			
    -*)                 printf "cat-log: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: cat-log [options] archive/revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  archive_revision="$1"
  shift
else
  archive_revision=
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root --accurate`"

if larch valid-patch-level-name "$archive_revision" ; then
  archive_revision="`larch tree-version`--$archive_revision"
fi

larch valid-package-name -e cat-log --patch-level "$archive_revision"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch "$archive_revision"`"
revision="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --non-arch "$archive_revision"`"
category=`larch parse-package-name --basename $revision`
branch=`larch parse-package-name $revision`
vsn=`larch parse-package-name --vsn $revision`
lvl=`larch parse-package-name --lvl $revision`

log_file="$category/$branch/$branch--$vsn/$archive/patch-log/$lvl"

################################################################
# Print the Log Entry
# 

cd "$wdroot/{arch}"

if test ! -e "$log_file" ; then

  printf "cat-log: no such log entry" 1>&2
  printf "  revision: %s/%s\\n" $archive $revision 1>&2
  printf "\\n" 1>&2
  exit 1

else

  cat "$log_file"

fi
