# 
# logs.sh - list patch logs in a project tree
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "list patch logs in a project tree\\n"
		printf "usage: logs [options]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -r --reverse                  sort from oldest to newest\\n"
		printf "\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf "\\n"
		printf " -a --arch ARCHIVE             list only logs for ARCHIVE\\n"
		printf " -c --category CATEGORY        list only logs in CATEGORY\\n"
		printf " -b --branch BRANCH            list only logs for BRANCH\\n"
		printf " -v --vsn VERSION-NUMBER       list only logs for VERSION-NUMBER\\n"
		printf "\\n"
		printf "Print the list of versions for which there are patch logs in\\n"
		printf "the project tree containing DIR (or the current directory).\\n"
		printf "\\n"
		printf "Version names are printed as fully-qualified names:\\n"    
		printf "\\n"
		printf "	\"%%s/%%s\\\\n\"  \$archive \$version\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=
dir=.

arch_limit=
category_limit=
branch_limit=
version_limit=

debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --debug)		shift
    			debug_opt=--debug
			printf "\n" 1>&2
			printf "logs: DEBUGGING ACTIVATED\n" 1>&2
			printf "\n" 1>&2
			set -x
			;;
			
    -d|--dir)           shift
                        if test $# -eq 0 ; then
                          printf "logs: -d and --dir require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        dir="$1"
                        shift
                        ;;

    -a|--arch)          shift
                        if test $# -eq 0 ; then
                          printf "logs: -a and --arch require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        arch_limit="$1"
                        shift
                        ;;

    -c|--category)      shift
                        if test $# -eq 0 ; then
                          printf "logs: -c and --category require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        category_limit="$1"
                        shift
                        ;;

    -b|--branch)        shift
                        if test $# -eq 0 ; then
                          printf "logs: -b and --branch require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        branch_limit="$1"
                        shift
                        ;;

    -v|--vsn)           shift
                        if test $# -eq 0 ; then
                          printf "logs: -v and --vsn require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        version_limit="$1"
                        shift
                        ;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "logs: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "logs: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    -*)			printf "logs: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 0 ; then
  printf "usage: logs [options]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi


################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root --accurate`"
cd "$wdroot"

if test -z "$category_limit" -a ! -z "$branch_limit" ; then
  category_limit="`larch parse-package-name --basename \"$branch_limit\"`"
fi


################################################################
# Print the list of versions as fully-qualified version names.
# 

cd "$wdroot/{arch}"

archdir="`pwd`"

for category in `ls | sort` ; do

  if   larch valid-package-name -b -- "$category" \
    && test -z "$category_limit" \
            -o "$category_limit" = "$category" \
  ; then

    catdir="$archdir/$category"
    cd "$catdir"

    for branch in `ls | sort` ; do

      if   larch valid-package-name -- "$branch" \
        && test -z "$branch_limit" \
                -o "$branch_limit" = "$branch" \
      ; then

        branchdir="$catdir/$branch"
	cd "$branchdir"

        for version in `ls \
			| sed -e '{
				    s/.*\([[:digit:]]\{1,\}\.[[:digit:]]\{1,\}\)/\1 &/
				    t
				    d
				  }' \
			| sort -k 1,1${reverse#-}n \
			| awk '{ print $2 }'` ; do

          if   larch valid-package-name -v -- "$version" \
            && test -z "$version_limit" \
                    -o "$category--$branch_limit--$version_limit" = "$version" \
          ; then

            vsndir="$branchdir/$version"
	    cd "$vsndir"

	    for archive in `ls | sort` ; do

	      if   larch valid-archive-name -- "$archive" \
                && test -z "$archive_limit" \
                        -o "$archive_limit" = "$archive"  \
              ; then

	        logdir="$vsndir/$archive"

  	        if test -d "$logdir/patch-log" ; then
	          printf "%s/%s\\n" "$archive" "$version"
	        fi

 	      fi
	    done

          fi
        done

      fi
    done

  fi
done
