/* with-ftp.c
 *
 ****************************************************************
 * Copyright (C) 2001, 2002  Tom Lord
 * 
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/os/errno.h"
#include "hackerlab/os/setjmp.h"
#include "hackerlab/os/stdlib.h"
#include "hackerlab/os/sys/time.h"
#include "hackerlab/os/signal.h"
#include "hackerlab/os/sys/types.h"
#include "hackerlab/os/sys/wait.h"
#include "hackerlab/os/unistd.h"

#include "hackerlab/hash/hash-utils.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/vu-network/url-socket.h"

#include "ftp-utils/client/protocol.h"



/* __STDC__ prototypes for static functions */
static void ftp_handle_command (t_uchar * command, int command_len);
static void local_handle_command (t_uchar * command, int command_len);
static long trim_crnl (t_uchar * str, long len);
static void child_changed (int sig);
static char * peer_addr_env (int fd);



extern char ** environ;

static t_uchar * program_name = "with-ftp";
static t_uchar * usage = "with-ftp [options] site command args...";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
				   "\n"
				   "Copyright 2001, 2002 Tom Lord\n"
				   "\n"
				   "This is free software; see the source for copying conditions.\n"
				   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
				   "PARTICULAR PURPOSE.\n"
				   "\n"
				   "Report bugs to <lord@regexps.com>.\n"
				   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.")

static t_uchar long_help[] = ("invoke a command from with an FTP(-style) client session\n"
                              "Begin an FTP session or other file system client session with SITE\n"
                              "then invoke COMMAND with ARGS.\n"
                              "\n"
                              "COMMAND can in turn invoke the various \"wftp-*\" programs (such as\n"
                              "\"wftp-ls\") as part of the FTP session.\n"
                              "\n"
                              "SITE should be specified as a directory name (for local file systems)\n"
                              "or as a URL using one of these methdos:\n"
                              "\n"
                              "         ftp:\n"
                              "\n"
                              "Several environment variables are set when COMMAND is invoked.  These\n"
                              "are:\n"
                              "\n"
                              "         WFTP            a (non-standard) \"unix:\" method URL\n"
                              "                         specifying the unix domain socket over\n"
                              "                         which \"wftp-*\" programs speak to \"with-ftp\"\n"
                              "\n"
                              "         WFTP_SITE       the SITE argument to \"with-ftp\"\n"
                              "\n"
                              "         WFTP_SECRET     a (plain-text) authentication cookie that\n"
                              "                         \"wftp-*\" programs send to \"with-ftp\"\n"
                              "\n"
                              "         WFTP_SITE_ADDR  the IP address associated with SITE\n"
                              "\n"
                              "         WFTP_MY_ADDR    the (local) IP address this host is using to\n"
                              "                         talk with SITE\n");


enum options
{
  OPTS (OPT_ENUM, OPT_IGN)  
};

struct opt_desc opts[] = 
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};


/* Remote starting directory.
 * 
 * Either ftp://site[/this]  or [/this]
 */
t_uchar * remote_home;

/* FTP fds */
int in_fd;
int out_fd;

/* server_fd, socket_path, my_url, my_url_env
 * 
 * server_fd: A server socket to which with-ftp clients (e.g. wftp-ls) connect.
 * 
 * socket_path: a path to the unix-domain socket which is server_fd.
 * 
 * my_url: that path in URI format.
 * 
 * my_url_env: the environment variable assignment string for my_url.
 * 
 * my_port, my_host_addr, and my_host are not currently used.  If server_fd is an
 * inet domain socket, they are used -- but that code is disabled.
 */
int server_fd;
char * socket_path = 0;
t_uchar * my_url;
t_uchar * my_url_env;

int my_port;
t_ulong my_host_addr;
t_uchar * my_host;


/* connect_in/connect_out
 * 
 * A RDONLY and WRONLY pair for the current wftp-ftp client connection.
 */
int connect_in;
int connect_out;


/* data_server_fd, data_socket_path, my_data_url
 * 
 * If data connections have to be served by this process (e.g., for local 
 * file access rather than FTP access, then data_server_fd is a server
 * socket to which passive mode clients can connect.
 * 
 * data_socket_path is the path to that (unix domain) socket.
 * 
 * my_data_url is a URI for data_socket_path.
 *
 */
int data_server_fd;
t_uchar * data_socket_path = 0;
t_uchar * my_data_url;


/* rename_from_path
 * 
 * Between RNFR and RNTO, this holds the path name from RNFR.
 */
t_uchar * rename_from_path = 0;


/* pid of the subprocess
 */
int subproc;

/* What to do on SIGCHLD 
 */
sigjmp_buf sigchld_jmp;
sigjmp_buf sigchld_jmp_tool;
int tool_running = 0;



int
main (int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * site;
  t_uchar * site_env;
  t_uchar * site_addr_env;
  char ** command_argv;
  int using_ftp;
  int unix_domain = 1;

  url_socket_push_server_handler (url_socket_unix, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0);

  using_ftp = 0;
  option = 0;

  safe_buffer_fd (0, 0, O_RDONLY, 0);
  safe_buffer_fd (1, 0, O_WRONLY, 0);

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, long_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
	break;
      switch (o)
	{
	default:
	  safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
	  panic ("internal error parsing arguments");

	usage_error:
	  opt_usage (2, argv[0], program_name, usage, 1);
	  panic_exit ();

	/* bogus_arg: */
	  safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
	  goto usage_error;
	}
    }

  if (argc < 3)
    goto usage_error;

  site = argv[1];
  site_env = str_alloc_cat (0, "WFTP_SITE=", site);
  
  command_argv = argv + 2;
  
  {
    t_uchar secret[65];
    t_uchar * secret_env;
    int home_fd;
    int errn;
    int code;
    t_uchar * repl_text;
    long repl_text_len;
    void (*handle_command) (t_uchar * command, int command_len);

    {
      struct timeval time;
      unsigned long hash;
      int z;

      if (0 > gettimeofday (&time, 0))
	{
	  panic ("gettimeofday failed");
	}

      srand ((unsigned int)(getpid() + (time.tv_sec ^ time.tv_usec)));

      for (z = 0; z < (sizeof (secret) - 1); ++z)
	{
	  hash = (unsigned long)rand();
	  secret[z] = (((hash & 0xf) < 10) ? '0' + (hash & 0xf) : 'a' + (hash & 0xf) - 10);
	  hash >>= 4;
	  secret[z] = (((hash & 0xf) < 10) ? '0' + (hash & 0xf) : 'a' + (hash & 0xf) - 10);
	  hash >>= 4;
	  hash = hash_ul (hash);
	  if (   (0 > gettimeofday (&time, 0))
	      || (0 > gettimeofday (&time, 0)))
	    {
	      panic ("gettimeofday failed");
	    }
	  hash ^= time.tv_sec ^ time.tv_usec;
	  srand ((unsigned int)hash);
	}
      secret[z] = 0;

      secret_env = str_alloc_cat (0, "WFTP_SECRET=", secret);
    }



    home_fd = -1;

    if (!str_cmp_prefix ("ftp://", site))
      {
	t_uchar * site_spec;
	t_uchar * site_spec_end;
	/* connect to the FTP server */

	using_ftp = 1;

	site_spec = site + 6;	/* over ftp:// */
	site_spec_end = str_chr_index (site_spec, '/');
	if (!site_spec_end)
	  {
	    remote_home = str_save (0, "/");
	    site_spec = str_save (0, site_spec);
	  }
	else
	  {
	    remote_home = str_save (0, site_spec_end);
	    site_spec = str_save_n (0, site_spec, site_spec_end - site_spec);
	  }

	handle_command = ftp_handle_command;
	
	repl_text = 0;
	if (0 > ftp_client_connect_and_login (&in_fd, &out_fd, 0, &errn, &code,
					      &repl_text, &repl_text_len,
					      site_spec))
					  
	  {
	    safe_printfmt (2, "unable to establish ftp connection to %s\n", site);
	    if (repl_text)
	      safe_printfmt (2, "%.*s\n", (int)repl_text_len, repl_text);
	    exit (1);
	  }

	lim_free (0, repl_text);

	site_addr_env = peer_addr_env (in_fd);

	if (0 > ftp_client_cwd (0, &errn, &code, &repl_text, &repl_text_len, in_fd, out_fd, remote_home))
	  {
	    safe_printfmt (2, "unable to cd to remote directory: %s\n", remote_home);
	    if (repl_text)
	      safe_printfmt (2, "%.*s\n", (int)repl_text_len, repl_text);
	    exit (1);
	  }

	lim_free (0, repl_text);
      }
    else
      {
	remote_home = str_save (0, site);
	handle_command = local_handle_command;
	home_fd = safe_open (".", O_RDONLY, 0);
	safe_chdir (site);
      }

    /* create a server socket for local clients */
    if (unix_domain)
      {
	t_ulong pid;
	char pid_str[sizeof (t_ulong) * 4];

	pid = getpid ();
	cvt_ulong_to_decimal (pid_str, pid);

	socket_path = str_save (0, getenv ("HOME"));
	socket_path = str_realloc_cat (0, socket_path, "/.with-ftp");
	vu_mkdir (&errn, socket_path, 0700);
	socket_path = str_realloc_cat (0, socket_path, "/");
	socket_path = str_realloc_cat (0, socket_path, pid_str);
	my_url = str_alloc_cat (0, "unix:", socket_path);
	my_url_env = str_alloc_cat (0, "WFTP=", my_url);
	data_socket_path = str_alloc_cat (0, socket_path, ".data");
	my_data_url = str_alloc_cat (0, my_url, ".data");

	if (!using_ftp)
	  {
	    vu_unlink (&errn, socket_path);
	    vu_unlink (&errn, data_socket_path);
	    server_fd = url_socket_create_server_socket (&errn, my_url);
	    data_server_fd = url_socket_create_server_socket (&errn, my_data_url);
	    if ((data_server_fd < 0) || (server_fd < 0))
	      {
		if (socket_path)
		  vu_unlink (&errn, socket_path);
		if (data_socket_path)
		  vu_unlink (&errn, socket_path);
		safe_printfmt (2, "with-ftp: error creating server sockets\n");
		safe_printfmt (2, "  error %d: %s", errn, errno_to_string (errn));
		panic ("i/o error");
	      }
	  }


      }
    else
      {
	t_uchar port_str[100];

	panic ("inet code disabled");
	server_fd = url_inet_server (0, &my_host_addr, &my_host, &my_port, &errn, 0, 0);

	if (server_fd < 0)
	  {
	    panic ("unable to create server socket\n");
	  }

	port_str[0] = ':';
	cvt_ulong_to_decimal (port_str + 1, my_port);

	my_url = str_alloc_cat (0, "inet://", my_host);
	my_url = str_realloc_cat (0, my_url, port_str);
	my_url_env = str_alloc_cat (0, "WFTP=", my_url);
      }

    if (!using_ftp)
      {
	struct sockaddr_in addr;
	t_uchar myhost[MAXHOSTNAMELEN + 1];

	if (0 > gethostname (myhost, sizeof (myhost)))
	  {
	    panic ("gethostname failed");
	  }

	{
	  struct hostent * hostentp;
	  char * addr_name;

	  hostentp = gethostbyname (myhost);
	  if (!hostentp || (hostentp->h_addrtype != AF_INET))
	    {
	      panic ("gethostbyname failed");
	    }
	  mem_move ((t_uchar *)&addr.sin_addr, hostentp->h_addr, hostentp->h_length);

	  addr_name = inet_ntoa (addr.sin_addr);
	  if (!addr_name)
	    panic ("inet_ntoa failed on my address");

	  site_addr_env = str_alloc_cat (0, "WFTP_SITE_ADDR=", addr_name);
	}
      }


    {
      sigset_t interesting_signal_mask;

      sigfillset (&interesting_signal_mask);
      sigdelset (&interesting_signal_mask, SIGHUP);
      sigdelset (&interesting_signal_mask, SIGINT);
      sigdelset (&interesting_signal_mask, SIGCHLD);
      sigdelset (&interesting_signal_mask, SIGTRAP);
      if (0 > sigprocmask (SIG_SETMASK, &interesting_signal_mask, 0))
	{
	  panic ("sigprocmask(2)");
	}
    }

    /* prepare for the subprocess' eventual death */
    if (sigsetjmp (sigchld_jmp, 1))
      {
	int wait_pid;
	int wait_status;
	
	/* In the body of this sigsetjmp, nothing may happen which may
	 * cause reentrancy problems.  This code can be reached asynchronously
	 * from just about anywhere.
	 */

	/* Did the subprocess exit?
	 * If so, exit, with status 0 iff subpocess had status 0.
	 */
	wait_pid = waitpid (subproc, &wait_status, WUNTRACED);
	if (socket_path)
	  unlink (socket_path);	/* can not use vu_unlink here since that may call malloc */
	if (data_socket_path)
	  unlink (socket_path);	/* can not use vu_unlink here since that may call malloc */
	if (wait_pid < 0)
	  panic ("with-ftp: error waiting for subprocess\n");
 	else if (wait_pid == 0)
	  panic ("with-ftp: got SIGCHLD but waitpid returns 0\n");
	else if (WIFEXITED (wait_status) && !WEXITSTATUS (wait_status))
	  {
	    _exit (0);
	  }
	else
	  {
	    _exit (1);
	  }
      }
    else
      signal (SIGCHLD, child_changed);


    /* fire up the sub-process */
    {
      t_uchar * path;
      t_uchar ** path_dirs;
      t_uchar * executable;
      t_uchar * addr_env;
      struct sockaddr_in my_addr;
      int my_addr_len;
      t_uchar * my_addr_name;


      if (using_ftp)
	{
	  my_addr_len = sizeof (my_addr);
	  if (0 > getsockname (out_fd, (struct sockaddr *)&my_addr, &my_addr_len))
	    {
	      panic ("getsockname failed?");
	    }
	  
	  my_addr_name = str_save (0, inet_ntoa (my_addr.sin_addr));
	  addr_env = str_alloc_cat (0, "WFTP_MY_ADDR=", my_addr_name);
	  lim_free (0, my_addr_name);
	}
      

      path = getenv ("PATH");
      if (!path)
	path = "/bin:/usr/bin";

      path_dirs = path_parse (0, path);

      executable = path_find_executable (0, path_dirs, command_argv[0]);

      if (!executable)
	{
	  safe_printfmt (2, "with-ftp: %s not found\n", command_argv[0]);
	  if (socket_path)
	    vu_unlink (&errn, socket_path);
	  if (data_socket_path)
	    vu_unlink (&errn, socket_path);
	  exit (1);
	}

      subproc = fork ();

      if (subproc < 0)
	panic ("unable to fork");

      if (subproc == 0)
	{
	  t_uchar ** new_env;
	  int x;

	  if (home_fd > 0)
	    {
	      safe_fchdir (home_fd);
	      safe_close (home_fd);
	    }

	  new_env = 0;

	  *(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = secret_env;
	  *(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = my_url_env;
	  *(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = site_env;
	  *(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = site_addr_env;
	  
	  if (using_ftp)
	    *(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = addr_env;

	  for (x = 0; environ[x]; ++x)
	    {
	      if (   str_cmp_prefix ("WFTP_MY_ADDR=", environ[x])
		  && str_cmp_prefix ("WFTP_SITE=", environ[x])
		  && str_cmp_prefix ("WFTP_SECRET=", environ[x])
		  && str_cmp_prefix ("WFTP_SITE_ADDR=", environ[x])
		  && str_cmp_prefix ("WFTP=", environ[x]))
		*(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = environ[x];
	    }

	  *(t_uchar **)ar_push ((void **)&new_env, 0, sizeof (t_uchar *)) = 0;
	  
	  
	  execve (executable, command_argv, (char **)new_env);

	  safe_printfmt (2, "with-ftp: unable to exec %s\n", executable);
	  if (socket_path)
	    vu_unlink (&errn, socket_path);
	  if (data_socket_path)
	    vu_unlink (&errn, socket_path);
	  exit (1);
	}
    }

    /* the main loop */
    while (1)
      {

	/* Wait for a connection from a local client.
	 * Meanwhile, ftp_keepalive runs on the alarm signal.
	 */
      retry_accept:
	connect_in = safe_open (my_url, O_RDWR, 0);

	/* connect_in = url_inet_server_accept (&errn, server_fd); */

	if (0 > connect_in)
	  {
	    /* errn *shouldn't* ever be EINTR, but if it were, this
	     * is how it should be handled, and this is safe.
	     */
	    if (errn == EINTR)
	      goto retry_accept;

	    safe_printfmt (2, "with-ftp: internal error during accept (%d)\n", errn);
	    if (socket_path)
	      vu_unlink (&errn, socket_path);
	    if (data_socket_path)
	      vu_unlink (&errn, socket_path);
	    exit (1);
	  }

	connect_out = safe_dup (connect_in);

	safe_buffer_fd (connect_in, 0, O_RDONLY, 0);
	safe_buffer_fd (connect_out, 0, O_WRONLY, 0);

	{
	  t_uchar * command;
	  long command_len;

	  /* authenticate the client */
	  safe_next_line (&command, &command_len, connect_in);

	  if (str_cmp_n (secret, sizeof (secret) - 1, command, (command_len ? command_len - 1 : 0)))
	    {
	      safe_close (connect_in);
	      safe_close (connect_out);
	      panic_msg ("");
	      panic_msg ("ATTENTION:");
	      panic_msg ("");
	      panic_msg ("with-ftp: INCORRECT AUTH TOKEN RECEIVED");
	      panic_msg ("");
	      panic_msg ("IS THIS A CRACK ATTEMPT?");
	      panic_msg ("");
	      kill (subproc, SIGKILL);
	      panic ("exitting");
	    }
	}
	
	/* Read a sequence of single line commands from the local
	 * client, handling each.  On end of file, close the local
	 * client connection and move on.
	 */
	while (1)
	  {
	    t_uchar * command;
	    long command_len;

	    safe_next_line (&command, &command_len, connect_in);

	    if (!str_cmp_prefix ("crack", command))
	      {
		panic_msg ("");
		panic_msg ("ATTENTION:");
		panic_msg ("");
		panic_msg ("with-ftp: CLIENT PROGRAM (e.g. wftp-ls) DETECTED BAD DATA PORT PEER");
		panic_msg ("");
		panic_msg ("IS THIS A CRACK ATTEMPT?");
		panic_msg ("");
		kill (subproc, SIGKILL);
		panic ("exitting");
	      }
		

	    if (!command)
	      {
		int ign;
		/* EOF on command connection */
		vu_close (&ign, connect_in);
		vu_close (&ign, connect_out);
		break;
	      }
	    else
	      {
		handle_command (command, command_len);
	      }
	  }
      }
  }
  
  return 0;
}


static void
ftp_handle_command (t_uchar * command, int command_len)
{
  int errn;
  int code;
  t_uchar * repl;
  long repl_len;

  command_len = trim_crnl (command, command_len);
  repl = 0;
  repl_len = 0;

  /* commands not requiring a data port */

  if (!str_cmp_prefix ("cd ", command))
    {
      /* "cd <path>" */
      if (0 > ftp_client_cwd (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, command + 3))
	{
	  if (errn)
	    {
	    ftp_protocol_error:
	      safe_printfmt (2, "with-ftp: FTP protocol error\n%s\n", repl ? repl : (t_uchar *)"  (no output from server)");
	      if (socket_path)
		vu_unlink (&errn, socket_path);
	      if (data_socket_path)
		vu_unlink (&errn, socket_path);
	      exit (1);
	    }
	  else
	    {
	    ftp_command_error:
	      if ((0 > printfmt (&errn, connect_out, "ERROR %ld\n", repl_len))
		  || (repl && (0 > vu_write_retry (&errn, connect_out, repl, repl_len))))
		{
		  if (errn != EPIPE)
		    panic ("I/O error in ftp_handle_command (ftp_command_error)");
		}
	    }
	}
      else
	{
	ftp_command_ok:
	  if ((0 > printfmt (&errn, connect_out, "OK %ld\n", repl_len))
	      || (repl && (0 > vu_write_retry (&errn, connect_out, repl, repl_len))))
	    {
	      if (errn != EPIPE)
		panic ("I/O error in ftp_handle_command (ftp_command_ok)");
	    }
	}
    }
  else if (!str_cmp_prefix ("cdup", command))
    {
      /* "cdup" */
      if (0 > ftp_client_cdup (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("delete ", command))
    {
      /* "delete <path>" */
      if (0 > ftp_client_dele (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, command + 7))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("home", command))
    {
      /* "home" */
      if (0 > ftp_client_cwd (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, remote_home))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("rmdir ", command))
    {
      /* "rmdir <path>" */
      if (0 > ftp_client_rmd (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, command + 6))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("mkdir ", command))
    {
      /* "mkdir <path>" */
      if (0 > ftp_client_mkd (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, command + 6))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("noop", command))
    {
      /* "noop" */
      if (0 > ftp_client_noop (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("pwd", command))
    {
      t_uchar * pwd;
      long pwd_len;
      
      /* "pwd" */
      if (0 > ftp_client_pwd (0, &errn, &code, &pwd, &pwd_len, &repl, &repl_len, in_fd, out_fd))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	{
	  /* in with-ftp protocol, pwd generates a non-stanard reply
	   * to spare clients from parsing ftp replies.  We don't just
	   * send the full text of the FTP reply.  Instead, the reply
	   * is:
	   *
	   * 	OK <str_length(pwd) + 1>\n
	   *	pwd\n
	   * 
	   * Note the trailing newline which is not part of the pwd.
	   */
	  if (0 > printfmt (&errn, connect_out, "OK %ld\n%.*s\n", 1 + pwd_len, (int)pwd_len, pwd))
	    {
	      if (errn != EPIPE)
		panic ("I/O error in ftp_handle_command (pwd)");
	    }
	  lim_free (0, pwd);
	}
    }
  else if (!str_cmp_prefix ("rnfr ", command))
    {
      if (rename_from_path)
	lim_free (0, rename_from_path);

      rename_from_path = str_save (0, command + 5);
      goto ftp_command_ok;
    }
  else if (!str_cmp_prefix ("rnto ", command))
    {
      if (!rename_from_path)
	goto bad_command;
      
      if (0 > ftp_client_rename (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, rename_from_path, command + 5))
	{
	  if (errn)
	    goto ftp_protocol_error;
	  else
	    goto ftp_command_error;
	}
      else
	goto ftp_command_ok;
    }  


  /* commands requiring a data port */

  else
    {
      t_uchar * host;
      t_uchar * comma;
      t_uchar * port;
      t_uchar * args;
      t_ulong h;
      t_uint p;
      char * pasv_url;


      /* <cmd> host,port args...
       * or
       * <cmd> pasv args...
       */

      pasv_url = 0;
      host = str_chr_index (command, ' ');
      if (!host)
	goto bad_command;

      ++host;

      if (!str_cmp_prefix ("pasv ", host))
	{
	  t_ulong host_addr;
	  t_uint16 port;

	  /* passive mode */
	  /* block postcondition: args points at the list of args */
	  if (0 > ftp_client_pasv (0, &errn, &code, &host_addr, &port, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }

	  {
	    char * pos;
	    char addr[64];

	    host_addr = ntohl (host_addr);
	    port = ntohs (port);

	    pos = addr;
	    cvt_ulong_to_decimal (pos, (host_addr >> 24) & 0xff);
	    pos += str_length (pos);
	    *pos = '.';
	    ++pos;
	    cvt_ulong_to_decimal (pos, (host_addr >> 16) & 0xff);
	    pos += str_length (pos);
	    *pos = '.';
	    ++pos;
	    cvt_ulong_to_decimal (pos, (host_addr >> 8) & 0xff);
	    pos += str_length (pos);
	    *pos = '.';
	    ++pos;
	    cvt_ulong_to_decimal (pos, host_addr & 0xff);
	    pos += str_length (pos);
	    *pos = ':';
	    ++pos;
	    cvt_ulong_to_decimal (pos, port);
	    pos += str_length (pos);
	    *pos = 0;
	    pasv_url = str_alloc_cat (0, "inet://", addr);
	    if ((0 > printfmt (&errn, connect_out, "%s\n", pasv_url))
		|| (0 > vfdbuf_flush (&errn, connect_out)))
	      {
		if (errn != EPIPE)
		  panic ("I/O error in ftp_handle_command (pasv)");
	      }
	  }
	  args = host + 5;	/* "pasv " */
	}
      else
	{
	  /* active mode */
	  /* block postcondition: args points at the list of args */
	  comma = str_chr_index (host, ',');
	  if (!comma)
	    goto bad_command;

	  if (cvt_decimal_to_ulong (&errn, &h, host, comma - host))
	    goto bad_command;

	  port = comma + 1;
	  
	  args = str_chr_index (port, ' ');
	  if (!args)
	    goto bad_command;

	  if (cvt_decimal_to_uint (&errn, &p, port, args - port))
	    goto bad_command;

	  args += 1;

	  if (0 > ftp_client_port (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, h, p))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      lim_free (0, repl);
	      repl = 0;
	    }
	}

      /* args points at the list of args */
      
      if (!str_cmp_prefix ("get ", command))
	{
	  if (0 > ftp_client_type_image (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      lim_free (0, repl);
	      repl = 0;
	    }

	  if (0 > ftp_client_begin_retr (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, args))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      if ((0 > printfmt (&errn, connect_out, "OK %ld\n", repl_len))
		  || (repl && (0 > vu_write_retry (&errn, connect_out, repl, repl_len)))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn != EPIPE)
		    panic ("I/O error in ftp_handle_command (get)");
		}
	      lim_free (0, repl);
	      repl = 0;
	    }

	  if (0 > ftp_client_end_retr (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    goto ftp_command_ok;
	}
      else if (!str_cmp_prefix ("put ", command))
	{
	  if (0 > ftp_client_type_image (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      lim_free (0, repl);
	      repl = 0;
	    }

	  if (0 > ftp_client_begin_stor (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, args))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      if ((0 > printfmt (&errn, connect_out, "OK %ld\n", repl_len))
		  || (repl && (0 > vu_write_retry (&errn, connect_out, repl, repl_len)))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn != EPIPE)
		    panic ("I/O error in ftp_handle_command (put)");
		}
	      lim_free (0, repl);
	      repl = 0;
	    }

	  if (0 > ftp_client_end_stor (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    goto ftp_command_ok;
	}
      else if (!str_cmp_prefix ("nlist ", command))
	{
	  if (0 > ftp_client_type_ascii (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      lim_free (0, repl);
	      repl = 0;
	    }

	  if (0 > ftp_client_begin_nlst (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, args))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      if ((0 > printfmt (&errn, connect_out, "OK %ld\n", repl_len))
		  || (repl && (0 > vu_write_retry (&errn, connect_out, repl, repl_len)))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn != EPIPE)
		    panic ("I/O error in ftp_handle_command (nlist)");
		}
	      lim_free (0, repl);
	      repl = 0;
	    }

	  if (0 > ftp_client_end_nlst (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      goto ftp_command_ok;
	    }
	}
      else if (!str_cmp_prefix ("list ", command))
	{
	  if (0 > ftp_client_type_ascii (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      lim_free (0, repl);
	      repl = 0;
	    }
	  if (0 > ftp_client_begin_list (0, &errn, &code, &repl, &repl_len, in_fd, out_fd, args))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      if ((0 > printfmt (&errn, connect_out, "OK %ld\n", repl_len))
		  || (repl && (0 > vu_write_retry (&errn, connect_out, repl, repl_len)))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn != EPIPE)
		    panic ("I/O error in ftp_handle_command (nlist)");
		}
	      lim_free (0, repl);
	      repl = 0;
	    }
	  if (0 > ftp_client_end_list (0, &errn, &code, &repl, &repl_len, in_fd, out_fd))
	    {
	      if (errn)
		goto ftp_protocol_error;
	      else
		goto ftp_command_error;
	    }
	  else
	    {
	      goto ftp_command_ok;
	    }
	}
      else
	{
	bad_command:
	  safe_printfmt (2, "with-ftp: bad command recieved (%s)\n", command);
	  panic ("with-ftp protocol error");
	}
    }
      

  lim_free (0, repl);
  if (0 > vfdbuf_flush (&errn, connect_out))
    panic ("vfdbuf_flush error in ftp_handle_command");
}


static void
local_handle_command (t_uchar * command, int command_len)
{
  int errn;
  t_uchar * message;
  int cmd_status;

  command_len = trim_crnl (command, command_len);

  /* commands not requiring a data port */

  if (!str_cmp_prefix ("cd ", command))
    {
      /* "cd <path>" */
      if (0 > (cmd_status = vu_chdir (&errn, command + 3)))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "chdir ok");
    }
  else if (!str_cmp_prefix ("cdup", command))
    {
      /* "cdup" */
      if (0 > (cmd_status = vu_chdir (&errn, "..")))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "chdir .. ok");
    }
  else if (!str_cmp_prefix ("delete ", command))
    {
      /* "delete <path>" */
      if (0 > (cmd_status = vu_unlink (&errn, command + 7)))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "unlink ok");
    }
  else if (!str_cmp_prefix ("home", command))
    {
      /* "home" */
      if (0 > (cmd_status = vu_chdir (&errn, remote_home)))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "home ok");
    }
  else if (!str_cmp_prefix ("rmdir ", command))
    {
      /* "rmdir <path>" */
      if (0 > (cmd_status = vu_rmdir (&errn, command + 6)))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "rmdir ok");
    }
  else if (!str_cmp_prefix ("mkdir ", command))
    {
      /* "mkdir <path>" */
      if (0 > (cmd_status = vu_mkdir (&errn, command + 6, 0777)))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "rmdir ok");
    }
  else if (!str_cmp_prefix ("noop", command))
    {
      /* "noop" */
      cmd_status = 0;
      message = str_save (0, "noop ok");
    }
  else if (!str_cmp_prefix ("pwd", command))
    {
      /* There needs to be a vu_ version of getcwd, but there 
       * isn't yet, so this cheats by using the libc version.
       */
      char * path;
      size_t sizeof_path;

      sizeof_path = 4096;
      path = lim_malloc (0, sizeof_path);

      while (1)
	{
	  if (getcwd (path, sizeof_path))
	    {
	      if ((0 > printfmt (&errn, connect_out, "OK %lu\n%s\n", (unsigned long)(1 + str_length (path)), path))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn != EPIPE)
		    panic ("I/O error in local_handle_command (pwd OK)\n");
		}
	      lim_free (0, path);
	      return;
	    }
	  else if (errno == ERANGE)
	    {
	      sizeof_path *= 2;
	      path = lim_realloc (0, path, sizeof_path);
	    }
	  else
	    {
	      cmd_status = -1;
	      message = str_save (0, errno_to_string (errno));
	      lim_free (0, path);
	      break;
	    }
	}
    }
  else if (!str_cmp_prefix ("rnfr ", command))
    {
      if (rename_from_path)
	lim_free (0, rename_from_path);

      rename_from_path = str_save (0, command + 5);
      message = str_save (0, "rnfr ok");
      cmd_status = 0;
    }
  else if (!str_cmp_prefix ("rnto ", command))
    {
      if (!rename_from_path)
	goto bad_command;
      
      if (0 > (cmd_status = vu_rename (&errn, rename_from_path, command + 5)))
	message = str_save (0, errno_to_string (errn));
      else
	message = str_save (0, "rename ok");
    }  


  /* commands requiring a data port */

  else
    {
      t_uchar * host;
      t_uchar * comma;
      t_uchar * port;
      t_uchar * args;
      t_ulong h;
      t_uint p;
      int data_fd;

      /* <cmd> host,port args...
       * or
       * <cmd> pasv args...
       */

      host = str_chr_index (command, ' ');
      if (!host)
	goto bad_command;

      ++host;

      if (!str_cmp_prefix ("pasv ", host))
	{
	  /* passive mode */
	  /* block postconditions: args points at the list of args, data_fd ready */
	  if ((0 > printfmt (&errn, connect_out, "%s\n", my_data_url))
	      || (0 > vfdbuf_flush (&errn, connect_out)))
	    {
	      if (errn == EPIPE)
		return;
	      else
		panic ("I/O error in local_handle_command (PASV url send)\n");
	    }
	  data_fd = safe_open (my_data_url, O_RDWR, 0);
	  args = host + 5;	/* "pasv " */
	}
      else
	{
	  /* non-passive mode */
	  /* block postconditions: args points at the list of args, data_fd ready */

	  comma = str_chr_index (host, ',');
	  if (!comma)
	    goto bad_command;

	  if (cvt_decimal_to_ulong (&errn, &h, host, comma - host))
	    goto bad_command;

	  port = comma + 1;

	  args = str_chr_index (port, ' ');
	  if (!args)
	    goto bad_command;

	  if (cvt_decimal_to_uint (&errn, &p, port, args - port))
	    goto bad_command;

	  args += 1;

	  data_fd = url_inet_client_addr (&errn, h, p);
	  if (data_fd < 0)
	    {
	      message = str_save (0, errno_to_string (errn));
	      cmd_status = -1;
	      goto cmd_done;
	    }
	}

      if (!str_cmp_prefix ("get ", command))
	{
	  int file_fd;

	  file_fd = vu_open (&errn, args, O_RDONLY, 0);
	  if (file_fd < 0)
	    {
	      int ign;

	    bad_data_cmd:
	      vu_close (&ign, data_fd);
	      message = str_save (0, errno_to_string (errn));
	      cmd_status = -1;
	      goto cmd_done;
	    }
	  else
	    {
	      message = str_save (0, "file opened for get");
	      if ((0 > printfmt (&errn, connect_out, "OK %lu\n %s\n", (unsigned long)(2 + str_length (message)), message))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn == EPIPE)
		    goto bad_data_cmd;
		  else
		    panic ("I/O error in local_handle_command (get OK)\n");
		}
	    }

	  {
	    char buf[4096];
	    size_t got;

	    while (1)
	      {
		got = vu_read_retry (&errn, file_fd, buf, sizeof (buf));
		if (   (got < 0)
		    || (   (got > 0)
			&& (got != vu_write_retry (&errn, data_fd, buf, got))))
		  {
		    safe_close (file_fd);
		    goto bad_data_cmd;
		  }
		else if (got == 0)
		  {
		    if (   (0 > vu_close (&errn, file_fd))
			|| (0 > vu_close (&errn, data_fd)))
		      goto bad_data_cmd;
		    break;
		  }
	      }
	  }

	  message = str_save (0, "get command ok");
	  cmd_status = 0;
	}
      else if (!str_cmp_prefix ("put ", command))
	{
	  int file_fd;

	  file_fd = vu_open (&errn, args, O_WRONLY | O_CREAT, 0666);
	  safe_buffer_fd (file_fd, 0, O_WRONLY, 0);
	  if (file_fd < 0)
	    goto bad_data_cmd;
	  else
	    {
	      message = str_save (0, "file opened for put");
	      if ((0 > printfmt (&errn, connect_out, "OK %lu\n %s\n", (unsigned long)(2 + str_length (message)), message))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn == EPIPE)
		    goto bad_data_cmd;
		  else
		    panic ("I/O error in local_handle_command (put OK)\n");
		}
	    }

	  {
	    char buf[4096];
	    size_t got;

	    while (1)
	      {
		got = vu_read_retry (&errn, data_fd, buf, sizeof (buf));
		if (   (got < 0)
		    || (   (got > 0)
			&& (got != vu_write_retry (&errn, file_fd, buf, got))))
		  {
		    safe_close (file_fd);
		    goto bad_data_cmd;
		  }
		else if (got == 0)
		  {
		    if (   (0 > vu_close (&errn, file_fd))
			|| (0 > vu_close (&errn, data_fd)))
		      goto bad_data_cmd;
		    break;
		  }
	      }
	  }

	  message = str_save (0, "put command ok");
	  cmd_status = 0;
	}
      else if (!str_cmp_prefix ("nlist ", command))
	{
	  DIR * dir;
	  int l;

	  l = str_length (args);

	  if (0 > vu_opendir (&errn, &dir, l ? args : (t_uchar *)"."))
	    goto bad_data_cmd;
	  else
	    {
	      message = str_save (0, "dir opened for nlist");
	      if ((0 > printfmt (&errn, connect_out, "OK %lu\n %s\n", (unsigned long)(2 + str_length (message)), message))
		  || (0 > vfdbuf_flush (&errn, connect_out)))
		{
		  if (errn == EPIPE)
		    goto bad_data_cmd;
		  else
		    panic ("I/O error in local_handle_command (nlist OK)\n");
		}
	    }

	  while (1)
	    {
	      char * file;

	      file = 0;

	      if (0 > vu_readdir (&errn, 0, &file, dir))
		{
		  safe_closedir (dir);
		  if (!errn)
		    break;
		  else
		    goto bad_data_cmd;
		}
	      else 
		{
		  if (   str_cmp (".", file)
		      && str_cmp ("..", file)
		      && (0 > printfmt (&errn, data_fd, "%s\r\n", file)))
		    {
		      lim_free (0, file);
		      safe_closedir (dir);
		      goto bad_data_cmd;
		    }
		  lim_free (0, file);
		}
	    }
	  if (0 > vu_close (&errn, data_fd))
	    goto bad_data_cmd;

	  message = str_save (0, "nlist command ok");
	  cmd_status = 0;
	}
      else if (!str_cmp_prefix ("list ", command))
	{
	  int ls_pid;

	  message = str_save (0, "beginning list");
	  if ((0 > printfmt (&errn, connect_out, "OK %lu\n %s\n", (unsigned long)(2 + str_length (message)), message))
	      || (0 > vfdbuf_flush (&errn, connect_out)))
	    {
	      if (errn == EPIPE)
		goto bad_data_cmd;
	      else
		panic ("I/O error in local_handle_command (list OK)\n");
	    }

	  ls_pid = 0;

	  if (sigsetjmp (sigchld_jmp_tool, 1))
	    {
	      int wait_pid;
	      int wait_status;

	      safe_close (data_fd);
	      tool_running = 0;
	      
	      if (!ls_pid)
		{
		  /* SIGCHLD must have been for the primary subprocess
		   */
		  siglongjmp (sigchld_jmp, 1);
		}

	      wait_pid = waitpid (ls_pid, &wait_status, WUNTRACED);

	      if (wait_pid < 0)
		panic ("error waiting for ls subprocess");

	      if (wait_pid == 0)
		{
		  /* must be the main subprocess that gave SIGCHLD
		   */
		  if (ls_pid)
		    {
		      ls_pid = 0;
		      kill (ls_pid, 9);
		    }
		  siglongjmp (sigchld_jmp, 1);
		}
	      else if (WIFEXITED (wait_status) && !WEXITSTATUS (wait_status))
		{
		  message = str_save (0, "list command ok");
		  cmd_status = 0;
		  goto cmd_done;
		}
	      else
		{
		  message = str_save (0, "abnormal exit from ls");
		  cmd_status = -1;
		  goto cmd_done;
		}
	    }

	  tool_running = 1;

	  /* Between this fork and the `sigsuspend', nothing may happen that
	   * can not be asynchronously interrupted by `SIGCHLD' with a 
	   * longjmp to `sigchld_jmp_tool' which is filled above.
	   *
	   * That longjmp handler usually resumes normal execution so anything
	   * tricky here would cause reentrancy problems there.
	   */

	  ls_pid = fork ();

	  if (ls_pid < 0)
	    panic ("unable to fork for ls");

	  if (ls_pid > 0)
	    {
	      sigset_t mask;

	      sigemptyset (&mask);
	      sigsuspend (&mask);
	      panic ("SIGCHLD didn't cause longjmp?!?!?!");
	    }
	  else
	    {
	      safe_move_fd (data_fd, 1);
	      execlp ("ls", "ls", "-l", 0);
	      panic ("execlp of ls failed");
	    }
	}
      else
	{
	bad_command:
	  safe_printfmt (2, "with-ftp: bad command recieved (%s)\n", command);
	  panic ("with-ftp protocol error");
	}
    }
      
 cmd_done:
  {
    int ign;
    printfmt (&ign, connect_out, "%s %lu\n %s\n", ((cmd_status < 0) ? "ERROR" : "OK"), (unsigned long)(2 + str_length (message)), message);
    vfdbuf_flush (&ign, connect_out);
    lim_free (0, message);
  }
}



static long
trim_crnl (t_uchar * str, long len)
{
  if (!len)
    return 0;

  if (str[len - 1] == '\n')
    {
      str[len - 1] = 0;
      --len;
    }

  if (!len)
    return 0;

  if (str[len - 1] == '\r')
    {
      str[len - 1] = 0;
      --len;
    }

  return len;
}

static void
child_changed (int sig)
{
  if (tool_running)
    siglongjmp (sigchld_jmp_tool, 1);
  else
    siglongjmp (sigchld_jmp, 1);
}

static char *
peer_addr_env (int fd)
{
  struct sockaddr_in addr;
  int size;
  char * addr_name;
	  

  size = sizeof (addr);
  if (0 > getpeername (fd, (struct sockaddr *)&addr, &size))
    panic ("unable to get servers address (getpeername)");

  addr_name = inet_ntoa (addr.sin_addr);
  if (!addr_name)
    panic ("inet_ntoa failed on server address");

  return str_alloc_cat (0, "WFTP_SITE_ADDR=", addr_name);
}



/* tag: Tom Lord Tue Dec  4 14:41:23 2001 (with-ftp.c)
 */
