/*
 * Copyright (c) 2000-2001 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 *
 * ra  - Read Argus
 *       This program read argus output streams, either through a socket,
 *       a piped stream, or in a file, filters and optionally writes the
 *       output to a file, its stdout or prints the binary records to
 *       stdout in ASCII.
 */


#ifndef ArgusClient
#define ArgusClient
#endif


#include <argus_client.h>
#include <signal.h>
#include <ctype.h>

int RaInitialized = 0;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      RaInitialized++;
      if (nflag) {
         hfield = 15;
         pfield = 5;
      }
   }
}


void
RaParseComplete (int sig)
{
   long long total_count, total_dst_count, total_src_count;
   long long total_bytes, total_dst_bytes, total_src_bytes;

   if (aflag) {
      putchar ('\n');

      total_src_count = tcp_src_count + udp_src_count + icmp_src_count +
         ip_src_count + arp_src_count + nonip_src_count;
      total_dst_count = tcp_dst_count + udp_dst_count + icmp_dst_count +
         ip_dst_count + arp_dst_count + nonip_dst_count;
      total_count = total_dst_count + total_src_count;

      total_src_bytes = tcp_src_bytes + udp_src_bytes + icmp_src_bytes +
         ip_src_bytes + arp_src_bytes + nonip_src_bytes;
      total_dst_bytes = tcp_dst_bytes + udp_dst_bytes + icmp_dst_bytes +
         ip_dst_bytes + arp_dst_bytes + nonip_dst_bytes;
      total_bytes = total_dst_bytes + total_src_bytes;

      if (total_count) {
         printf ("             total_pkts         src_pkts         dst_pkts      total_bytes        src_bytes        dst_bytes\n");
         printf ("tcp    %16lld %16lld %16lld %16lld %16lld %16lld\n",
            tcp_dst_count + tcp_src_count, tcp_src_count, tcp_dst_count,
            tcp_dst_bytes + tcp_src_bytes, tcp_src_bytes, tcp_dst_bytes);

         printf ("udp    %16lld %16lld %16lld %16lld %16lld %16lld\n",
            udp_dst_count + udp_src_count, udp_src_count, udp_dst_count,
            udp_dst_bytes + udp_src_bytes, udp_src_bytes, udp_dst_bytes);

         printf ("icmp   %16lld %16lld %16lld %16lld %16lld %16lld\n",
            icmp_dst_count + icmp_src_count, icmp_src_count, icmp_dst_count,
            icmp_dst_bytes + icmp_src_bytes, icmp_src_bytes, icmp_dst_bytes);

         printf ("ip     %16lld %16lld %16lld %16lld %16lld %16lld\n",
            ip_dst_count + ip_src_count, ip_src_count, ip_dst_count,
            ip_dst_bytes + ip_src_bytes, ip_src_bytes, ip_dst_bytes);

         printf ("arp    %16lld %16lld %16lld %16lld %16lld %16lld\n",
            arp_dst_count + arp_src_count, arp_src_count, arp_dst_count,
            arp_dst_bytes + arp_src_bytes, arp_src_bytes, arp_dst_bytes);

         printf ("non-ip %16lld %16lld %16lld %16lld %16lld %16lld\n",
            nonip_dst_count + nonip_src_count, nonip_src_count, nonip_dst_count,
            nonip_dst_bytes + nonip_src_bytes, nonip_src_bytes, nonip_dst_bytes);

         printf ("sum    %16lld %16lld %16lld %16lld %16lld %16lld",
            total_count, total_src_count, total_dst_count,
            total_bytes, total_src_bytes, total_dst_bytes);

         printf ("\n\n");

      } else
         printf ("No data seen.\n");

      fflush (stdout);
   }

   if (sig == SIGINT)
      _exit(0);
}


void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout()\n");
#endif
}

void
parse_arg (argc, argv)
int argc;
char**argv;
{}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ra Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -a             print record summaries on termination.\n");
   fprintf (stderr, "         -A             print application bytes.\n");
   fprintf (stderr, "         -b             dump packet-matching code.\n");
   fprintf (stderr, "         -c             print packet and byte counts.\n");
   fprintf (stderr, "         -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -d <bytes>     print number of <bytes> from user data capture buffer.\n");
   fprintf (stderr, "               format:  num | s<num> | d<num> | s<num>:d<num>\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>     specify debug level\n");
#endif
   fprintf (stderr, "         -e <encode>    convert user data using <encode> method.\n");
   fprintf (stderr, "                        Supported types are <Ascii> and <Encode64>.\n");
   fprintf (stderr, "         -E <file>      write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "         -g             print record time duration.\n");
   fprintf (stderr, "         -G             print both start and last time values.\n");
   fprintf (stderr, "         -h             print help.\n");
   fprintf (stderr, "         -i             print source probe id.\n");
   fprintf (stderr, "         -I             print transaction state and option indicators.\n");
   fprintf (stderr, "         -l             print last time values [default is start time].\n");
   fprintf (stderr, "         -m             print MAC addresses.\n");
   fprintf (stderr, "         -n             don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "         -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R             print out response data when availabile.\n");
   fprintf (stderr, "         -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "         -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                             [yyyy/]mm/dd\n");
   fprintf (stderr, "         -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z             print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


int RaLabelCounter = 0;

char RaLabelStr[256], *RaLabel = NULL;
char *RaDefaultFlowLabel = "     SrcAddr  Sport  Dir         DstAddr  Dport ";
char *RaMatrixFlowLabel  = "     SrcAddr         Dir         DstAddr        ";
char *RaTopNFlowLabel    = "     SrcAddr           ";

char *RacOptionLabel = " SrcPkt   Dstpkt    SrcBytes     DstBytes   ";
char *RAcOptionLabel = " SrcPkt   Dstpkt    SAppBytes    DAppBytes  ";
char *RamOptionLabel = "    SrcMacAddr       DstMacAddr       ";
char *RaROptionLabel = " SrcPkt   Dstpkt    Response Information    ";
char *RaTOptionLabel  = " Stos Dtos Sttl Dttl   SMinIdle    SMeanIdl    SMaxIdle     DMinIdle    DMeanIdl    DMaxIdle ";

void
RaProcessRecord (struct ArgusRecord *argus)
{
   int i, gflagvar = gflag, Gflagvar = Gflag;
   char RaPrecisionPad[32], RaTimePad[32], date[128];

   bzero (date, sizeof (date));
   bzero (RaTimePad, sizeof (RaTimePad));
   bzero (RaPrecisionPad, sizeof (RaPrecisionPad));

   if (Lflag && !(qflag)) {
      if (RaLabel == NULL) {
         gflag = 0; Gflag = 0;
         print_date (argus, date);
         gflag = gflagvar; Gflag = Gflagvar;
         if (strlen(date) > 10)
            for (i = 0; i < (strlen(date) - 10)/2; i++)
               strcat (RaTimePad, " ");
          
         for (i = 0; i < pflag/2; i++)
            strcat (RaPrecisionPad, " ");

         bzero (RaLabelStr, sizeof(RaLabelStr));
         RaLabel = RaLabelStr;

         if (idflag)
            strcat (RaLabel, "     ProbeId    ");

         strcat (RaLabel, RaTimePad);
         if (lflag)
            strcat (RaLabel, " Last_Time ");
         else
            strcat (RaLabel, "Start_Time ");

         if (pflag && !uflag)
            if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0'))
               strcat (RaLabel, "Fraction ");

         strcat (RaLabel, RaTimePad);
         if (strlen(date) & 0x01) strcat (RaLabel, " ");

         if (gflag) {
            strcat (RaLabel, RaPrecisionPad);
            if (pflag > 0)
               strcat (RaLabel, " Duration ");
            else
               strcat (RaLabel, "Duration ");

            strcat (RaLabel, RaPrecisionPad);
            if (pflag &0x01) strcat (RaLabel, " ");
         } else
         if (Gflag) {
            strcat (RaLabel, RaTimePad);
            strcat (RaLabel, " Last_Time ");
            strcat (RaLabel, RaTimePad);
            if (strlen(date) & 0x01) strcat (RaLabel, " ");
         } else
            strcat (RaLabel, " ");

         if (Iflag) strcat (RaLabel, " Flgs  ");

         strcat (RaLabel, "Type  ");
         if (mflag) strcat (RaLabel, RamOptionLabel);

         if (ArgusInput->ArgusManStart.ahdr.type & ARGUS_RMON) {
            if (ArgusInput->ArgusManStart.ahdr.status & ARGUS_TOPN)
               strcat (RaLabel, RaTopNFlowLabel);
            else
               strcat (RaLabel, RaMatrixFlowLabel);

         } else {
            
            sprintf (&RaLabel[strlen(RaLabel)], "%*sSrcAddr%*s%*sSport %*sDir ", (hfield - 8)/2, " ", (hfield - 8)/2, " ", (pfield - 4)/2, " ", (pfield - 4)/2, " ");
            sprintf (&RaLabel[strlen(RaLabel)], "  %*s DstAddr%*s%*sDport%*s", (hfield - 8)/2, " ", (hfield - 8)/2, " ", (pfield - 4)/2, " ", (pfield - 4)/2, " ");
         }

         if (cflag && Rflag) {
            strcat (RaLabel, RaROptionLabel);
         } else {
            if (cflag && Aflag) {
               strcat (RaLabel, RAcOptionLabel);
            } else {
               if (cflag) {
                  strcat (RaLabel, RacOptionLabel);
               }
            }
         }

         if (zflag || Zflag) {
            strcat (RaLabel, " Status");
         } else {
            strcat (RaLabel, "State");
         }

         if (dflag) {
            int slen = ArgusSrcUserDataLen, dlen = ArgusDstUserDataLen;

            if (eflag == ARGUS_ENCODE_64) {
               if (slen > 0)
                  slen += slen/2;
               if (dlen > 0)
                  dlen += dlen/2;
            }

            if (slen) {
               sprintf (&RaLabel[strlen(RaLabel)], " %*s%s", ((slen + 8) - 7)/2, " ", "SrcData");
            }

            if (dlen) {
               sprintf (&RaLabel[strlen(RaLabel)], "%*s ", ((slen + 8) - 7)/2, " ");
               sprintf (&RaLabel[strlen(RaLabel)], " %*s%s", ((dlen + 8) - 7)/2, " ", "DstData");
            }
         }
      }

      if ((RaFieldDelimiter != ' ') && (RaFieldDelimiter != '\0')) {
         char tmpbuf[256], *ptr = tmpbuf, *str = RaLabel;
         bzero (tmpbuf, sizeof(tmpbuf));

         while (isspace((int)*str)) str++;

         while (*str) {
            if (*str == ' ') {
               *ptr++ = RaFieldDelimiter;
               while (isspace((int)*str)) str++;
            }
            *ptr++ = *str++;
         }
         bzero (RaLabelStr, sizeof(RaLabelStr));
         bcopy (tmpbuf, RaLabelStr, strlen(tmpbuf));
      }

      if (!(RaLabelCounter++ % Lflag)) {
         printf ("%s\n", RaLabel);
      }

      if (Lflag < 0)
         Lflag = 0;
   }
}


void
process_man (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_man_string (argus);
      (void) printf ("%s", str);

      printf ("\n");
      fflush (stdout);
   }
}


void
process_frag (struct ArgusRecord *argus)
{
}


void
process_tcp (struct ArgusRecord *argus)
{
   if (aflag) {
      tcp_src_count += argus->argus_far.src.count;
      tcp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         tcp_src_bytes += argus->argus_far.src.appbytes;
         tcp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         tcp_src_bytes += argus->argus_far.src.bytes;
         tcp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   RaProcessRecord (argus);

   if (!qflag) {
      printf ("%s", get_tcp_string (argus));
      if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
         printf ("%s", RaGetUserDataString(argus));

      printf ("\n");
   }

   fflush(stdout);
}


void
process_icmp (struct ArgusRecord *argus)
{
   if (aflag) {
      icmp_src_count += argus->argus_far.src.count;
      icmp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         icmp_src_bytes += argus->argus_far.src.appbytes;
         icmp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         icmp_src_bytes += argus->argus_far.src.bytes;
         icmp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   RaProcessRecord (argus);

   if (!qflag) {
      printf ("%s", get_icmp_string (argus));
      if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
         printf ("%s", RaGetUserDataString(argus));

      printf ("\n");
   }

   fflush(stdout);
}


void
process_udp (struct ArgusRecord *argus)
{
   if (aflag) {
      udp_src_count += argus->argus_far.src.count;
      udp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         udp_src_bytes += argus->argus_far.src.appbytes;
         udp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         udp_src_bytes += argus->argus_far.src.bytes;
         udp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   RaProcessRecord (argus);

   if (!qflag) {
      printf ("%s", get_ip_string (argus));
      if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
         printf ("%s", RaGetUserDataString(argus));
   
      printf ("\n");
   }
   fflush(stdout);
}


void
process_ip (struct ArgusRecord *argus)
{
   if (aflag) {
      ip_src_count += argus->argus_far.src.count;
      ip_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         ip_src_bytes += argus->argus_far.src.appbytes;
         ip_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         ip_src_bytes += argus->argus_far.src.bytes;
         ip_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   RaProcessRecord (argus);

   if (!qflag) {
      printf ("%s", get_ip_string (argus));
      if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
         printf ("%s", RaGetUserDataString(argus));
   
      printf ("\n");
   }
   fflush(stdout);
}


void
process_arp (struct ArgusRecord *argus)
{
   if (aflag) {
      arp_src_count += argus->argus_far.src.count;
      arp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         arp_src_bytes += argus->argus_far.src.appbytes;
         arp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         arp_src_bytes += argus->argus_far.src.bytes;
         arp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   RaProcessRecord (argus);

   if (!qflag) {
      printf ("%s", get_arp_string (argus));
      if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
         printf ("%s", RaGetUserDataString(argus));
   
      printf ("\n");
   }
   fflush(stdout);
}


void
process_non_ip (struct ArgusRecord *argus)
{
   if (aflag) {
      nonip_src_count += argus->argus_far.src.count;
      nonip_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         nonip_src_bytes += argus->argus_far.src.appbytes;
         nonip_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         nonip_src_bytes += argus->argus_far.src.bytes;
         nonip_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   RaProcessRecord (argus);

   if (!qflag) {
      printf ("%s", get_nonip_string (argus));
      if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
         printf ("%s", RaGetUserDataString(argus));
   
      printf ("\n");
   }
   fflush(stdout);
}

