/*
 * Copyright (c) 2000-2001 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * ragator - argus record aggregator  with flow model manipulation.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#ifndef ArgusClient
#define ArgusClient
#endif


#include <argus_client.h>
#include <ragator.h>

#include <math.h>

#define RA_HASHTABLESIZE	65536

int RaFlowModelFile = 0;
int RaAllocHashTableHeaders = 0;
int RaAllocArgusRecordStore = 0;
int RaAllocArgusRecord      = 0;

#define RA_SVCPASSED	0x0100
#define RA_SVCFAILED	0x0200
#define RA_SVCTEST	(RA_SVCFAILED|RA_SVCPASSED)


struct ArgusServiceRecord {
   u_int status;
   struct ArgusRecord *argus;
};

struct ArgusHoldingBuffer {
   int size, count;
   struct ArgusListStruct **array;
};
 
void RaProcessRecord (struct ArgusRecord *argus) { }
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
struct ArgusRecordStore *RaThisArgusStore = NULL;
struct ArgusHoldingBuffer *RaHoldingBuffer = NULL;

int RaInitialized = 0;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaWriteOut = 0;
   
      bzero ((char *) RaFlowArray,  sizeof(RaFlowArray));
      bzero ((char *) RaModelArray, sizeof(RaModelArray));
   
      if (ArgusFlowModelFile)
         if ((RaFlowModelFile = RaReadFlowModelFile()) < 0)
            exit(0);
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      if (Bflag) {
         if ((RaHoldingBuffer = (struct ArgusHoldingBuffer *) ArgusCalloc (1, sizeof(*RaHoldingBuffer))) != NULL) {
            if ((RaHoldingBuffer->array = (struct ArgusListStruct **) ArgusCalloc (Bflag, sizeof(void *))) != NULL) {
               RaHoldingBuffer->size = Bflag;
            } else {
               ArgusFree (RaHoldingBuffer);
               RaHoldingBuffer = NULL;
               exit (-1);
            }
         } else
            exit (-1);
      }

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }

      RaInitialized++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if (!RaParseCompleting) {
      RaParseCompleting++;
      if (RaModelerQueue)
         RaProcessQueue (RaModelerQueue, ARGUS_STOP);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ragator Version %s\n", version);
   fprintf (stderr, "usage:  %s [-f flowfile]\n", ArgusProgramName);
   fprintf (stderr, "usage:  %s [-f flowfile] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -f <flowfile>  read flow model from <flowfile>.\n");
   fprintf (stderr, "ra-options: -a             print record summaries on termination.\n");
   fprintf (stderr, "            -A             print application bytes.\n");
   fprintf (stderr, "            -b             dump packet-matching code.\n");
   fprintf (stderr, "            -c             print packet and byte counts.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>     specify debug level\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -g             print record time duration.\n");
   fprintf (stderr, "            -G             print both start and last time values.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -I             print transaction state and option indicators.\n");
   fprintf (stderr, "            -l             print last time values [default is start time].\n");
   fprintf (stderr, "            -m             print MAC addresses.\n");
   fprintf (stderr, "            -n             don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R             print out response data when availabile.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                            mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z             print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


struct ArgusRecordStore *RaFindRevArgusRecord(struct ArgusRecord *);
struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
long long RaGetActiveDuration (struct ArgusRecord *);
long long RaGetuSecDuration (struct ArgusRecord *);

struct ArgusRecord *
RaCopyArgusRecord (struct ArgusRecord *argus)
{
   int length = 0;
   struct ArgusRecord *retn = NULL;

   if (argus && ((length = argus->ahdr.length) > 0))
      if ((retn = (struct ArgusRecord *) ArgusCalloc(1, length)) != NULL)
         bcopy ((char *) argus, (char *) retn, length);

   return (retn);
}


long long
RaGetActiveDuration (struct ArgusRecord *argus)
{
   return (RaGetuSecDuration (argus));
}

long long
RaGetuSecDuration (struct ArgusRecord *argus)
{
   struct ArgusAGRStruct *agr;
   long long retn = 0;
   int sec, usec;

   if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
      int ArgusThisMultiplier = 1000;

      if (agr->status & ARGUS_AGR_USECACTTIME)
         ArgusThisMultiplier = 1;

      if (agr->act.n)
         return (retn = agr->act.mean * ArgusThisMultiplier);
   }

   sec  = argus->argus_far.time.last.tv_sec  - argus->argus_far.time.start.tv_sec;
   usec = argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec;

   if (usec < 0) {
      sec--; usec += 1000000;
   }

   retn = (sec * 1000000) + usec;

   return (retn);
}


long long RaThisActiveDuration = 0;
int RaThisActiveIndex = 0;
int RaTimeout = 0x1FFFFFFF;
int RaIdleTimeout = 0;


void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;

   RaThisActiveDuration = RaGetActiveDuration(argus);

   if (Hflag) {
      RaThisActiveIndex = (RaThisActiveDuration < 100)        ? 0 :
                          (RaThisActiveDuration < 500)        ? 1 :
                          (RaThisActiveDuration < 1000)       ? 2 :
                          (RaThisActiveDuration < 5000)       ? 3 :
                          (RaThisActiveDuration < 10000)      ? 4 :
                          (RaThisActiveDuration < 50000)      ? 4 :
                          (RaThisActiveDuration < 100000)     ? 5 :
                          (RaThisActiveDuration < 500000)     ? 6 :
                          (RaThisActiveDuration < 100000)     ? 7 :
                          (RaThisActiveDuration < 250000)     ? 8 :
                          (RaThisActiveDuration < 500000)     ? 9 :
                          (RaThisActiveDuration < 750000)     ? 10:
                          (RaThisActiveDuration < 1000000)    ? 11:
                          (RaThisActiveDuration < 5000000)    ? 12:
                          (RaThisActiveDuration < 10000000)   ? 13:
                          (RaThisActiveDuration < 50000000)   ? 14: 15;
   } else
      RaThisActiveIndex = 0;

   if ((rap = RaFlowModelOverRides(argus)) != NULL) {
      RaModifyFlow(rap, argus);
      RaTimeout = rap->ArgusTimeout;
   } else 
      RaModifyDefaultFlow(argus);

   if ((store = RaFindArgusRecord(argus)) == NULL)
      store = RaFindRevArgusRecord(argus);

   if (store) {
      RaThisArgusStore = store;
      RaRemoveFromQueue(RaModelerQueue, &store->qhdr);
      RaPushQueue(RaModelerQueue, &store->qhdr);

      RaCheckTimeout (store, argus);

      if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
         RaSendArgusRecord(store);
         store->status &= ~RA_SVCTEST;
         store->status |= (srv->status & RA_SVCTEST);
      }

      if (!(store->data[RaThisActiveIndex])) {
         struct ArgusRecordData *data = NULL;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord(argus, store->data[RaThisActiveIndex]);

      RaUpdateArgusStore(argus, store);

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         if ((store->rahtblhdr = RaAddHashEntry (store)) != NULL)
            RaAddToQueue(RaModelerQueue, &store->qhdr);

         store->ArgusTimeout     = RaTimeout;
         store->ArgusIdleTimeout = RaIdleTimeout;

         RaThisArgusStore = store;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.mean * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.mean * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[RaThisActiveIndex] = data;

         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaProcessRecord: done\n");
#endif
}


void
process_man (struct ArgusRecord *argus)
{
}


struct ArgusServiceRecord ArgusThisSrv;

void
process_tcp (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   RaIdleTimeout = 0;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
   RaIdleTimeout = 0;
}


void
process_icmp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_udp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) 
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_arp (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessSrvRecord (&ArgusThisSrv);
}


void
process_non_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusAGRStruct *agr = NULL;
   int i;

   for (i = 0; i < ARGUS_TIMESERIES; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.mean = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.mean = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);
            }

            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) buf;

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) == NULL) {
               if (ArgusFlowModelFile && (data->agr.count > 1)) {
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);

               } else {
                  argus->ahdr.status &= ~(ARGUS_MERGED);
               }
            }
      
            if (wfile) {

#ifdef _LITTLE_ENDIAN
               ArgusHtoN(argus);
#endif

               ArgusWriteNewLogfile (wfile, argus);
      
#ifdef _LITTLE_ENDIAN
               ArgusNtoH(argus);
#endif

            } else {
      
               if (argus->ahdr.type & ARGUS_MAR)
                  printf ("%s\n", get_man_string (argus));
                  
               else {
                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP:              
                              printf ("%s", get_tcp_string (argus));
                              break;
            
                           case IPPROTO_ICMP:              
                              printf ("%s", get_icmp_string (argus));
                              break;
            
                           default:
                              printf ("%s", get_ip_string (argus));
                              break;
                        }
                        break;
            
                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                        printf ("%s", get_arp_string (argus));
                        break;
            
                     default:
                        printf ("%s", get_nonip_string (argus));
                        break;
                  }

                  if (ArgusSrcUserDataLen || ArgusDstUserDataLen)
                     printf ("%s\n", RaGetUserDataString(argus));
                  else
                     printf ("\n");
               }
               fflush (stdout);
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_TIME_DSR: {
                        struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                        time->src.act.n = 0;
                        time->src.act.mean = 0;
                        time->src.act.stdev = 0;
                        time->src.act.max = 0;
                        time->src.act.min = 0x7FFFFFFF;
                        time->src.idle.n = 0;
                        time->src.idle.mean = 0;
                        time->src.idle.stdev = 0;
                        time->src.idle.max = 0;
                        time->src.idle.min = 0x7FFFFFFF;

                        time->dst.act.n = 0;
                        time->dst.act.mean = 0;
                        time->dst.act.stdev = 0;
                        time->dst.act.max = 0;
                        time->dst.act.min = 0x7FFFFFFF;
                        time->dst.idle.n = 0;
                        time->dst.idle.mean = 0;
                        time->dst.idle.stdev = 0;
                        time->dst.idle.max = 0;
                        time->dst.idle.min = 0x7FFFFFFF;

                        break;
                     }

                     case ARGUS_VLAN_DSR: {
                        struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                        vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_MPLS_DSR: {
                        struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                        mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.min = 0x7FFFFFFF;   agr->act.mean = 0;
                        agr->act.stdev = 0;  agr->act.max = 0;
                        agr->idle.n = 0;
                        agr->idle.min = 0x7FFFFFFF; agr->idle.mean = 0;
                        agr->idle.stdev = 0; agr->idle.max = 0;
                        break;
                     }
                  }

                  if ((farlen = farhdr->length) == 0)
                     break;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }

            data->agr.count = 0;
            data->agr.act.n = 0;
            data->agr.act.mean = 0;
            data->agr.act.stdev = 0;
            data->agr.act.max = 0;
            data->agr.act.min = 0x7FFFFFFF;
            data->agr.idle.n = 0;
            data->agr.idle.mean = 0;
            data->agr.idle.stdev = 0;
            data->agr.idle.max = 0;
            data->agr.idle.min = 0x7FFFFFFF;
      
            data->act.n = 0;
            data->act.sumtime = 0;
            data->act.sumsqrd = 0;
            data->idle.n = 0;
            data->idle.sumtime = 0;
            data->idle.sumtime = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}

int
RaCIDRAddrMatches (u_int addr, struct RaCIDRAddr *cidr)
{
   int retn = 1;

   if (cidr->addr)
      if ((addr & cidr->mask) != cidr->addr)
         retn = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCIDRAddrMatches(0x%x, 0x%x) returned %d\n", addr, cidr, retn);
#endif

   return (retn);
}

int
RaProtoMatches (u_short p, int proto)
{
   int retn = 1;

   if (proto != 0xFF)
      retn = (p == proto) ? 1 : 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProtoMatches (%d, %d) returned %d\n", p, proto, retn);
#endif

   return (retn);
}


int
RaPortMatches (u_short p1, u_short p2)
{
   int retn = 1;
   
   if (p2 != 0xFFFF)
      retn = (p1 == p2) ? 1 : 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPortMatches (%d, %d) returned %d\n", p1, p2, retn);
#endif

   return (retn);
}

int
RaPolicyMatch (struct ArgusRecord *argus, struct RaPolicyStruct *rap)
{
   int retn = 0;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   u_char proto = flow->ip_flow.ip_p;

   if (RaCIDRAddrMatches (flow->ip_flow.ip_src, &rap->src))

   if (RaCIDRAddrMatches (flow->ip_flow.ip_dst, &rap->dst))

   if (RaProtoMatches (flow->ip_flow.ip_p, rap->proto))

   switch (proto) {
      case IPPROTO_TCP:
      case IPPROTO_UDP:
         if (RaPortMatches (flow->ip_flow.sport, rap->sport))
         if (RaPortMatches (flow->ip_flow.dport, rap->dport))
            retn = 1;
         break;

      case IPPROTO_ICMP:
         if (RaPortMatches (flow->icmp_flow.type, rap->sport))
         if (RaPortMatches (flow->icmp_flow.code, rap->dport))
         retn = 1;
         break;

      default:
         retn = 1;
         break;
   }


#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPolicyMatch(0x%x, 0x%x) returned %d\n", argus, rap, retn);
#endif

   return (retn);
}


struct RaPolicyStruct *
RaFlowModelOverRides (struct ArgusRecord *argus)
{
   struct RaPolicyStruct *retn = NULL;
   int i;

   if (RaThisFlowNum) {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
         case ETHERTYPE_IP:
            for (i = 0; i < RaThisFlowNum; i++) {
               if (RaPolicyMatch (argus, RaFlowArray[i])) {
                  retn = RaFlowArray[i];
                  break;
               }
            }
            break;

         default:
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaFlowModelOverRides(0x%x) returned 0x%x\n", argus, retn);
#endif

   return (retn);
}

#if !defined(__OpenBSD__) || !defined(_NETINET_IP_ICMP_H_)
#include <netinet/ip_icmp.h>
#define _NETINET_IP_ICMP_H_
#endif

void
RaModifyDefaultFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;

   switch (argus->ahdr.status & 0xFFFF) {
      case ETHERTYPE_IP:
         if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_FRAG_FLOWTAG) {
            argus->argus_far.flow.ip_flow.tp_p = 0;
            argus->argus_far.flow.ip_flow.ip_id = 0;
         }
         
         switch (argus->argus_far.flow.ip_flow.ip_p) {
            case IPPROTO_ICMP:
               if (flow->icmp_flow.type != ICMP_UNREACH)
                  flow->icmp_flow.id = 0xFFFF;

               flow->icmp_flow.ip_id = 0xFFFF;
               break;

            case IPPROTO_UDP:
            case IPPROTO_TCP:
            default:
               flow->icmp_flow.ip_id = 0;
               break;
         }
         break;

      case ETHERTYPE_ARP:
      case ETHERTYPE_REVARP:
         break;

      default:
         break;
   }
}

int RaFlowMajorModified = 0;

void
RaModifyFlow (struct RaPolicyStruct *rap, struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   struct RaPolicyStruct *model = RaModelArray[rap->RaModelId];
   u_short proto;

   RaFlowMajorModified = 0;

   switch (argus->ahdr.status & 0xFFFF) {
      case ETHERTYPE_IP: {
         if (flow->ip_flow.tp_p == ARGUS_FRAG_FLOWTAG) {
            flow->ip_flow.tp_p = 0;
            flow->ip_flow.ip_id = 0;
         }

         if (model) {
            flow->ip_flow.ip_src = flow->ip_flow.ip_src & model->src.addr;
            flow->ip_flow.ip_dst = flow->ip_flow.ip_dst & model->dst.addr;

            proto = flow->ip_flow.ip_p;
            if (!(model->proto)) {
               flow->ip_flow.ip_id = 0;
               flow->ip_flow.ip_p = 0;
               flow->ip_flow.tp_p = 0;
               flow->ip_flow.sport = 0xFFFF;
               flow->ip_flow.dport = 0xFFFF;

            } else {
               flow->ip_flow.ip_p  &= model->proto;

               switch (proto) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                  case IPPROTO_IGMP:
                     flow->ip_flow.ip_id = 0;
                     switch (flow->ip_flow.ip_p) {
                        case IPPROTO_TCP:
                        case IPPROTO_UDP:
                        case IPPROTO_IGMP:
                           if (!(model->sport))
                              flow->ip_flow.sport = 0xFFFF;
                           if (!(model->dport)) {
                              flow->ip_flow.dport = 0xFFFF;
                              RaFlowMajorModified = 1;
                           }
                           break;

                        default:
                           flow->ip_flow.sport = 0xFFFF;
                           flow->ip_flow.dport = 0xFFFF;
                           break;
                     }
                     break;
       
                  case IPPROTO_ICMP:
                     if (!(model->sport))
                        flow->icmp_flow.type = 0xFF;
         
                     if (!(model->dport))
                        flow->icmp_flow.code = 0xFF;
         
                     flow->icmp_flow.id = 0xFFFF;
                     flow->icmp_flow.ip_id = 0xFFFF;
                     break;
         
                  default:
                     flow->ip_flow.ip_id = 0;
                     break;
               }

         
               if (flow->ip_flow.ip_p == 0) {
                  flow->ip_flow.tp_p  = 0;
                  flow->ip_flow.ip_id = 0;
               }
            }
         }
         break;
      }

      case ETHERTYPE_REVARP:
      case ETHERTYPE_ARP:
         flow->arp_flow.arp_tpa = flow->arp_flow.arp_tpa & model->src.addr;
         flow->arp_flow.arp_spa = flow->arp_flow.arp_spa & model->dst.addr;

         if ((model->src.addr != 0xFFFFFFFF) || (model->src.addr != 0xFFFFFFFF))
            bzero (flow->arp_flow.etheraddr, 6);
         break;

      default:
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaModifyFlow(0x%x, 0x%x) done.\n", rap, argus);
#endif
}


int RaThisHash = 0;
struct ArgusFlow *RaThisFlow = NULL;
int RaThisFlowRev = 0;

struct ArgusRecordStore *
RaFindRevArgusRecord(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   struct ArgusFlow flow = *RaThisFlow;
   struct ArgusTCPObject *tcp = NULL;

   RaThisFlowRev = 0;

   switch (flow.ip_flow.ip_p) {
      case IPPROTO_UDP:              
         RaThisFlow->ip_flow.ip_src = flow.ip_flow.ip_dst;
         RaThisFlow->ip_flow.ip_dst = flow.ip_flow.ip_src;
         RaThisFlow->ip_flow.sport  = flow.ip_flow.dport;
         RaThisFlow->ip_flow.dport  = flow.ip_flow.sport;
         RaThisFlowRev++;

         if ((hashEntry = RaFindHashObject ()) != NULL)
            retn = hashEntry->storeobj;

         *RaThisFlow = flow;
         break;

      case IPPROTO_TCP:              
         if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
            if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL)
               if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT))
                  break;

         RaThisFlow->ip_flow.ip_src = flow.ip_flow.ip_dst;
         RaThisFlow->ip_flow.ip_dst = flow.ip_flow.ip_src;
         RaThisFlow->ip_flow.sport  = flow.ip_flow.dport;
         RaThisFlow->ip_flow.dport  = flow.ip_flow.sport;
         RaThisFlowRev++;

         if ((hashEntry = RaFindHashObject ()) != NULL)
            retn = hashEntry->storeobj;

         *RaThisFlow = flow;
         break;

      default:
         break;
   }


   return (retn);
}

struct ArgusRecordStore *
RaFindArgusRecord(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct RaHashTableHeader *hashEntry;
   unsigned short hash = 0, *ptr;
   int i, len;

   RaThisFlowRev = 0;
   RaThisFlow = &argus->argus_far.flow;
   ptr = (unsigned short *) RaThisFlow;

   for (i = 0, len = (sizeof(*RaThisFlow)) / sizeof(unsigned short); i < len; i++)
      hash += *ptr++;

   RaThisHash = hash;

   if ((hashEntry = RaFindHashObject ()) != NULL)
      retn = hashEntry->storeobj;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaFindArgusRecord: returning 0x%x\n", retn);
#endif

   return (retn);
}


int RaTotalRecords = 0;


struct ArgusRecordStore *
RaNewArgusStore(struct ArgusRecord *argus)
{
   struct ArgusRecordStore *retn = NULL;
   struct timeval lasttime;

   if ((retn = (struct ArgusRecordStore *) ArgusCalloc (1, sizeof(struct ArgusRecordStore))) != NULL) {
      RaAllocArgusRecordStore++;
      lasttime = argus->argus_far.time.last;

      retn->qhdr.lasttime = lasttime;
      retn->qhdr.logtime  = lasttime;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaNewArgusStore: added 0x%x\n", retn);
#endif

   return (retn);
}

int RaAllocArgusRecordData = 0;
int RaTotalDataRecords = 0;

struct ArgusRecordData *
RaNewArgusData(struct ArgusRecord *argus)
{
   struct ArgusRecordData *retn = NULL;
   struct timeval lasttime;

   if ((retn = (struct ArgusRecordData *) ArgusCalloc (1, sizeof (*retn))) != NULL) {
      RaAllocArgusRecordData++;
   
      if ((retn->argus = RaCopyArgusRecord(argus)) != NULL) {
         RaTotalDataRecords++;

         retn->agr.laststartime = argus->argus_far.time.start;
         lasttime = argus->argus_far.time.last;

         retn->agr.type     = ARGUS_AGR_DSR;
         retn->agr.length   = sizeof(retn->agr);
         retn->agr.lasttime = lasttime;
         retn->agr.lasttime = lasttime;
         retn->agr.act.min  = RaThisActiveDuration;
         retn->agr.act.max  = RaThisActiveDuration;
         retn->act.sumtime  = RaThisActiveDuration;
         retn->act.sumsqrd  = pow (RaThisActiveDuration, 2.0);
         retn->act.n        = 1;
         retn->agr.count    = 1;

         retn->agr.idle.min = 0x7FFFFFFF;
         retn->agr.status  |= ARGUS_AGR_USECACTTIME;
         retn->agr.status  |= ARGUS_AGR_USECIDLETIME;

      } else {
         ArgusFree (retn);
         RaAllocArgusRecordData--;
         retn = NULL;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaNewArgusData: added 0x%x\n", retn);
#endif

   return (retn);
}

void
RaRemoveArgusRecord(struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveArgusRecord: done\n");
#endif
}


void
RaMergeArgusRecord(struct ArgusRecord *a1, struct ArgusRecordData *data)
{
   u_char buf[MAXSTRLEN];
   struct ArgusRecord *a2 = NULL, *tmp = (struct ArgusRecord *)buf;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   struct ArgusFarStruct *far1 = NULL;
   struct ArgusFarStruct *far2 = NULL;
   struct ArgusRecordStore store;
   struct ArgusFarStruct *ArgusThisDataFar;
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0;
   unsigned int i, status = 0, TimeMultiplier;
   long long duration;

   bzero ((char *)&store, sizeof(store));
   store.data[0] = data;

   if ((a2 = data->argus) != NULL) {
      bzero (buf, sizeof(buf));
      bcopy ((char *) a2, buf, a2->ahdr.length);

      ArgusThisDataFar = (struct ArgusFarStruct *) data->farhdrs[ARGUS_FAR_DSR_INDEX];

      a1DSRStatus = ArgusIndexRecord (a1,  a1farhdr);
      a2DSRStatus = ArgusIndexRecord (tmp, a2farhdr);

      if (RaFlowMajorModified) {
         a1DSRStatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
         data->farhdrstatus &= (ARGUS_FAR_DSR_INDEX || ARGUS_AGR_DSR_INDEX);
      }

      status = a1DSRStatus & a2DSRStatus;

      far1 = (struct ArgusFarStruct *) a1farhdr[ARGUS_FAR_DSR_INDEX];
      far2 = (struct ArgusFarStruct *) a2farhdr[ARGUS_FAR_DSR_INDEX];

      if (!(RaThisFlowRev)) {
         if (((far2->src.count + far1->src.count) < far2->src.count) ||
             ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
 
      } else {
         if (((far2->src.count + far1->dst.count) < far2->src.count) ||
             ((far2->dst.count + far1->src.count) < far2->dst.count) ||
             ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
             ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes)) {
            goto doneRaMergeArgus;
         }
      }

      for (i = 0; i < 32; i++) {
         if (status & 0x01) {
            switch (i) {
               case ARGUS_FAR_DSR_INDEX: {
                  u_int diffstartime = 0;
                  int diffusec = 0;

                  far1 = (struct ArgusFarStruct *) a1farhdr[i];
                  far2 = (struct ArgusFarStruct *) a2farhdr[i];

                  if (!(RaThisFlowRev)) {
                     if (((far2->src.count + far1->src.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->src.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->dst.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->dst.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(&store);

                     far2->src.count    += far1->src.count;
                     far2->src.bytes    += far1->src.bytes;
                     far2->dst.count    += far1->dst.count;
                     far2->dst.bytes    += far1->dst.bytes;
                     far2->src.appbytes += far1->src.appbytes;
                     far2->dst.appbytes += far1->dst.appbytes;

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                     }

                  } else {
                     if (((far2->src.count + far1->dst.count) < far2->src.count) ||
                         ((far2->src.bytes + far1->dst.bytes) < far2->src.bytes) ||
                         ((far2->dst.count + far1->src.count) < far2->dst.count) ||
                         ((far2->dst.bytes + far1->src.bytes) < far2->dst.bytes))
                        RaSendArgusRecord(&store);

                     if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) {
                        if (far1->attr_ip.sttl != far2->attr_ip.dttl) far2->attr_ip.dttl = 0xFF;
                        if (far1->attr_ip.dttl != far2->attr_ip.sttl) far2->attr_ip.sttl = 0xFF;
                        if (far1->attr_ip.stos != far2->attr_ip.dtos) far2->attr_ip.dtos = 0xFF;
                        if (far1->attr_ip.dtos != far2->attr_ip.stos) far2->attr_ip.stos = 0xFF;
                     }

                     far2->src.count    += far1->dst.count;
                     far2->src.bytes    += far1->dst.bytes;
                     far2->dst.count    += far1->src.count;
                     far2->dst.bytes    += far1->src.bytes;
                     far2->src.appbytes += far1->dst.appbytes;
                     far2->dst.appbytes += far1->src.appbytes;
                  }

                  diffstartime  = (far1->time.start.tv_sec  - data->agr.laststartime.tv_sec);
                  if ((diffusec = (far1->time.start.tv_usec - data->agr.laststartime.tv_usec)) < 0) {
                     diffstartime--;
                     diffusec += 1000000;
                  }

                  data->agr.laststartime = far1->time.start;

                  if (a1farhdr[ARGUS_AGR_DSR_INDEX] == NULL) {
                     if (diffstartime > 0) {
                        if (diffstartime > 2147) {
                           if (data->agr.status & ARGUS_AGR_USECIDLETIME) {
                              if (data->agr.idle.min != 0x7FFFFFFF)
                                 data->agr.idle.min /= 1000;
                              data->agr.idle.max /= 1000;
                              data->idle.sumtime /= 1000;
                              data->idle.sumsqrd /= 1000000.0;
                              data->agr.status &= ~ARGUS_AGR_USECIDLETIME;
                              data->agr.status |=  ARGUS_AGR_MSECIDLETIME;
                           }
                        }

                        if (data->agr.status & ARGUS_AGR_USECIDLETIME)
                           TimeMultiplier = 1000000;
                        else
                           TimeMultiplier = 1000;

                        diffstartime *= TimeMultiplier;
                        diffstartime += (diffusec * 1000) / (TimeMultiplier / 1000);

                        if ((data->agr.idle.min > diffstartime) && (diffstartime > 0))
                           data->agr.idle.min = diffstartime;

                        if ((data->agr.idle.max < diffstartime) && (diffstartime > 0))
                           data->agr.idle.max = diffstartime;

                        data->idle.sumtime += diffstartime;
                        data->idle.sumsqrd += pow(diffstartime, 2.0);
                        data->idle.n++;
                     }

                     duration = RaGetuSecDuration(a1);
                     if (duration > 0x6FFFFFFF) {
                        if (data->agr.status & ARGUS_AGR_USECIDLETIME) {
                            if (data->agr.idle.min != 0x7FFFFFFF)
                               data->agr.act.min /= 1000;
                            data->agr.act.max /= 1000;
                            data->act.sumtime /= 1000;
                            data->act.sumsqrd /= 1000000.0;
                            data->agr.status &= ~ARGUS_AGR_USECACTTIME;
                            data->agr.status |=  ARGUS_AGR_MSECACTTIME;
                        }
                     }

                     if (data->agr.status & ARGUS_AGR_USECACTTIME)
                        TimeMultiplier = 1;
                     else
                        TimeMultiplier = 1000;

                     duration /= TimeMultiplier;

                     if ((data->agr.act.min > duration) && (duration > 0))
                        data->agr.act.min = duration;

                     if ((data->agr.act.max < duration) && (duration > 0))
                        data->agr.act.max = duration;

                     data->act.sumtime += duration;
                     data->act.sumsqrd += pow(duration, 2.0);
                     data->act.n++;

                     data->agr.count++;
                  }

                  if (far1->ArgusTransRefNum != far2->ArgusTransRefNum)
                     far2->ArgusTransRefNum = 0;

                  if ((far2->time.start.tv_sec  > far1->time.start.tv_sec) ||
                     ((far2->time.start.tv_sec == far1->time.start.tv_sec) &&
                     (far2->time.start.tv_usec  > far1->time.start.tv_usec)))

                     far2->time.start = far1->time.start;

                  if ((far2->time.last.tv_sec  < far1->time.last.tv_sec) ||
                     ((far2->time.last.tv_sec == far1->time.last.tv_sec) &&
                     (far2->time.last.tv_usec  < far1->time.last.tv_usec)))

                     far2->time.last = far1->time.last;

                  if (RaThisArgusStore->qhdr.lasttime.tv_sec != far2->time.last.tv_sec) {
                     RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     data->agr.lasttime = far2->time.last;
                  } else
                     if (RaThisArgusStore->qhdr.lasttime.tv_usec < far2->time.last.tv_usec) {
                        RaThisArgusStore->qhdr.lasttime = far2->time.last;
                     }

                  break;
               }

#define TCP_FIN_STATE_TIME	360

               case ARGUS_TCP_DSR_INDEX: {
                  struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
                  struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

                  if (ArgusFlowModelFile == NULL) {
                     if ((tcp2->state & (ARGUS_FIN | ARGUS_FIN_ACK | ARGUS_NORMAL_CLOSE | ARGUS_RESET)) > 0) {
                        if ((a2->argus_far.time.last.tv_sec > (a1->argus_far.time.start.tv_sec + TCP_FIN_STATE_TIME)) ||
                            (a1->argus_far.time.last.tv_sec > (a2->argus_far.time.start.tv_sec + TCP_FIN_STATE_TIME))) {

                           goto doneRaMergeArgus;
                        }
                     } else {
/*
                        if ((a2->argus_far.ArgusTransRefNum != a1->argus_far.ArgusTransRefNum)) {
                           goto doneRaMergeArgus;
                        }
*/
                     }
                  }

                  tcp2->state |= tcp1->state;

                  if (!(tcp2->synAckuSecs))
                     tcp2->synAckuSecs = tcp1->synAckuSecs;
                  else
                     if (tcp2->synAckuSecs > tcp1->synAckuSecs)
                        if (tcp1->synAckuSecs)
                           tcp2->synAckuSecs = tcp1->synAckuSecs;

                  if (!(tcp2->ackDatauSecs))
                     tcp2->ackDatauSecs = tcp1->ackDatauSecs;
                  else
                     if (tcp2->ackDatauSecs > tcp1->ackDatauSecs)
                        if (tcp1->ackDatauSecs)
                           tcp2->ackDatauSecs = tcp1->ackDatauSecs;

                  tcp2->src.seqbase   = 0;
                  tcp2->dst.seqbase   = 0;

                  tcp2->src.ackbytes += tcp1->src.ackbytes;
                  tcp2->dst.ackbytes += tcp1->dst.ackbytes;
                  tcp2->src.bytes    += tcp1->src.bytes;
                  tcp2->dst.bytes    += tcp1->dst.bytes;
                  tcp2->src.rpkts    += tcp1->src.rpkts;
                  tcp2->dst.rpkts    += tcp1->dst.rpkts;        

                  if (tcp2->src.win > tcp1->src.win);
                     tcp2->src.win = tcp1->src.win;
                  if (tcp2->dst.win > tcp1->dst.win);
                     tcp2->dst.win = tcp1->dst.win;        

                  tcp2->src.flags    |= tcp1->src.flags;
                  tcp2->dst.flags    |= tcp1->dst.flags;        

                  break;
               }

               case ARGUS_RTP_DSR_INDEX: {
                  struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
                  struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];

                  if ((((rtp2->sdrop + rtp1->sdrop) < rtp2->sdrop) || ((rtp2->ddrop + rtp1->ddrop) < rtp2->ddrop)) ||
                      (((rtp2->ssdev + rtp1->ssdev) < rtp2->ssdev) || ((rtp2->dsdev + rtp1->dsdev) < rtp2->dsdev))) {
                     RaSendArgusRecord(&store);
                     rtp2->sdrop = rtp1->sdrop;
                     rtp2->ddrop = rtp1->ddrop;
                     rtp2->ssdev = rtp1->ssdev;
                     rtp2->dsdev = rtp1->dsdev;

                  } else {
                     rtp2->sdrop += rtp1->sdrop;
                     rtp2->ddrop += rtp1->ddrop;
                     rtp2->ssdev += rtp1->ssdev;
                     rtp2->dsdev += rtp1->dsdev;
                  }
                  break;
               }

               case ARGUS_ICMP_DSR_INDEX:
               case ARGUS_IGMP_DSR_INDEX:
               case ARGUS_ARP_DSR_INDEX:
               case ARGUS_FRG_DSR_INDEX:
                  break;

               case ARGUS_VLAN_DSR_INDEX: {
                  struct ArgusVlanStruct *vlan1 = (struct ArgusVlanStruct *) a1farhdr[i];
                  struct ArgusVlanStruct *vlan2 = (struct ArgusVlanStruct *) a2farhdr[i];

                  if ((vlan2->status & ARGUS_SRC_VLAN) && (vlan1->status & ARGUS_SRC_VLAN))
                     if (vlan2->sid != vlan1->sid)
                        vlan2->status |= ARGUS_SRC_CHANGED;

                  if ((vlan2->status & ARGUS_DST_VLAN) && (vlan1->status & ARGUS_DST_VLAN))
                     if (vlan2->did != vlan1->did)
                        vlan2->status |= ARGUS_DST_CHANGED;
                  
                  vlan2->status |= vlan1->status;
                  vlan2->sid = vlan1->sid;
                  vlan2->did = vlan1->did;
                  break;
               }

               case ARGUS_MPLS_DSR_INDEX: {
                  struct ArgusMplsStruct *mpls1 = (struct ArgusMplsStruct *) a1farhdr[i];
                  struct ArgusMplsStruct *mpls2 = (struct ArgusMplsStruct *) a2farhdr[i];

                  if ((mpls2->status & ARGUS_SRC_MPLS) && (mpls1->status & ARGUS_SRC_MPLS))
                     if (mpls2->slabel != mpls1->slabel)
                        mpls2->status |= ARGUS_SRC_CHANGED;

                  if ((mpls2->status & ARGUS_DST_MPLS) && (mpls1->status & ARGUS_DST_MPLS))
                     if (mpls2->dlabel != mpls1->dlabel)
                        mpls2->status |= ARGUS_DST_CHANGED;
                  
                  mpls2->status |= mpls1->status;
                  mpls2->slabel = mpls1->slabel;
                  mpls2->dlabel = mpls1->dlabel;
                  break;
               }

               case ARGUS_ESP_DSR_INDEX: {
                  struct ArgusESPStruct *esp1 = (struct ArgusESPStruct *) a1farhdr[i];
                  struct ArgusESPStruct *esp2 = (struct ArgusESPStruct *) a2farhdr[i];

                  if (esp1 && esp2) {
                     if (esp1->src.spi != esp2->src.spi)
                        esp2->src.spi = 0;

                     if (esp1->dst.spi != esp2->dst.spi)
                        esp2->dst.spi = 0;

                     esp2->src.lastseq = esp1->src.lastseq;
                     esp2->dst.lastseq = esp1->dst.lastseq;
                     esp2->src.lostseq = esp1->src.lostseq;
                     esp2->dst.lostseq = esp1->dst.lostseq;
                  }
                  break;
               }
    
               case ARGUS_TIME_DSR_INDEX: {
                  struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
                  struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];
                  long long sumtime, sumtime1, sumtime2;
                  double sumsqrd, sumsqrd1, sumsqrd2;

                  if (time1 && time2) {
                     if ((time2->src.act.n += time1->src.act.n) > 0) {
                        sumtime1 = (time1->src.act.mean * time1->src.act.n);
                        sumtime2 = (time2->src.act.mean * time2->src.act.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->src.act.mean  = sumtime/time2->src.act.n;
                        if (time1->src.act.n)
                           sumsqrd1 = (time1->src.act.n * pow(time1->src.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.act.n * pow(time2->src.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.act.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->src.act.stdev = sqrt (sumsqrd/time2->src.act.n - pow (sumtime/time2->src.act.n, 2.0));
                        if (time2->src.act.max < time1->src.act.max)
                           time2->src.act.max = time1->src.act.max; 
                        if (time2->src.act.min > time1->src.act.min)
                           time2->src.act.min = time1->src.act.min; 
                     }

                     if ((time2->dst.act.n += time1->dst.act.n) > 0) {
                        sumtime1 = (time1->dst.act.mean * time1->dst.act.n);
                        sumtime2 = (time2->dst.act.mean * time2->dst.act.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->dst.act.mean  = sumtime/time2->dst.act.n;
                        if (time1->dst.act.n)
                           sumsqrd1 = (time1->dst.act.n * pow(time1->dst.act.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.act.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.act.n * pow(time2->dst.act.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.act.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->dst.act.stdev = sqrt (sumsqrd/time2->dst.act.n - pow (sumtime/time2->dst.act.n, 2.0));
                        if (time2->dst.act.max < time1->dst.act.max)
                           time2->dst.act.max = time1->dst.act.max; 
                        if (time2->dst.act.min > time1->dst.act.min)
                           time2->dst.act.min = time1->dst.act.min; 
                     }

                     if ((time2->src.idle.n += time1->src.idle.n) > 0) {
                        sumtime1 = (time1->src.idle.mean * time1->src.idle.n);
                        sumtime2 = (time2->src.idle.mean * time2->src.idle.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->src.idle.mean  = sumtime/time2->src.idle.n;
                        if (time1->src.idle.n)
                           sumsqrd1 = (time1->src.idle.n * pow(time1->src.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->src.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->src.idle.n * pow(time2->src.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->src.idle.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->src.idle.stdev = sqrt (sumsqrd/time2->src.idle.n - pow (sumtime/time2->src.idle.n, 2.0));
                        if (time2->src.idle.max < time1->src.idle.max)
                           time2->src.idle.max = time1->src.idle.max; 
                        if (time2->src.idle.min > time1->src.idle.min)
                           time2->src.idle.min = time1->src.idle.min; 
                     }

                     if ((time2->dst.idle.n += time1->dst.idle.n) > 0) {
                        sumtime1 = (time1->dst.idle.mean * time1->dst.idle.n);
                        sumtime2 = (time2->dst.idle.mean * time2->dst.idle.n);
                        if ((sumtime = sumtime1 + sumtime2) < sumtime1)
                           goto doneRaMergeArgus;
  
                        time2->dst.idle.mean  = sumtime/time2->dst.idle.n;
                        if (time1->dst.idle.n)
                           sumsqrd1 = (time1->dst.idle.n * pow(time1->dst.idle.stdev, 2.0)) + pow(sumtime1, 2.0)/time1->dst.idle.n;
                        else
                           sumsqrd1 = 0;
  
                        sumsqrd2 = (time2->dst.idle.n * pow(time2->dst.idle.stdev, 2.0)) + pow(sumtime2, 2.0)/time2->dst.idle.n;
                        if ((sumsqrd = sumsqrd1 + sumsqrd2) < sumsqrd1)
                           goto doneRaMergeArgus;
  
                        time2->dst.idle.stdev = sqrt (sumsqrd/time2->dst.idle.n - pow (sumtime/time2->dst.idle.n, 2.0));
                        if (time2->dst.idle.max < time1->dst.idle.max)
                           time2->dst.idle.max = time1->dst.idle.max; 
                        if (time2->dst.idle.min > time1->dst.idle.min)
                           time2->dst.idle.min = time1->dst.idle.min; 
                     }
                  }

                  break;
               }

               case ARGUS_SRCUSRDATA_DSR_INDEX:
               case ARGUS_DSTUSRDATA_DSR_INDEX:
                  break;

               case ARGUS_AGR_DSR_INDEX: {
                  struct ArgusAGRStruct *agr1 = (struct ArgusAGRStruct *) a1farhdr[i];
                  unsigned int agr1sumtime, sumtime;
                  double agr1sumsqrd, sumsqrd;
                  int overlapping = 1;
                  long long newvalue = 0;

                  if (!((data->agr.status & ARGUS_AGR_USECACTTIME) &&
                            (agr1->status & ARGUS_AGR_USECACTTIME))) {

                     if (data->agr.status & ARGUS_AGR_USECACTTIME) {
                         data->agr.act.max   /= 1000;
                         data->agr.act.min   /= 1000;
                         data->agr.act.mean  /= 1000;
                         data->agr.act.stdev /= 1000;
                         data->act.sumtime /= 1000;
                         data->act.sumsqrd /= 1000000.0;
                         data->agr.status &= ~ARGUS_AGR_USECACTTIME;
                         data->agr.status |=  ARGUS_AGR_MSECACTTIME;
                     }

                     if (agr1->status & ARGUS_AGR_USECACTTIME) {
                         agr1->act.max   /= 1000;
                         agr1->act.min   /= 1000;
                         agr1->act.mean  /= 1000;
                         agr1->act.stdev /= 1000;
                         agr1->status &= ~ARGUS_AGR_USECACTTIME;
                         agr1->status |=  ARGUS_AGR_MSECACTTIME;
                     }
                  }

                  if (data->agr.act.min > agr1->act.min)
                     data->agr.act.min = agr1->act.min;
                  if (data->agr.act.max < agr1->act.max)
                     data->agr.act.max = agr1->act.max;

                  if (agr1->act.n) {
                     agr1sumtime = agr1->act.mean * agr1->act.n;
                     agr1sumsqrd = (agr1->act.n * pow(agr1->act.stdev, 2.0)) +
                                    pow(agr1sumtime, 2.0)/agr1->act.n;

                     data->act.sumtime += agr1sumtime;
                     data->act.sumsqrd += agr1sumsqrd;

                     sumtime = data->act.sumtime;
                     sumsqrd = data->act.sumsqrd;

                     data->agr.act.n += agr1->act.n;
                     data->act.n += agr1->act.n;

                     data->agr.act.mean = sumtime/data->act.n;
                     data->agr.act.stdev = sqrt (sumsqrd/data->act.n -
                                           pow (sumtime/data->act.n, 2.0));
                  }

                  if (!((data->agr.status & ARGUS_AGR_USECIDLETIME) &&
                            (agr1->status & ARGUS_AGR_USECIDLETIME))) {

                     if (data->agr.status & ARGUS_AGR_USECIDLETIME) {
                         data->agr.idle.max   /= 1000;
                         data->agr.idle.min   /= 1000;
                         data->agr.idle.mean  /= 1000;
                         data->agr.idle.stdev /= 1000;
                         data->idle.sumtime /= 1000;
                         data->idle.sumsqrd /= 1000000.0;
                         data->agr.status &= ~ARGUS_AGR_USECIDLETIME;
                         data->agr.status |=  ARGUS_AGR_MSECIDLETIME;
                     }

                     if (agr1->status & ARGUS_AGR_USECIDLETIME) {
                         agr1->idle.max   /= 1000;
                         agr1->idle.min   /= 1000;
                         agr1->idle.mean  /= 1000;
                         agr1->idle.stdev /= 1000;
                         agr1->status &= ~ARGUS_AGR_USECIDLETIME;
                         agr1->status |=  ARGUS_AGR_MSECIDLETIME;
                     }
                  }

                  if (data->agr.status & ARGUS_AGR_USECIDLETIME)
                     TimeMultiplier = 1;
                  else
                     TimeMultiplier = 1000;

                  overlapping = 1;

                  if ((a2->argus_far.time.start.tv_sec < a1->argus_far.time.start.tv_sec) ||
                     ((a2->argus_far.time.start.tv_sec == a1->argus_far.time.start.tv_sec) &&
                      (a2->argus_far.time.start.tv_usec < a1->argus_far.time.start.tv_usec))) {

                     if ((a2->argus_far.time.last.tv_sec < a1->argus_far.time.start.tv_sec) ||
                        ((a2->argus_far.time.last.tv_sec == a1->argus_far.time.start.tv_sec) &&
                         (a2->argus_far.time.last.tv_usec < a1->argus_far.time.start.tv_usec))) {

                        newvalue = (((agr1->laststartime.tv_sec  - data->agr.laststartime.tv_sec) * 1000000) +
                                     (agr1->laststartime.tv_usec - data->agr.laststartime.tv_usec))/TimeMultiplier;
                        overlapping = 0;
                     }

                  } else {
                     if ((a1->argus_far.time.last.tv_sec < a2->argus_far.time.start.tv_sec) ||
                        ((a1->argus_far.time.last.tv_sec == a2->argus_far.time.start.tv_sec) &&
                         (a1->argus_far.time.last.tv_usec < a2->argus_far.time.start.tv_usec))) {
                        newvalue = (((data->agr.laststartime.tv_sec  - agr1->laststartime.tv_sec) * 1000000) +
                                     (data->agr.laststartime.tv_usec - agr1->laststartime.tv_usec))/TimeMultiplier;
                        overlapping = 0;
                     }
                  }

                  if (!(overlapping)) {
                     sumtime = (agr1->idle.mean * agr1->idle.n) + newvalue;
                     sumsqrd = ((agr1->idle.n * pow(agr1->idle.stdev, 2.0)) + pow(sumtime, 2.0)/agr1->idle.n) +
                                 pow(newvalue, 2.0);

                     data->idle.sumtime += sumtime;
                     data->idle.sumsqrd += sumsqrd;
                     data->idle.n += agr1->idle.n + 1;

                     if (newvalue) {
                        if (newvalue > data->agr.idle.max) 
                           data->agr.idle.max = newvalue;
                        else
                        if (newvalue < data->agr.idle.min)
                           data->agr.idle.min = newvalue;
                     }

                     data->agr.count += agr1->count;
                  } else {
                     if (data->agr.idle.min >= agr1->idle.min) {
                        data->idle.sumtime += agr1->idle.min;
                        data->idle.sumsqrd += pow(agr1->idle.min, 2.0);
                        data->idle.n++;
                        data->agr.idle.min = agr1->idle.min;
                     }
                  }

                  if ((data->agr.laststartime.tv_sec > agr1->laststartime.tv_sec) ||
                      ((data->agr.laststartime.tv_sec == agr1->laststartime.tv_sec) &&
                       (data->agr.laststartime.tv_usec > agr1->laststartime.tv_usec))) {
                     data->agr.laststartime = agr1->laststartime;
                  }

                  if ((data->agr.lasttime.tv_sec < agr1->lasttime.tv_sec) ||
                      ((data->agr.lasttime.tv_sec == agr1->lasttime.tv_sec) &&
                       (data->agr.lasttime.tv_usec < agr1->lasttime.tv_usec))) {

                     RaThisArgusStore->qhdr.lasttime = agr1->lasttime;
                     data->agr.lasttime = agr1->lasttime;
                  }

                  break;
               }
            }
         }
         status >>= 1;
      }

      ArgusFree(data->argus);
      data->argus = RaCopyArgusRecord(tmp);

      data->status |= RA_MODIFIED;
      data->argus->ahdr.status |= ARGUS_MERGED;
      data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);

#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: merged\n");
#endif
      return;

   } else {
      data->argus = RaCopyArgusRecord(a1);
      data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);

#ifdef ARGUSDEBUG
      ArgusDebug (5, "RaMergeArgusRecord: swapped");
#endif
      return;
   }

doneRaMergeArgus:
   RaSendArgusRecord(&store);
   ArgusFree(data->argus);

   data->argus = RaCopyArgusRecord(a1);
   data->status |= RA_MODIFIED;
   data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaMergeArgusRecord: dumped and swapped");
#endif
}

void
RaUpdateArgusStore(struct ArgusRecord *argus, struct ArgusRecordStore *store)
{
   store->status |= RA_MODIFIED;
   store->qhdr.lasttime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaUpdateArgusStore: \n");
#endif
}

void
RaTimeoutArgusStore (struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus;
   int i;

   if (store != NULL) {
      if (store->status & RA_MODIFIED)
         RaSendArgusRecord(store);

      if (store->rahtblhdr)
         RaRemoveHashEntry (store->rahtblhdr);

      for (i = 0; i < ARGUS_TIMESERIES; i++) {
         struct ArgusRecordData *data = NULL;
         
         if ((data = store->data[i]) != NULL) {
            if ((argus = data->argus) != NULL) {
               RaRemoveArgusRecord(argus);
               ArgusFree (argus);
            }
            ArgusFree(data);
         }
      }

      ArgusFree (store);
      RaAllocArgusRecordStore--;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaTimeoutArgusStore: 0x%x\n", store);
#endif
}


int
RaCheckTimeout (struct ArgusRecordStore *store, struct ArgusRecord *argus)
{
   int retn = 0, lapseTime;
   struct ArgusRecordData *data = NULL;
   struct ArgusRecord *sargus = NULL;

   struct timeval *tvp = &ArgusGlobalTime;

   int lapseTimeSecs, lapseTimeuSecs;
   int starTimeSecs, starTimeuSecs;
   int lastTimeSecs, lastTimeuSecs;

   if (store->ArgusTimeout > 0) {
      lapseTime = store->qhdr.logtime.tv_sec + store->ArgusTimeout;

      if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
              (tvp->tv_usec > store->qhdr.logtime.tv_usec))) {
         RaSendArgusRecord(store);

      } else {
         if (argus != NULL) {
            if ((data = store->data[RaThisActiveIndex]) != NULL) {
               sargus = data->argus;

               starTimeSecs  = sargus->argus_far.time.start.tv_sec;
               starTimeuSecs = sargus->argus_far.time.start.tv_usec;
         
               if (argus->argus_far.time.last.tv_sec > sargus->argus_far.time.last.tv_sec) {
                  lastTimeSecs  = argus->argus_far.time.last.tv_sec;
                  lastTimeuSecs = argus->argus_far.time.last.tv_usec;
               } else {
                  lastTimeSecs  = sargus->argus_far.time.last.tv_sec;
                  if (argus->argus_far.time.last.tv_sec == sargus->argus_far.time.last.tv_sec) {
                     if (argus->argus_far.time.last.tv_usec > sargus->argus_far.time.last.tv_usec) {
                        lastTimeuSecs  = argus->argus_far.time.last.tv_usec;
                     } else
                        lastTimeuSecs  = sargus->argus_far.time.last.tv_usec;
                  } else
                     lastTimeuSecs  = sargus->argus_far.time.last.tv_usec;
               }

               lapseTimeSecs  = (lastTimeSecs  - starTimeSecs);
               lapseTimeuSecs = (lastTimeuSecs - starTimeuSecs);

               if (lapseTimeuSecs < 0) {
                  lapseTimeSecs--;
                  lapseTimeuSecs += 1000000;
               }

               if (lapseTimeSecs >= store->ArgusTimeout)
                  RaSendArgusRecord(store);
            }
         }
      }
   }

   if (store->ArgusIdleTimeout > 0) {
      if ((data = store->data[RaThisActiveIndex]) != NULL) {
         sargus = data->argus;
         lastTimeSecs  = sargus->argus_far.time.last.tv_sec;
         lastTimeuSecs = sargus->argus_far.time.last.tv_usec;

         lapseTimeSecs  = (tvp->tv_sec  - lastTimeSecs);
         lapseTimeuSecs = (tvp->tv_usec - lastTimeuSecs);

         if (lapseTimeSecs >= store->ArgusTimeout)
            retn++;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

struct RaCIDRAddr *
RaParseCIDRAddr (char *str)
{
   int error = 0, vlen, notIPAddr = 0, i;
   struct RaCIDRAddr *retn = NULL;
   char *ptr = NULL, chr = '\0';
   unsigned int **alist;
   int maskIndex;

   RaCIDRPtr->addr = 0;
   RaCIDRPtr->mask = 0xFFFFFFFF;

   if ((ptr = strchr(str, '*')) == NULL) {
      if ((ptr = strchr (str, ':')) != NULL) {
         chr = *ptr;
         *ptr++ = 0;
      } else {
         if ((ptr = strchr (str, '/')) != NULL) {
            chr = *ptr;
            *ptr++ = 0;
         }
      }

      for (i = 0; i < strlen(str); i++)
        if (!(isdigit((int)str[i]) || (str[i] == '.')))
           notIPAddr++;

      if (!(notIPAddr)) {
         vlen = __argus_atoin(str, &RaCIDRPtr->addr);

         /* Promote short net number */

         while (RaCIDRPtr->addr && (RaCIDRPtr->addr & 0xff000000) == 0) {
            RaCIDRPtr->addr <<= 8;
            RaCIDRPtr->mask <<= 8;
         }

      } else {
         if ((alist = argus_nametoaddr(str)) != NULL)  {
            RaCIDRPtr->addr = **alist;
         } else
            error++;
      }
   
      if (ptr && (((chr == ':') || (chr == '/')) && isdigit((int) *ptr))) {
         maskIndex = atoi(ptr);
   
         if ((maskIndex) > 0) {
            RaCIDRPtr->mask = 0xFFFFFFFF;
            RaCIDRPtr->mask <<= (32 - maskIndex);
         }

         if (RaCIDRPtr->mask)
            RaCIDRPtr->addr &= RaCIDRPtr->mask;
      }
   }

   if (!error)
      retn = RaCIDRPtr;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParseCIDRAddr: returning 0x%x \n", retn);
#endif
   
   return (retn);
}


struct RaPolicyStruct *
RaParsePolicyEntry (char *str)
{
   struct RaPolicyStruct *retn = NULL;
   struct RaCIDRAddr *addr = NULL;
   int i, type, policyelements, error = 0;
   char buf[MAXSTRLEN], *ptr = NULL, *tmp = buf;

   if (strncmp (str, RA_FLOWTAGSTR, strlen(RA_FLOWTAGSTR)) == 0) {
      type = RA_FLOWLIST;
      policyelements = RA_FLOWPOLICYFIELDNUM;

   } else {
      if ((strncmp (str, RA_MODELTAGSTR, strlen(RA_MODELTAGSTR))) == 0) {
         type = RA_MODELIST;
         policyelements = RA_MODELPOLICYFIELDNUM;

      } else 
         return (NULL);
   }

   if ((retn = (struct RaPolicyStruct *) ArgusCalloc (1, sizeof (*retn))) == NULL)
      return retn;

   bzero (buf, MAXSTRLEN);
   bcopy ((char *) str, buf, strlen(str));

   for (i = 0; ((i < policyelements) && !(error)); i++) {
      if ((ptr = strtok (tmp, " \t")) && (*ptr != '\n')) {
         switch (i) {
            case RA_LABELSTRING: 
               switch (type) {
                  case RA_FLOWLIST:
                     if (strcmp (ptr,  RA_FLOWTAGSTR))
                        error++;
                     else 
                        retn->RaEntryType = RA_FLOWLIST;
                     break;

                  case RA_MODELIST:
                     if (strcmp (ptr, RA_MODELTAGSTR))
                        error++;
                     else 
                        retn->RaEntryType = RA_MODELIST;
                     break;
               }
               break;
   
            case RA_POLICYID: 
               if (!(retn->RaPolicyId = atoi (ptr))) error++;
               break;

            case RA_POLICYSRCADDR:
               if ((addr = RaParseCIDRAddr (ptr)) != NULL)
                  retn->src = *addr;
               else
                  error++;
               break;
   
            case RA_POLICYDSTADDR:
               if ((addr = RaParseCIDRAddr (ptr)) != NULL)
                  retn->dst = *addr;
               else
                  error++;
               break;
   
            case RA_POLICYPROTO: 
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->proto = atoi(ptr);
                     } else {
                        if (*ptr == '*') {
                           retn->proto = 0xFF;
                        } else {
                           struct protoent *proto;
                           if ((proto = getprotobyname(ptr)) != NULL)
                              retn->proto = proto->p_proto;
                           else
                              error++;
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->proto = 0xFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->proto = 0x00;
                        else
                           error++;
                     break;
               }
               break;

            case RA_POLICYSRCPORT:
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->sport = (arg_uint16) atoi (ptr);
                     } else {
                        if (*ptr == '*')
                           retn->sport = 0xFFFF;
                        else {
                           int proto, port;
                           if (retn->proto == IPPROTO_ICMP) {
                              if (!(strcmp (ptr, "echo"))) retn->sport = ICMP_ECHO; else
                              if (!(strcmp (ptr, "unreach"))) retn->sport = ICMP_UNREACH; else
                              if (!(strcmp (ptr, "squench"))) retn->sport = ICMP_SOURCEQUENCH; else
                              if (!(strcmp (ptr, "redirect"))) retn->sport = ICMP_REDIRECT; else
                              if (!(strcmp (ptr, "timexed"))) retn->sport = ICMP_TIMXCEED; else
                              if (!(strcmp (ptr, "param"))) retn->sport = ICMP_PARAMPROB; else
                              if (!(strcmp (ptr, "timestmp"))) retn->sport = ICMP_TSTAMP; else
                              if (!(strcmp (ptr, "info"))) retn->sport = ICMP_IREQ; else
                              if (!(strcmp (ptr, "advert"))) retn->sport = ICMP_ROUTERADVERT; else
                              if (!(strcmp (ptr, "solicit"))) retn->sport = ICMP_ROUTERSOLICIT; else
                                 error++;
                           } else {
                              if (argus_nametoport(ptr, &port, &proto)) {
                                 retn->sport = port;
                              } else
                                 error++;
                           }
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->sport = 0xFFFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->sport = 0x0000;
                        else
                           error++;
                     break;
               }
               break;
   
            case RA_POLICYDSTPORT:
               switch (type) {
                  case RA_FLOWLIST:
                     if (isdigit((int)*ptr)) {
                        retn->dport = (arg_uint16) atoi (ptr);
                     } else {
                        if (*ptr == '*')
                           retn->dport = 0xFFFF;
                        else {
                           if (retn->proto == IPPROTO_ICMP) {
                           } else {
                              int proto, port;
                              if (argus_nametoport(ptr, &port, &proto)) {
                                 retn->dport = port;
                              } else
                                 error++;
                           }
                        }
                     }
                     break;

                  case RA_MODELIST:
                     if ((strstr(ptr, "yes")) != NULL)
                        retn->dport = 0xFFFF;
                     else
                        if ((strstr(ptr, "no")) != NULL)
                           retn->dport = 0x0000;
                        else
                           error++;
                     break;
               }
               break;
      
            case RA_POLICYMODELST:
               if (type == RA_FLOWLIST) {
                  if (isdigit((int)*ptr))
                     retn->RaModelId = (arg_uint32) atoi (ptr);
                  else
                     error++;
               }
               break;

            case RA_POLICYTIMEOUT:
               if (type == RA_FLOWLIST) {
                  if (isdigit((int)*ptr))
                     retn->ArgusTimeout = (arg_uint32) atoi (ptr);
                  else
                     error++;
               }
               break;
         }
         tmp = NULL;

      } else
         break;
   }

   if (error) {
      ArgusFree(retn);
      retn = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParsePolicyEntry: returning 0x%x \n", retn);
#endif

   return (retn);
}


int
RaCreatePolicyEntry (int index, char *str)
{
   int retn = 0;
   struct RaPolicyStruct *rap;

   if ((rap = RaParsePolicyEntry (str)) != NULL) {
      rap->str = str;

      switch (rap->RaEntryType) {
         case RA_FLOWLIST:
            RaFlowArray[RaThisFlowNum++] = rap;
            break;

         case RA_MODELIST:
            RaModelArray[RaThisModelNum++] = rap;
            break;
      }
   } else {
      fprintf (stderr, "%s: RaCreatePolicyEntry: format error\n", ArgusProgramName);
      fprintf (stderr, "line %d %s", index, str);
      exit (-1);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCreatePolicyEntry: returning %d\n", retn);
#endif

   return (retn);
}


int
RaReadFlowModelFile ()
{
   int retn = 0, i, x;
   char buf[MAXSTRLEN], *str = NULL;
   FILE *fd;
   int RaPolicyLines = 0, RaPolicyLineNumber = 0;

   if ((fd = fopen (ArgusFlowModelFile, "r")) != NULL) {
      while ((str = fgets (buf, MAXSTRLEN, fd)) != NULL) {
         RaPolicyLineNumber++;

         while (isspace((int)*str))
            str++;
         if (strlen(str)) {
            switch (buf[0]) {
               case '#':
               case '\n':
               case '!':
                  break;
   
               default:
                  if ((str = strdup(str)) != NULL) {
                     if (RaCreatePolicyEntry(RaPolicyLineNumber, str) < 0) {
                        fprintf (stderr, "%s: RaReadFlowModeleFile: parse error line %d\n", ArgusProgramName,
                                   RaPolicyLineNumber);
                        fprintf (stderr, "%s: RaReadFlowModeleFile: %s\n", ArgusProgramName,
                                   RaPolicyError[RaParseError]);
                        exit (-1);
                     } else
                        RaPolicyLines++;
                    
                  } else {
                     fprintf (stderr, "%s: RaReadFlowModeleFile: error %s\n", ArgusProgramName,
                                   strerror(errno));
                     exit (-1);
                  }
            }
         }
      }

      for (i = 0; i < RaThisFlowNum; i++) {
         if (RaFlowArray[i]) {
            int found = 0;
            for (x = 0; x < RaThisModelNum; x++) {
               if (RaModelArray[x])
                  if (RaFlowArray[i]->RaModelId == RaModelArray[x]->RaPolicyId) {
                     RaFlowArray[i]->RaModelId = x;
                     found++;
                     break;
                  }
            }
            if (!found) {
               fprintf (stderr, "%s: RaReadFlowModeleFile: Model Index %d not found\n",
                                ArgusProgramName, RaFlowArray[i]->RaModelId);
               exit (-1);
            }
         }
      }

      retn = RaThisFlowNum;
      fclose(fd);

   } else
      fprintf (stderr, "%s: RaReadFlowModeleFile: open '%s': %s\n", ArgusProgramName,
                                ArgusFlowModelFile,  strerror(errno));

#ifdef ARGUSDEBUG
   ArgusDebug (3, "RaReadFlowModelFile: returning RaFlowArray[%d]  RaModelArray[%d]\n", RaThisFlowNum, RaThisModelNum);
#endif

   return (retn);
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  256
#define RA_MAXQSIZE  25000
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);

         break;

      default:
         while (queue->count > RA_MAXQSIZE) {
            obj = (struct ArgusRecordStore *) RaRemoveFromQueue(RaModelerQueue, RaModelerQueue->start->prv);
            RaTimeoutArgusStore(obj);
         }

         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


struct RaHashTableHeader *
RaFindHashObject ()
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *head = NULL, *target;

   if ((target = RaHashTable.array[RaThisHash % RaHashTable.size]) != NULL) {
      head = target;
      do {
         if (!(bcmp ((char *) RaThisFlow, (char *) &target->flow, sizeof(*RaThisFlow)))) {
               retn = target;
               break;
            } else
               target = target->nxt;
      } while (target != head);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaFindHashObject: returning 0x%x\n", retn);
#endif

   return (retn);
}



struct RaHashTableHeader *
RaAddHashEntry (struct ArgusRecordStore *store)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *start = NULL;

   if ((retn = (struct RaHashTableHeader *) ArgusCalloc (1, sizeof (struct RaHashTableHeader))) != NULL) {
      RaAllocHashTableHeaders++;
      bcopy((char *) RaThisFlow, (char *)&retn->flow, sizeof (*RaThisFlow));
      retn->storeobj = store;
      retn->hash = RaThisHash;
      
      if ((start = RaHashTable.array[RaThisHash % RaHashTable.size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      RaHashTable.array[RaThisHash % RaHashTable.size] = retn;
      RaHashTable.count++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaAddHashEntry (0x%x) returning 0x%x\n", store, retn);
#endif

   return (retn);
}

void
RaRemoveHashEntry (struct RaHashTableHeader *rahtblhdr)
{
   unsigned short hash = rahtblhdr->hash;

   rahtblhdr->prv->nxt = rahtblhdr->nxt;
   rahtblhdr->nxt->prv = rahtblhdr->prv;

   if (rahtblhdr == RaHashTable.array[hash % RaHashTable.size]) {
      if (rahtblhdr == rahtblhdr->nxt)
         RaHashTable.array[hash % RaHashTable.size] = NULL;
      else
         RaHashTable.array[hash % RaHashTable.size] = rahtblhdr->nxt;

      RaHashTable.count--;
   }

   ArgusFree (rahtblhdr);
   RaAllocHashTableHeaders--;
 
#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaRemoveHashObject (0x%x) returning\n", rahtblhdr);
#endif
}


struct RaQueueStruct *
RaNewQueue ()
{
   struct RaQueueStruct *retn =  NULL;

   if ((retn = (struct RaQueueStruct *) ArgusCalloc (1, sizeof (struct RaQueueStruct))) != NULL) {
      retn->count = 0;
      retn->start = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaNewQueue () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
RaDeleteQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *obj = NULL;

   if (queue != NULL) {
      while ((obj = RaPopQueue(queue)))
         ArgusFree(obj);

      ArgusFree(queue);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaDeleteQueue (0x%x) returning\n", queue);
#endif
}


int
RaGetQueueCount(struct RaQueueStruct *queue)
{

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaGetQueueCount (0x%x) returning %d\n", queue, queue->count);
#endif

   return (queue->count);
}

void
RaPushQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int retn = 0;

   if ((retn = RaAddToQueue (queue, obj)) > 0)
      queue->start = queue->start->prv;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaPushQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}


int
RaAddToQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int retn = 0;

   if (queue && obj) {
      if (obj->queue == NULL) {
         if (queue->start != NULL) {
            obj->prv = queue->start->prv;
            queue->start->prv = obj;
            obj->nxt = queue->start;
            obj->prv->nxt = obj;
         } else {
            queue->start = obj;
            obj->nxt = obj;
            obj->prv = obj;
         }
         queue->count++;
         obj->queue = queue;
         retn = 1;

      } else
         ArgusLog (LOG_ERR, "RaAddToQueue (0x%x, 0x%x) obj in queue 0x%x\n", queue, obj, obj->queue);
   } else
      ArgusLog (LOG_ERR, "RaAddToQueue (0x%x, 0x%x) parameter error\n", queue, obj);

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaAddToQueue (0x%x, 0x%x) returning %d\n", queue, obj, retn);
#endif

   return (retn);
}


struct ArgusQueueHeader *
RaPopQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *retn = NULL;
   struct ArgusQueueHeader *obj = NULL;

   if (queue && queue->count) {
      if ((obj = (struct ArgusQueueHeader *) queue->start) != NULL)
         retn = RaRemoveFromQueue(queue, obj);
      else
         ArgusLog (LOG_ERR, "RaPopQueue(0x%x) internal queue error count %d start NULL\n", queue, queue->count);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "RaPopQueue (0x%x) returning 0x%x\n", queue, retn);
#endif
   
   return(retn);
}


struct ArgusQueueHeader *
RaRemoveFromQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct ArgusQueueHeader *retn = NULL;

   if ((queue != NULL) && (obj != NULL)) {
      if (queue->count && (obj->queue == queue)) {
         queue->count--;

         if (queue->count) {
            if (queue->start == obj)
               queue->start = obj->nxt;

            obj->prv->nxt = obj->nxt;
            obj->nxt->prv = obj->prv;

         } else
            queue->start = NULL;

         obj->prv = NULL;
         obj->nxt = NULL;
         obj->queue = NULL;
         retn = obj;

      } else
         ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) obj not in queue\n", queue, obj);
   } else
      ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) parameter error\n", queue, obj);

#ifdef ARGUSDEBUG
   ArgusDebug (9, "RaRemoveFromQueue (0x%x, 0x%x) returning 0x%x\n", queue, obj, obj);
#endif

   return (retn);
}


struct ArgusListStruct *
ArgusNewList ()
{
   struct ArgusListStruct *retn = NULL;
 
   if ((retn = (struct ArgusListStruct *) ArgusCalloc (1, sizeof (struct ArgusListStruct))) != NULL) {
      retn->start = NULL;
      retn->count = 0;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusNewList () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
ArgusDeleteList (struct ArgusListStruct *list)
{
   if (list) {
      while (list->start)
         ArgusPopFrontList(list);

      ArgusFree (list);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusDeleteList (0x%x) returning\n", list);
#endif
}

int
ArgusListEmpty (struct ArgusListStruct *list)
{
   return (list->start == NULL);
}

int
ArgusGetListCount(struct ArgusListStruct *list)
{
   return (list->count);
}

void
ArgusPushFrontList(struct ArgusListStruct *list, void *obj)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if (list && obj) {
      if ((lobj = (struct ArgusListObjectStruct *) ArgusCalloc (1, sizeof(*lobj))) != NULL) {
         lobj->obj = obj;
   
         if (list->start) {
            lobj->nxt = list->start;
            lobj->prv = list->start->prv;
            lobj->nxt->prv = lobj;
            lobj->prv->nxt = lobj;
         } else {
            lobj->prv = lobj;
            lobj->nxt = lobj;
         }
   
         list->start = lobj;
         list->count++;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPushFrontList (0x%x, 0x%x) returning 0x%x\n", list, obj);
#endif
}

void
ArgusPushBackList(struct ArgusListStruct *list, void *obj)
{
   ArgusPushFrontList(list, obj);
   list->start = list->start->nxt;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPushBackList (0x%x, 0x%x) returning 0x%x\n", list, obj);
#endif
}

void *
ArgusFrontList(struct ArgusListStruct *list)
{
   void *retn = NULL;

   if (list->start)
      retn = list->start->obj;

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusFrontList (0x%x) returning 0x%x\n", list, retn);
#endif

   return (retn);
}


void
ArgusPopBackList(struct ArgusListStruct *list)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if ((lobj = list->start)) {
      list->start = list->start->prv;
      ArgusPopFrontList(list);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPopBackList (0x%x) returning\n", list);
#endif
}

void
ArgusPopFrontList(struct ArgusListStruct *list)
{
   struct ArgusListObjectStruct *lobj = NULL;

   if ((lobj = list->start)) {
      if (--list->count > 0) {
         if (lobj->prv)
            lobj->prv->nxt = lobj->nxt;
 
         if (lobj->nxt)
            lobj->nxt->prv = lobj->prv;
 
        list->start = lobj->nxt;
 
      } else
         list->start = NULL;
 
      ArgusFree(lobj);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusPopFrontList (0x%x) returning\n", list);
#endif
}
