/* Cflow.xs - perl XS module for processing raw flow files
 * Copyright (C) 2001-2002  Dave Plonka
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: Cflow.xs,v 1.22 2002/01/14 17:17:30 dplonka Exp $
 * Dave Plonka <plonka@doit.wisc.edu>
 */

#ifdef __cplusplus
extern "C" {
#endif
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
#ifdef __cplusplus
}
#endif

#if 0
#define ARGUS /* This is now the responsibility of Makefile.PL */
#endif

#if 0
#ifndef ARGUS
#define OSU /* This is now the responsibility of Makefile.PL */
#endif
#endif

#include <errno.h> /* errno, ENOENT */
#include <fcntl.h> /* fcntl, open, O_RDONLY */
#include <limits.h> /* PATH_MAX */
#include <stdio.h> /* FILE, stderr, fdopen, fread, fclose, sprintf */
#include <string.h> /* strncpy, strcmp, strerror */
#include <sys/types.h> /* size_t */
#include <unistd.h> /* STDIN_FILENO, close */

#include "cflow5.h"

#ifdef OSU /* { */

/* { kludge since flow-tools config.h macros collide with perl's/xsubpp's */
#  define SAVE_PACKAGE PACKAGE
#  undef PACKAGE
#  define SAVE_VERSION VERSION
#  undef VERSION
/* } */

#  include "config.h"
#  include "ftlib.h"

/* { kludge revisited */
#  undef PACKAGE
#  undef VERSION
#  define PACKAGE SAVE_PACKAGE
#  define VERSION SAVE_VERSION
/* } */

#elif defined ARGUS /* }{ */

#  include <arpa/inet.h> /* for inet_addr */

/* { these were needed w/perl 5.004_04 under Solaris 2.6: */
#  include <sys/socket.h> /* for struct sockaddr */
#  include <net/if.h> /* for struct ifnet */
/* } */

#  include <argus_client.h>
#  include <argus_util.h> /* for ARGUS_INPUT */

#  include <argus_parse.h>
extern struct ARGUS_INPUT *ArgusRemoteFDs[]; /* allocated in "argus_parse.c" */

#if 0 /* don't include <argus_filter.h> since its yy* stuff collides w/perl's */
#  include <argus_filter.h> /* for ArgusThisFarStatus, etc. */
#else
extern u_int ArgusThisFarStatus;
extern struct ArgusFarHeaderStruct *ArgusThisFarHdrs[];
#endif

#endif /* } */


/*
 * This is the value for index that we *expect* using Cisco flow-export
 * version 5 and cflowd-2.x.  This was determined by simple experimentation
 * with just "COLLECT: { flows }" in "cflowd.conf".
 */
uint32_t expected_entry_mask =
   k_routerMask |
   k_srcIpAddrMask |
   k_dstIpAddrMask |
   k_inputIfIndexMask |
   k_outputIfIndexMask |
   k_srcPortMask |
   k_dstPortMask |
   k_pktsMask |
   k_bytesMask |
   k_ipNextHopMask |
   k_startTimeMask |
   k_endTimeMask |
   k_protocolMask |
   k_tosMask |
   k_srcAsMask |
   k_dstAsMask |
   k_srcMaskLenMask |
   k_dstMaskLenMask |
   k_tcpFlagsMask |
   k_engineTypeMask |
   k_engineIdMask;

static int
not_here(s)
char *s;
{
    croak("%s not implemented on this architecture", s);
    return -1;
}

static double
constant(name, arg)
char *name;
int arg;
{
    errno = 0;
    switch (*name) {
    }
    errno = EINVAL;
    return 0;

not_there:
    errno = ENOENT;
    return 0;
}

static cflowrec flow;
static size_t n; /* return value (number of flows matched) */
static size_t total; /* return value (total number of flows) */
#                  define INDEX       0
#                  define UNIX_SECS   1
#                  define EXPORTER    2
#                  define SRCADDR     3
#                  define DSTADDR     4
#                  define INPUT_IF    5
#                  define OUTPUT_IF   6
#                  define SRCPORT     7
#                  define DSTPORT     8
#                  define PKTS        9
#                  define BYTES       10
#                  define NEXTHOP     11
#                  define STARTIME    12
#                  define ENDTIME     13
#                  define PROTOCOL    14
#                  define TOS         15
#                  define SRC_AS      16
#                  define DST_AS      17
#                  define SRC_MASK    18
#                  define DST_MASK    19
#                  define TCP_FLAGS   20
#                  define ENGINE_TYPE 21
#                  define ENGINE_ID   22
#                  define RAW         23
#                  define ICMPTYPE    24
#                  define ICMPCODE    25
#                  define START_MSECS 26
#                  define END_MSECS   27
#                  define DURATION_SECS 28
#                  define DURATION_MSECS 29
#                  define RESERVED    30 /* placeholder for last var */
static SV *vars[RESERVED],
	  *wanted;

#ifdef ARGUS /* [ */

static unsigned long argusid,
		     argusid_nl,
                     nexthop,
		     nexthop_nl;
static unsigned short ifIndex = 42, /* isn't this the answer? ;^) */
                      ifIndex_ns;

/*******************************************************************************
 * { These functions are all Argus call-backs:
 *   (See "argus-2.0.4/clients/ratemplate.c" for examples.)
 */

void
usage() {
}

/* ArgusClientInit
 *
 * Unlike typical ra* clients, this one has its own main function.  One
 * side-effect is that ArgusClientInit does not get called automagically.
 * So, instead we call it "manually" now from the find function, even
 * though it has external linkage (i.e. it is not static).
 */
void
ArgusClientInit() {
   /* Here, we just set nexthop to some non-zero value.
    * This is because, using NetFlow as a precedent, a zero nexthop usually
    * means that the packet wasn't forwarded.
    */
   nexthop_nl = inet_addr("127.0.0.1");
   nexthop = ntohl(nexthop_nl);

   ifIndex_ns = htons(ifIndex);
}

void
RaParseComplete(int sig) {  
}

void
ArgusClientTimeout() {
}

/* RaProcessRecord
 *
 * This function sets $Cflow::raw (the packed raw flow record in cflowd-
 * format) to the contents of the flow data object.
 * 
 * It's primary purpose is invoke the user-defined "wanted" subroutine,
 * which provides the perl script with access the flow attributes.
 *
 * If an Argus flow contains bidirectional information, the appropriate
 * "process_" function will call this function twice, once for each
 * unidirectional flow.  It is the responsibility of the "process_"
 * function to swap src/dst values around as appropriate for the reverse
 * unidirectional flow.
 */
void
RaProcessRecord(struct ArgusRecord *argus) {
   dSP;
   size_t results;

   /* set up $Cflow::raw now that, presumably, flow has been populated */
   sv_setpvn(vars[RAW], (char *)&flow, (sizeof flow) - 1);

   total++;

   /* { call-back to the user-defined perl "wanted" subroutine: */

   ENTER;
   SAVETMPS;

   PUSHMARK(sp);
   PUTBACK;
   results = perl_call_sv(wanted, G_SCALAR|G_NOARGS);
   SPAGAIN;
   if (1 == results && POPi != 0) {
      n++;
   }
   PUTBACK;

   FREETMPS;
   LEAVE;

   /* } */
}

/* process_far
 *
 * This local function performs general preparation of the flow data
 * object and perl flows variables for the forward unidirectional flow
 * in the current FAR.  
 *
 * The protocol-specific "process_" functions should perform
 * protocol-specific set up of the flow data object and perl flow
 * variables before calling RaProcessRecord.
 *
 * This function also converts the byte counts in the argus record
 * from ether-bytes to IP-bytes, so it should be called by each of
 * the "process_" call-back functions before they use either
 * argus->argus_far.src.bytes or argus->argus_far.dst.bytes.
 */
static void
process_far(struct ArgusRecord *argus, cflowrec *flowp) {
   struct timeval diff;

   /* adjust the (ethernet) byte counts so that they are IP bytes
    * (14 == 2 * sizeof mac_address + sizeof ethertype)
    */
   argus->argus_far.src.bytes -= 14 * argus->argus_far.src.count;
   argus->argus_far.dst.bytes -= 14 * argus->argus_far.dst.count;

   /* { required for $Cflow::raw: */
   flow.index = htonl(expected_entry_mask);
   flow.rtraddr = argusid_nl;
   flow.srcaddr = htonl(argus->argus_far.flow.ip_flow.ip_src);
   flow.dstaddr = htonl(argus->argus_far.flow.ip_flow.ip_dst);
   flow.nexthop = nexthop_nl;
   flow.input_if = ifIndex_ns;  /* dummy value - just so its non-zero */
   flow.output_if = ifIndex_ns; /* dummy value - just so its non-zero */
   flow.pkts = htonl(argus->argus_far.src.count);
   flow.bytes = htonl(argus->argus_far.src.bytes);
   flow.starttime = htonl(argus->argus_far.time.start.tv_sec);
   flow.endtime = htonl(argus->argus_far.time.last.tv_sec);
   flow.srcport = 0;
   flow.dstport = 0;
   flow.protocol = argus->argus_far.flow.ip_flow.ip_p;
   flow.tos = argus->argus_far.attr_ip.stos;
   /* { argus doesn't have these */
   flow.src_as = 0;
   flow.dst_as = 0;
   /* FIXME? do class A,B,C rules?: */
   flow.src_mask = 0;
   flow.dst_mask = 0;
   /* } */
   flow.tcp_flags = 0;
   /* } */

   sv_setuv(vars[INDEX], expected_entry_mask);
   sv_setuv(vars[EXPORTER], argusid);
   sv_setuv(vars[SRCADDR], argus->argus_far.flow.ip_flow.ip_src);
   sv_setuv(vars[DSTADDR], argus->argus_far.flow.ip_flow.ip_dst);
   sv_setuv(vars[INPUT_IF], ifIndex);
   sv_setuv(vars[OUTPUT_IF], ifIndex);
   sv_setuv(vars[PKTS], argus->argus_far.src.count);
   sv_setuv(vars[BYTES], argus->argus_far.src.bytes);
   sv_setuv(vars[NEXTHOP], nexthop);
   sv_setuv(vars[STARTIME], argus->argus_far.time.start.tv_sec);
   sv_setuv(vars[START_MSECS], argus->argus_far.time.start.tv_usec / 1000);
   sv_setuv(vars[ENDTIME], argus->argus_far.time.last.tv_sec);
   sv_setuv(vars[END_MSECS], argus->argus_far.time.last.tv_usec / 1000);
   sv_setuv(vars[PROTOCOL], argus->argus_far.flow.ip_flow.ip_p);
   sv_setuv(vars[TOS], argus->argus_far.attr_ip.stos);
   /* argus has no f5data */

   sv_setuv(vars[UNIX_SECS], argus->argus_far.time.last.tv_sec);
   diff.tv_sec  = argus->argus_far.time.last.tv_sec -
                  argus->argus_far.time.start.tv_sec;

   diff.tv_usec = argus->argus_far.time.last.tv_usec -
                  argus->argus_far.time.start.tv_usec;
   if (diff.tv_usec < 0) {
      diff.tv_sec--;
      diff.tv_usec += 1000000;
   }
   sv_setuv(vars[DURATION_SECS], diff.tv_sec);
   sv_setuv(vars[DURATION_MSECS], diff.tv_usec / 1000); /* usec -> msec */
}

void
process_man(struct ArgusRecord *argus) {
   argusid = argus->argus_mar.argusid; /* remember the {argus,probe}id */
   argusid_nl = htonl(argusid);
}

void
process_tcp(struct ArgusRecord *argus) {
   unsigned char src_flags,
		 dst_flags;
   /* see "clients/raxml.c" for example of how to find the TCP flags */
   struct ArgusTCPObject *tcp =
      (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

   process_far(argus, &flow);

   if ((void *)0 != tcp) {
      src_flags = tcp->src.flags;
      dst_flags = tcp->dst.flags;
   } else {
      src_flags = 0;
      dst_flags = 0;
   }

   /* { tcp-specific: */
   flow.srcport = htons(argus->argus_far.flow.ip_flow.sport);
   sv_setuv(vars[SRCPORT], argus->argus_far.flow.ip_flow.sport);
   flow.dstport = htons(argus->argus_far.flow.ip_flow.dport);
   sv_setuv(vars[DSTPORT], argus->argus_far.flow.ip_flow.dport);
   flow.tcp_flags = src_flags;
   sv_setuv(vars[TCP_FLAGS], src_flags);
   /* } */
   RaProcessRecord(argus);

   if (argus->argus_far.dst.count) { /* report the reverse unidir flow */
      /* { swap 'em */
      flow.srcaddr = htonl(argus->argus_far.flow.ip_flow.ip_dst);
      sv_setuv(vars[SRCADDR], argus->argus_far.flow.ip_flow.ip_dst);
      flow.dstaddr = htonl(argus->argus_far.flow.ip_flow.ip_src);
      sv_setuv(vars[DSTADDR], argus->argus_far.flow.ip_flow.ip_src);
      flow.tos = argus->argus_far.attr_ip.dtos;
      sv_setuv(vars[TOS], argus->argus_far.attr_ip.dtos);
      flow.pkts = htonl(argus->argus_far.dst.count);
      sv_setuv(vars[PKTS], argus->argus_far.dst.count);
      flow.bytes = htonl(argus->argus_far.dst.bytes);
      sv_setuv(vars[BYTES], argus->argus_far.dst.bytes);
      /* { tcp-specific: */
      flow.srcport = htons(argus->argus_far.flow.ip_flow.dport);
      sv_setuv(vars[SRCPORT], argus->argus_far.flow.ip_flow.dport);
      flow.dstport = htons(argus->argus_far.flow.ip_flow.sport);
      sv_setuv(vars[DSTPORT], argus->argus_far.flow.ip_flow.sport);
      flow.tcp_flags = dst_flags;
      sv_setuv(vars[TCP_FLAGS], dst_flags);
      /* } */
      /* } */
      RaProcessRecord(argus);
   }
}

void
process_icmp(struct ArgusRecord *argus) {
   unsigned char type, code;

   process_far(argus, &flow);

   /* see "clients/raxml.c" for example */
   /* { icmp-specific: */
   if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS) {
      struct ArgusICMPObject *icmp =
	 (struct ArgusICMPObject *)ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];
      type = icmp->icmp_type;
      code = argus->argus_far.flow.icmp_flow.code;
   } else {
      type = argus->argus_far.flow.icmp_flow.type;
      code = argus->argus_far.flow.icmp_flow.code;
   }
   flow.srcport = 0;
   sv_setuv(vars[SRCPORT], 0);
   /* embed the ICMP type and code in dstport like NetFlow does: */
   flow.dstport = (type << 8) | code;
   sv_setuv(vars[DSTPORT], htons(flow.dstport));
   sv_setuv(vars[ICMPTYPE], type);
   sv_setuv(vars[ICMPCODE], code);
   /* } */

   RaProcessRecord(argus);

   if (argus->argus_far.dst.count) { /* report the reverse unidir flow */
      /* { swap 'em */
      flow.srcaddr = htonl(argus->argus_far.flow.ip_flow.ip_dst);
      sv_setuv(vars[SRCADDR], argus->argus_far.flow.ip_flow.ip_dst);
      flow.dstaddr = htonl(argus->argus_far.flow.ip_flow.ip_src);
      sv_setuv(vars[DSTADDR], argus->argus_far.flow.ip_flow.ip_src);
      flow.tos = argus->argus_far.attr_ip.dtos;
      sv_setuv(vars[TOS], argus->argus_far.attr_ip.dtos);
      flow.pkts = htonl(argus->argus_far.dst.count);
      sv_setuv(vars[PKTS], argus->argus_far.dst.count);
      flow.bytes = htonl(argus->argus_far.dst.bytes);
      sv_setuv(vars[BYTES], argus->argus_far.dst.bytes);
      /* { icmp-specific */
      /* FIXME? Where do I find the ICMP type & code for the reverse
       *        unidirectional flow? (or is the reverse flow always ECHOREPLY?)
       * For now we'll just set it to zero... which happens to be ECHOREPLY.
       */
      sv_setuv(vars[ICMPTYPE], 0);
      sv_setuv(vars[ICMPCODE], 0);
      flow.dstport = 0;
      /* FIXME? OR-in the ICMP type/code like NetFlow does */
      sv_setuv(vars[DSTPORT], htons(flow.dstport));
      /* } */
      /* } */
      RaProcessRecord(argus);
   }
}

void
process_udp(struct ArgusRecord *argus) {
   process_far(argus, &flow);

   /* { udp-specific: */
   flow.srcport = htons(argus->argus_far.flow.ip_flow.sport);
   sv_setuv(vars[SRCPORT], argus->argus_far.flow.ip_flow.sport);
   flow.dstport = htons(argus->argus_far.flow.ip_flow.dport);
   sv_setuv(vars[DSTPORT], argus->argus_far.flow.ip_flow.dport);
   flow.tcp_flags = 0;
   sv_setuv(vars[TCP_FLAGS], 0);
   /* } */

   RaProcessRecord(argus);

   if (argus->argus_far.dst.count) {
      /* { swap 'em */
      flow.srcaddr = htonl(argus->argus_far.flow.ip_flow.ip_dst);
      sv_setuv(vars[SRCADDR], argus->argus_far.flow.ip_flow.ip_dst);
      flow.dstaddr = htonl(argus->argus_far.flow.ip_flow.ip_src);
      sv_setuv(vars[DSTADDR], argus->argus_far.flow.ip_flow.ip_src);
      flow.tos = argus->argus_far.attr_ip.dtos;
      sv_setuv(vars[TOS], argus->argus_far.attr_ip.dtos);
      flow.pkts = htonl(argus->argus_far.dst.count);
      sv_setuv(vars[PKTS], argus->argus_far.dst.count);
      flow.bytes = htonl(argus->argus_far.dst.bytes);
      sv_setuv(vars[BYTES], argus->argus_far.dst.bytes);
      /* { udp-specific: */
      flow.srcport = htons(argus->argus_far.flow.ip_flow.dport);
      sv_setuv(vars[SRCPORT], argus->argus_far.flow.ip_flow.dport);
      flow.dstport = htons(argus->argus_far.flow.ip_flow.sport);
      sv_setuv(vars[DSTPORT], argus->argus_far.flow.ip_flow.sport);
      /* } */
      /* } */

      RaProcessRecord(argus);
   }
}

void
process_ip(struct ArgusRecord *argus) {
   process_far(argus, &flow);

   /* { ip-specific */
   flow.srcport = 0;
   sv_setuv(vars[SRCPORT], 0);
   flow.dstport = 0;
   sv_setuv(vars[DSTPORT], 0);
   flow.tcp_flags = 0;
   sv_setuv(vars[TCP_FLAGS], 0);
   /* } */

   RaProcessRecord(argus);

   if (argus->argus_far.dst.count) {
      /* { swap 'em */
      flow.srcaddr = htonl(argus->argus_far.flow.ip_flow.ip_dst);
      sv_setuv(vars[SRCADDR], argus->argus_far.flow.ip_flow.ip_dst);
      flow.dstaddr = htonl(argus->argus_far.flow.ip_flow.ip_src);
      sv_setuv(vars[DSTADDR], argus->argus_far.flow.ip_flow.ip_src);
      flow.tos = argus->argus_far.attr_ip.dtos;
      sv_setuv(vars[TOS], argus->argus_far.attr_ip.dtos);
      flow.pkts = htonl(argus->argus_far.dst.count);
      sv_setuv(vars[PKTS], argus->argus_far.dst.count);
      flow.bytes = htonl(argus->argus_far.dst.bytes);
      sv_setuv(vars[BYTES], argus->argus_far.dst.bytes);
      /* } */

      RaProcessRecord(argus);
   }
}

void
process_arp(struct ArgusRecord *argus) { 
}

void
process_non_ip(struct ArgusRecord *argus) {
} 

void
parse_arg(int argc, char**argv) {
}

/* }
 ******************************************************************************/

#endif /* ] */

MODULE = Cflow		PACKAGE = Cflow		

PROTOTYPES: ENABLE

char *
find(wanted_arg, ...)
	SV* wanted_arg
	PREINIT:
		size_t arg = 1; /* current argument */
		char retval[sizeof "4294967295/4294967295"]; /* UINT_MAX */
		SV *perfile = (SV *)0;
		unsigned verbose;
#            ifdef OSU /* { */
                struct ftio fs;
                struct fts3rec_gen *fdata;
                /* f5data is just a casted (sic) alias for fdata: */
#               define f5data ((struct fts3rec_v5 *)fdata)
		unsigned int has_f5data; /* boolean: is f5data is valid? */
                struct ftver ftv;
#            elif defined ARGUS /* }{ */
                static struct ARGUS_INPUT input_s,
                                          *input = &input_s;
#            endif /* } */
	CODE:
#            ifdef ARGUS /* { */
		ArgusClientInit();
                ArgusProgramName = "Cflow"; /* FIXME - should be perl caller */
#            endif /* } */
		wanted = wanted_arg;
		n = 0;
		total = 0;
		if (2 > items ||
                    !SvROK(wanted) ||
                    SVt_PVCV != SvTYPE(SvRV(wanted))) {
		   croak("Usage: find(CODEREF, [CODEREF], FILE [...])");
		}

		if (SvROK(ST(arg)) &&
                    SVt_PVCV == SvTYPE(SvRV(ST(arg)))) { /* CODE */
		   perfile = ST(arg);
		   arg++;
		}

		verbose = SvIV(perl_get_sv("Cflow::verbose", TRUE));

		vars[INDEX] = perl_get_sv("Cflow::index", TRUE);
		vars[UNIX_SECS] = perl_get_sv("Cflow::unix_secs", TRUE);
		vars[EXPORTER] = perl_get_sv("Cflow::exporter", TRUE);
		vars[SRCADDR] = perl_get_sv("Cflow::srcaddr", TRUE);
		vars[DSTADDR] = perl_get_sv("Cflow::dstaddr", TRUE);
		vars[INPUT_IF] = perl_get_sv("Cflow::input_if", TRUE);
		vars[OUTPUT_IF] = perl_get_sv("Cflow::output_if", TRUE);
		vars[SRCPORT] = perl_get_sv("Cflow::srcport", TRUE);
		vars[DSTPORT] = perl_get_sv("Cflow::dstport", TRUE);
		vars[PKTS] = perl_get_sv("Cflow::pkts", TRUE);
		vars[BYTES] = perl_get_sv("Cflow::bytes", TRUE);
		vars[NEXTHOP] = perl_get_sv("Cflow::nexthop", TRUE);
		vars[STARTIME] = perl_get_sv("Cflow::startime", TRUE);
		vars[START_MSECS] = perl_get_sv("Cflow::start_msecs", TRUE);
		vars[ENDTIME] = perl_get_sv("Cflow::endtime", TRUE);
		vars[END_MSECS] = perl_get_sv("Cflow::end_msecs", TRUE);
		vars[DURATION_SECS] = perl_get_sv("Cflow::duration_secs", TRUE);
		vars[DURATION_MSECS] = perl_get_sv("Cflow::duration_msecs", TRUE);
		vars[PROTOCOL] = perl_get_sv("Cflow::protocol", TRUE);
		vars[TOS] = perl_get_sv("Cflow::tos", TRUE);
		vars[SRC_AS] = perl_get_sv("Cflow::src_as", TRUE);
		vars[DST_AS] = perl_get_sv("Cflow::dst_as", TRUE);
		vars[SRC_MASK] = perl_get_sv("Cflow::src_mask", TRUE);
		vars[DST_MASK] = perl_get_sv("Cflow::dst_mask", TRUE);
		vars[TCP_FLAGS] = perl_get_sv("Cflow::tcp_flags", TRUE);
		vars[ENGINE_TYPE] = perl_get_sv("Cflow::engine_type", TRUE);
		vars[ENGINE_ID] = perl_get_sv("Cflow::engine_id", TRUE);
		vars[RAW] = perl_get_sv("Cflow::raw", TRUE);
		vars[ICMPTYPE] = perl_get_sv("Cflow::ICMPType", TRUE);
		vars[ICMPCODE] = perl_get_sv("Cflow::ICMPCode", TRUE);

		for (; arg < items; arg++) {
		   size_t len;
                   char *namep;
		   char name[PATH_MAX];
		   FILE *fp = (FILE *)0;
                   int fd;

		   if (SVt_PV != SvTYPE(ST(arg))) {
		      croak("Usage: find(CODEREF, [CODEREF], FILE [...])");
		   }

                   namep = SvPV(ST(arg), len);
		   strncpy(name, namep, len);
		   name[len] = '\0';

		   if (0 == strcmp("-", name)) {
		      fd = STDIN_FILENO;
		   } else {
                      if (-1 == (fd = open(name, O_RDONLY, 0))) {
                         warn("open \"%s\": %s\n", name, strerror(errno));
                         continue;
                      }
		   }
                   /* xsubpp doesn't like this line to be blank */
#            ifdef OSU /* [ */
                   fterr_setid("Cflow"); /* FIXME - should be perl caller */
		   if (verbose) {
		      fterr_setfile(1, stderr);
		   } else {
		      fterr_setfile(0, (FILE *)0);
		   }
                   if (0 <= ftio_init(&fs, fd, FT_IO_FLAG_READ) &&
                       0 <= ftio_check_generic(&fs)) {
		      ftio_get_ver(&fs, &ftv);
		      has_f5data = (5 == ftv.d_version ||
			            6 == ftv.d_version ||
			            7 == ftv.d_version);
                   } else { /* FIXME - ifdef OSU, reading cflowd from stdin? */
                      /* assume the file is in cflowd's v5 format: */
                      ftio_close(&fs);
                      if (-1 == (fd = open(name, O_RDONLY, 0))) {
                         warn("open \"%s\": %s\n", name, strerror(errno));
                         continue;
                      }
		      fp = fdopen(fd, "r");
		      if ((FILE *)0 == fp) {
		         warn("fdopen \"%s\": %s", name, strerror(errno));
                         continue;
		      }
                   }
#            elif defined ARGUS /* ][ */
		   /* see main in "argus-2.0.4/common/argus_parse.c" */
		   input->fd = fd;
		   /* suppress type-checking on 2nd argument due to
		    * conflicting (incorrect prototype in <argus_parse.h>
		    * supplied with argus-2.0.4:
		    */
		   if (ArgusReadConnection(input, (void *)name) < 0) {
                      /* not a Argus-2.0 data stream (or other error) */
                      close(fd);
                      if (-1 == (fd = open(name, O_RDONLY, 0))) {
                         warn("open \"%s\": %s\n", name, strerror(errno));
                         continue;
                      }
		      fp = fdopen(fd, "r");
		      if ((FILE *)0 == fp) {
		         warn("fdopen \"%s\": %s", name, strerror(errno));
                         continue;
		      }
		   }
#            else /* ][ */
                   /* assume the file is in cflowd's v5 format: */
		   fp = fdopen(fd, "r");
		   if ((FILE *)0 == fp) {
		      warn("fdopen \"%s\": %s", name, strerror(errno));
                      continue;
		   }
#            endif /* ] */

		   /* call the per-file "hook" if one was supplied. */
		   if ((SV *)0 != perfile) {
                      ENTER;
                      SAVETMPS;

		      PUSHMARK(sp);
		      XPUSHs(sv_mortalcopy(ST(arg)));
		      PUTBACK;
		      perl_call_sv(perfile, G_VOID|G_DISCARD);
		      SPAGAIN;
		      PUTBACK;

                      FREETMPS;
                      LEAVE;
		   }
#            ifdef ARGUS /* [ */
                      if ((FILE *)0 == fp) {
			 ArgusRemoteFDs[0] = input;
			 ArgusReadStream();
		      }
		      else
#            endif /* ] */
                   for/*ever*/ (;;) {
		      size_t results;
                      uint32_t index;
#            ifdef OSU /* [ */
		      struct fttime first, last;
                      if ((FILE *)0 == fp) {
		         fdata = ftio_read(&fs);
                         if ((void *)0 == fdata) {
                            break;
                         }
			 first = ftltime(fdata->sysUpTime,
				         fdata->unix_secs,
				         fdata->unix_nsecs,
				         fdata->First);
			 last =  ftltime(fdata->sysUpTime,
				         fdata->unix_secs,
				         fdata->unix_nsecs,
				         fdata->Last);

			 /* { required for $Cflow::raw: */
			 /*
			  * FIXME? make $Cflow::raw a tied variable so
			  * that a performance penalty (because of all
			  * the htonl, htons calls below) is not incurred
			  * if the user doesn't even reference $Cflow::raw.
			  */
                         flow.index = htonl(expected_entry_mask);
                         flow.rtraddr = htonl(fdata->exaddr);
                         flow.srcaddr = htonl(fdata->srcaddr);
                         flow.dstaddr = htonl(fdata->dstaddr);
                         flow.nexthop = htonl(fdata->nexthop);
                         flow.input_if = htons(fdata->input);
                         flow.output_if = htons(fdata->output);
                         flow.pkts = htonl(fdata->dPkts);
                         flow.bytes = htonl(fdata->dOctets);
                         flow.starttime = htonl(first.secs);
                         flow.endtime = htonl(last.secs);
                         flow.srcport = htons(fdata->srcport);
                         flow.dstport = htons(fdata->dstport);
                         flow.protocol = fdata->prot;
                         flow.tos = fdata->tos;
                         flow.tcp_flags = fdata->tcp_flags;
#if 0 /* FIXME */
  u_int8  tcp_retx_cnt;   /* Number of mis-seq with delay > 1sec */
  u_int8  tcp_retx_secs;  /* Cumulative secs between mis-sequenced pkts */
  u_int8  tcp_misseq_cnt; /* Number of mis-sequenced tcp pkts seen */
#endif
			 if (has_f5data) {
                            flow.src_as = htons(f5data->src_as);
                            flow.dst_as = htons(f5data->dst_as);
                            flow.src_mask = f5data->src_mask;
                            flow.dst_mask = f5data->dst_mask;
                            flow.engine_type = f5data->engine_type;
                            flow.engine_id = f5data->engine_id;
			 } else { /* this info is unavailable: */
                            flow.src_as = 0;
                            flow.dst_as = 0;
			    /* FIXME? do class A,B,C rules?: */
                            flow.src_mask = 0;
                            flow.dst_mask = 0;
			 }
                         /* FIXME? handle fdata->drops */
		         sv_setpvn(vars[RAW], (char *)&flow, (sizeof flow) - 1);
			 /* } */
		         sv_setuv(vars[INDEX], expected_entry_mask);
		         sv_setuv(vars[EXPORTER], fdata->exaddr);
		         sv_setuv(vars[SRCADDR], fdata->srcaddr);
		         sv_setuv(vars[DSTADDR], fdata->dstaddr);
		         sv_setuv(vars[INPUT_IF], fdata->input);
		         sv_setuv(vars[OUTPUT_IF], fdata->output);
		         sv_setuv(vars[SRCPORT], fdata->srcport);
		         sv_setuv(vars[DSTPORT], fdata->dstport);
		         sv_setuv(vars[PKTS], fdata->dPkts);
		         sv_setuv(vars[BYTES], fdata->dOctets);
		         sv_setuv(vars[NEXTHOP], fdata->nexthop);
		         sv_setuv(vars[STARTIME], first.secs);
		         sv_setuv(vars[ENDTIME], last.secs);
			 sv_setuv(vars[START_MSECS], first.msecs);
			 sv_setuv(vars[END_MSECS], last.msecs);
		         sv_setuv(vars[PROTOCOL], fdata->prot);
		         sv_setuv(vars[TOS], fdata->tos);
			 if (has_f5data) {
		            sv_setuv(vars[SRC_AS], f5data->src_as);
		            sv_setuv(vars[DST_AS], f5data->dst_as);
		            sv_setuv(vars[SRC_MASK], f5data->src_mask);
		            sv_setuv(vars[DST_MASK], f5data->dst_mask);
		            sv_setuv(vars[ENGINE_TYPE], f5data->engine_type);
		            sv_setuv(vars[ENGINE_ID], f5data->engine_id);
			 }
		         sv_setuv(vars[TCP_FLAGS], fdata->tcp_flags);

		         sv_setuv(vars[UNIX_SECS], fdata->unix_secs);
		         if (1 == fdata->prot) { /* ICMP */
		            sv_setuv(vars[ICMPTYPE], fdata->dstport >> 8);
		            sv_setuv(vars[ICMPCODE], 0xff & fdata->dstport);
		         }
		         sv_setuv(vars[DURATION_SECS],
				  (fdata->Last - fdata->First) / 1000);
		         sv_setuv(vars[DURATION_MSECS],
				  (fdata->Last - fdata->First) % 1000);
                      } else {
#            endif /* ] */
		         if (1 != fread(&flow, (sizeof flow) - 1, 1, fp)) {
                            break;
                         }
                         index = ntohl(flow.index);
                         if (index != expected_entry_mask) {
                            if (verbose) {
                             warn("%s: Invalid index in cflowd flow file: 0x%X!"
				    " Version 5 flow-export is required"
				    " with *all* fields being saved.\n",
				    name, index);
                            }
                            goto next_file_label;
                         }
		         sv_setuv(vars[INDEX], index);
		         sv_setpvn(vars[RAW], (char *)&flow, (sizeof flow) - 1);
		         sv_setuv(vars[EXPORTER], ntohl(flow.rtraddr));
		         sv_setuv(vars[SRCADDR], ntohl(flow.srcaddr));
		         sv_setuv(vars[DSTADDR], ntohl(flow.dstaddr));
		         sv_setuv(vars[INPUT_IF], ntohs(flow.input_if));
		         sv_setuv(vars[OUTPUT_IF], ntohs(flow.output_if));
		         sv_setuv(vars[SRCPORT], ntohs(flow.srcport));
		         sv_setuv(vars[DSTPORT], ntohs(flow.dstport));
		         sv_setuv(vars[PKTS], ntohl(flow.pkts));
		         sv_setuv(vars[BYTES], ntohl(flow.bytes));
		         sv_setuv(vars[NEXTHOP], ntohl(flow.nexthop));
		         sv_setuv(vars[STARTIME], ntohl(flow.starttime));
		         sv_setuv(vars[ENDTIME], ntohl(flow.endtime));
		         sv_setuv(vars[PROTOCOL], flow.protocol);
		         sv_setuv(vars[TOS], flow.tos);
		         sv_setuv(vars[SRC_AS], ntohs(flow.src_as));
		         sv_setuv(vars[DST_AS], ntohs(flow.dst_as));
		         sv_setuv(vars[SRC_MASK], flow.src_mask);
		         sv_setuv(vars[DST_MASK], flow.dst_mask);
		         sv_setuv(vars[TCP_FLAGS], flow.tcp_flags);
		         sv_setuv(vars[ENGINE_TYPE], flow.engine_type);
		         sv_setuv(vars[ENGINE_ID], flow.engine_id);

		         sv_setuv(vars[UNIX_SECS], ntohl(flow.endtime));
		         if (1 == flow.protocol) { /* ICMP */
		            sv_setuv(vars[ICMPTYPE], ntohs(flow.dstport) >> 8);
		            sv_setuv(vars[ICMPCODE], 0xff & ntohs(flow.dstport));
		         }
		         sv_setuv(vars[DURATION_SECS],
				  ntohl(flow.endtime) - ntohl(flow.starttime));
#            ifdef OSU /* [ */
                      }
#            endif /* ] */
		      total++;

                      ENTER;
                      SAVETMPS;

		      PUSHMARK(sp);
		      PUTBACK;
		      results = perl_call_sv(wanted, G_SCALAR|G_NOARGS);
		      SPAGAIN;
		      if (1 == results && POPi != 0) {
			 n++;
		      }
		      PUTBACK;

                      FREETMPS;
                      LEAVE;
		   }

                next_file_label:
                   if ((FILE *)0 != fp) {
		      if (STDIN_FILENO != fd) {
		         fclose(fp);
                         fp = (FILE *)0;
			 fd = -1;
                      }
		   } else {
		      if (STDIN_FILENO != fd) {
                         close(fd);
#            ifdef OSU /* [ */
                         ftio_close(&fs);
#            endif /* ] */
                      }
                      fd = -1;
                   }
		}
		sprintf(retval, "%u/%u", (unsigned)n, (unsigned)total);
		RETVAL = retval;
	OUTPUT:
		RETVAL

