#ifndef CLIENT_UIRENDERER_H
#define CLIENT_UIRENDERER_H

#include <Ark/ArkRenderer.h>

namespace Client
{

   /**
    * The UIRenderer is a wrapper around the renderer class, intended for UI
    * elements. It uses direct drawing (ie primitives don't use indexed
    * vertex lists), and the vertex buffers are setup for UI-only use : 
    * they contain vertex coordinate, color, and texture coordinates.
    *
    * \par Example
    * Drawing a textured triangle is as simple as writing those few lines :
    * \begincode
        ui->SetTexture (mytexture);
	ui->BeginTriangles ();
	  ui->Vertex (0,0, 0.0, 0.0);
	  ui->Vertex (5,0, 1.0, 0.0);
	  ui->Vertex (5,5, 1.0, 1.0);
        ui->Flush();
    * \endcode
    */
   class UIRenderer
   {
	 Ark::Renderer *m_Renderer;
	 Ark::VertexBuffer m_VBuffer;
	 Ark::Material m_Material;
	 int m_NumVertices;

      public:
	 UIRenderer (Ark::Renderer *renderer,
		     int vbuffersize, int width, int height);
	 ~UIRenderer ();

	 ///////////////////////////////////////////////////////////////

	 void SetTexture (const Ark::TexturePtr& texture);

	 void BeginTriangles ();

	 void Vertex (int x, int y,
		      scalar u = 0.0, scalar v = 0.0,
		      uchar r = 255, uchar g = 255, uchar b = 255,
		      uchar a = 255);

	 void Flush ();

	 ///////////////////////////////////////////////////////////////

	 void DrawTexturedRectangle (int minx, int miny,
				     int w, int h);

	 ///////////////////////////////////////////////////////////////

	 /**
	  * Returns a font given a descriptive string.
	  * NAME-SIZE-COLOR.
	  * For a font located in {game}/data/fonts/arial1.glf in white:
	  * \begincode
	  * ui->GetFont("arial1-10-ffffff");
	  * \endcode
	  */
	 Ark::FontPtr GetFont (const Ark::String &name);

	 /**
	  * Draws a String with a given font
	  * \param font The font to use
	  * \param str The string to draw
	  * \param x The x coordinate
	  * \param y The y coordinate
	  */
	 void DrawString (Ark::Font *font,
			  const Ark::String &str, int x, int y);

	 /**
	  * Draws a String with a given font name (cf. GetFont)
	  * \param font The font to use
	  * \param str The string to draw
	  * \param x The x coordinate
	  * \param y The y coordinate
	  */
	 void DrawString (const Ark::String &name,
			  const Ark::String &str, int x, int y);

	 int GetStringWidth (Ark::Font *font, const Ark::String &str);
	 int GetStringHeight (Ark::Font *font, const Ark::String &str);

	 ///////////////////////////////////////////////////////////////

	 Ark::Renderer *Rdr ();
 	 int width;
	 int height;
   };

   extern UIRenderer *g_UI;

}

#endif
