#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <glib.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <pthread.h>

#include <asdutil.h>
#include <dsp.h>
#include <asd.h>
#include <recursive-lock.h>

#define NFRAGS 2

typedef struct 
{
  gchar *dsp;
  int fd;
  gboolean r;
  gboolean r_inuse;
  gboolean w;
  gboolean w_inuse;
  SampleType sample_type;
} Dsp;

GSList *_dsps = NULL;

GStaticMutex _dsps_mutex = G_STATIC_MUTEX_INIT;
GStaticPrivate _dsps_private = G_STATIC_PRIVATE_INIT;

#define _dsps_lock() recursive_lock(&_dsps_mutex, &_dsps_private)
#define _dsps_unlock() recursive_unlock(&_dsps_mutex, &_dsps_private)

static void _dsps_unlock_cleanup(void *p)
{
  _dsps_unlock();
}

static gboolean _open(gchar *device, gboolean r, Dsp *dsp)
{
  int caps, format, channels, speed, arg, n, fragsize, bps;
  audio_buf_info ai;

  g_assert(device && dsp);

  if ((dsp->fd = open(device, O_RDWR|O_NDELAY)) < 0)
    {
      if ((dsp->fd = open(device, (r ? O_RDONLY : O_WRONLY) | O_NDELAY)) < 0)
        {
          g_message("DSP: Could not open device '%s' for %s.", device, r ? "reading" : "writing");
          return FALSE;
        }
      
      g_message("DSP: Could open device '%s' only for %s.", device, r ? "reading" : "writing");
      dsp->w = !(dsp->r = r);
    }
  else
    {
      pthread_cleanup_push(fd_cleanup_handler, &dsp->fd);
      ioctl(dsp->fd, SNDCTL_DSP_SETDUPLEX, 0);

      if (ioctl(dsp->fd, SNDCTL_DSP_GETCAPS, &caps) == -1)
	{
	  close(dsp->fd);
	  return FALSE;
	}
      pthread_cleanup_pop(0);
      
      if (caps & DSP_CAP_DUPLEX)
	dsp->w = dsp->r = TRUE;
      else
	dsp->w = !(dsp->r = r);
    }

  pthread_cleanup_push(fd_cleanup_handler, &dsp->fd);  
  
  // Switch of NDELAY-flag
  if ((n = fcntl(dsp->fd, F_GETFL, 0)) == -1)
    {
      close(dsp->fd);
      return FALSE;
    }
  
  if (fcntl(dsp->fd, F_SETFL, n & ~O_NDELAY) == -1)
    {
      close(dsp->fd);
      return FALSE;
    }

  ioctl(dsp->fd, SNDCTL_DSP_RESET, 0);
  
  // Stolen from XMMS:
  fragsize = 0;
  bps = sample_type_bps(&default_sample_type);
  while ((1L << fragsize) < bps / 25)
    fragsize++;
  fragsize--;
  g_message("DSP: Fragment sizes for OSS/ASD: %ib/%ib", 1 << fragsize, default_block_size);
  
  arg = (NFRAGS << 16) | fragsize; 
  ioctl(dsp->fd, SNDCTL_DSP_SETFRAGMENT, &arg);

  // Set bitdepth, first try 16, than 8
  format = AFMT_S16_LE;
  if (ioctl(dsp->fd, SNDCTL_DSP_SETFMT, &format) == -1) 
    {
      close(dsp->fd);
      return FALSE;
    }
  
  if (format == AFMT_S16_LE) 
    dsp->sample_type.bits = 16;
  else
    {
      format = AFMT_S8;
      if (ioctl(dsp->fd, SNDCTL_DSP_SETFMT, &format) == -1)
	{
	  close(dsp->fd);
	  return FALSE;
	}
      
      if (format != AFMT_S8)
	{
	  close(dsp->fd);
	  return FALSE;
	}
      dsp->sample_type.bits = 8;
    }

  dsp->sample_type.be = FALSE;
  dsp->sample_type.sign = TRUE;

  // Set channels, first try stereo, than mono
  channels = 2;
  if (ioctl(dsp->fd, SNDCTL_DSP_CHANNELS, &channels) == -1) 
    {
      close(dsp->fd);
      return FALSE;
    }
  
  if (channels == 2) 
    dsp->sample_type.channels = 2;
  else
    {
      channels = 1; 
      if (ioctl(dsp->fd, SNDCTL_DSP_CHANNELS, &channels) == -1) 
	{
	  close(dsp->fd);
	  return FALSE;
	}
      
      if (channels != 1) 
	{
	  close(dsp->fd);
	  return FALSE;
	}

      dsp->sample_type.channels = 1;
    }

  speed = 44100; 
  if (ioctl(dsp->fd, SNDCTL_DSP_SPEED, &speed) == -1)
    {
      close(dsp->fd);
      return FALSE;
    }
  
  if ((speed < (44100*0.9)) || (speed > (44100*1.1)))
    {
      close(dsp->fd);
      return FALSE;
    }

  dsp->sample_type.rate = speed;

  if (ioctl(dsp->fd, SNDCTL_DSP_GETOSPACE, &ai) != -1)
    g_message("DSP: SNDCTL_DSP_GET_OSPACE says fragment size: %i; fragments: %i", ai.fragsize, ai.fragstotal);

  pthread_cleanup_pop(0);

  return TRUE;
}


static gboolean _func(gconstpointer a, gconstpointer b)
{
  Dsp *p;
  g_assert(a && b);

  p = (Dsp*) a;
  return strcmp(p->dsp, (gchar*) b) != 0;
}

static Dsp* _find(gchar *device)
{
  GSList *l;

  l = g_slist_find_custom(_dsps, device, _func);
  if (l)
    return (Dsp*) l->data;

  return NULL;
}


int dsp_open(gchar *device, gboolean r, SampleType *t)
{ 
  int ret = -1;
  Dsp *p;

  g_assert(device);

  _dsps_lock();

  if ((p = _find(device)))
    {
      if (r && p->r && !p->r_inuse)
	{
	  p->r_inuse = TRUE;
	  ret = p->fd;
	}
      else if (!r && p->w && !p->w_inuse)
	{
	  p->w_inuse = TRUE;
	  ret = p->fd;
	}
    }
  else
    {
      p = g_new0(Dsp, 1);
      pthread_cleanup_push(g_free, p);
      pthread_cleanup_push(_dsps_unlock_cleanup, NULL);
      ret = _open(device, r, p) ? p->fd : -1;
      pthread_cleanup_pop(0);
      pthread_cleanup_pop(0);
      
      if (ret < 0)
	g_free(p);
      else
	{
	  p->dsp = g_strdup(device);
	  p->w_inuse = 	!(p->r_inuse = r);
	  _dsps = g_slist_prepend(_dsps, p);
	}
    }

  *t = p->sample_type;

  _dsps_unlock();

  return ret;
}

void dsp_close(gchar *device, gboolean r)
{
  Dsp *p;
  
  _dsps_lock();
  pthread_cleanup_push(_dsps_unlock_cleanup, NULL);
  
  g_assert(p = _find(device));
  g_assert((r && p->r_inuse) || (!r && p->w_inuse));

  if (r)
    p->r_inuse = FALSE;
  else
    p->w_inuse = FALSE;
    
  
  if (!p->r_inuse && !p->w_inuse)
    {
      _dsps = g_slist_remove(_dsps, p);
      close(p->fd);
      g_free(p->dsp);
      g_free(p);
    }

  pthread_cleanup_pop(1);
}
