#include <unistd.h>
#include <fcntl.h>

#include <sink-default.h>
#include <sample-convert.h>
#include <asd.h>

SinkDefaultPrivate* sink_default_alloc_private(gchar *name)
{
  SinkDefaultPrivate *p;

  g_assert(p = g_new0(SinkDefaultPrivate, 1));
  p->filename = name ? g_strndup(name, PATH_MAX) : NULL;
  p->is_open = FALSE;
  p->buffer = NULL;
  p->buffer_size = 0;
  p->error = FALSE;
  return p;
}

gboolean sink_default_open_impl(Sink *s)
{  
  SinkDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SinkDefaultPrivate*) s->private_data);
  g_assert(p->filename && !p->is_open);
  g_assert(s->mode == SINK_DISABLED);

  if ((p->is_open = (p->fd = open(p->filename, O_WRONLY)) >= 0))
    {
      g_message("Sink '%s' successfully opened.", s->shortname);

      if (!sample_type_equal(&s->sample_type, &default_sample_type))
	{
	  p->buffer_size = sample_convert_length(&default_sample_type, &s->sample_type, default_block_size, FALSE);
	  p->buffer = g_new(guint8, p->buffer_size);
	}
    }
  else
    g_message("Sink '%s' open failed.", s->shortname);
  
  sink_set_mode(s, p->is_open ? SINK_RUNNING : SINK_DISABLED);
  s->byte_counter_since_open = 0;

  return p->is_open;
}

void sink_default_close_impl(Sink *s)
{ 
  SinkDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SinkDefaultPrivate*) s->private_data);
  
  if (p->is_open)
    {
      close(p->fd);
      p->is_open = FALSE;
      g_message("Sink '%s' closed.", s->shortname);
    }

  sink_set_mode(s, SOURCE_DISABLED);
  s->byte_counter_since_open = 0;
}

void sink_default_flush_impl(Sink *s)
{
  g_assert(s);
}

gboolean sink_default_reopen_impl(Sink *s)
{
  g_assert(s);
  if (s->close)
    s->close(s);
  
  if (s->open)
    return s->open(s);

  return FALSE;
}

glong sink_default_write_impl(Sink *s, guchar*d, gulong l)
{
  gulong i = 0;
  SinkDefaultPrivate *p;
  g_assert(s && d && (l > 0));
  g_assert(p = (SinkDefaultPrivate*) s->private_data);

  if (p->error)
    return -1;
  
  while (i < l)
    {
      ssize_t r;
      r = write(p->fd, d, l-i);
      
      if (r <= 0)
	{
	  if (s->reopen)
	    if (s->reopen(s))
	      continue;

	  p->error = TRUE;
	  break;
    	}

      d += r;
      i += r;
    }

  throughput_measure(s->throughput, i);

  return i;
}


gboolean sink_default_write_block_impl(Sink *s, Block *b)
{  
  SinkDefaultPrivate *p;
  gboolean ok = TRUE;

  g_assert(s);
  g_assert(p = (SinkDefaultPrivate*) s->private_data);

  g_assert(s->write);

  if (p->buffer) // Conversion needed
    {
      glong l;
      sample_convert_copy(&default_sample_type, block_get_data(b), b->size, &s->sample_type, p->buffer, FALSE);
      l = sample_convert_length(&default_sample_type, &s->sample_type, b->size, FALSE);
      
      if (s->write(s, p->buffer, l) <= 0)
	ok = FALSE;
    }
  else
    {
      if (s->write(s, block_get_data(b), b->size) <= 0)
	ok = FALSE;
    }

  return ok;
}


void sink_default_free_private_impl(Sink *s)
{  
  SinkDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SinkDefaultPrivate*) s->private_data);

  g_free(p->filename);
  g_free(p->buffer);
  g_free(p);
  s->private_data = NULL;
}

void sink_default_save_private_impl(Sink *s, xmlNodePtr node)
{
  g_error("Save not yet implemented.");
}
