------------------------------------------------------------------------------
--                                                                          --
--                 ASIS-for-GNAT IMPLEMENTATION COMPONENTS                  --
--                                                                          --
--   A S I S . D A T A _ D E C O M P O S I T I O N . E X T E N S I O N S    --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--            Copyright (c) 1995-1999, Free Software Foundation, Inc.       --
--                                                                          --
-- ASIS-for-GNAT is free software; you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. ASIS-for-GNAT is distributed  in the hope  that it will be use- --
-- ful, but WITHOUT ANY WARRANTY; without even the implied warranty of MER- --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License  distributed with ASIS-for-GNAT; see file     --
-- COPYING. If not, write to the Free Software Foundation,  59 Temple Place --
-- - Suite 330,  Boston, MA 02111-1307, USA.                                --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- ASIS-for-GNAT was originally developed  by the ASIS-for-GNAT team at the --
-- Software  Engineering  Laboratory  of  the Swiss  Federal  Institute  of --
-- Technology (LGL-EPFL) in Lausanne,  Switzerland, in cooperation with the --
-- Scientific  Research  Computer  Center of  Moscow State University (SRCC --
-- MSU), Russia,  with funding partially provided  by grants from the Swiss --
-- National  Science  Foundation  and  the  Swiss  Academy  of  Engineering --
-- Sciences.  ASIS-for-GNAT is now maintained by  Ada Core Technologies Inc --
-- (http://www.gnat.com).                                                   --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with System;            use System;

with Asis.Exceptions;   use Asis.Exceptions;
with Asis.Set_Get;      use Asis.Set_Get;

with A4G.Int_Knds;      use A4G.Int_Knds;
with A4G.Vcheck;        use A4G.Vcheck;
with A4G.DDA_Aux;       use A4G.DDA_Aux;

with Einfo;             use Einfo;
with Uintp;             use Uintp;
with Urealp;            use Urealp;
with Namet;             use Namet;

package body Asis.Data_Decomposition.Extensions  is

   Package_Name : constant String := "Asis.Data_Decomposition.Extensions.";

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Write_Real_To_Buffer (U : Ureal);
   --  Puts nnn/ddd to Name_Buffer, where nnn and ddd are integer values of
   --  the normalized numerator and denominator of the given real value.
   --  This procedure is supposed to be used to output images of positive
   --  real values only, so it assumes, that nnn is always positive.

   procedure Add_Uint_To_Buffer (U : Uint);
   --  Add image of universal integer to Name_Buffer, updating Name_Len
   --  (This procedure is the simplified version of the local procedure
   --  Exp_Dbug.Add_Uint_To_Buffer - it always consider its argument
   --  being positive)

   ------------------------
   -- Add_Uint_To_Buffer --
   ------------------------

   procedure Add_Uint_To_Buffer (U : Uint) is
   begin
      UI_Image (U, Decimal);
      Add_Str_To_Name_Buffer (UI_Image_Buffer (1 .. UI_Image_Length));
   end Add_Uint_To_Buffer;

   -----------------
   -- Delta_Value --
   -----------------

   function Delta_Value
     (Fixed_Point_Subtype : Asis.Element)
      return String
   is
      Arg_Node      : Node_Id;
      Res_Ureal     : Ureal;
   begin
      Check_Validity (Fixed_Point_Subtype, Package_Name & "Delta_Value");

      Arg_Node := R_Node (Fixed_Point_Subtype);

      if not (Int_Kind (Fixed_Point_Subtype) = A_Defining_Identifier and then
              Is_Fixed_Point_Type (Arg_Node))
      then
         Raise_ASIS_Inappropriate_Element
           (Diagnosis => Package_Name & "Delta_Value");
      end if;

      Res_Ureal := Einfo.Delta_Value (Arg_Node);

      Write_Real_To_Buffer (Res_Ureal);

      return Namet.Name_Buffer (1 .. Namet.Name_Len);

   exception
      when ASIS_Inappropriate_Element =>
         raise;

      when others =>

         Raise_ASIS_Failed (
            Argument  => Fixed_Point_Subtype,
            Diagnosis => Package_Name & "Delta_Value");
   end Delta_Value;

   function Delta_Value
     (Fixed_Point_Subtype : Asis.Element)
      return Fraction
   is
      Arg_Node  : Node_Id;
      Res_Ureal : Ureal;

      Result    : Fraction;

   begin
      Check_Validity (Fixed_Point_Subtype, Package_Name & "Delta_Value");

      Arg_Node := R_Node (Fixed_Point_Subtype);

      if not (Int_Kind (Fixed_Point_Subtype) = A_Defining_Identifier and then
              Is_Fixed_Point_Type (Arg_Node))
      then
         Raise_ASIS_Inappropriate_Element
           (Diagnosis => Package_Name & "Delta_Value");
      end if;

      Res_Ureal := Einfo.Delta_Value (Arg_Node);

      Result.Num   := UI_To_Aint (Norm_Num (Res_Ureal));
      Result.Denum := UI_To_Aint (Norm_Den (Res_Ureal));

      return Result;

   exception
      when ASIS_Inappropriate_Element =>
         raise;

      when Invalid_Data =>
         Raise_ASIS_Failed (
            Argument  => Fixed_Point_Subtype,
            Diagnosis => Package_Name &
                        "Delta_Value (result value can not be " &
                        "represented as a fraction)");

      when others =>

         Raise_ASIS_Failed (
            Argument  => Fixed_Point_Subtype,
            Diagnosis => Package_Name & "Delta_Value");
   end Delta_Value;

   ------------------
   -- Digits_Value --
   ------------------

   function Digits_Value
     (Floating_Point_Subtype : Asis.Element)
      return ASIS_Natural
   is
      Arg_Node      : Node_Id;
   begin
      Check_Validity (Floating_Point_Subtype, Package_Name & "Digits_Value");

      Arg_Node := R_Node (Floating_Point_Subtype);

      if not (Int_Kind (Floating_Point_Subtype) = A_Defining_Identifier
            and then
             (Is_Decimal_Fixed_Point_Type (Arg_Node) or else
              Is_Floating_Point_Type (Arg_Node)))
      then
         Raise_ASIS_Inappropriate_Element
           (Diagnosis => Package_Name & "Digits_Value");
      end if;

      return ASIS_Natural (UI_To_Int (Einfo.Digits_Value (Arg_Node)));

   exception
      when ASIS_Inappropriate_Element =>
         raise;

      when others =>
         Raise_ASIS_Failed (
            Argument  => Floating_Point_Subtype,
            Diagnosis => Package_Name & "Digits_Value");
   end Digits_Value;

   --------------------------
   --  Portable_Data_Value --
   --------------------------

   function Portable_Data_Value
     (Value : Constrained_Subtype)
      return Portable_Data
   is

      Local_Value : aliased Constrained_Subtype := Value;

      subtype Result_Portable_Data is
         Portable_Data (1 .. (Local_Value'Size + 7) / 8);

      type Result_Portable_Data_Access is access Result_Portable_Data;

      function To_Result_Portable_Data_Access is new
         Ada.Unchecked_Conversion (Address, Result_Portable_Data_Access);

      Result : constant Result_Portable_Data_Access :=
         To_Result_Portable_Data_Access (Value'Address);

   begin

      return Result.all;

   end Portable_Data_Value;

   -----------------
   -- Scale_Value --
   -----------------

   function Scale_Value
     (Desimal_Fixed_Point_Subtype : Asis.Element)
      return ASIS_Natural
   is
      Arg_Node      : Node_Id;

   begin
      Check_Validity
        (Desimal_Fixed_Point_Subtype, Package_Name & "Scale_Value");

      Arg_Node := R_Node (Desimal_Fixed_Point_Subtype);

      if not (Int_Kind (Desimal_Fixed_Point_Subtype) = A_Defining_Identifier
            and then
              Is_Decimal_Fixed_Point_Type (Arg_Node))
      then
         Raise_ASIS_Inappropriate_Element
           (Diagnosis => Package_Name & "Scale_Value");
      end if;

      return ASIS_Natural (UI_To_Int (Einfo.Scale_Value (Arg_Node)));

   exception
      when ASIS_Inappropriate_Element =>
         raise;

      when others =>
         Raise_ASIS_Failed (
            Argument  => Desimal_Fixed_Point_Subtype,
            Diagnosis => Package_Name & "Scale_Value");
   end Scale_Value;

   -----------------
   -- Small_Value --
   -----------------

   function Small_Value
     (Fixed_Point_Subtype : Asis.Element)
      return String
   is
      Arg_Node  : Node_Id;
      Res_Ureal : Ureal;

   begin
      Check_Validity (Fixed_Point_Subtype, Package_Name & "Small_Value");

      Arg_Node := R_Node (Fixed_Point_Subtype);

      if not (Int_Kind (Fixed_Point_Subtype) = A_Defining_Identifier and then
              Is_Fixed_Point_Type (Arg_Node))
      then
         Raise_ASIS_Inappropriate_Element
           (Diagnosis => Package_Name & "Small_Value");
      end if;

      Res_Ureal := Einfo.Small_Value (Arg_Node);

      Write_Real_To_Buffer (Res_Ureal);

      return Namet.Name_Buffer (1 .. Namet.Name_Len);

   exception
      when ASIS_Inappropriate_Element =>
         raise;

      when others =>
         Raise_ASIS_Failed (
            Argument  => Fixed_Point_Subtype,
            Diagnosis => Package_Name & "Small_Value");
   end Small_Value;

   function Small_Value
     (Fixed_Point_Subtype : Asis.Element)
      return Fraction
   is
      Arg_Node  : Node_Id;
      Res_Ureal : Ureal;

      Result    : Fraction;
   begin
      Check_Validity (Fixed_Point_Subtype, Package_Name & "Small_Value");

      Arg_Node := R_Node (Fixed_Point_Subtype);

      if not (Int_Kind (Fixed_Point_Subtype) = A_Defining_Identifier and then
              Is_Fixed_Point_Type (Arg_Node))
      then
         Raise_ASIS_Inappropriate_Element
           (Diagnosis => Package_Name & "Small_Value");
      end if;

      Res_Ureal := Einfo.Small_Value (Arg_Node);

      Result.Num   := UI_To_Aint (Norm_Num (Res_Ureal));
      Result.Denum := UI_To_Aint (Norm_Den (Res_Ureal));

      return Result;

   exception
      when ASIS_Inappropriate_Element =>
         raise;

      when Invalid_Data =>
         Raise_ASIS_Failed (
            Argument  => Fixed_Point_Subtype,
            Diagnosis => Package_Name &
                        "Small_Value (result value can not be " &
                        "represented as a fraction)");

      when others =>
         Raise_ASIS_Failed (
            Argument  => Fixed_Point_Subtype,
            Diagnosis => Package_Name & "Small_Value");
   end Small_Value;

   --------------------------
   -- Write_Real_To_Buffer --
   --------------------------

   procedure Write_Real_To_Buffer (U : Ureal) is
   begin
      Namet.Name_Len := 0;

      Add_Uint_To_Buffer (Norm_Num (U));
      Add_Str_To_Name_Buffer ("/");
      Add_Uint_To_Buffer (Norm_Den (U));
   end Write_Real_To_Buffer;

end Asis.Data_Decomposition.Extensions;