/** \file libassogiate/mime-type.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "mime-type.hh"

#include <glib/gutils.h>
#include <gtkmm/icontheme.h>
#include <libxml++/nodes/element.h>
#include <libxml++/nodes/textnode.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
/* struct Message                                                             */
/******************************************************************************/

ustring
Message::get_untranslated() const throw()
{
	const_iterator C = find(ustring());
	return (C == end()) ? ustring() : C->second;
}

ustring
Message::get_translated() const throw()
{
	for (const gchar* const *i = g_get_language_names(); *i; ++i) { // unwrapped
		const_iterator j = find(*i);
		if (j != end()) return j->second;
	}

	return get_untranslated();
}

void
Message::set_from_user(const ustring& value) throw()
{
	(*this)[ustring()] = value;

	for (const gchar* const *i = g_get_language_names(); *i; ++i) // unwrapped
		if (ustring(*i).find('.') == ustring::npos) { /* without encoding */
			(*this)[*i] = value;
			break;
		}
}

/******************************************************************************/
/* struct MatchSet                                                            */
/******************************************************************************/

MatchSet::~MatchSet()
{
	FOREACH(std::list<MagicMatch*>, submatches, i)
		delete *i;
}

/******************************************************************************/
/* struct XMLRoot                                                             */
/******************************************************************************/

bool
XMLRoot::operator==(const XMLRoot& other) const throw()
{
	return (namespace_uri == other.namespace_uri &&
		local_name == other.local_name);
}

/******************************************************************************/
/* class MimeType                                                             */
/******************************************************************************/

std::pair<ustring, ustring>
MimeType::split_name(const ustring& full_name) throw(std::invalid_argument)
{
	std::vector<ustring> parts = misc::ustrsplit(full_name, "/", 2);
	if (parts.size() != 2)
		throw std::invalid_argument("expected / in MIME type name");
	return std::pair<ustring, ustring>(parts[0], parts[1]);
}

MimeType::MimeType(const ustring& type, const ustring& subtype) throw()
:	m_type(type), m_subtype(subtype), m_description(),
	m_aliases(), m_superclasses(),
	m_glob_patterns(), m_magics(), m_xml_roots(),
	m_location_nodes()
{}

MimeType::~MimeType()
{
	FOREACH(std::list<Magic*>, m_magics, i)
		delete *i;
}

ustring
MimeType::get_full_name() const throw()
{	return m_type + "/" + m_subtype; }

ustring
MimeType::get_icon_name() const throw()
{
	RefPtr<Gtk::IconTheme> theme = Gtk::IconTheme::get_default();

	ustring result = m_type + "-" + m_subtype;
	if (theme->has_icon(result))
		return result;
	
	result.insert(0, "gnome-mime-");
	if (theme->has_icon(result))
		return result;
	
	if (m_type == "inode") {
		if (m_subtype == "blockdevice")
			return "drive-harddisk";
		else if (m_subtype == "symlink")
			return "emblem-symbolic-link";
		else if (m_subtype == "directory")
			return "folder";
	}
	
	result = m_type + "-x-generic";
	if (theme->has_icon(result))
		return result;

	return "gtk-file";
}

Location
MimeType::get_locations() const throw()
{
	Location result = NO_LOCATION;
	if (is_in_location(SYSTEM_STANDARD)) result |= SYSTEM_STANDARD;
	if (is_in_location(SYSTEM_OVERRIDE)) result |= SYSTEM_OVERRIDE;
	if (is_in_location(USER_STANDARD)) result |= USER_STANDARD;
	if (is_in_location(USER_OVERRIDE)) result |= USER_OVERRIDE;
	return result;
}

bool
MimeType::is_in_location(Location location) const throw()
{	return (m_location_nodes.find(location) != m_location_nodes.end()); }

MimeType*
MimeType::limit_locations(Location locations) const throw()
{
	MimeType *result = new MimeType(m_type, m_subtype);
	bool have_result = false;
	
	CONST_FOREACH(LocationNodesMap, m_location_nodes, i)
		if (i->first & locations) {
			have_result = true;
			result->accumulate(*i->second, i->first);
		}

	if (have_result)
		return result;
	else {
		delete result;
		return NULL;
	}
}

void
MimeType::accumulate(const xmlpp::Node& node, Location source)
	throw(xmlpp::exception)
{
	m_location_nodes.insert(LocationNodesMap::value_type(source, &node));

	xmlpp::Node::NodeList children = node.get_children();
	FOREACH(xmlpp::Node::NodeList, children, i) {
		if cast(*i, xmlpp::Element, el) {
			ustring name = el->get_name();
			if (name == "glob") {
				xmlpp::Attribute *pattern = el->get_attribute("pattern");
				if (pattern != NULL) {
					ustring value = pattern->get_value();
					if (std::find(m_glob_patterns.begin(),
						m_glob_patterns.end(), value) == m_glob_patterns.end())
						m_glob_patterns.push_back(value);
				}
			} else if (name == "magic") {
				Magic *magic = new Magic();
				xmlpp::Attribute *prio_a = el->get_attribute("priority");
				magic->priority = 50;
				if (prio_a != NULL) magic->priority =
					compose::decompose<int>(prio_a->get_value(), 50);
				xmlpp::Node::NodeList children2 = el->get_children("match");
				FOREACH(xmlpp::Node::NodeList, children2, j)
					if cast(*j, xmlpp::Element, el2)
						magic->submatches.push_back(parse_match(*el2));
				m_magics.push_back(magic);
			} else if (name == "alias") {
				xmlpp::Attribute *type = el->get_attribute("type");
				if (type != NULL) {
					ustring value = type->get_value();
					if (std::find(m_aliases.begin(), m_aliases.end(), value) ==
						m_aliases.end())
						m_aliases.push_back(value);
				}
			} else if (name == "sub-class-of") {
				xmlpp::Attribute *type = el->get_attribute("type");
				if (type != NULL) {
					ustring value = type->get_value();
					if (std::find(m_superclasses.begin(), m_superclasses.end(),
						value) == m_superclasses.end())
						m_superclasses.push_back(value);
				}
			} else if (name == "comment") {
				xmlpp::Attribute *lang_a = el->get_attribute("lang", "xml");
				if (el->get_child_text())
					m_description
						[(lang_a!=NULL) ? lang_a->get_value() : ustring()]
						= el->get_child_text()->get_content();
			} else if (name == "root-XML") {
				xmlpp::Attribute *nsuri_a = el->get_attribute("namespaceURI"),
					*lname_a = el->get_attribute("localName");
				if (nsuri_a != NULL) {
					XMLRoot root;
					root.namespace_uri = nsuri_a->get_value();
					if (lname_a != NULL) root.local_name = lname_a->get_value();
					if (std::find(m_xml_roots.begin(), m_xml_roots.end(),
						root) == m_xml_roots.end())
						m_xml_roots.push_back(root);
				}
			}
		}
	}
}

void
MimeType::output(xmlpp::Element& node) const throw()
{
	node.set_attribute("type", get_full_name());
	
	xmlpp::Node::NodeList children = node.get_children();
	FOREACH(xmlpp::Node::NodeList, children, i)
		if cast(*i, xmlpp::Element, el) {
			if (!el->get_namespace_prefix().empty()) continue;
			ustring name = el->get_name();
			if (name == "glob" || name == "magic" || name == "alias" ||
				name == "sub-class-of" || name == "comment" ||
				name == "root-XML")
				node.remove_child(el);
		}
	
	CONST_FOREACH(std::list<ustring>, m_glob_patterns, i)
		node.add_child("glob")->set_attribute("pattern", *i);

	CONST_FOREACH(std::list<Magic*>, m_magics, i) {
		xmlpp::Element *sub = node.add_child("magic");
		sub->set_attribute("priority", compose::ucompose1((*i)->priority));
		CONST_FOREACH(std::list<MagicMatch*>, (*i)->submatches, j)
			output_match(*sub, **j);
	}
	
	CONST_FOREACH(std::list<ustring>, m_aliases, i)
		node.add_child("alias")->set_attribute("type", *i);
	
	CONST_FOREACH(std::list<ustring>, m_superclasses, i)
		node.add_child("sub-class-of")->set_attribute("type", *i);

	CONST_FOREACH(Message, m_description, i) {
		xmlpp::Element *sub = node.add_child("comment");
		if (!i->first.empty())
			sub->set_attribute("lang", i->first, "xml");
		sub->set_child_text(i->second);
	}
	
	CONST_FOREACH(std::list<XMLRoot>, m_xml_roots, i) {
		xmlpp::Element *sub = node.add_child("root-XML");
		sub->set_attribute("namespaceURI", i->namespace_uri);
		sub->set_attribute("localName", i->local_name);
	}
}

MagicMatch*
MimeType::parse_match(const xmlpp::Element& node) throw(xmlpp::exception)
{
	xmlpp::Attribute *type_a = node.get_attribute("type"),
		*offset_a = node.get_attribute("offset"),
		*value_a = node.get_attribute("value"),
		*mask_a = node.get_attribute("mask");
	if (type_a == NULL || offset_a == NULL || value_a == NULL)
		throw xmlpp::parse_error("missing attribute(s)");

	MagicMatch *match = new MagicMatch();

	ustring type = type_a->get_value();
	if (type == "string") match->type = MagicMatch::STRING;
	else if (type == "host16") match->type = MagicMatch::HOST16;
	else if (type == "host32") match->type = MagicMatch::HOST32;
	else if (type == "big16") match->type = MagicMatch::BIG16;
	else if (type == "big32") match->type = MagicMatch::BIG32;
	else if (type == "little16") match->type = MagicMatch::LITTLE16;
	else if (type == "little32") match->type = MagicMatch::LITTLE32;
	else if (type == "byte") match->type = MagicMatch::BYTE;
	else throw xmlpp::parse_error("invalid attribute value(s)");
	
	std::vector<ustring> parts = misc::ustrsplit(offset_a->get_value(), ":", 2);
	if (parts.size() < 1)
		throw xmlpp::parse_error("invalid attribute value(s)");
	match->first_offset = compose::decompose<int>(parts[0], -1);
	if (match->first_offset == -1)
		throw xmlpp::parse_error("invalid attribute value(s)");
	match->last_offset = (parts.size() == 2)
		? compose::decompose<int>(parts[1], -1) : -1;
	
	match->value = value_a->get_value();
	match->mask = (mask_a != NULL) ? mask_a->get_value() : ustring();
	
	xmlpp::Node::NodeList children = node.get_children("match");
	FOREACH(xmlpp::Node::NodeList, children, i)
		if cast(*i, xmlpp::Element, el)
			match->submatches.push_back(parse_match(*el));
	
	return match;
}

void
MimeType::output_match(xmlpp::Element& parent, const MagicMatch& match) throw()
{
	xmlpp::Element *node = parent.add_child("match");

	ustring type;
	if (match.type == MagicMatch::STRING) type = "string";
	else if (match.type == MagicMatch::HOST16) type = "host16";
	else if (match.type == MagicMatch::HOST32) type = "host32";
	else if (match.type == MagicMatch::BIG16) type = "big16";
	else if (match.type == MagicMatch::BIG32) type = "big32";
	else if (match.type == MagicMatch::LITTLE16) type = "little16";
	else if (match.type == MagicMatch::LITTLE32) type = "little32";
	else if (match.type == MagicMatch::BYTE) type = "byte";

	if (!type.empty()) node->set_attribute("type", type);
	node->set_attribute("offset", (match.last_offset == -1)
		? compose::ucompose1(match.first_offset)
		: compose::ucompose("%1:%2", match.first_offset, match.last_offset));
	node->set_attribute("value", match.value);
	if (!match.mask.empty()) node->set_attribute("mask", match.mask);

	CONST_FOREACH(std::list<MagicMatch*>, match.submatches, i)
		output_match(*node, **i);
}

/******************************************************************************/
/* class NodeMapExtender                                                      */
/******************************************************************************/

NodeMapExtender::~NodeMapExtender()
{}

} /* namespace assoGiate */
