/*
 * util.c - utility function library
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "util.h"
#include "log.h"

/** Remove leading and trailing whitespace */
void
trim_string(char* str)
{
	int i = strlen(str), nl = 0;
	/* trailing */
	while (i>0)
	{
		--i;
		if (str[i] == '\n')
			nl = 1;
		if (str[i] == ' ' || str[i] == '\t' || str[i] == '\n')
			str[i] = '\0';
		else
			break;
	}
	if (nl)
		str[++i] = '\n';

	/* leading */
	i = 0;
	while (str[i] == ' ' || str[i] == '\t')
		i++;
	while (*(str+i) != '\0')
	{
		*str = *(str+i);
		str++;
	}
	*str = '\0';
}

/** Remove comments */
void
remove_comments(char* str, char c)
{
	while (*str != '\0' && *str != c)
		str++;
	/* everything after c is comment */
	if (*str == c)
	{
		*str = '\n';
		str++;
		*str = '\0';
	}
}

/** Position in string */
int
position_in_string(char *str, const char* sub)
{
	int pos = -1, i = 0, j = 0;
	char* walk;
	const char* restore = sub;
	while (*str != '\0')
	{
		walk = str;
		j = 0;
		while (*sub != '\0' && *walk == *sub)
		{
			sub++;
			walk++;
			j++;
		}

		if (*sub == '\0' && j > 0)
			pos = i;

		sub = restore;
		j = 0;
		i++;
		str++;
	}
	if (pos < 0)
		return pos;
	return pos + strlen(sub);
}

/** MAX */
uint32_t
max(uint32_t a, uint32_t b)
{
	return (a<b?b:a);
}

/** MIN */
uint32_t
min(uint32_t a, uint32_t b)
{
	return (a>b?b:a);
}

/** Compare two strings on length and equality */
int
my_strcmp(const char* a, const char* b)
{
	log_assert(a && b);

	return strcmp(a, b);
}

/** Read pidfile */
pid_t
readpid(const char* pidfile, size_t* len)
{
	FILE* fd;
	pid_t pid;
	char pidbuf[16];
	char *t;
	size_t l;

	*len = 0;
	if ((fd = fopen(pidfile, "r")) == NULL)
		return -1;
	if ((l = fread(pidbuf, sizeof(char), 16, fd)) == 0)
	{
		fclose(fd);
		return -1;
	}
	fclose(fd);
	/* Empty pidfile means no pidfile... */
	if (l == 0)
	{
		errno = ENOENT;
		return -1;
	}

	pid = strtol(pidbuf, &t, 10);
	*len = l;
	if (*t && *t != '\n')
		return -1;
	return pid;
}

/* BIND STUFF */

static int
bind_is_bindspecial(int c)
{
	switch(c) {
		case '{':
		case '}':
		case '"':
		case ';':
			return 1;
		default:
			return 0;
	}
	/* not reached */
	return 0;
}

static void
bind_skip_to_eol(FILE* fd)
{
	int c;
	while((c = getc(fd)) != EOF)
		if (c == '\n')
			return;
}

int
bind_skip_to_special(FILE* fd, ldns_buffer* buf, int* line, char c)
{
	int rdlen;
	ldns_buffer_clear(buf);
	while((rdlen=bind_read_keyword(fd, buf, line, 1, 0)))
	{
		if (rdlen == 1 && isspace((int)*ldns_buffer_begin(buf)))
		{
			ldns_buffer_clear(buf);
			continue;
		}
		if (rdlen != 1 || *ldns_buffer_begin(buf) != (uint8_t)c)
		{
			ldns_buffer_write_u8(buf, 0);
			return 0;
		}
		return 1;
	}
	return 0;
}

static int
bind_read_string(FILE* fd, ldns_buffer* buf, int* line)
{
	int c, numdone=0;
	while((c = getc(fd)) != EOF)
	{
		if (isspace(c))
			continue;
		/* not space, must be a quote then */
		numdone++;
		if (numdone == 1)
		{
			if (c != '"')
				return 0;
			continue;
		}
		else if (numdone > 1)
		{
			if (c == '"')
				return numdone;
			if (c == '\n')
			{
				c = ' ';
				(*line)++;
			}
			if (ldns_buffer_remaining(buf) < 2)
				return 0;
			ldns_buffer_write_u8(buf, (uint8_t)c);
		}
	}
	return 0;
}

int
bind_read_keyword(FILE* fd, ldns_buffer* buf, int* line, int comments,
	int newlines)
{
	int c;
	int numdone = 0;
	while((c = getc(fd)) != EOF)
	{
		if(comments && c == '#')
		{
			bind_skip_to_eol(fd);
			(*line)++;
			continue;
        	}
		else if(comments && c=='/' && numdone > 0 &&
			ldns_buffer_read_u8_at(buf,
				ldns_buffer_position(buf)-1) == '/')
		{
			ldns_buffer_skip(buf, -1);
			numdone--;
			bind_skip_to_eol(fd);
			(*line)++;
			continue;
		}
		else if(comments && c=='*' && numdone>0 &&
			ldns_buffer_read_u8_at(buf,
				ldns_buffer_position(buf)-1) == '/')
		{
			ldns_buffer_skip(buf, -1);
			numdone--;
			/* skip to end of comment */
			while(c != EOF && (c=getc(fd)) != EOF)
			{
				if(c == '*' && (c=getc(fd)) == '/')
					break;
				if(c == '\n')
					(*line)++;
			}
			continue;
		}
		/* not a comment, complete the keyword */
		if (numdone > 0 && (isspace(c) || bind_is_bindspecial(c)))
		{
			ungetc(c, fd);
			return numdone;
		}
		if (c == '\n')
		{
			if (!newlines)
				c = ' ';
			(*line)++;
		}
		if (ldns_buffer_remaining(buf) < 2)
			return 0;
		ldns_buffer_write_u8(buf, (uint8_t)c);
		numdone++;
		/* read white space */
		if (isspace(c))
		{
			while((c = getc(fd)) != EOF)
			{
				if (c == '\n')
					(*line)++;
				if (!isspace(c))
				{
					ungetc(c, fd);
					break;
				}

				if (newlines)
				{
					ldns_buffer_write_u8(buf, (uint8_t)c);
					numdone++;
				}
			}
			return numdone;
		}
		if (bind_is_bindspecial(c))
			return numdone;
	}
	return numdone;
}

int
bind_process_contents(tp_set_t* anchors, char* file, ldns_buffer* buf,
	int* line, FILE* fd, int writing)
{
	/* one content contains: dname number number number "string" ; */
	int column=0;	/*  0: expect dname
			 	 	 *  1,2,3: expect number
					 *  4: expect key in str format
					 */
	int comments=1, rdlen;
	char* str = 0;
	ta_t* ta = NULL;
	tp_t* tp = NULL;

	ldns_buffer_clear(buf);
	while ((rdlen = bind_read_keyword(fd, buf, line, comments, 0)) != 0)
	{
		if(rdlen == 1 && isspace((ldns_buffer_current(buf)[-1])))
		{
			ldns_buffer_write(buf, " ", 1);
			continue;
		}
		if (rdlen == 0)
			return 0;
		if (column == 5 && rdlen == 1 &&
		    ldns_buffer_current(buf)[-1] == ';')
		{
			ldns_buffer_skip(buf, -1);
			ldns_buffer_write_u8(buf, 0);
			str = strdup((char*)ldns_buffer_begin(buf));
			if(!str)
				return 0;
			trim_string(str);
			debug(("add trusted key: %s", str));
			ta = add_trustanchor_frm_str(anchors, str, &tp, 0);
			if (!ta)
			{
				free(str);
				return 0;
			}

			if (!writing)
			{
				ta->s = STATE_VALID;
				tp->file = (const char*) file;
				debug(("set trusted keys file for %s to %s", tp->name,
					tp->file));
			}

			free(str);
			ldns_buffer_clear(buf);
			column = 0;
			continue;
		}
		if (column == 0 && rdlen == 1 &&
					ldns_buffer_current(buf)[-1] == '}')
			return 1;
		if (rdlen == 1 &&
			bind_is_bindspecial((ldns_buffer_current(buf)[-1])))
		{
			if (ldns_buffer_current(buf)[-1] == '"') {
				/* read owner name as string */
				ldns_buffer_set_position(buf,
					(ldns_buffer_position(buf)-1));
				ungetc('"', fd);
				rdlen = bind_read_string(fd, buf, line);
				if (rdlen == 0)
					return 0;
			}
			else
				return 0;
		}
		/* good keyword */
		column++;
		if(ldns_buffer_remaining(buf) < 8+1)
			return 0;
		if (column == 1)
			ldns_buffer_write(buf, " DNSKEY ", 8);
		if (column == 4)
		{
			ldns_buffer_write(buf, " ", 1);
			rdlen = bind_read_string(fd, buf, line);
			if (rdlen == 0)
				return 0;
			column++;
		}
		/* next */
	}
	return 0;
}

/** rdf 2 str 2 buf + update position + free str */
static void
bind_rdf2buf_str_free(ldns_buffer* buf, ldns_rdf* rdf, int* at, const char* sp)
{
	char* str = ldns_rdf2str(rdf);
	ldns_buffer_write_string_at(buf, *at, str);
	(*at) += strlen(str);
	free(str);
	ldns_buffer_write_string_at(buf, *at, sp);
	(*at) += 1;
}

/** Write trusted keys content */
int
bind_write_trusted_keys(tp_set_t* anchors, char* file, ldns_buffer* buf,
	int* at)
{
        /* trust point data */
	tp_t* tp = NULL;
	/* trust point node */
	rbnode_t* node = NULL;
	/* tree walking */
	size_t i;
	/* write a newline at the beginning of the trusted-keys clause */
	ldns_buffer_write_string_at(buf, *at, "\n");
	(*at) += 1;
	/* write trusted keys */
	if (anchors)
		node = rbt_first(anchors->sep);
	while (node)
	{
		tp = (tp_t*) node->key;

		if (my_strcmp((const char*) file, tp->file))
		{
			debug(("file %s not equal to trust point file %s", file, tp->file));
			node = rbt_successor(node);
			continue;
		}

		debug(("update trust point %s (%u anchors)", tp->name, tp->count));

		for (i=0; i < tp->count; i++)
		{
			ldns_rdf* rdf;
			ldns_rr* rr = tp->anchorlist[i]->rr;
			char* keytag;

			/* only put DNSKEYS in bind trusted-keys clause */
			if (ldns_rr_get_type(rr) != LDNS_RR_TYPE_DNSKEY)
				continue;

			/* only put trust anchors that resolver may use */
			if (get_trustanchor_state(tp->anchorlist[i]) !=
								STATE_VALID &&
				get_trustanchor_state(tp->anchorlist[i]) !=
								STATE_MISSING)
				continue;

			/* only put ZSKs if no KSKs are known */
			if (!rr_is_dnskey_sep(tp->anchorlist[i]->rr)  &&
						(tp->valid || tp->missing))
				continue;

			/* 100 is plenty enough, but should be calculated */
			keytag = (char*) malloc(sizeof(char)*100);
                        /* print dname */
			rdf = ldns_rr_owner(rr);
			bind_rdf2buf_str_free(buf, rdf, at, "\t");
			/* print flags */
			rdf = ldns_rr_dnskey_flags(rr);
			bind_rdf2buf_str_free(buf, rdf, at, " ");
			/* print protocol */
			rdf = ldns_rr_dnskey_protocol(rr);
			bind_rdf2buf_str_free(buf, rdf, at, " ");
			/* print algorithm */
			rdf = ldns_rr_dnskey_algorithm(rr);
			bind_rdf2buf_str_free(buf, rdf, at, " ");
			/* print key data */
			ldns_buffer_write_string_at(buf, *at, "\"");
			(*at) += 1;
			rdf = ldns_rr_dnskey_key(rr);
			bind_rdf2buf_str_free(buf, rdf, at, "\"");
			ldns_buffer_write_string_at(buf, *at, " ; ");
			(*at) += 3;
			/* write keytag */
			sprintf(keytag, "#{id = %d (ksk), size = %db}\n",
				ldns_calc_keytag(rr),
				ldns_rr_dnskey_key_size(rr));
				ldns_buffer_write_string_at(buf, *at, keytag);
			(*at) += strlen(keytag);
			free(keytag);
		}
		node = rbt_successor(node);
	}
	/* close the trusted-keys clause */
	ldns_buffer_write_string_at(buf, *at, "}");
	(*at) += 1;
	return 0; /* STATUS_OK */
}
