<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    optlist-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Handle command-line option arguments conveniently.      */
/*                                                                      */
/* ******************************************************************** */
/** @package utils */

// ----------------------------------------------------------------------
/**
* A class to make handling command line options a bit easier and more
* consistent.
* We support the following syntaxes:
*   programname --myoptname
*   programname --myoptname myvalue
*   programname --myoptname=myvalue
*   programname -t myvalue
*   programname --myoptname "myvalue words"
* In the first, only the presence of the option is required, as some kind
* of flag. The second has a value associated with the option, and the
* third is the same but using "=" as the separator. The fourth uses the
* old "-" prefix, and the fifth shows a string as the value.
* @package utils
*/
class optlist {
  // Public
  // Private
  /** Array of options
      @access private */
  var $opts = array();
  /** Number of options
      @access private */
  var $optcount = 0;
  /** The current program name
      @access private */
  var $progname = "";
  // .....................................................................
  /**
  * Instantiate the optlist object. This does all the work since we expect
  * that this is being done to process command line arguments/options.
  */
  function optlist() {
    global $argc, $argv;
    $this->progname = basename(array_shift($argv));
    $argc -= 1;
    if (isset($argc) && $argc > 0) {
      $optstr = implode(" ", $argv);
      $optstr = str_replace("--", "~", $optstr);
      $optstr = str_replace(" -", "~", $optstr);
      $opts = explode("~", $optstr);
      foreach ($opts as $opt) {
        if ($opt != "") {
          $pos = strpos($opt, "=");
          if ($pos > -1) {
            $optname = trim(substr($opt, 0, $pos));
            $optval  = trim(substr($opt, $pos + 1));
          }
          else {
            $pos = strpos($opt, " ");
            if ($pos > -1) {
              $optname = trim(substr($opt, 0, $pos));
              $optval  = trim(substr($opt, $pos + 1));
            }
            else {
              $optname = trim($opt);
              $optval = "";
            }
          }
          $optval = str_replace("\"", "", $optval);
          $optval = str_replace("'", "", $optval);
          $this->opts[$optname] = $optval;
        }
      }
      $this->optcount = count($this->opts);
    }
  } // optlist
  // .....................................................................
  /**
  * Return the value of the named option. Returns the string associated
  * with this option, or false if it does not exist.
  * @param string $optname Name of the option
  * @return mixed The string value of the option, or false if not valid
  */
  function opt_value($optname) {
    if (isset($this->opts[$optname])) {
      return $this->opts[$optname];
    }
    else {
      return false;
    }
  } // opt_value
  // .....................................................................
  /**
  * Return status of the named option. Returns true if the option exists
  * or false if it does not exist.
  * @param string $optname Name of the option
  * @return boolean True if it exists or false if it does not.
  */
  function opt_exists($optname) {
    return (isset($this->opts[$optname]));
  } // opt_exists
} // optlist class

// -----------------------------------------------------------------------
?>