/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "bakery_gnomeui/App/App_GnomeUI.h"
#include <algorithm>
#include <libgnome/gnome-i18n.h>
//#include <libgnomeui/gnome-stock/h>

namespace Bakery
{

//Initialize static member data:
Glib::ustring App_GnomeUI::m_strCommandLine_0;
Gnome::UI::About* App_GnomeUI::m_pAbout = 0;

App_GnomeUI::App_GnomeUI(const Glib::ustring& appname, const Glib::ustring& title)
 : Gnome::UI::App(appname, title)
{
  init_app_name(appname);

  set_resizable(); //resizable
  set_default_size(640, 400); //A sensible default.
 
  //Connect signals:

  //GnomeClient Session management:
  Gnome::UI::Client* pClient = Gnome::UI::Client::master_client(); //static method.
  if(pClient)
  {
    pClient->signal_die().connect(SigC::slot(*this, &App_GnomeUI::on_session_die));
    pClient->signal_save_yourself().connect(SigC::slot(*this, &App_GnomeUI::on_session_save_yourself));
  }
}

App_GnomeUI::~App_GnomeUI()
{
  //If this was the last instance:
  if(m_AppInstanceManager.get_app_count() == 0)
  {
    //Delete shared static widgets if this was the last instance:
    if(m_pAbout)
    {
      delete m_pAbout;
      m_pAbout = 0;
    }
  }
}


void App_GnomeUI::init()
{
  //set_wmclass(m_strAppName, m_strTitle); //The docs say "Don't use this".

  set_statusbar(m_Status);

  init_menus();
  init_toolbars();

  show_all();

  //on_document_load(); //Show the document (even if it is empty).
}

void App_GnomeUI::init_menus()
{
  init_menus_file();
  init_menus_edit();
  init_menus_help();

  create_menus(m_menu_UI_Infos);
  install_menu_hints();

  //Override this to add more menus.
}

void App_GnomeUI::init_toolbars()
{
  using namespace Gnome::UI::Items;

  m_toolbar_UI_Infos.push_back(Gnome::UI::Items::Item(Gnome::UI::Items::Icon(Gtk::Stock::NEW),
                         N_("New "),
                         SigC::slot(*this, &App_GnomeUI::on_menu_file_new),
                         N_("Create a new " + m_strAppName)));

  create_toolbar(m_toolbar_UI_Infos);
}


bool App_GnomeUI::on_delete_event(GdkEventAny* e)
{
  //Clicking on the [x] in the title bar should be like choosing File|New
  on_menu_file_close();

  return true; // true = don't hide, don't destroy
}

void App_GnomeUI::init_menus_file()
{
  // File menu
  type_vecGnome_UI_Info menu_file;

  //Build menu:
  menu_file.push_back(Gnome::UI::MenuItems::New(_("New Instance"), _("Create a new instance"), SigC::slot(*this, &App_GnomeUI::on_menu_file_new)));
  menu_file.push_back(Gnome::UI::MenuItems::Close(SigC::slot(*this, &App_GnomeUI::on_menu_file_close)));
  menu_file.push_back(Gnome::UI::MenuItems::Exit(SigC::slot(*this, &App_GnomeUI::on_menu_file_exit)));

  //Add menu:
  m_menu_UI_Infos.push_back(Gnome::UI::Menus::File(menu_file));
}

void App_GnomeUI::init_menus_edit()
{
  //Edit menu
  type_vecGnome_UI_Info menu_edit;

  //Build menu:
  menu_edit.push_back(Gnome::UI::MenuItems::Cut(SigC::slot(*this, &App_GnomeUI::on_menu_edit_cut)));
  menu_edit.push_back(Gnome::UI::MenuItems::Copy(SigC::slot(*this, &App_GnomeUI::on_menu_edit_copy)));
  menu_edit.push_back(Gnome::UI::MenuItems::Paste(SigC::slot(*this, &App_GnomeUI::on_menu_edit_paste)));
  menu_edit.push_back(Gnome::UI::MenuItems::Clear(SigC::slot(*this, &App_GnomeUI::on_menu_edit_clear)));

  //Add menu:
  m_menu_UI_Infos.push_back(Gnome::UI::Menus::Edit(menu_edit));
}

void App_GnomeUI::init_menus_help()
{
  //Help menu
  type_vecGnome_UI_Info menu_help;

  //Build menu:
  menu_help.push_back(Gnome::UI::Items::Help(m_strAppName));
  menu_help.push_back(Gnome::UI::MenuItems::About(SigC::slot(*this, &App_GnomeUI::on_menu_help_about)));

  //Add menu:
 
  m_menu_UI_Infos.push_back(Gnome::UI::Menus::Help(menu_help));
}



void App_GnomeUI::on_menu_help_about()
{
  if(m_pAbout && m_bAboutShown) // "About" box hasn't been closed, so just raise it
  {
    m_pAbout->set_transient_for(*this);

    Glib::RefPtr<Gdk::Window> about_win = m_pAbout->get_window();
    about_win->show();
    about_win->raise();
  }
  else
  {
    // not yet wrapped:
    Glib::RefPtr<Gdk::Pixbuf> refLogo = Gdk::Pixbuf::create_from_file("icon.png");

    //Re-create About box:
    if(m_pAbout)
    {
      delete m_pAbout;
      m_pAbout = 0;
    }
    m_pAbout = new Gnome::UI::About(m_strAppName, m_HelpInfo.m_strVersion,
                               m_HelpInfo.m_strCopyright,
                               m_HelpInfo.m_vecAuthors, m_HelpInfo.m_vecDocumenters,
                               m_HelpInfo.m_strDescription, m_HelpInfo.m_strTranslatorCredits,
                               refLogo);

    m_pAbout->set_transient_for(*this);
    m_pAbout->signal_hide().connect(SigC::slot(*this, &App_GnomeUI::on_about_close));
    m_bAboutShown = true;
    m_pAbout->show();
  }
}

void App_GnomeUI::on_about_close()
{
  m_bAboutShown = false;
}


void App_GnomeUI::on_session_die()
{
  //"I expect you to die, Mr. Bond."

  //TODO: Questions:
  //Do we need to handle this as well as the delete event?
  //Does GnomeClient want us to close this instance or all instances?
  on_menu_file_close();
}

bool App_GnomeUI::on_session_save_yourself(int phase, Gnome::UI::SaveStyle save_style, bool is_shutdown, Gnome::UI::InteractStyle interact_style, bool is_fast)
{
  //But let the women and children save themselves first.

  //TODO: Save command line to reload the same document via the command line
  //- Need to implement command line stuff.

  Gnome::UI::Client* pClient = Gnome::UI::Client::master_client(); //static method.
  if(pClient)
  {
    if(m_strCommandLine_0.size()) //Need to App_GnomeUI::call set_command_line_args().
    {
      std::vector<Glib::ustring> vecArgs;

      vecArgs.push_back(m_strCommandLine_0);

      pClient->set_clone_command (vecArgs);
      pClient->set_restart_command (vecArgs);

      //TODO: Open the same documents?
    }
  }

  return true;
}

void App_GnomeUI::set_command_line_args(int argc, char **&argv)
{
  if( (argc > 0) && argv[0])
    m_strCommandLine_0 = (char*)argv[0];
}


void App_GnomeUI::on_hide()
{
  ui_signal_hide().emit();
}

void App_GnomeUI::ui_hide()
{
  hide();
}

void App_GnomeUI::ui_bring_to_front()
{
  get_window()->raise();
}



} //namespace
