package org.basex.query.func;

import static org.basex.query.util.Err.*;

import org.basex.query.*;
import org.basex.query.expr.*;
import org.basex.query.iter.*;
import org.basex.query.util.*;
import org.basex.query.value.*;
import org.basex.query.value.item.*;
import org.basex.query.value.map.*;
import org.basex.query.value.node.*;
import org.basex.query.value.type.*;
import org.basex.util.*;

/**
 * Dynamic function call.
 *
 * @author BaseX Team 2005-12, BSD License
 * @author Leo Woerteler
 */
public final class DynamicFunc extends Arr {
  /**
   * Function constructor.
   * @param ii input info
   * @param fun function expression
   * @param arg arguments
   */
  public DynamicFunc(final InputInfo ii, final Expr fun, final Expr[] arg) {
    super(ii, Array.add(arg, fun));
  }

  @Override
  public Expr compile(final QueryContext ctx) throws QueryException {
    super.compile(ctx);
    final int ar = expr.length - 1;
    final Expr f = expr[ar];
    final Type t = f.type().type;
    if(t instanceof FuncType) {
      final FuncType ft = (FuncType) t;
      if(ft.args != null && ft.args.length != ar)
        throw INVARITY.thrw(info, f, ar);
      if(ft.ret != null) type = ft.ret;
    }
    // maps can only contain fully evaluated Values, so this is safe
    return allAreValues() && f instanceof Map ? optPre(value(ctx), ctx) : this;
  }

  @Override
  public Item item(final QueryContext ctx, final InputInfo ii) throws QueryException {
    return getFun(ctx).invItem(ctx, ii, argv(ctx));
  }

  @Override
  public Value value(final QueryContext ctx) throws QueryException {
    return getFun(ctx).invValue(ctx, info, argv(ctx));
  }

  @Override
  public Iter iter(final QueryContext ctx) throws QueryException {
    return getFun(ctx).invIter(ctx, info, argv(ctx));
  }

  /**
   * Evaluates all arguments.
   * @param ctx query context
   * @return array of argument values
   * @throws QueryException query exception
   */
  private Value[] argv(final QueryContext ctx) throws QueryException {
    final Value[] argv = new Value[expr.length - 1];
    for(int i = argv.length; --i >= 0;) argv[i] = ctx.value(expr[i]);
    return argv;
  }

  /**
   * Evaluates and checks the function item.
   * @param ctx query context
   * @return function item
   * @throws QueryException query exception
   */
  private FItem getFun(final QueryContext ctx) throws QueryException {
    final int ar = expr.length - 1;
    final Item it = checkItem(expr[ar], ctx);
    if(!it.type.isFunction()) throw Err.type(this, FuncType.arity(ar), it);
    final FItem fit = (FItem) it;
    if(fit.arity() != ar) throw INVARITY.thrw(info, fit, ar);
    return fit;
  }

  @Override
  public void plan(final FElem plan) {
    final FElem el = planElem();
    addPlan(plan, el, expr[expr.length - 1]);
    for(int i = 0; i < expr.length - 1; i++) expr[i].plan(el);
  }

  @Override
  public String description() {
    return expr[expr.length - 1].description() + "(...)";
  }

  @Override
  public String toString() {
    final TokenBuilder tb = new TokenBuilder(expr[expr.length - 1].toString()).add('(');
    for(int i = 0; i < expr.length - 1; i++) {
      tb.add(expr[i].toString());
      if(i < expr.length - 2) tb.add(", ");
    }
    return tb.add(')').toString();
  }
}
