/* branches.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "commands/cmd.h"
#include "commands/branches.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [[archive/]category]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) 

t_uchar arch_cmd_branches_help[] = N_("list the branches in an archive category\n"
                                    "Print a list of the branches within CATEGORY\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_branches (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_branches_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;
        }
    }

  if (argc > 2)
    goto usage_error;


  {
    t_uchar * cat_spec = 0;
    t_uchar * cat = 0;
    struct arch_archive * arch = 0;
    rel_table branches = 0;


    if (argc == 2)
      {
        cat_spec = str_save (0, argv[1]);
      }
    else
      {
        cat_spec = arch_try_tree_version (program_name);
      }

    arch = arch_archive_connect_branch (cat_spec, &cat);
    
    if (!arch)
      {
	safe_printfmt (2, _("Could not connect to archive %s\n"), cat_spec);
	exit (2);
      }
    
    if (!arch_valid_package_name (cat, arch_maybe_archive, arch_req_category, 1))
      {
	safe_printfmt (2, _("%s: invalid category name (%s)\n"),
		       argv[0], cat);
	exit (2);
      }

    cat = str_replace (cat, arch_parse_package_name (arch_ret_category, 0, cat));

    branches = arch_archive_branches (arch, cat);
    rel_print_table (1, branches);

    lim_free (0, cat_spec);
    lim_free (0, cat);
    arch_archive_close (arch);
  }

  return 0;
}




/* tag: Tom Lord Tue May 20 14:38:14 2003 (branches.c)
 */
