package de.fub.bytecode.classfile;

import  de.fub.bytecode.Constants;
import  java.io.*;

/**
 * This class represents a stack map entry recording the types of
 * local variables and the the of stack items at a given byte code offset.
 *
 * @version $Id: StackMapEntry.java,v 1.2 2001/08/02 12:44:54 dahm Exp $
 * @author  <A HREF="http://www.berlin.de/~markus.dahm/">M. Dahm</A>
 * @see     StackMap
 * @see     StackMapType
 */
public final class StackMapEntry implements Cloneable {
  private int            byte_code_offset;
  private int            number_of_locals;
  private StackMapType[] types_of_locals;
  private int            number_of_stack_items;
  private StackMapType[] types_of_stack_items;

  /**
   * Construct object from file stream.
   * @param file Input stream
   * @throw IOException
   */
  StackMapEntry(DataInputStream file) throws IOException
  {
    this(file.readShort(), file.readShort(), null, -1, null);

    types_of_locals = new StackMapType[number_of_locals];
    for(int i=0; i < number_of_locals; i++)
      types_of_locals[i] = new StackMapType(file);

    number_of_stack_items = file.readShort();
    types_of_stack_items = new StackMapType[number_of_stack_items];
    for(int i=0; i < number_of_stack_items; i++)
      types_of_stack_items[i] = new StackMapType(file);
  }

  public StackMapEntry(int byte_code_offset, int number_of_locals,
		       StackMapType[] types_of_locals,
		       int number_of_stack_items,
		       StackMapType[] types_of_stack_items) {
    this.byte_code_offset = byte_code_offset;
    this.number_of_locals = number_of_locals;
    this.types_of_locals = types_of_locals;
    this.number_of_stack_items = number_of_stack_items;
    this.types_of_stack_items = types_of_stack_items;
  }

  /**
   * Dump stack map entry
   *
   * @param file Output file stream
   * @throw IOException
   */ 
  public final void dump(DataOutputStream file) throws IOException
  {
    file.writeShort(byte_code_offset);

    file.writeShort(number_of_locals);
    for(int i=0; i < number_of_locals; i++)
      types_of_locals[i].dump(file);

    file.writeShort(number_of_stack_items);
    for(int i=0; i < number_of_stack_items; i++)
      types_of_stack_items[i].dump(file);
  }

  /**
   * @return String representation.
   */ 
  public final String toString() {
    StringBuffer buf = new StringBuffer("StackMapEntry(" + byte_code_offset +
					", " + number_of_locals + "{");

    for(int i=0; i < number_of_locals; i++) {
      buf.append(types_of_locals[i]);
      if(i < number_of_locals - 1)
	buf.append(", ");
    }

    buf.append("},");

    for(int i=0; i < number_of_stack_items; i++) {
      buf.append(types_of_stack_items[i]);
      if(i < number_of_stack_items - 1)
	buf.append(", ");
    }

    buf.append("})");

    return buf.toString();    
  }


  public void           setByteCodeOffset(int b)               { byte_code_offset = b; }
  public int            getByteCodeOffset()                    { return byte_code_offset; }
  public void           setNumberOfLocals(int n)               { number_of_locals = n; }
  public int            getNumberOfLocals()                    { return number_of_locals; }
  public void           setTypesOfLocals(StackMapType[] t)     { types_of_locals = t; }
  public StackMapType[] getTypesOfLocals()                     { return types_of_locals; }
  public void           setNumberOfStackItems(int n)           { number_of_stack_items = n; }
  public int            getNumberOfStackItems()                { return number_of_stack_items; }
  public void           setTypesOfStackItems(StackMapType[] t) { types_of_stack_items = t; }
  public StackMapType[] getTypesOfStackItems()                 { return types_of_stack_items; }

  /**
   * @return deep copy of this object
   */
  public StackMapEntry copy() {
    try {
      return (StackMapEntry)clone();
    } catch(CloneNotSupportedException e) {}

    return null;
  }
}
