;*=====================================================================*/
;*    serrano/prgm/project/bigloo/api/pthread/src/Llib/pthread.scm     */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Mon Feb  4 11:49:11 2002                          */
;*    Last change :  Mon Nov  7 05:28:22 2005 (serrano)                */
;*    Copyright   :  2002-05 Manuel Serrano                            */
;*    -------------------------------------------------------------    */
;*    The public Posix Thread implementation.                          */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __pth_thread

   (option  (set! *dlopen-init* #t))
   
   (include "pthread.sch")
   
   (export (abstract-class thread
	      ;; the name of the thread
	      (name read-only)
	      ;; the result of the thread
	      (end-result (default #unspecified))
	      ;; field for storing uncaught exceptions
	      (end-exception (default #unspecified)))

	   (class pthread::thread
	      (%setup-pthread!)
	      ;; the user thunk
	      (body::procedure read-only)
	      ;; the actual native thread
	      ($builtin::$pthread (default ($pthread-nil)))
	      ;; is the thread detached
	      (detachedp::bool (default #f)))

	   (class &thread-error::&error)
	   
	   (class uncaught-exception::&exception
	      (reason::obj read-only))

	   (class terminated-thread-exception::&exception))
   
   (export (%setup-pthread! ::pthread)
	   (make-thread ::procedure . ::obj)
	   (thread-start! ::pthread)
	   (thread-start-joinable! ::pthread)
	   (thread-yield!)
	   (thread-sleep! ::date)
	   (thread-join! ::pthread)
	   (thread-terminate! ::pthread)
	   (current-thread::obj)
	   (thread-specific::obj ::pthread)
	   (thread-specific-set!::obj ::pthread ::obj)
	   (thread-cleanup::obj ::pthread)
	   (thread-cleanup-set!::obj ::pthread ::procedure)))

;*---------------------------------------------------------------------*/
;*    object-display ::pthread ...                                     */
;*---------------------------------------------------------------------*/
(define-method (object-display o::pthread . port)
   (with-output-to-port (if (pair? port) (car port) (current-output-port))
      (lambda ()
         (with-access::pthread o (name)
            (display* "#<thread:" name ">")))))

;*---------------------------------------------------------------------*/
;*    object-write ::pthread ...                                       */
;*---------------------------------------------------------------------*/
(define-method (object-write o::pthread . port)
   (with-output-to-port (if (pair? port) (car port) (current-output-port))
      (lambda ()
         (with-access::pthread o (name)
            (display* "#<thread:" name ">")))))

;*---------------------------------------------------------------------*/
;*    object-print ::pthread ...                                       */
;*---------------------------------------------------------------------*/
(define-method (object-print o::pthread port print-slot)
   (object-write o port))

;*---------------------------------------------------------------------*/
;*    %setup-pthread! ...                                              */
;*---------------------------------------------------------------------*/
(define (%setup-pthread! o::pthread)
   (if (not (bigloo-initialized?))
       (error 'make-thread
	      "Threads cannot be created until modules are initialized (see the documentation)"
	      o)
       (with-access::pthread o ($builtin body end-result end-exception name)
	  (let ((b (lambda ()
		      (let ((id (if (symbol? name)
				    (symbol-append '& name)
				    (gensym '&thread-))))
			 (let ()
			    ($push-trace id)
			    ($set-uncaught-exception-handler!
			     (lambda (val)
				(error (format "unwind-until!:~a" o)
				       "exit out of dynamic scope"
				       val)))
			    (bind-exit (terminate)
			       (with-exception-handler
				  (lambda (e)
				     (let ((u (instantiate::uncaught-exception
						 (reason e))))
					(set! end-exception  u)
					(exception-notify e)
					(terminate #f)))
				  (lambda ()
				     (set! end-result (body))))))))))
	     (set! $builtin ($pthread-new b))))))
    
;*---------------------------------------------------------------------*/
;*    object-write ::pthread ...                                       */
;*---------------------------------------------------------------------*/
(define-method (object-write o::pthread . port)
   (with-output-to-port (if (pair? port) (car port) (current-output-port))
      (lambda ()
         (with-access::pthread o (name detachedp)
            (display* "#<thread:" name "," detachedp ">")))))

;*---------------------------------------------------------------------*/
;*    make-thread ...                                                  */
;*---------------------------------------------------------------------*/
(define (make-thread thunk . obj)
   (letrec ((th (instantiate::pthread
		   (body thunk)
		   (name (if (pair? obj) (car obj) (gensym 'pthread))))))
      th))

;*---------------------------------------------------------------------*/
;*    thread-start! ...                                                */
;*---------------------------------------------------------------------*/
(define (thread-start! t)
   (pthread-detachedp-set! t #t)
   ($pthread-start! (pthread-$builtin t) t #t)
   t)

;*---------------------------------------------------------------------*/
;*    thread-start-joinable! ...                                       */
;*---------------------------------------------------------------------*/
(define (thread-start-joinable! t)
   (pthread-detachedp-set! t #f)
   ($pthread-start! (pthread-$builtin t) t #f)
   t)

;*---------------------------------------------------------------------*/
;*    thread-yield! ...                                                */
;*---------------------------------------------------------------------*/
(define (thread-yield!)
   ($pthread-sched-yield))

;*---------------------------------------------------------------------*/
;*    thread-sleep! ...                                                */
;*---------------------------------------------------------------------*/
(define (thread-sleep! d)
   (let* ((cdt (date->seconds (current-date)))
	  (dt (date->seconds d))
	  (a (-second dt cdt)))
      (if (>second a #e0)
	  (sleep (elong->fixnum a)))))
 
;*---------------------------------------------------------------------*/
;*    thread-join! ...                                                 */
;*---------------------------------------------------------------------*/
(define (thread-join! t)
   (if (pthread-detachedp t)
       (raise (instantiate::&thread-error
		 (proc 'thread-join!)
		 (msg "detatched thread")
		 (obj t)))
       (with-access::pthread t ($builtin end-result end-exception)
	  ($pthread-join! $builtin)
	  (if (&exception? end-exception)
	      (raise end-exception)
	      end-result))))

;*---------------------------------------------------------------------*/
;*    thread-terminate! ...                                            */
;*---------------------------------------------------------------------*/
(define (thread-terminate! t)
   (with-access::pthread t ($builtin end-exception)
      (when ($pthread-terminate! $builtin)
	 (set! end-exception (instantiate::terminated-thread-exception)))
      t))

;*---------------------------------------------------------------------*/
;*    current-thread ...                                               */
;*---------------------------------------------------------------------*/
(define (current-thread)
   ($pthread-current-thread))

;*---------------------------------------------------------------------*/
;*    thread-specific ...                                              */
;*---------------------------------------------------------------------*/
(define (thread-specific t)
   ($pthread-specific (pthread-$builtin t)))

;*---------------------------------------------------------------------*/
;*    thread-specific-set! ...                                         */
;*---------------------------------------------------------------------*/
(define (thread-specific-set! t v)
   ($pthread-specific-set! (pthread-$builtin t) v)
   v)

;*---------------------------------------------------------------------*/
;*    thread-cleanup ...                                               */
;*---------------------------------------------------------------------*/
(define (thread-cleanup t)
   ($pthread-cleanup (pthread-$builtin t)))

;*---------------------------------------------------------------------*/
;*    thread-cleanup-set! ...                                          */
;*---------------------------------------------------------------------*/
(define (thread-cleanup-set! t p)
   (if (correct-arity? p 1)
       (begin
	  ($pthread-cleanup-set! (pthread-$builtin t) p)
	  p)
       (error 'thread-cleanup-set! "Illegal procedure arity" p)))
