
/* to avoid parse error before `__extension__' in <linux/string.h> included by
 * <linux/genhd.h> (conflicts with <bits/string2.h> on sparc at least).
 */
#define __NO_STRING_INLINES

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <linux/unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <mntent.h>
#include <assert.h>
#include <syslog.h>
#include <linux/cdrom.h>
#include <linux/major.h>
#include <linux/genhd.h>

#include "fdisk.h"

#define SIZE(a) (sizeof(a)/sizeof(a[0]))

/*
 * XXX: Some 2.2.x kernel headers (ie. libc6-dev) do
 * not contain this for some reason -- Ben
 */
#ifndef MAX_DISKNAME_LEN
#define MAX_DISKNAME_LEN PATH_MAX
#endif

/* this array translates partition type numbers to file system types */
static struct fstypes {
    int ptype;
    int fstype;
} ptype2fstype[] = {
#if HAVE_MSDOS_PARTITION
    /* PC */
    { PTYPE_DOS12,		FSTYPE_MSDOS },
    { PTYPE_DOS16S,		FSTYPE_MSDOS },
    { PTYPE_DOSEXT,		FSTYPE_EXTPART },
    { PTYPE_DOS16L,		FSTYPE_MSDOS },
    { PTYPE_NTFS, 		FSTYPE_MSDOS },
    { PTYPE_WFAT32,		FSTYPE_MSDOS },
    { PTYPE_WFAT32L,		FSTYPE_MSDOS },
    { PTYPE_WEXT,		FSTYPE_EXTPART },
    { PTYPE_MINIX,		FSTYPE_MINIX },
    { PTYPE_LSWAP,		FSTYPE_SWAP },
    { PTYPE_LINUX,		FSTYPE_EXT2 },
    { PTYPE_LINUXEXT,		FSTYPE_EXTPART },
    { PTYPE_GPT,		FSTYPE_UNKNOWN },
    { PTYPE_EFI,		FSTYPE_MSDOS },
#endif /* HAVE_MSDOS_PARTITION */
#if HAVE_BSD_DISKLABEL
    { PTYPE_BSD_MSDOS,		FSTYPE_MSDOS },
    { PTYPE_BSD_ADOS,		FSTYPE_AFFS },
    { PTYPE_BSD_HFS,		FSTYPE_HFS },
#endif /* HAVE_BSD_PARTITION */
#if HAVE_SOLARIS_X86_PARTITION
    /* only type PTYPE_SOLARIS_X86 not mountable */
#endif /* HAVE_SOLARIS_X86_PARTITION */
#if HAVE_OSF_PARTITION
    { PTYPE_OSF_SWAP,		FSTYPE_SWAP },
    { PTYPE_OSF_EXT2,		FSTYPE_EXT2 },
    { PTYPE_OSF_OTHER,		FSTYPE_MSDOS },
    { PTYPE_OSF_ADOS,		FSTYPE_AFFS },
    { PTYPE_OSF_HFS,		FSTYPE_HFS },
#endif /* HAVE_OSF_PARTITION */
#if HAVE_SUN_PARTITION
    { PTYPE_SUN_L_MINIX,	FSTYPE_MINIX },
    { PTYPE_SUN_L_SWAP,		FSTYPE_SWAP },
    { PTYPE_SUN_L_NATIVE,	FSTYPE_EXT2 },
    /* other partitions would be ufs, but that's not supported yet */
#endif /* HAVE_SUN_PARTITION */
#if HAVE_AMIGA_PARTITION
    { PTYPE_AMIGA_OFS,		FSTYPE_AFFS },
    { PTYPE_AMIGA_FFS,		FSTYPE_AFFS },
    { PTYPE_AMIGA_OFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_FFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_OFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_FFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MUFS,		FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_OFS,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_FFS,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_OFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_FFS_INTL,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_OFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_MU_FFS_DC,	FSTYPE_AFFS },
    { PTYPE_AMIGA_EXT2,		FSTYPE_EXT2 },
    { PTYPE_AMIGA_LINUX,	FSTYPE_EXT2 },
    { PTYPE_AMIGA_SWAP,		FSTYPE_SWAP },
    { PTYPE_AMIGA_SWP,		FSTYPE_SWAP },
    { PTYPE_AMIGA_MINIX,	FSTYPE_MINIX },
#endif /* HAVE_AMIGA_PARTITION */
#if HAVE_ATARI_PARTITION
    /* Atari */
    { PTYPE_GEMDOSS,		FSTYPE_MSDOS },
    { PTYPE_GEMDOSL,		FSTYPE_MSDOS },
    { PTYPE_ATARI_LINUX,	FSTYPE_EXT2 },
    { PTYPE_ATARI_SWAP,		FSTYPE_SWAP },
    { PTYPE_ATARI_MINIX1,	FSTYPE_MINIX },
    { PTYPE_ATARI_MINIX2,	FSTYPE_MINIX },
    { PTYPE_ATARI_HFS,		FSTYPE_HFS },
#endif /* HAVE_ATARI_PARTITION */
#if HAVE_MAC_PARTITION
    { PTYPE_MAC_HFS,		FSTYPE_HFS },
    { PTYPE_MAC_SCRATCH,	FSTYPE_SWAP },
/*  { PTYPE_MAC_PRODOS,		FSTYPE_MSDOS }, (??) */
    { PTYPE_MAC_SWAP,		FSTYPE_SWAP },
    { PTYPE_MAC_MSDOS,		FSTYPE_MSDOS },
    { PTYPE_MAC_MINIX,		FSTYPE_MINIX },
    { PTYPE_MAC_AFFS,		FSTYPE_AFFS },
    { PTYPE_MAC_EXT2,		FSTYPE_EXT2 },
    /* some people used partitions with names "A/UX ..." for Linux in the past
     * make them the transition easier :-) */
    { PTYPE_MAC_AUX,		FSTYPE_EXT2 },
#endif /* HAVE_MAC_PARTITION */
#if HAVE_ACORN_PARTITION
    { PTYPE_ACORN_ADFS,		FSTYPE_ADFS },
    { PTYPE_ACORN_EXT2,		FSTYPE_EXT2 },
    { PTYPE_ACORN_SWAP,		FSTYPE_SWAP },
    { PTYPE_ACORN_ICS_ADFS,	FSTYPE_ADFS },
    { PTYPE_ACORN_ICS_EXT2,	FSTYPE_EXT2 },
    { PTYPE_ACORN_ICS_SWAP,	FSTYPE_SWAP },
#endif /* HAVE_ACORN_PARTITION */
#if HAVE_SGI_PARTITION
    { PTYPE_SGI_L_SWAP,         FSTYPE_SWAP },
    { PTYPE_SGI_L_NATIVE,       FSTYPE_EXT2 },
    { PTYPE_SGI_VOLHDR,         FSTYPE_UNKNOWN },
    { PTYPE_SGI_VOL,            FSTYPE_UNKNOWN },
#endif /* HAVE_SGI_PARTITION */
#if HAVE_IBM_PARTITION
    { PTYPE_IBM_L_SWAP,         FSTYPE_SWAP },
    { PTYPE_IBM_L_NATIVE,       FSTYPE_EXT2 },
    { PTYPE_IBM_LINUX,          FSTYPE_UNKNOWN },
#endif /* HAVE_IBM_PARTITION */
#if HAVE_LOOP_PARTITION
    /* Fake partition types for loop mounts */
    { PTYPE_LOOP_EXT2,		FSTYPE_EXT2 },
    { PTYPE_LOOP_SWAP,		FSTYPE_SWAP },
    { PTYPE_LOOP_MSDOS,		FSTYPE_MSDOS },
    { PTYPE_LOOP_MINIX,		FSTYPE_MINIX },
    { PTYPE_LOOP_AFFS,		FSTYPE_AFFS },
    { PTYPE_LOOP_HFS,		FSTYPE_HFS },
#endif /* HAVE_LOOP_PARTITION */
    /* Fake partition type for NFS mounts */
    { PTYPE_NFS,		FSTYPE_NFS },
};

/*
 * List of system Id's, copied from sfdisk 3.05, itself adapted
 * from fdisk 2.0d and <linux/genhd.h> and SFS and several other sources.
 */

static struct systypes {
    unsigned int type;
    char *name;
} sys_types[] = {
#if HAVE_MSDOS_PARTITION
    /* PC */
    {0, "Empty"},
    {1, "DOS 12-bit FAT"},
    {2, "XENIX root"},
    {3, "XENIX usr"},
    {4, "DOS 16-bit FAT <32M"},
    {5, "DOS Extended"},                /* DOS 3.3+ extended partition */
    {6, "DOS 16-bit FAT >=32M"},
    {7, "HPFS / NTFS"},
    {8, "AIX boot or SplitDrive"},
    {9, "AIX data or Coherent"},
    {0x0a, "OS/2 Boot Manager"},
    {0x0b, "Win95 FAT32"},
    {0x0c, "Win95 FAT32 (LBA)"},
    {0x0e, "Win95 FAT16 (LBA)"},
    {0x0f, "Win95 Extended (LBA)"},
    {0x10, "OPUS"},
    {0x11, "Hidden DOS FAT12"},
    {0x12, "Compaq diagnostics"},
    {0x14, "Hidden FAT16 <32M"},
    {0x16, "Hidden FAT16"},
    {0x17, "Hidden HPFS/NTFS"},
    {0x18, "AST Windows SmartSleep"},
    {0x1b, "Hidden Win95 FAT32"},
    {0x1c, "Hidden Win95 FAT32 (LBA)"},
    {0x1e, "Hidden Win95 FAT16 (LBA)"},
    {0x24, "NEC DOS"},
    {0x39, "Plan 9"},
    {0x3c, "PartitionMagic recovery"},
    {0x40, "Venix 80286"},
    {0x41, "PPC PReP Boot"},
    {0x42, "Windows 2000 Dynamic Partition (Formerly SFS)"},
    {0x43, "Linux native (sharing disk with DRDOS)"},
    {0x4d, "QNX4.x"},
    {0x4e, "QNX4.x 2nd part"},
    {0x4f, "QNX4.x 3rd part"},
    {0x50, "OnTrack Disk Manager"},
    {0x51, "OnTrack DM6 Aux1 (or novell)"},
    {0x52, "CP/M or Microport SysV/AT"},
    {0x53, "OnTrack DM6 Aux3"},
    {0x54, "OnTrack DM6"},
    {0x55, "EZ-Drive disk manager"},
    {0x56, "Golden Bow disk manager"},
    {0x5c, "Priam Edisk disk manager"},
    {0x61, "SpeedStor"},
    {0x63, "GNU HURD or Mach or Sys V/386 (such as ISC UNIX)"},
    {0x64, "Novell Netware 286"},
    {0x65, "Novell Netware 386"},
    {0x70, "DiskSecure Multi-Boot"},
    {0x75, "PC/IX"},
    {0x80, "MINIX prior to 1.4a"},
    {0x81, "MINIX 1.4b or later / old Linux"},
    {0x82, "Linux swap"},
    {0x83, "Linux native"},
    {0x84, "OS/2 hidden C: drive"},
    {0x85, "Linux extended"},
    {0x86, "NTFS volume set"},
    {0x87, "NTFS volume set"},
    {0x8e, "Linux LVM"},
    {0x93, "Amoeba"},
    {0x94, "Amoeba Bad Block Table"},               /* (bad block table) */
    {0x9f, "BSD/OS"},		/* BSDI */
    {0xa0, "IBM Thinkpad hibernation"}, /* according to dan@fch.wimsey.bc.ca */
    {0xa5, "BSD/386"},                  /* 386BSD */
    {0xa6, "OpenBSD"},
    {0xa7, "NeXTSTEP"},
    {0xb7, "BSDI fs"},
    {0xb8, "BSDI swap"},
    {0xbb, "Boot Wizard hidden"},
    {0xc1, "DRDOS/sec (FAT-12)"},
    {0xc4, "DRDOS/sec (FAT-16) < 32M"},
    {0xc6, "DRDOS/sec (FAT-16)"},
    {0xc7, "Syrinx"},
    {0xda, "Non-FS data"},
    {0xdb, "CP/M or Concurrent CP/M or Concurrent DOS or CTOS"},
    {0xde, "Dell Utility"},
    {0xdf, "BootIt"},
    {0xe1, "DOS access or SpeedStor 12-bit FAT extended partition"},
    {0xe3, "DOS R/O or SpeedStor"},
    {0xe4, "SpeedStor 16-bit FAT extended partition < 1024 cyl."},
    {0xeb, "BeOS fs"},
    {0xee, "Intel EFI GUID Partition Table"},
    {0xef, "Intel EFI System Partition (FAT-12/16/32)"},
    {0xf0, "Linux/PA-RISC boot"},
    {0xf1, "SpeedStor"},
    {0xf2, "DOS 3.3+ secondary"},
    {0xf4, "SpeedStor large partition"},
    {0xfd, "Linux RAID (autodetect)"},
    {0xfe, "SpeedStor >1024 cyl. or LANstep"},
    {0xff, "Xenix Bad Block Table"},
#endif /* HAVE_MSDOS_PARTITION */
#if HAVE_BSD_DISKLABEL
    { PTYPE_BSD_SWAP, 	"BSD swap" },
    { PTYPE_BSD_V6,	"BSD version 6" },
    { PTYPE_BSD_V7,	"BSD version 7" },
    { PTYPE_BSD_SYSV,	"BSD--System V" },
    { PTYPE_BSD_V71K,	"4.1BSD" },
    { PTYPE_BSD_V8,	"8th edition BSD" },
    { PTYPE_BSD_BSDFFS,	"4.2BSD" },
    { PTYPE_BSD_MSDOS,	"BSD--MSDOS" },
    { PTYPE_BSD_BSDLFS,	"4.4BSD LFS" },
    { PTYPE_BSD_OTHER,	"BSD--unknown" },
    { PTYPE_BSD_HPFS,	"BSD--OS/2 HPFS" },
    { PTYPE_BSD_ISOFS,	"BSD--iso9660" },
    { PTYPE_BSD_BOOT,	"BSD boot" },
    { PTYPE_BSD_ADOS,	"BSD--Amiga AFFS" },
    { PTYPE_BSD_HFS,	"BSD--Apple HFS" },
#endif /* HAVE_BSD_DISKLABEL */
#if HAVE_SOLARIS_X86_PARTITION
    { PTYPE_SOLARIS_X86,"Solaris/x86" },
#endif /* HAVE_SOLARIS_X86_PARTITION */
#if HAVE_OSF_PARTITION
    { PTYPE_OSF_SWAP, 	"swap" },
    { PTYPE_OSF_V6,	"BSD version 6" },
    { PTYPE_OSF_V7,	"BSD version 7" },
    { PTYPE_OSF_SYSV,	"System V" },
    { PTYPE_OSF_V71K,	"4.1BSD" },
    { PTYPE_OSF_V8,	"8th edition BSD" },
    { PTYPE_OSF_BSDFFS,	"4.2BSD" },
    { PTYPE_OSF_EXT2,	"Linux native" },
    { PTYPE_OSF_BSDLFS,	"4.4BSD LFS" },
    { PTYPE_OSF_OTHER,	"unknown (msdos?)" },
    { PTYPE_OSF_HPFS,	"OS/2 HPFS" },
    { PTYPE_OSF_ISOFS,	"iso9660" },
    { PTYPE_OSF_BOOT,	"boot" },
    { PTYPE_OSF_ADOS,	"Amiga AFFS" },
    { PTYPE_OSF_HFS,	"Apple HFS" },
    { PTYPE_OSF_ADVFS,	"Digital AdvFS" },
#endif /* HAVE_OSF_PARTITION */
#if HAVE_SUN_PARTITION
    { PTYPE_SUN_EMPTY,		"empty" },
    { PTYPE_SUN_BOOT,		"Boot" },
    { PTYPE_SUN_S_ROOT,		"SunOS root" },
    { PTYPE_SUN_S_SWAP,		"SunOS swap" },
    { PTYPE_SUN_S_USR,		"SunOS usr" },
    { PTYPE_SUN_WHOLE,		"Whole disk" },
    { PTYPE_SUN_S_STAND,	"SunOS stand" },
    { PTYPE_SUN_S_VAR,		"SunOS var" },
    { PTYPE_SUN_S_HOME,		"SunOS home" },
    { PTYPE_SUN_L_MINIX,	"Linux minix" },
    { PTYPE_SUN_L_SWAP,		"Linux swap" },
    { PTYPE_SUN_L_NATIVE,	"Linux native" },
#endif /* HAVE_SUN_PARTITION */
#if HAVE_AMIGA_PARTITION
    { PTYPE_AMIGA_BOOT,		"generic boot" },
    { PTYPE_AMIGA_OFS,		"OFS" },
    { PTYPE_AMIGA_FFS,		"FFS" },
    { PTYPE_AMIGA_OFS_INTL,	"OFS INTL" },
    { PTYPE_AMIGA_FFS_INTL,	"FFS INTL" },
    { PTYPE_AMIGA_OFS_DC,	"OFS DC" },
    { PTYPE_AMIGA_FFS_DC,	"FFS DC" },
    { PTYPE_AMIGA_MUFS,		"MU FFS INTL" },
    { PTYPE_AMIGA_MU_OFS,	"MU OFS" },   
    { PTYPE_AMIGA_MU_FFS,	"MU FFS" },   
    { PTYPE_AMIGA_MU_OFS_INTL,	"MU OFS INTL" },
    { PTYPE_AMIGA_MU_FFS_INTL,	"MU FFS INTL" },
    { PTYPE_AMIGA_MU_OFS_DC,	"MU OFS DC" },
    { PTYPE_AMIGA_MU_FFS_DC,	"MU FFS DC" },
    { PTYPE_AMIGA_LINUX,	"Linux native" },
    { PTYPE_AMIGA_EXT2,		"Linux ext2" },
    { PTYPE_AMIGA_SWAP,		"Swap" },
    { PTYPE_AMIGA_SWP,		"Swap" },
    { PTYPE_AMIGA_AMIX0,	"Amix \\0" },
    { PTYPE_AMIGA_AMIX1,	"Amix \\1" },
    { PTYPE_AMIGA_NETBSD_ROOT,	"NetBSD root" },
    { PTYPE_AMIGA_NETBSD_SWAP,	"NetBSD swap" },
    { PTYPE_AMIGA_NETBSD_OTH,	"NetBSD other" },
    { PTYPE_AMIGA_PFS0,		"Professional FS" },
    { PTYPE_AMIGA_PFS1,		"Professional FS" },
    { PTYPE_AMIGA_AFS0,		"AmiFile Safe" },
    { PTYPE_AMIGA_AFS1,		"AmiFile Safe (exp.)" },
#endif /* HAVE_AMIGA_PARTITION */
#if HAVE_ATARI_PARTITION
    /* Atari */
    { PTYPE_GEMDOSS,		"GEMDOS (< 32M)" },
    { PTYPE_GEMDOSL,		"GEMDOS (> 32M)" },
    { PTYPE_ATARI_LINUX,	"Linux" },
    { PTYPE_ATARI_SWAP,		"Swap" },
    { PTYPE_ATARI_MINIX1,	"Minix" },
    { PTYPE_ATARI_MINIX2,	"Minix" },
    { PTYPE_ATARI_HFS,		"Mac HFS" },
    { PTYPE_ATARI_SYSV,		"Atari SysV Unix" },
#endif /* HAVE_ATARI_PARTITION */
#if HAVE_MAC_PARTITION
    { PTYPE_MAC_PMAP,		"partition map" },
    { PTYPE_MAC_DRIVER,		"MacOS Driver" },
    { PTYPE_MAC_DRIVER43,	"MacOS Driver 4.3" },
    { PTYPE_MAC_ATADRIVER,      "MacOS ATA Driver" },
    { PTYPE_MAC_FWDRIVER,       "MacOS FW Driver" },
    { PTYPE_MAC_IODRIVER,       "MacOS IOKit" },
    { PTYPE_MAC_PATCH,          "MacOS Patches" },
    { PTYPE_MAC_OSXBOOT,        "MacOSX bootloader" },
    { PTYPE_MAC_OSXLOADER,      "MacOSX loader" },
    { PTYPE_MAC_UFS,            "UFS" },
    { PTYPE_MAC_HFS,		"HFS" },
    { PTYPE_MAC_MFS,		"MFS" },
    { PTYPE_MAC_SCRATCH,	"Scratch" },
    { PTYPE_MAC_PRODOS,		"ProDOS" },
    { PTYPE_MAC_FREE,		"Free" },
    { PTYPE_MAC_SWAP,		"Linux swap" },
    { PTYPE_MAC_AUX,		"A/UX" },
    { PTYPE_MAC_MSDOS,		"MS-DOS" },
    { PTYPE_MAC_MINIX,		"Minix" },
    { PTYPE_MAC_AFFS,		"Amiga AFFS" },
    { PTYPE_MAC_EXT2,		"Linux native" },
    { PTYPE_MAC_BOOT,		"NewWorld bootblock" },
#endif /* HAVE_MAC_PARTITION */
#if HAVE_ACORN_PARTITION
    { PTYPE_ACORN_LINUX,	"ARM Linux Partition Map" },
    { PTYPE_ACORN_ADFS,		"ADFS" },
    { PTYPE_ACORN_EXT2,		"Linux ext2 (ARM Linux)" },
    { PTYPE_ACORN_SWAP,		"Linux swap (ARM Linux)" },
    { PTYPE_ACORN_ICS_ADFS,	"ADFS (ICS/APDL)" },
    { PTYPE_ACORN_ICS_EXT2,	"Linux ext2 (ICS/APDL)" },
    { PTYPE_ACORN_ICS_SWAP,	"Linux swap (ICS/APDL)" },
#endif /* HAVE_ACORN_PARTITION */
#if HAVE_SGI_PARTITION
    { PTYPE_SGI_VOLHDR,          "SGI volhdr" },
    { PTYPE_SGI_TREPL,           "SGI trkrepl" },
    { PTYPE_SGI_SREPL,           "SGI secrepl" },   
    { PTYPE_SGI_SWAP,            "SGI swap" },    
    { PTYPE_SGI_BSD,             "BSD " },   
    { PTYPE_SGI_SYSV,            "System V" },
    { PTYPE_SGI_VOL,             "SGI disk" },
    { PTYPE_SGI_EFS,             "SGI efs" },
    { PTYPE_SGI_LVOL,            "SGI lvol" },
    { PTYPE_SGI_RLVOL,           "SGI rlvol" },
    { PTYPE_SGI_XFS,             "SGI xfs" },
    { PTYPE_SGI_XLVOL,           "SGI xlvol" },
    { PTYPE_SGI_RXLVOL,          "SGI rxlvol" },
    { PTYPE_SGI_L_SWAP,          "Linux Swap" },
    { PTYPE_SGI_L_NATIVE,        "Linux native" },
#endif /* HAVE_SGI_PARTITION */
#if HAVE_IBM_PARTITION
    { PTYPE_IBM_L_SWAP,          "Linux Swap" },
    { PTYPE_IBM_L_NATIVE,        "Linux native" },
    { PTYPE_IBM_LINUX,           "Linux native or swap" },
#endif /* HAVE_IBM_PARTITION */
#if HAVE_LOOP_PARTITION 
    { PTYPE_LOOP_EXT2,		"Linux native (loop mounted)" },
    { PTYPE_LOOP_SWAP,		"Linux swap (loop mounted)" },
    { PTYPE_LOOP_MSDOS,		"MS-DOS (loop mounted)" },
    { PTYPE_LOOP_MINIX,		"Minix (loop mounted)" },
    { PTYPE_LOOP_AFFS,		"Amiga AFFS (loop mounted)" },
    { PTYPE_LOOP_HFS,		"Mac HFS (loop mounted)" },
#endif /* HAVE_LOOP_PARTITION */
    /* Fake partition type for NFS mounts */
    { PTYPE_NFS,		"NFS" },
};

char *fstype_name[FSTYPE_MAX] = {
    "ext2",
    "ext2", /* this will have to be changed when (if?) ReiserFS gets its own partition type */
    "swap",
    "msdos",
    "minix",
    "affs",
    "hfs",
    "adfs",
    "nfs",
    NULL, /* FSTYPE_EXTPART is not a valid filesystem, so return "UNKNOWN" */
    NULL /* FSTYPE_UNKNOWN */
};

#define SIZE(a) (sizeof(a)/sizeof(a[0]))

/* translate partition type number to description */
char *
fdisk_sysname(int type) {
    struct systypes *s;

    for( s = sys_types+SIZE(sys_types)-1; s >= sys_types; s-- )
      if (s->type == type)
        return s->name;
    return "Unknown";
}

struct fdisk_disk *fdisk_disks = NULL;

/* internal state */
static struct fdisk_disk *last_disk;
static struct fdisk_partition *last_partition;
static struct fdisk_partition *last_partitions_by_type[255];
static struct fdisk_partition *last_swapon_partition;

void fdisk_init() {
  int i;
  struct fdisk_disk *d1,*d2;
  struct fdisk_partition *p1,*p2;

  if (NULL != fdisk_disks) {
    d1 = fdisk_disks;
    while ( d1 ) {
      d2 = d1;
      d1 = d1->next;
      free (d2);
    }
    p1 = fdisk_partitions;
    while ( p1 ) {
      p2 = p1;
      p1 = p1->next;
      free (p2);
    }
  }

  for (i=0; i<FSTYPE_MAX; i++) {
    fdisk_partitions_by_type[i] = NULL;
    last_partitions_by_type[i] = NULL;
  }

  fdisk_disks = last_disk = NULL;
  fdisk_partitions = last_partition = NULL;
}


struct fdisk_disk *
fdisk_add_disk(const char *name, unsigned long size){
  struct fdisk_disk *disk;
  
  disk = (struct fdisk_disk *) malloc(sizeof(struct fdisk_disk));
  if (!disk) return NULL;
  
  disk->name = strdup(name);
  disk->size = size;
  disk->next = NULL;
  disk->partitions = NULL;
  disk->last_partition = NULL;
  disk->partition_format = NULL;

  if (NULL == fdisk_disks) {
    fdisk_disks = disk;
  }

  if (NULL != last_disk) {
    last_disk->next = disk;
  }
  last_disk = disk;


#ifdef DEBUG
  printf( "added disk %s, size %ld\n", name, size );
#endif

  return(disk);
}

int
fdisk_fstype_of(unsigned int ptype)
{
    struct fstypes *p;
    
    for( p = ptype2fstype+SIZE(ptype2fstype)-1; p >= ptype2fstype; p-- ) {
	if (ptype == p->ptype)
	return p->fstype;
    }
    return FSTYPE_UNKNOWN;
}

char *
fdisk_fstype_name_of(unsigned int ptype)
{
    int fstype = fdisk_fstype_of(ptype);
    return fstype_name[fstype];
}


/* get_part_info: read and parse line in /proc/partitions format */
static int get_part_info(FILE *f, dev_t *dev, char *name)
{
    char line[PATH_MAX];
    unsigned int major, minor;

    while (fgets(line, sizeof line, f)) {
	if (sscanf(line, "%4u  %4u %*u %s", &major, &minor, name) == 3) {
	    *dev = MKDEV(major, minor);
	    return 1;
	}
    }
    return 0;
}


static struct DEVICE_BLOCK {
    dev_t dev;	/* initial device number */
    int n_dev;	/* number of drives in this block */
    int step;	/* step in device number between drives */
} device_blocks[] = {
    /* all the r/w block devices known to the kernel */
    /* according to linux/Documentation/devices.txt  */
    {  3 << 8,   2, 64 },	/* hda      - hdb       */
    {  8 << 8,  16, 16 },	/* sda      - sdp       */
    { 13 << 8,   2, 64 },	/* xda      - xhb       */
    { 14 << 8,   4, 64 },	/* dos_hda  - dos_hdd   */
    { 21 << 8,   2, 64 },	/* mfma     - mfmb      */
    { 22 << 8,   2, 64 },	/* hdc      - hdd       */
    { 28 << 8,  16, 16 },	/* ada      - adp       */
    { 33 << 8,   2, 64 },	/* hde      - hdf       */
    { 34 << 8,   2, 64 },	/* hdg      - hdh       */
    { 36 << 8,   4, 64 },	/* eda      - edd       */
    { 44 << 8,  16, 16 },	/* ftla     - ftlp      */
    { 45 << 8,   4, 16 },	/* pdaa     - pdad      */
    { 48 << 8, 256,  8 },	/* rd/c0d0  - rd/c8d31  */
    { 56 << 8,   2, 64 },	/* hdi      - hdj       */
    { 57 << 8,   2, 64 },	/* hdk      - hdl       */
    { 65 << 8, 112, 16 },	/* sdq      - sddx      */
    { 72 << 8, 128, 16 },	/* ida/c0d0 - ida/c7d15 */
    { 94 << 8,  26,  4 },	/* dasda    - dasdz     */
    {104 << 8, 256, 16 },       /* cciss/c0d0 - cciss/c7d15 */
    {114 << 8,   8, 16 },       /* ataraid/d0 - ataraid/d7 */
    {       0,   0,  0 }
};


/* check_part_offset: return true iff partition number is within range */
static int check_part_offset(dev_t disk, unsigned int part)
{
    struct DEVICE_BLOCK *d;
    int i;

    for (d = device_blocks; d->dev && disk >= d->dev; d++)
	for (i = 0; i < d->n_dev; i++)
	    if (disk == d->dev + i * d->step)
		return part < d->step;

    return 0;
}


/* strdupcat: return concatenation of two strings in malloc-ed buffer */
static char *strdupcat(char *s1, char *s2)
{
    char *out = malloc(strlen(s1) + strlen(s2) + 1);
    if (out) {
	strcpy(out, s1);
	strcat(out, s2);
    }
    return out;
}


/* part_name: return partition device name from disk and partition number */
char *part_name(struct fdisk_disk *disk, unsigned int part)
{
    FILE *f;
    char *pname = NULL, *dname = disk->name + 5;

    assert(strncmp(disk->name, "/dev/", 5) == 0);

#ifdef DEBUG
    printf("looking for partition %d of disk %s\n", part, dname );
#endif

    /* scan /proc/partitions for likely devices */
    f = fopen("/proc/partitions", "r");
    if (f) {
	char name[MAX_DISKNAME_LEN];
	dev_t dev, disk_dev = 0;

	while (get_part_info(f, &dev, name)) {
	    /* look for target partition or real disk containing it */
	    if (! disk_dev) {
		/* look for real disk containing target partition */
		if (strcmp(name, dname) == 0) {
		    if (! check_part_offset(dev, part)) {
			syslog(LOG_WARNING, "disk %s partition %u is not "
		               "supported (no such device)", dname, part);
			break;
		    }
		    disk_dev = dev;
		}
	    } else {
		/* look for partition with same number as target */
		if (disk_dev + part == dev) {
#ifdef DEBUG
		    printf("found partition %s: major=%d, minor=%d\n",
			   name, (int)MAJOR(dev), (int)MINOR(dev));
#endif
		    pname = strdupcat("/dev/", name);
		    break;
		}
	    }
	}
	fclose(f);
    }
    return pname;
}


struct fdisk_partition *fdisk_add_partition(const char *name, int minor,
 unsigned int type, unsigned long size)
{
    int fstype;
    struct fdisk_partition *partition = malloc(sizeof(struct fdisk_partition));

    if (!partition)
        return NULL;

    partition->name = NULL;
    partition->mount_point= NULL;
    partition->type = type;
    partition->size = size;
    partition->in_use = 0;
    partition->disk = NULL;
    partition->next = NULL;
    partition->next_by_disk = NULL;
    partition->next_by_type = NULL;
    partition->next_in_use = NULL;

    if (minor == -1) {
        partition->name = strdup(name);
        assert(partition->name);
    } else {
	partition->disk = fdisk_find_disk(name);
	assert(partition->disk);
	assert(minor >= 0);
	partition->name = part_name(partition->disk, (unsigned) minor);
	if (!partition->name) {
	    syslog(LOG_WARNING, "no device name for disk %s partition %d "
		   "(skipped)", name, minor);
	    free(partition);
	    return NULL;
	}

#ifdef DEBUG
        printf("adding partition %s, to disk %s\n",
               partition->name, partition->disk->name);
#endif

        if (! partition->disk->partitions)
            partition->disk->partitions = partition;
        if (partition->disk->last_partition)
            partition->disk->last_partition->next_by_disk = partition;
        partition->disk->last_partition = partition;
    }
  
    if (NULL == fdisk_partitions)
        fdisk_partitions = partition;
    if (NULL != last_partition)
        last_partition->next = partition;
    last_partition = partition;

    fstype = fdisk_fstype_of(type);
    if (NULL == fdisk_partitions_by_type[fstype])
        fdisk_partitions_by_type[fstype] = partition;
    if (NULL != last_partitions_by_type[fstype])
        last_partitions_by_type[fstype]->next_by_type = partition;
    last_partitions_by_type[fstype] = partition;

#ifdef DEBUG
    printf("added partition %s, size %ld, type %08x=\"%s\"\n",
	   partition->name, size, type, fdisk_sysname(type) );
#endif
    return partition;
}


struct fdisk_disk *
fdisk_find_disk(const char *name) {
  struct fdisk_disk *d;

  d = fdisk_disks;

  while (d) {
    if (0 == strcmp(name, d->name)) {
      return (d);
    }
    d = d->next;
  }

  return(NULL);
}

struct fdisk_partition*
fdisk_find_partition_by_mntpoint(const char* mount_point)
{
  struct fdisk_partition* p;

  if (mount_point) {
    mounted_reread();
    p = mounted_partitions;

    while (p) {
      if (0 == strcmp(mount_point, p->mount_point))
        return p;
      p = p->next_in_use;
    }
  }

  return (NULL);
}

struct fdisk_partition *
fdisk_find_partition_by_name(const char *name){
  struct fdisk_partition *p;

  if (name) {
    p = fdisk_partitions;

    while (p) {
      if (0 == strcmp(name, p->name))
        return(p);
      p = p->next;
    }
  }

  return(NULL);
}

struct fdisk_partition *
fdisk_find_partition_by_type(const int type){
  struct fdisk_partition *p;

  p = fdisk_partitions;

  while (p) {
    if (p->type == type)
      return(p);
    p = p->next;
  }

  return(NULL);
} 

void
mounted_reread(){
  FILE *mtabFile;
  struct mntent *mnt_ent;
  struct fdisk_partition *p; 
  int type;
  
  mounted_partitions = NULL;

  if (NULL == (mtabFile = setmntent("/proc/mounts", "r"))) {
    return ;
  }

  while ( NULL != (mnt_ent = getmntent(mtabFile)) ) {
#ifdef DEBUG
    fprintf(stderr,"mounted partition: %s, on: %s, type: %s , options: %s ... ", 
	mnt_ent->mnt_fsname, mnt_ent->mnt_dir, mnt_ent->mnt_type, mnt_ent->mnt_opts);
#endif
    /* If mnt_ent is proc, devfs or devpts, ignore it */
    if ( (strcmp(mnt_ent->mnt_type,"proc") == 0) ||
         (strcmp(mnt_ent->mnt_type,"devfs") == 0) ||
	 (strcmp(mnt_ent->mnt_type,"devpts") == 0) ) {
#ifdef DEBUG
      fprintf(stderr,"proc/devfs/devpts fs. Ignoring it.\n");
#endif
      continue;
    }
    /* If mnt_ent is our current root fs, ignore it */
    if (strcmp(mnt_ent->mnt_dir,"/") == 0) {
#ifdef DEBUG
      fprintf(stderr,"root fs. Ignoring it.\n");
#endif
      continue;
    }
    /* If mnt_ent is not a partition on the local disks add it to the
     * partitions list*/
    p = fdisk_find_partition_by_name(mnt_ent->mnt_fsname); 
    if ( NULL == p) {
      for (type=0;fstype_name[type];type++) {
	if (strcmp(mnt_ent->mnt_type,fstype_name[type]) == 0) break;
      }
      if ( type == FSTYPE_UNKNOWN ) {
#ifdef DEBUG
        fprintf(stderr,"unknown fstype. Ignoring it.\n");
#endif
        continue;
      }
#ifdef DEBUG
      fprintf(stderr,"NOT found on local disks. Adding it.\n");
#endif
      if (hasmntopt(mnt_ent,"loop")) {
	type=(PTYPE_PREFIX_LOOP|type);
      } else {
	switch (type) {
	  case FSTYPE_NFS:
	    type=PTYPE_NFS;
	    break;
/* FIXME For cases where there's a one-to-many mapping between FSTYPEs
 * and PTYPEs, I guess it should be better to define a new set of PTYPEs
 * for non-well-detected partitions, instead of forcing a possibly wrong
 * PTYPE, as I do below */
	  case FSTYPE_EXT2:
	    type=PTYPE_LINUX;
	    break;
	  case FSTYPE_MSDOS:
	    type=PTYPE_DOS16L;
	    break;
	  case FSTYPE_MINIX:
	    type=PTYPE_MINIX;
	    break;
        }
      }
      p=fdisk_add_partition(mnt_ent->mnt_fsname,-1,type,0);
    }
#ifdef DEBUG
    else {
      fprintf(stderr,"found on local disks.\n");
    }
#endif
    p->in_use=-1;
    p->mount_point=strdup(mnt_ent->mnt_dir);
    if (mounted_partitions) {
    /* reverse ordered, as this is the order in which they would be
     * un-mounted. */
        p->next_in_use=mounted_partitions;
    }
    mounted_partitions=p; 
  }
  endmntent(mtabFile);
}

#ifdef OLDKERNEL	/* For 2.0.x kernels */

void
swapon_reread(){
  FILE *swapsFile;
  struct mntent *swap_ent;

  struct fdisk_partition *p;

  swapon_partitions = last_swapon_partition = NULL;

  if (NULL == (swapsFile = setmntent("/etc/swaps", "r"))) {
    return ;
  }

  while ( NULL != (swap_ent = getmntent(swapsFile)) ) {
#ifdef DEBUG
    fprintf(stderr,"Swap partition: %s, type: %s ... ",
        swap_ent->mnt_fsname, swap_ent->mnt_type);
#endif
    /* If swap_ent is not a partition on the local disks, then ignore it */
    p = fdisk_find_partition_by_name(swap_ent->mnt_fsname);
    if ( NULL != p) {
#ifdef DEBUG
      fprintf(stderr,"found on local disks.\n");
#endif
      p->in_use=-1;
      if (NULL == swapon_partitions)
        swapon_partitions=p;
      else {
        last_swapon_partition->next_in_use=p;
      }
      last_swapon_partition=p;
    }
#ifdef DEBUG
    else {
      fprintf(stderr,"NOT found on local disks. Ignored.\n");
    }
#endif
  }
  endmntent(swapsFile);
}

#else /* For 2.1.x kernels */

struct swpent
  {
    char *name;
    int type; /* 0 = partition, 1 = file */
    int size;
    int used;
    int prio;
  };

struct swpent *
getswpent (FILE *stream, struct swpent *sp)
{
  char *head;
#define BUFSIZE 8192
  char buffer[BUFSIZE];

  do
    {
      char *end_ptr;

      if (fgets (buffer, BUFSIZE, stream) == NULL)
        return NULL;

      end_ptr = strchr (buffer, '\n');
      if (end_ptr != NULL)      /* chop newline */
        *end_ptr = '\0';
      else
        {
          /* Not the whole line was read.  Do it now but forget it.  */
          char tmp[1024];
          while (fgets (tmp, sizeof tmp, stream) != NULL)
            if (strchr (tmp, '\n') != NULL)
              break;
        }
      head = buffer;
      /* skip empty lines and comment lines:  */
    } while (head[0] != '/');

  head += strcspn (head, " \t");
  *head++='\0';
  sp->name = strdup(buffer);
  head += strspn (head, " \t");
  sp->type = (strncmp(head, "file", 4) == 0) ? 1 : 0;
  head += strcspn (head, " \t");
  sscanf(head," %d %d %d ", &sp->size, &sp->used, &sp->prio);

  return sp;
}

void
swapon_reread(){
  FILE *swapsFile;
  struct swpent swap_ent;

  struct fdisk_partition *p;

  swapon_partitions = last_swapon_partition = NULL;

  if (NULL == (swapsFile = setmntent("/proc/swaps", "r"))) {
    return ;
  }

  while ( NULL != getswpent(swapsFile,&swap_ent) ) {
#ifdef DEBUG
    fprintf(stderr,"Swap partition: %s, type: %s , size: %d, used: %d, prio: %d ... ",
	swap_ent.name, (swap_ent.type==0)?"partition":"file", swap_ent.size,
	swap_ent.used, swap_ent.prio);
#endif
    /* If swap_ent is not a partition on the local disks, then ignore it */
    p = fdisk_find_partition_by_name(swap_ent.name);
    free(swap_ent.name);
    if ( NULL != p) {
#ifdef DEBUG
      fprintf(stderr,"found on local disks.\n");
#endif
      p->in_use=-1;
      if (NULL == swapon_partitions)
        swapon_partitions=p;
      else {
        last_swapon_partition->next_in_use=p;
      }
      last_swapon_partition=p;
    }
#ifdef DEBUG
    else {
      fprintf(stderr,"NOT found on local disks. Ignored.\n");
    }
#endif
  }
  fclose(swapsFile);
}
#endif /* OLDKERNEL */

/*
 * sseek: seek to specified sector - return 0 on failure
 *
 * For >4GB disks lseek needs a > 32bit arg, and we have to use llseek.
 * On the other hand, a 32 bit sector number is OK until 2TB.
 * The routines _llseek and sseek below are the only ones that
 * know about the loff_t type.
 *
 * sseek() uses the _llseek system call directly (if available), because of
 * some libc5/libc6 mismatch stuff.
 */

#ifdef __NR__llseek
static _syscall5( int, _llseek, uint, fd, ulong, hi, ulong, lo,
		  loff_t *, res, uint, wh );

/* seek to a sector */
static int sseek(int fd, unsigned long s)
{
    loff_t in, out;

    in = (loff_t)s * 512;
    out = 1;
    if (_llseek (fd, in>>32, in & 0xffffffff, &out, SEEK_SET) != 0) {
	perror( "llseek" );
	return 0;
    }
    if (in != out) {
	fprintf( stderr, "seek error: wanted 0x%08x%08x, got 0x%08x%08x\n",
		 (uint)(in>>32), (uint)(in & 0xffffffff),
		 (uint)(out>>32), (uint)(out & 0xffffffff) );
	return 0;
    }
    return 1;
}
#else
#if !defined(__alpha__) && !defined(__ia64__)
/* Better print an error if _llseek isn't available at compile time, the
 * resulting binary wouldn't be able to handle disks > 2GB */
#error No _llseek available
/* Sorry, kernel doesn't know _llseek call :-((( */
#endif
static int sseek(int fd, unsigned long s)
{
    off_t out;

    out = lseek( fd, (off_t)(s*512), SEEK_SET );
    if ((off_t)s*512 != out)
	return 0;
    return 1;
}
#endif

int sread(int fd, unsigned long sector, void *buffer)
{
    if (!sseek( fd, sector ))
	return 0;
    if (read( fd, buffer, 512 ) != 512) {
	/* don't report sector zero read errors */
	if (sector != 0 || errno != EIO)
	    fprintf( stderr, "libfdisk: error reading sector %ld: %s\n",
		 sector, strerror(errno) );
	return 0;
    }
    return 1;
}

static struct {
    char *name;
    int (*func)(char *device, int fd);
} parsers[] = {
/* Put GPT before MSDOS, as GPT lives inside an msdos type EE partition */
#ifdef HAVE_GPT_PARTITION
    { "gpt", parse_gpt_partition },
#endif
#ifdef HAVE_MSDOS_PARTITION
    { "msdos", parse_msdos_partition },
#endif
#ifdef HAVE_OSF_PARTITION
    { "osf",   parse_osf_partition },
#endif
#ifdef HAVE_SUN_PARTITION
    { "sun",   parse_sun_partition },
#endif
#ifdef HAVE_AMIGA_PARTITION
    { "amiga", parse_amiga_partition },
#endif
#ifdef HAVE_ATARI_PARTITION
    { "atari", parse_atari_partition },
#endif
#ifdef HAVE_MAC_PARTITION
    { "mac",   parse_mac_partition },
#endif
#ifdef HAVE_ACORN_PARTITION
    { "acorn", parse_acorn_partition },
#endif
#ifdef HAVE_SGI_PARTITION
    { "sgi",   parse_sgi_partition },
#endif
#ifdef HAVE_IBM_PARTITION
    { "ibm",   parse_ibm_partition },
#endif
};

static void
parse_partition(char *device, int fd)
{
    int i, rv;

    for( i = 0; i < SIZE(parsers); ++i ) {
	rv = parsers[i].func( device, fd );
	if (rv < 0)
	    return;
	if (rv > 0) {
	    struct fdisk_disk *disk;
	    if ((disk = fdisk_find_disk(device)))
		disk->partition_format = parsers[i].name;
	    return;
	}
    }
}


/* ide_is_disk: return true if IDE device is a disk drive */
static int ide_is_disk(char *device)
{
    char pname[32];
    int n;
    FILE *f;
    int result = 0;

    /* /dev/hd? => /proc/ide/hd?/media */
    device = strrchr(device, '/');
    assert(device);
    n = snprintf(pname, sizeof pname, "/proc/ide/%s/media", device + 1);
    assert(n > 0);

    /* open /proc/ide/hd?/media and look for ide_disk text */
    f = fopen(pname, "r");
    if (f) {
	char line[8];
	result = fgets(line, sizeof line, f) && strcmp(line, "disk\n") == 0;
	fclose(f);
    }
    return result;
}


/* is_disk: return true if device is a disk drive */
static int is_disk(char *device, int fd)
{
    struct stat stat_buf;
    struct cdrom_volctrl cdvol;

    /* reject anything un-stat-able, or not a block device */
    if (fstat(fd, &stat_buf) || ! S_ISBLK(stat_buf.st_mode))
	return 0;

    /* IDE devices */
    switch (MAJOR(stat_buf.st_rdev)) {
    case IDE0_MAJOR: case IDE1_MAJOR: case IDE2_MAJOR:
    case IDE3_MAJOR: case IDE4_MAJOR: case IDE5_MAJOR:
	return ide_is_disk(device);
    }

    /* SCSI disk driver will only accept disks and mag-opt drives */
    if (SCSI_DISK_MAJOR(MAJOR(stat_buf.st_rdev))) 
	return 1;

    /* DASD disk driver will only accept disks */
    if (DASD_MAJOR == MAJOR(stat_buf.st_rdev)) 
	return 1;

#if #cpu (i386)
    /* ESDI devices assumed to all be disks */
    if (MAJOR(stat_buf.st_rdev) == PS2ESDI_MAJOR)
	return 1;
#elif #cpu (m68k)
    /* ACSI devices assumed to all be disks */
    if (MAJOR(stat_buf.st_rdev) == ACSI_MAJOR)
	return 1;
#endif

    /** TODO ***************************************************************/
    /* I'm not sure how to check for non-disk devices on dac960 or smart2. */
    /* *********************************************************************/

    /* supposedly anything failing CDROMVOLREAD for the right reasons is ok */
    return ioctl(fd, CDROMVOLREAD, &cdvol) && (errno == EINVAL ||
					       errno == EPERM  ||
					       errno == EIO    ||
					       errno == EBADRQC);
}


static void alarmed(int signo) { }


/* try_device: check specific device by name, add to disk/part lists if ok */
static void try_device(char *device)
{
    int fd;

    signal(SIGALRM, alarmed);
    alarm(30);
    fd = open(device, O_RDONLY);
    alarm(0);

    if (fd >= 0) {
        if (is_disk(device, fd)) {
	    unsigned int size;
	    if (ioctl(fd, BLKGETSIZE, &size))
		size = 0;   /* don't complain, size not really used yet */
	    size >>= 1;     /* convert from 512-byte sectors to kB */
	    fdisk_add_disk(device, size);

	    parse_partition(device, fd);
	} else {
	    /* Don't need to do anything for individual partitions because   */
	    /* parse_partition() above calls fdisk_add_partition() for every */
	    /* partition on the 'whole-disk' device.  We could eliminate a   */
	    /* huge amount of code and complexity if we knew the partition   */
	    /* type right now.						     */
	}
	close(fd);
    }
}


/* is_drive: return true if device is a 'whole disk', not just a partition */
static int is_drive(dev_t dev)
{
    struct DEVICE_BLOCK *d;
    int i;

    /* search for device in each block of devices */
    for (d = device_blocks; d->dev && dev >= d->dev; d++)
	for (i = 0; i < d->n_dev; i++)
	    if (dev == d->dev + i * d->step)
		return 1;

    return 0;
}


/* fdisk_reread: build up disk/partition lists from scratch */
void fdisk_reread(void)
{
    FILE *f;

    fdisk_init();

    /* scan /proc/partitions for likely device names */
    f = fopen("/proc/partitions", "r");
    if (f) {
	char name[MAX_DISKNAME_LEN];
	dev_t dev;

	while (get_part_info(f, &dev, name)) {
	    /* check device number against known 'whole-disk' devices */
	    if (is_drive(dev)) {
		static char path[] = "/dev/";
		char dev_name[PATH_MAX];

		/* try to build partition list for device */
		strcpy(dev_name, path);
		strcat(dev_name, name);
		try_device(dev_name);
	    }
	}
	fclose(f);
    }

    mounted_reread();
    swapon_reread();
}
