#include <stdio.h>
#include <sys/sysmacros.h>
#include <sys/ioctl.h>
#include <asm/types.h>
#include <linux/serial.h>
#include "dbootstrap.h"
#include "lang.h"
#include <syslog.h> 
#include "util.h"
#include "notail.h"

#define SEDCOMMAND "sed"


static int require_swap (void) {
  if ( (swapon_partitions == NULL) &&
       (fdisk_partitions_by_type[FSTYPE_SWAP] == NULL) &&
       (noSwap == 0) ) {
    problemBox(_("Your swap partition has not been set up.\nBefore you begin installation, you must first partition your disk, and then initialize and mount your root filesystem, and initialize a swap partition.\nPlease use the main menu to complete any of those steps that you have not done, and then use the main menu to return to this step."), _("No swap partition found"));
    return 1;
  }
  return 0;
}

/* XXX: This does not work with read-only filesystems, which is most
 * likely only solvable from modconf, why don't we chroot when we run
 * modconf, we can simply copy the libraries, IMO - Ben */
static int update_cdrom_symlink (void) {
  struct stat statbuf;
  int pos;
  
  if ( (! lstat("/dev/cdrom", &statbuf) ) &&
       S_ISLNK((&statbuf)->st_mode) && 
       NAME_ISDIR("/target/dev", &statbuf) ) {
    pos = readlink("/dev/cdrom", prtbuf, 20);
    prtbuf[pos] = '\0';
    symlink(prtbuf, "/target/dev/cdrom");
  }
  return 0;
}

#ifdef SCSI_FLOPPY
/* XXX: neither does this - Ben */
static int update_sfd0_symlink (void) {
  struct stat statbuf;
  int pos;
  
  if ( (! lstat("/dev/sfd0", &statbuf) ) &&
       S_ISLNK((&statbuf)->st_mode) && 
       NAME_ISDIR("/target/dev", &statbuf) ) {
    pos = readlink("/dev/sfd0", prtbuf, 20);
    prtbuf[pos] = '\0';
    symlink(prtbuf, "/target/dev/sfd0");
  }
  return 0;
}
#endif

#if #cpu (m68k)
static int update_fdisk_symlink (void) {
  char *target;

  /* determine target of symlink */
  if (strcmp(Arch2, "Atari") == 0)
	  target = "atari-fdisk";
  else if (strcmp(Arch2, "Amiga") == 0)
	  target = "amiga-fdisk";
  else if (strcmp(Arch2, "Macintosh") == 0)
	  target = "mac-fdisk";
  else if (strcmp(Arch2, "VME") == 0)
	  target = "../usr/sbin/pmac-fdisk";
  else
	  return -1;

  /* remove an existing /sbin/fdisk (ignore if not existed) */
  unlink("/target/sbin/fdisk");
  symlink(target, "/target/sbin/fdisk");
  return 0;
}
#endif

#if #cpu (powerpc)
static int update_fdisk_symlink (void) {
  char *target;

  /* determine target of symlink */
  if (strcmp(Arch2, "apus") == 0)
	  target = "amiga-fdisk";
  else if (strcmp(Arch2, "BeBox") == 0)
	  target = "ddisk";
  else if (strcmp(Arch2, "CHRP") == 0)
	  target = "ddisk";
  else if (strcmp(Arch2, "MBX") == 0)
	  target = "ddisk";
  else if (strstr(Arch2, "PowerMac"))
	  target = "mac-fdisk";
  else if (strcmp(Arch2, "PReP") == 0)
	  target = "ddisk";
  else
	  return -1;

  /* remove an existing /sbin/fdisk (ignore if not existed) */
  unlink("/target/sbin/fdisk");
  symlink(target, "/target/sbin/fdisk");
  return 0;
}
#endif

#if #cpu (s390)
static int update_fdisk_symlink (void) {
  /* remove an existing /sbin/fdisk (ignore if not existed) */
  unlink("/target/sbin/fdisk");
  symlink("/target/sbin/fdasd", "/target/sbin/fdisk");
  return 0;
}
#endif

#ifdef SERIAL_CONSOLE
static int
update_console_info (void)
{
  char tty[16];
  int ttyline = -1;
  int ttyspeed = 9600;
  int err = 0;
  FILE *F;
  /* serial console setup:
   *  1/ relink /dev/console to /dev/ttyS<n> *only* if < 2.0 kernel
   *  2/ edit /target/etc/inittab to enable a T<n> line and set emulation type
   *  2b/ do the same thing with /target/etc/inittab.real
   *  3/ add ttyS<n> & cua<n> to /target/etc/securetty
   */
  if (!strncmp(kver, "2.0", 3)) {
    strcpy(tty, ttyname(0));
    if (!strncmp(tty, "/dev/cua", 8))
      ttyline = tty[8] - '0';
    else if (!strncmp(tty, "/dev/ttyS", 9))
      ttyline = tty[9] - '0';
  }
  else {
	struct serial_struct sr;
    if (ioctl(0, TIOCGSERIAL, &sr) < 0)
      strcpy(tty, "/dev/console");
    else {
      ttyline = sr.line;
      sprintf(tty, "/dev/ttyS%d", ttyline);
    }
  }
  if (ttyline >= 0) {
    char *term = getenv("TERM");
    /* XXX the following line will _break_ if a 2.0 kernel is used
     * with a /dev/cuaX device. since none of the kernel versions are
     * 2.0, i don't feel too bad about this    jaqque@debian.org */
    if (bootargs.console && bootargs.console[5]==',') ttyspeed=atoi(bootargs.console+6);
    if (!term||!*term) term="vt100";
    /* 1 */
    if (!strncmp(kver, "2.0", 3)) {
      /* 2.0 kernel only */
      unlink("/target/dev/console");
      symlink(tty, "/target/dev/console");
    }
    /* 2 */
    sprintf(prtbuf,
	    SEDCOMMAND
	    " -e "
	    "'"
	    "s/^#\\(T%d:.\\+base-config.*\\)/\\1/; "
	    "s/^\\([1-6]\\):/#\\1:/"
	    "'"
	    "< %s > %s",
	    ttyline,
	    "/target/etc/inittab",
	    "/target/etc/inittab.tmp");
    if (execlog(prtbuf, LOG_INFO) ||
	rename("/target/etc/inittab.tmp", "/target/etc/inittab"))
    {
      problemBox(_("Error writing /target/etc/inittab"), _("Write error"));
      return 1;
    }
    /* 2b */
    sprintf(prtbuf,
	    SEDCOMMAND
	    " -e 's/^#T0\\(.*ttyS\\).*/T%d\\1%d %d %s/'"
	    " -e 's/^\\([1-6]\\):/#\\1:/' < %s > %s",
	    ttyline, ttyline, ttyspeed, term,
	    "/target/etc/inittab.real",
	    "/target/etc/inittab.real.tmp");
    if (execlog(prtbuf, LOG_INFO) ||
	rename("/target/etc/inittab.real.tmp","/target/etc/inittab.real"))
    {
      problemBox(_("Error writing /target/etc/inittab.real"), _("Write error"));
      return 1;
    }
    /* 3 */
    F = fopen("/target/etc/securetty", "a");
    if (F) {
      err = fprintf(F, "ttyS%d\ncua%d\n", ttyline, ttyline) <= 0;
      fclose(F);
    }
    else
      err = 1;
    if (err) {
      problemBox(_("Error writing /target/etc/securetty"), _("Write error"));
      return 1;
    }
  }
  return 0;
}
#endif

#if #cpu (s390)
static int
update_s390_console_info (void)
{
  /* s390 console setup:
   *  1/ edit /target/etc/inittab.real to add a special s390 getty entry
   */
  /* 1 */
  sprintf(prtbuf,
          SEDCOMMAND
          " -e 's/^\\([1-6]\\):/#\\1:/'"
          " -e '/^#T3/a\\\n"
          "\\\n"
          "# s390 console\\\n"
          "1:2345:respawn:/sbin/getty 38400 console dumb' < %s > %s",
          "/target/etc/inittab.real",
          "/target/etc/inittab.real.tmp");
  if (execlog(prtbuf, LOG_INFO) ||
        rename("/target/etc/inittab.real.tmp","/target/etc/inittab.real"))
  {
    problemBox(_("Error writing /target/etc/inittab.real"), _("Write error"));
    return 1;                                                                   
  }
  return 0;
}
  

static int
update_s390_info (void)
{
  int err = 0;
  int i = 0;
  FILE *F;
  /* s390 setup:
   *  1/ add base-config call to .profile
   *  2/ disable password check  
   *  3/ allow logon of root on pts
   *   
   *  all these changes are undone during base-config execution
   */
  /* 1 */
  F = fopen("/target/root/.profile", "a");
  err = 0;
  if (F) {
    err = fprintf(F, "if [ `ps ax | grep -c base-config` -eq 1 ]; then if [ \"$TERM\" != dumb ]; then /usr/sbin/base-config; fi; fi\n") <= 0;
    fclose(F);
  }
  else
    err = 1;
  if (err) {
    problemBox(_("Error writing /target/root/.profile"), _("Write error"));
    return 1;
  }
  /* 2 */
  sprintf(prtbuf,
        SEDCOMMAND
            " -e 's/password   required   pam_unix.so nullok obscure min=4 max=8/#password   required   pam_unix.so nullok obscure min=4 max=8/'"
            "< %s > %s",
            "/target/etc/pam.d/passwd",
            "/target/etc/pam.d/passwd.tmp");
  if (execlog(prtbuf, LOG_INFO) ||
        rename("/target/etc/pam.d/passwd.tmp","/target/etc/pam.d/passwd"))
  {
    problemBox(_("Error writing /target/etc/pam.d/passwd"), _("Write error"));
    return 1;
  }
  /* 3 */
  F = fopen("/target/etc/securetty", "a");
  err = 0;
  if (F) {
    for(i=0; i<8 && !err; i++)
      err = fprintf(F, "pts/%d\n", i) <= 0;
    fclose(F);
  }
  else
    err = 1;
  if (err) {
    problemBox(_("Error writing /target/etc/securetty"), _("Write error"));
    return 1;
  }
  return 0;
}
#endif

static int write_fstab (void) {
  FILE *fstab, *proc;
  struct fdisk_partition *p;
  char* fsname;
  char* fsopts;
  char dev[80], dir[80], type[80], opt[80];
  int dump, pass;

  if (NULL == (fstab = fopen("/target/etc/fstab", "w"))) {
    problemBox(_("Can't open /target/etc/fstab"), _("Write error"));
    return 1;
  }

  if (is_nfs_partition("/target")) {
	fsname = "nfs";
	fsopts = "rw,rsize=8192,wsize=8192,nolock";
	pass = 1;
  }
  else if (is_fstype("/target", "xfs")) {
    fsname = "xfs";
    fsopts = "defaults";
    pass = 0; /* automatic fsck is innappropriate, thus fsck.xfs is /bin/true */
  }
  else if (is_fstype("/target", "ext3")) {
    fsname = "ext3";
    fsopts = "errors=remount-ro";
    pass = 1;
  }
  else if (is_fstype("/target", "ext2")) {
    fsname = "ext2";
    fsopts = "errors=remount-ro";
    pass = 1;
  }
  else if (is_fstype("/target", "reiserfs")) {
    fsname = "reiserfs";
    if (need_notail("/target"))
      fsopts = "notail";
    else
      fsopts = "defaults";
    pass = 0; /* fsck is unusable */
  }
  else {
    problemBox(_("Could not determine filesystem type of /"), _("Problem"));
    return -1;
  }

  fprintf(fstab,
	  _("# /etc/fstab: static file system information.\n"
	    "#\n"
	    "# <file system>\t<mount point>\t<type>\t<options>\t\t<dump>\t<pass>\n"));
  fprintf(fstab,
	  "%s\t/\t\t%s\t%s\t0\t%d\n",
	  Root->name, fsname, fsopts, pass);

  if (swapon_partitions != NULL) {
    p = swapon_partitions;
    while (p) {
      fprintf(fstab,
	      "%s\tnone\t\tswap\tsw\t\t\t0\t0\n",
	      p->name);
      p=p->next_in_use;
    }
  }

  fprintf(fstab,
	  "proc\t\t/proc\t\tproc\tdefaults\t\t0\t0\n");
  fprintf(fstab,
	    "/dev/fd0\t/floppy\t\tauto\tuser,noauto\t\t0\t0\n"
	    "/dev/cdrom\t/cdrom\t\tiso9660\tro,user,noauto\t\t0\t0\n");

  proc = fopen ("/proc/mounts", "r");
  if ( proc == NULL )
  {
    perrorBox(_("Can't read /proc/mounts"));
    fclose (fstab);
    return -1;
  }
  while ( EOF != fscanf (proc, "%s %s %s %s %i %i\n",
          dev, dir, type, opt, &dump, &pass ))
  {
    if (strncmp ("/target/", dir, 8))
    {
      continue; /* next record if it is not a subdir of /target */
    }

    if (!strcmp ("rw", opt))
      strcpy (opt, "defaults");
    
    if (need_notail(dir))
      strncat (opt, ",notail", 71);

    if (!strcmp("xfs", type))
      pass = 0;
    else if (!strcmp("reiserfs", type))
      pass = 0;
    else
      pass = 2;
    fprintf (fstab, "%s\t%s\t%s\t%s\t\t\t%i\t%i\n",
                    dev, dir+7, type, opt, dump, pass);
  }
  fclose (proc);
  fclose (fstab);
  return 0;
}

#ifdef FONT
static int write_etc_kbd (void) {
  FILE *etc_kbd;
  if (NULL == (etc_kbd = fopen("/target/etc/console-tools/config", "a"))) {
    problemBox(_("Can't open /target/etc/console-tools/config"), _("Write error"));
    return 1;
  }
  fprintf(etc_kbd, "CONSOLE_FONT=%s\n", FONT);
  fclose(etc_kbd);
  return 0;
}
#endif

int configure_base (void) {
  struct stat statbuf;

  if ( require_swap() ) return 1;

  if (! NAME_ISREG( "/target/etc/debian_version", &statbuf) ||
      ! NAME_ISREG( "/target/sbin/init", &statbuf) ) {
    problemBox(_("You have to install base for configuring it"),_("Base not installed"));
    return 1;
  }

  if ( write_fstab() ) return 1;

#if #cpu (m68k)
  /* don't do keymapping for VME serial console */
  if (strcmp(Arch2, "VME") == 0)
    unlink("/target/bin/loadkeys");
#endif

#ifdef FONT
    if ( write_etc_kbd() ) return 1;
#endif

  update_cdrom_symlink();

#ifdef SCSI_FLOPPY
  update_sfd0_symlink();
#endif

  if (!strncmp(kver, "2.0", 3)) {
    /* 2.0 kernel only: first update /dev/console then eventually
     * change it to /dev/ttyS<n> latter in update_console_info() */
    unlink("/target/dev/console");
    symlink("tty0", "/target/dev/console");
  }

#if #cpu (m68k)
  if (strcmp(Arch2, "VME") == 0)
    update_console_info();
  update_fdisk_symlink();
#endif

#if #cpu (powerpc)
  update_console_info();
  update_fdisk_symlink();
#endif

#if #cpu (sparc)
  update_console_info();
#endif

#if #cpu (i386)
  update_console_info();
#endif

#if #cpu (arm)
  update_console_info();
#endif

#if #cpu (alpha)
  update_console_info();
#endif

#if #cpu (mips)
    update_console_info();
#endif

#if #cpu (mipsel)
    update_console_info();
#endif

#if #cpu (hppa)
    if (serialConsole >= 0)
      update_console_info();
#endif

#if #cpu (ia64)
    if (serialConsole >= 0)
      update_console_info();
#endif

#if #cpu (s390)
    update_s390_console_info();
    update_fdisk_symlink();
#endif

  /* If on serial console or on s390, get rid of kbd files and setserial 
     startup.  The kbd package will be purged during final configuration. */
#if !(#cpu(s390))
  if (serialConsole >= 0)
#endif
  {
    unlink("/target/etc/rcS.d/S05keymaps-lct.sh");
    unlink("/target/etc/rcS.d/S48console-screen.sh");
    write_userconfig("SERIALCONSOLE", "true");
    /* Temporary hack, we should fix the setserial script so it doesn't
       trash serial consoles. MDP */
    unlink("/target/etc/rcS.d/S30setserial");
    unlink("/target/etc/rc0.d/K30setserial");
    unlink("/target/etc/rc6.d/K30setserial");
  }

#if #cpu (s390)
    update_s390_info();
#endif
        

  unlink("/target/sbin/unconfigured.sh");

  return 0;
}

int configure_drivers (void) {
  struct stat statbuf;
#if #cpu (s390)
  char buffer1[PATH_MAX];
  char buffer2[PATH_MAX];
#endif
  char *old_locale, *old_lc_messages, *old_lc_ctype, *p;

  if(! bootargs.isquiet)
     problemBox(_("Remember that many drivers are already included in the running kernel. If you don't see a required module in the modules configuration, this driver may already have been loaded and you can use the hardware without loading the driver manually."), _("Note about loaded drivers"));

  if (! NAME_ISEXE("/usr/bin/whiptail", &statbuf))
      symlink("/target/usr/bin/whiptail", "/usr/bin/whiptail");

  snprintf(prtbuf, sizeof (prtbuf), "/target/lib/modules/%s", kver);
  if (! NAME_ISDIR(prtbuf, &statbuf)) {
    snprintf(prtbuf, sizeof (prtbuf), 
             _("No modules were found in /target/lib/modules/%s that could be configured. Please install the kernel modules first, by running the \"Configure Device Driver Modules\" step"), 
             kver);
    problemBox(prtbuf, _("Problem"));
    return -1;
  }

#if #cpu (s390)
  /* copy available network modules */
  sprintf(buffer1, "/lib/modules.old/%s/kernel/drivers/s390/net/qdio.o", kver);       sprintf(buffer2, "/target/lib/modules/%s/kernel/drivers/s390", kver);
  if(NAME_ISREG (buffer1, &statbuf) && NAME_ISDIR (buffer2, &statbuf)) {
    sprintf(prtbuf, "cp -a /lib/modules.old/%s/kernel/drivers/s390/net/qdio.o /target/lib/modules/%s/kernel/drivers/s390/", kver, kver);
    execlog(prtbuf, LOG_INFO);
  }
  sprintf(buffer1, "/lib/modules.old/%s/kernel/drivers/s390/net/qeth.o", kver);   sprintf(buffer2, "/target/lib/modules/%s/kernel/drivers/s390/net", kver);
  if(NAME_ISREG (buffer1, &statbuf) && NAME_ISDIR (buffer2, &statbuf)) {
    sprintf(prtbuf, "cp -a /lib/modules.old/%s/kernel/drivers/s390/net/qeth.o /target/lib/modules/%s/kernel/drivers/s390/net/", kver, kver);
    execlog(prtbuf, LOG_INFO);
  }
  sprintf(buffer1, "/lib/modules.old/%s/kernel/drivers/s390/net/lcs.o", kver);
  sprintf(buffer2, "/target/lib/modules/%s/kernel/drivers/s390/net", kver);
  if(NAME_ISREG (buffer1, &statbuf) && NAME_ISDIR (buffer2, &statbuf)) {
    sprintf(prtbuf, "cp -a /lib/modules.old/%s/kernel/drivers/s390/net/lcs.o /target/lib/modules/%s/kernel/drivers/s390/net/", kver, kver);
    execlog(prtbuf, LOG_INFO);
  }

  /* copy module configuration from initial network setup */ 
  execlog("mkdir -p /target/etc/modutils/arch", LOG_INFO);
  execlog("cp -a /etc/modutils/arch/s390 /target/etc/modutils/arch", LOG_INFO);  
  execlog("cp -a /etc/chandev.conf /target/etc/", LOG_INFO);
  execlog("cp -a /etc/modules.conf /target/etc/", LOG_INFO);
  execlog("cp -a /etc/modutils/* /target/etc/modutils/", LOG_INFO);
#endif

  execlog("depmod -a", LOG_INFO);

  /* Enable kmod so that auto-loading of (e.g.) parport modules
     will work.  See bug #78750 */
  execlog("echo /sbin/modprobe >/proc/sys/kernel/modprobe", LOG_INFO);

#ifdef USE_LANGUAGE_CHOOSER
  old_locale = getenv("LANG");
  old_lc_messages = getenv("LC_MESSAGES");
  old_lc_ctype = getenv("LC_CTYPE");

  strcpy(prtbuf, lang->locale);
  p = strchr(prtbuf, '.');
  if (p)
    strcpy(p+1, "UTF-8");

  setenv("LANG", "C@utf-8", 1);
  setenv("LC_MESSAGES", prtbuf, 1);
  setenv("LC_CTYPE", "C@utf-8", 1);
#endif
  
  /* copy the preconfigured modules file to target/etc */
  execlog("test -f /target/etc/modules || cp /etc/modules /target/etc/", LOG_INFO);

  fullscreen_execlog("/target/usr/sbin/modconf"
                     " --exclude-section pcmcia"
                     " --target /target"
                     " --run-shell cdromsymlink"
                     " --libdir /tmp/drivers"
                     " --helpdir /target/usr/share/modconf"
                     " --tty /dev/tty");

  /* Turn kmod off again */
  execlog("echo /bin/true >/proc/sys/kernel/modprobe", LOG_INFO);

#ifdef USE_LANGUAGE_CHOOSER
  /* Restore the old locale */
  if (old_locale)
    setenv("LANG", old_locale, 1);
  else
    unsetenv("LANG");
  if (old_lc_messages)
    setenv("LC_MESSAGES", old_lc_messages, 1);
  else
    unsetenv("LC_MESSAGES");
  if (old_lc_ctype)
    setenv("LC_CTYPE", old_lc_ctype, 1);
  else
    unsetenv("LC_CTYPE");
#endif

  update_cdrom_symlink();

  return 0;
}

#ifdef _TESTING_
int
main ()
{
  LOAD_TRMFILE ("test.trm");
  InstallationRootDevice = block_device ("/");
  get_kver ();
  boxInit ();
  return configure_base ();
}

#endif /* _TESTING_ */
