
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
//#define BRISTOL_DBG
/*
 * Need to have basic template for an operator. Will consist of
 *
 *	envelopeinit()
 *	operate()
 *	reset()
 *	destroy()
 *
 *	destroy() is in the library.
 *
 * Operate will be called when all the inputs have been loaded, and the result
 * will be an output buffer written to the next operator.
 */

#include "bristol.h"
#include "envelope.h"

/*
 * The name of this operator, IO count, and IO names.
 */
#define OPNAME "ENV"
#define OPDESCRIPTION "Digital ADSR Envelope Generator"
#define PCOUNT 6
#define IOCOUNT 1

#define ENV_OUT_IND 0

/*
 * Reset any local memory information.
 */
static int destroy(bristolOP *operator)
{
#ifdef BRISTOL_DBG
	printf("reset(%x)\n", operator);
#endif

	/*
	 * Unmalloc anything we added to this structure
	 */
	bristolfree(operator->specs);

	/*
	 * Free any local memory. We should also free ourselves, since we did the
	 * initial allocation.
	 */
	cleanup(operator);
}

/*
 * Reset any local memory information.
 */
static int reset(bristolOP *operator, bristolOPParams *param)
{
#ifdef BRISTOL_DBG
	printf("reset(%x)\n", operator);
#endif

	param->param[4].float_val = 1.0;
	param->param[5].int_val = 1;
}

/*
 * Alter an internal parameter of an operator.
 */
static int param(bristolOP *operator, bristolOPParams *param,
	unsigned char index, float value)
{
	int offset = (value * (CONTROLLER_RANGE - 1)) + 1;

	if (offset >= CONTROLLER_RANGE)
		offset = CONTROLLER_RANGE - 1;
	else if (offset < 0)
		offset = 0;

#ifdef BRISTOL_DBG
	printf("param(%x, %f)\n", operator, value);
#endif

	switch (index) {
		case 0:
			/*
			 * Attack.
			 */
			param->param[index].float_val = gainTable[offset].rate;
			break;
		case 1:
		case 3:
			/*
			 * Decay and Release.
			 */
			param->param[index].float_val = 1/gainTable[offset].rate;
			break;
		case 2:
			offset = value * (CONTROLLER_RANGE - 1);
			/*
			 * Sustain
			 */
			param->param[index].float_val = gainTable[offset].gain
				* BRISTOL_VPO;
			break;
		case 4:
			/*
			 * Gain
			 */
			param->param[index].float_val = gainTable[offset].gain;
			break;
		case 5:
			/*
			 * Keytracking
			 */
			if (value == 0)
				param->param[index].int_val = 0;
			else
				param->param[index].int_val = 1;
	}
#ifdef BRISTOL_DBG
	printf("a %f, d %f, s %0.10f r %f g %f %i, %i\n",
		param->param[0].float_val, param->param[1].float_val,
		param->param[2].float_val, param->param[3].float_val,
		param->param[4].float_val, param->param[5].int_val, offset);
#endif

	return(0);
}

/*
 * Pure output signal generator - will drive amps, filters, DCOs, whatever.
 */
static int operate(register bristolOP *operator, bristolVoice *voice,
	bristolOPParams *param, void *lcl)
{
	register bristolENVlocal *local = lcl;
	register float cgain, attack, decay, sustain, release, *ob, gain, egain;
	register int count, rampup = -1;
	bristolENV *specs;

	specs = (bristolENV *) operator->specs;

#ifdef BRISTOL_DBG
	printf("envelope(%x, %x, %x, %x)\n", operator, voice, param, local);
#endif

	attack = param->param[0].float_val;
	decay = param->param[1].float_val;
	sustain = 1.0 + param->param[2].float_val;
	release = param->param[3].float_val;
	/*
	 * Adjust the gain depending on the key velocity, and include both channel
	 * and polypressure - we will typically have one or the other, but both
	 * is highly unlikely.
	 *
	 * This gain figure should be related to an option for key tracking?
	 */
	if (param->param[5].int_val)
		gain = param->param[4].float_val
			* (voice->velocity + voice->press + voice->chanpressure);
	else
		gain = param->param[4].float_val;

	/*
	 * Egain is to smooth over changes to the velocity gain when we retrigger
	 * the voice.
	 */
	egain = local->egain;
	if (egain < gain)
		rampup = 1;
	else if (egain > gain)
		rampup = 0;

	count = specs->spec.io[ENV_OUT_IND].samplecount;
	cgain = local->cgain;
	ob = specs->spec.io[ENV_OUT_IND].buf;

	if (voice->flags & BRISTOL_KEYOFF)
	{
//printf("keyoff %x\n", voice->flags);
		voice->flags |= BRISTOL_KEYOFFING;
		local->cstate = STATE_RELEASE;
	}

	if (voice->flags & BRISTOL_KEYON)
	{
//printf("keyon %x\n", voice->flags);
		cgain = 1.0;
		local->cstate = STATE_ATTACK;
	} else if (voice->flags & BRISTOL_KEYREON)
		local->cstate = STATE_ATTACK;
	if (voice->flags & (BRISTOL_KEYON|BRISTOL_KEYREON))
		local->cstate = STATE_ATTACK;

	/*
	 * We need some state indicator to say when we are leading into the attack.
	 * This is actually in the MIDI voice structure, but we do not have access
	 * to this. We should assume that a MIDI event sets our cstate as necessary?
	 */
	while (count > 0)
	{
		switch (local->cstate)
		{
			case STATE_RELEASE:
				/*
				 * Assume key off - start releasing gain.
				 */
				while (cgain > 1.0)
				{
					if (count-- > 0)
						*ob++ = ((cgain *= release) - 1) * egain;
					else
						break;
					if (rampup >= 0) {
						if (rampup)
						{
							if ((egain+=0.01) > gain)
							{
								egain = gain;
								rampup = -1;
							}
						} else {
							if ((egain-=0.01) < gain)
							{
								egain = gain;
								rampup = -1;
							}
						}
					}
				}
				if (cgain <= 1.0)
				{
					cgain = 1.0;
					local->cstate = STATE_DONE;
					voice->flags |= BRISTOL_KEYDONE;
					voice->flags &= ~BRISTOL_KEYOFFING;
				} else
					voice->flags &= ~BRISTOL_KEYDONE;
				break;
			case STATE_START:
			case STATE_ATTACK:
				/*
				 * Just triggered - start attack ramp from current gain level.
				 */
				while (cgain < BRISTOL_VPO)
				{
					if (count-- > 0)
						*ob++ = ((cgain *= attack) - 1) * egain;
					else
						break;
					if (rampup >= 0) {
						if (rampup)
						{
							if ((egain+=0.01) > gain)
							{
								egain = gain;
								rampup = -1;
							}
						} else {
							if ((egain-=0.01) < gain)
							{
								egain = gain;
								rampup = -1;
							}
						}
					}
				}
				if (cgain >= BRISTOL_VPO) local->cstate = STATE_DECAY;
				/*
				 * This just ensures that it is the last envelope in any given
				 * bristolSound chain that decides when to stop the audio
				 * stream.
				 */
				voice->flags &= ~BRISTOL_KEYDONE;
				break;
			case STATE_DECAY:
				/*
				 * Decay state. Ramp down to sustain level.
				 */
				while (cgain > sustain)
				{
					if (count-- > 0)
						*ob++ = ((cgain *= decay) - 1) * egain;
					else
						break;
					if (rampup >= 0) {
						if (rampup)
						{
							if ((egain+=0.01) > gain)
							{
								egain = gain;
								rampup = -1;
							}
						} else {
							if ((egain-=0.01) < gain)
							{
								egain = gain;
								rampup = -1;
							}
						}
					}
				}
				if (cgain <= sustain)
					local->cstate = STATE_SUSTAIN;
				voice->flags &= ~BRISTOL_KEYDONE;
				break;
			case STATE_SUSTAIN:
				/*
				 * Sustain - fixed output signal.
				 */
				cgain = sustain;
				while (count-- > 0)
				{
					*ob++ = (sustain - 1) * egain;
					if (rampup >= 0) {
						if (rampup)
						{
							if ((egain+=0.01) > gain)
							{
								egain = gain;
								rampup = -1;
							}
						} else {
							if ((egain-=0.01) < gain)
							{
								egain = gain;
								rampup = -1;
							}
						}
					}
				}
				voice->flags &= ~BRISTOL_KEYDONE;
				break;
			default:
				/*
				 * If we have an unknown state, configure it for OFF.
				 */
				local->cstate = STATE_DONE;
				while (count-- > 0)
					*ob++ = 0.0;
				voice->flags |= BRISTOL_KEYDONE;
				break;
		}
	}
	local->cgain = cgain;
	local->egain = egain;
}

/*
 * Setup any variables in our OP structure, in our IO structures, and malloc
 * any memory we need.
 */
bristolOP *
envinit(bristolOP **operator, int index, int samplerate, int samplecount)
{
	bristolENV *specs;

#ifdef BRISTOL_DBG
	printf("envelopeinit(%x(%x), %i, %i, %i)\n",
		operator, *operator, index, samplerate, samplecount);
#endif

	*operator = bristolOPinit(operator, index, samplecount);

	/*
	 * Then the local parameters specific to this operator. These will be
	 * the same for each operator, but must be inited in the local code.
	 */
	(*operator)->operate = operate;
	(*operator)->destroy = destroy;
	(*operator)->reset = reset;
	(*operator)->param = param;

	specs = (bristolENV *) bristolmalloc0(sizeof(bristolENV));
	(*operator)->specs = (bristolOPSpec *) specs;
	(*operator)->size = sizeof(bristolENV);

	/*
	 * These are specific to this operator, and will need to be altered for
	 * each operator.
	 */
	specs->spec.opname = OPNAME;
	specs->spec.description = OPDESCRIPTION;
	specs->spec.pcount = PCOUNT;
	specs->spec.iocount = IOCOUNT;
	specs->spec.localsize = sizeof(bristolENVlocal);

	/*
	 * Now fill in the envelope specs for this operator. These are specific to
	 * an ADSR.
	 */
	specs->spec.param[0].pname = "attack";
	specs->spec.param[0].description = "Initial ramp up rate";
	specs->spec.param[0].type = BRISTOL_FLOAT;
	specs->spec.param[0].low = 0;
	specs->spec.param[0].high = 1;
	specs->spec.param[0].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[1].pname = "decay";
	specs->spec.param[1].description = "Decay to sustain rate";
	specs->spec.param[1].type = BRISTOL_FLOAT;
	specs->spec.param[1].low = 0;
	specs->spec.param[1].high = 1;
	specs->spec.param[1].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[2].pname = "sustain";
	specs->spec.param[2].description = "Output level steady state";
	specs->spec.param[2].type = BRISTOL_FLOAT;
	specs->spec.param[2].low = 0;
	specs->spec.param[2].high = 1;
	specs->spec.param[2].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[3].pname = "release";
	specs->spec.param[3].description = "Final decay rate";
	specs->spec.param[3].type = BRISTOL_FLOAT;
	specs->spec.param[3].low = 0;
	specs->spec.param[3].high = 1;
	specs->spec.param[3].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[4].pname = "gain";
	specs->spec.param[4].description = "Overall signal level";
	specs->spec.param[4].type = BRISTOL_FLOAT;
	specs->spec.param[4].low = 0;
	specs->spec.param[4].high = 1;
	specs->spec.param[4].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	/*
	 * Now fill in the dco IO specs.
	 */
	specs->spec.io[0].ioname = "output";
	specs->spec.io[0].description = "ADSR Envelope Output Signal";
	specs->spec.io[0].samplerate = samplerate;
	specs->spec.io[0].samplecount = samplecount;
	specs->spec.io[0].flags = BRISTOL_DC|BRISTOL_OUTPUT;

	return(*operator);
}

