// Piece demo, to test the framerate of current piece models

#ifdef WIN32
#include <windows.h>
#endif
#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <vector>
#include <string>
#include <math.h>

#include "SDL.h"
#include "glhead.h"
#include "objfile.h"

using namespace std;

// screen width, height, and bit depth
#define SCREEN_WIDTH  640
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16

#define WINDOW_TITLE "piecedemo"

// This is our SDL surface
SDL_Surface *surface;


double xrot=0, yrot=0, zrot=0;
double zoom = -60;
int width=SCREEN_WIDTH, height=SCREEN_HEIGHT;

// Key array
bool keys[SDLK_LAST] = {false};
// Currently SDL only has 5 mouse constants, numbered 1 to 5
bool mouseb[6] = {false};
bool mouseout = false;
int mousex, mousey;
int mousedeltax = 0, mousedeltay = 0;

// Float to store fps count
GLfloat fps = 0;

// Storage for loaded pieces
ObjFile pawn, rook, knight, bishop, queen, king;

void Quit(int);


// function to release/destroy our resources and restoring the old desktop
void quit( int returnCode )
{
	// clean up the window
	SDL_Quit( );

	// and exit appropriately
	exit( returnCode );
}

int initGL( GLvoid );

// function to reset our viewport after a window resize
int resizeWindow( int width, int height )
{
	cout << "Resizing window" << endl;
	// Height / width ration
	GLfloat ratio;
 
	// Protect against a divide by zero
	if ( height == 0 )
		height = 1;

	ratio = ( GLfloat )width / ( GLfloat )height;

	// Setup our viewport.
	glViewport( 0, 0, ( GLsizei )width, ( GLsizei )height );

	// change to the projection matrix and set our viewing volume.
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity( );

	// Set our perspective
	gluPerspective( 45.0f, ratio, 20.0f, 100.0f );

	// Make sure we're chaning the model view and not the projection
	glMatrixMode( GL_MODELVIEW );

	// Reset The View
	glLoadIdentity( );

	initGL();
	return( true );
}


// general OpenGL initialization function
int initGL( GLvoid )
{
	cout << "Initializing OpenGL" << endl;
	// Enable smooth shading
	glShadeModel( GL_SMOOTH );

	// Set the background black
	glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );

	// Depth buffer setup
	glClearDepth( 1.0f );

	// Enables Depth Testing
	glEnable( GL_DEPTH_TEST );

	// The Type Of Depth Test To Do
	glDepthFunc( GL_LEQUAL );

	// Really Nice Perspective Calculations
	glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

	// Make sure that culling is enabled (must be disabled for transparency)
	glEnable(GL_CULL_FACE);

	// Material settings and light positions
	GLfloat light_position[] = { 3.0, 3.0, 5.0, 0.0 };
	GLfloat light_ambient[]= { 0.1f, 0.1f, 0.1f, 1.0f };
	GLfloat light_diffuse[]= { 1.0f, 1.0f, 1.0f, 1.0f };
	GLfloat light_specular[]= {1.0f, 1.0f, 1.0f, 1.0f };
	GLfloat specularmat[]= {1.0f, 1.0f, 1.0f, 1.0f };
	glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
	glEnable( GL_LIGHTING );
	glEnable( GL_LIGHT0 );
	glEnable( GL_NORMALIZE );
	glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
	glEnable( GL_COLOR_MATERIAL );
	glMaterialfv(GL_FRONT, GL_SPECULAR, specularmat);
	glMaterialf(GL_FRONT, GL_SHININESS, 128);

	// Make all the pieces have the same base size
	rook.setScale( pawn.scale() );
	knight.setScale( pawn.scale() );
	bishop.setScale( pawn.scale() );
	queen.setScale( pawn.scale() );
	king.setScale( pawn.scale() );

	
	// Build display lists for all pieces
	pawn.build();
	rook.build();
	knight.build();
	bishop.build();
	queen.build();
	king.build();
	
	return( true );
}

// Here goes our drawing code
int drawGLScene( GLvoid )
{
	// These are to calculate our fps
	static GLint T0     = 0;
	static GLint Frames = 0;
	static int startTime = SDL_GetTicks();
	static int start = SDL_GetTicks();

	// Make sure depth buffer writing before trying to clear it
	glDepthMask(GL_TRUE);
	// Clear The Screen And The Depth Buffer
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	// This resets the matrix and applies global rotation
	glLoadIdentity( );
	glTranslatef( 0.0f, 0.0f, zoom );
	glRotatef( xrot, 1.0f, 0.0f, 0.0f);
	glRotatef( yrot, 0.0f, 1.0f, 0.0f );
	glRotatef( zrot, 0.0f, 0.0f, 1.0f );

	// Storage for arcball matrix
	static double matrix[] = { 	1, 0, 0, 0, 
								0, 1, 0, 0,
								0, 0, 1, 0,
								0, 0, 0, 1 };


	static bool leftdown = false;
	static Vector quant;
	static double quantw;
	if( mouseb[SDL_BUTTON_LEFT] ) {
		// Calculate quaternion from arcball
		static Vector i;
		if( !leftdown ) {
			i.x = ( mousex * 1.0 / (( width - 1) * 0.5 ) ) - 1;
			i.y = 1 - ( mousey * 1.0 / (( height - 1) * 0.5 ) );
			i.z = 0;
			if( i.magnitude() > 1 )
				i.normalize();
			else
				i.z = sqrt(1 - (i.x*i.x + i.y*i.y) );
			leftdown = true;

		}
		Vector e;
		e.x = ( mousex * 1.0 / (( width - 1) * 0.5 ) ) - 1;
		e.y = 1 - ( mousey * 1.0 / (( height - 1) * 0.5 ) );
		e.z = 0;
		if( e.magnitude() > 1 )
			e.normalize();
		else
			e.z = sqrt(1 - (e.x*e.x + e.y*e.y));

		Vector perp;
		perp = i.cross( e );
		if( perp.magnitude() > 1e-5 ) {
			quant = perp;
			quantw = i.dot( e );
		} else {
			quant.x = quant.y = quant.z = 0;
			quantw = 0;
		}

		quant.normalize();
		if( quantw != 0 )
			// Apply rotation, converting from radians
			glRotated( 57.2958*2 * acos( quantw ), quant.x, quant.y, quant.z );

		mousedeltax = 0;
		mousedeltay = 0;
	} else {
		if( leftdown ) {
			// Mouse button released, store cumulative rotation
			glPushMatrix();
			glLoadIdentity();
			if( quantw != 0 )
				glRotated( 57.2958*2 * acos( quantw ), quant.x, quant.y, quant.z );
			glMultMatrixd( matrix );
			glGetDoublev( GL_MODELVIEW_MATRIX, matrix );
			glPopMatrix();

		}
		leftdown = false;
	}

	// Apply previous changes
	glMultMatrixd( matrix );
	
	static double scale = 5;
	glScaled( scale, scale, scale );
	glColor3d( 0.8, 0.8, 0.8 );
	glTranslatef( -3.5*1.0, 0, -1.5*1.0 );

	// Draw the pieces in their proper positions
	for( int j = 0; j < 4; j++ ) {
		for( int i = 0; i < 8; i++ ) {
			glPushMatrix();
			if( j == 1 || j == 2 )
				pawn.draw();
			else {
				if( i == 0 || i == 7 )
					rook.draw();
				else if( i == 1 || i == 6 )
					knight.draw();
				else if( i == 2 || i == 5 )
					bishop.draw();
				else if( i == 3 )
					queen.draw();
				else if( i == 4 )
					king.draw();
			}
			glPopMatrix();
			glTranslatef( 1.0, 0, 0 );
		}
		glTranslatef( -8, 0, 1 );
		if( j == 1 )
			glColor3d( 0.2, 0.2, 0.2 );
	}

	// Right mouse button scales
	if( mouseb[SDL_BUTTON_RIGHT] ) {
		scale += mousedeltay / 2.0;
		mousedeltay = 0;
		mousedeltax = 0;
	}

	// Up and down arrows zoom
	if( keys[SDLK_UP] )
		zoom += 1;
	
	if( keys[SDLK_DOWN] )
		zoom -= 1;

	// Draw it to the screen
	SDL_GL_SwapBuffers( );

	// Gather our frames per second
	Frames++;
	{
		GLint t = SDL_GetTicks();
		if (t - T0 >= 5000) {
		    GLfloat seconds = (t - T0) / 1000.0;
		    fps = Frames / seconds;
		    cout << Frames << " frames in " << seconds << " seconds = " << fps << " FPS" << endl;
		    T0 = t;
		    Frames = 0;
		}
	}
	return( true );
}

int main( int argc, char **argv )
{
	int x;
	vector<string> args(argc);
	for(x=0; x < argc; x++) 
		args[x]=argv[x];

	// Recenter pieces so that the center is at the base of the piece
	pawn.setRecenter( true, false, true );
	knight.setRecenter( true, false, true );
	rook.setRecenter( true, false, true );
	bishop.setRecenter( true, false, true );
	queen.setRecenter( true, false, true );
	king.setRecenter( true, false, true );

	// Load all of the piece models
	if( !pawn.load( "pawn.obj" ) ) {
		cout << "Failed to load pawn.obj" << endl;
		return 1;
	}
	
	if( !rook.load( "rook.obj" ) ) {
		cout << "Failed to load rook.obj" << endl;
		return 1;
	}
	
	if( !knight.load( "knight.obj" ) ) {
		cout << "Failed to load knight.obj" << endl;
		return 1;
	}
	
	if( !bishop.load( "bishop.obj" ) ) {
		cout << "Failed to load bishop.obj" << endl;
		return 1;
	}
	
	if( !queen.load( "queen.obj" ) ) {
		cout << "Failed to load queen.obj" << endl;
		return 1;
	}
	
	if( !king.load( "king.obj" ) ) {
		cout << "Failed to load king.obj" << endl;
		return 1;
	}

	// Generate smooth normals for the models
	pawn.findNorms();
	rook.findNorms();
	knight.findNorms();
	bishop.findNorms();
	queen.findNorms();
	king.findNorms();
	
	// Flags to pass to SDL_SetVideoMode
    int videoFlags;
	// main loop variable
	int done = false;
	// used to collect events
	SDL_Event event;
	// this holds some info about our display
	const SDL_VideoInfo *videoInfo;
	// whether or not the window is active
	int isActive = true;
	
	// initialize SDL
	if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
		cerr << "Video initialization failed:" << SDL_GetError() << endl;
		quit( 1 );
	}


	// Fetch the video info
	videoInfo = SDL_GetVideoInfo( );

    	if ( !videoInfo )
	{
		cerr << "Video query failed: " << SDL_GetError() << endl;
	 	quit( 1 );
	}

	// the flags to pass to SDL_SetVideoMode
	videoFlags  = SDL_OPENGL;          // Enable OpenGL in SDL
	videoFlags |= SDL_GL_DOUBLEBUFFER; // Enable double buffering
	videoFlags |= SDL_HWPALETTE;       // Store the palette in hardware
	videoFlags |= SDL_RESIZABLE;       // Enable window resizing

	// This checks to see if surfaces can be stored in memory
	if ( videoInfo->hw_available )
		videoFlags |= SDL_HWSURFACE;
	else
		videoFlags |= SDL_SWSURFACE;

	// This checks if hardware blits can be done
	if ( videoInfo->blit_hw )
		videoFlags |= SDL_HWACCEL;

	// Sets up OpenGL double buffering
	SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

	// get a SDL surface
	SDL_WM_SetCaption(WINDOW_TITLE,NULL);
	surface = SDL_SetVideoMode( SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_BPP,
			videoFlags );

	// Verify there is a surface
	if ( !surface )
	{
		cerr << "Video mode set failed: " << SDL_GetError( ) << endl;
		quit( 1 );
	}

	// initialize OpenGL
	initGL( );

	// resize the initial window
	resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

	SDL_GetMouseState(&mousex, &mousey);
	// wait for events
	int frameCount = 0;

	while ( !done )
	{
	
		// handle the events in the queue

		while ( SDL_PollEvent( &event ) )
		{
			switch( event.type )
			{
				case SDL_MOUSEBUTTONDOWN:
					mousex = event.button.x;
					mousey = event.button.y;	
					mouseb[event.button.button] = true;
					break;
				case SDL_MOUSEBUTTONUP:
					mousex = event.button.x;
					mousey = event.button.y;
					mouseb[event.button.button] = false;
					break;
				case SDL_MOUSEMOTION:	
			        //printf("Mouse moved to (%d,%d)\n", event.motion.x, event.motion.y);
					mousex = event.motion.x;
					if( event.motion.state ) {
						mousedeltax += event.motion.xrel;
						mousedeltay += event.motion.yrel;
					}
					mousey = event.motion.y;
					break;
				case SDL_ACTIVEEVENT:
					// Something's happend with our focus
					// If we lost focus or we are iconified, 					 
					// we shouldn't draw the screen
					if ( event.active.gain == 0 && event.active.state!=SDL_APPMOUSEFOCUS) {
						cout << "inact" <<endl;
						isActive = false;
					} else
						isActive = true;

					if ( event.active.gain == 0 && 
						event.active.state ==
						SDL_APPMOUSEFOCUS ) {
						mouseout = true;
						cout << "mouseout" << endl;
					} else if ( event.active.gain == 1 && 
						event.active.state ==
						SDL_APPMOUSEFOCUS ) {
						mouseout = false;
						cout << "mousein" << endl;
					}
					break;			    
				case SDL_VIDEORESIZE:
					// handle resize event
					surface = SDL_SetVideoMode( 
							event.resize.w,
							event.resize.h,
							16, videoFlags );
					if ( !surface )
					{
				    		cerr << "Could not get a surface after resize: " << SDL_GetError( ) << endl;
						quit( 1 );
					}
			    		resizeWindow( event.resize.w, 
							event.resize.h );
					width = event.resize.w;
					height = event.resize.h;
					break;
				case SDL_KEYDOWN:
					keys[ event.key.keysym.sym ] = true;
					// handle key presses
					break;
				case SDL_KEYUP:
					keys[ event.key.keysym.sym ] = false;
					break;
				case SDL_QUIT:
					// handle quit requests
					done = true;
					break;
				default:
				    break;
			}
		}
	
		if(keys[SDLK_ESCAPE] || keys[SDLK_q])
			quit( 0 );
		if(keys[SDLK_F1]) {
			//This only works with X
			//In order to get it to work with Win32 the surface must be recreated
			SDL_WM_ToggleFullScreen( surface );
			keys[SDLK_F1]=false;
		}
		if ( isActive )
			drawGLScene( );
	}

	// clean ourselves up and exit
	quit( 0 );

	// Should never get here
	return( 0 );
}
