/********************************************************************
 * Copyright (C) 2005,2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
--------------------------------------------------------------------*/
#include "Shared.h"
#include "Events.h"
#include "WipeFile.h"
#include <qdir.h>
#include <qstring.h>
#include <qiconset.h>
#include <qfileinfo.h>
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qapplication.h>
#include <qfile.h>
#include <qapplication.h>
#include <qdesktopwidget.h>
#include <qeventloop.h>

/*------- constants:
--------------------------------------------------------------------*/
const int     Shared::LayoutMargin          = 4;
const int     Shared::LayoutSpacing         = 4;
const int     Shared::OverStretch           = 100;
//...................................................................
const char* const Shared::MyDir                 = ".bsc";
const char* const Shared::MyTmpDir              = "/.tmp";
const char* const Shared::MyPackDir             = "/.pack";
const char* const Shared::LicenceFileName       = "licence.txt";
const char* const Shared::ThisDir               = ".";
const char* const Shared::ParentDir             = "..";
const char* const Shared::ShellCd               = "cd";
const char* const Shared::DirSep                = "/";
const char* const Shared::DateTimeMuster        = "dd/MM/yyyy hh:mm:ss";
//...................................................................
const char* const Shared::CmpDirIcon            = "cmpdir.png";
const char* const Shared::CloseIcon             = "close.png";
const char* const Shared::AcceptIcon            = "accept.png";
const char* const Shared::SaveIcon              = "filesave.png";
const char* const Shared::ReloadIcon            = "reload.png";
const char* const Shared::ApplyIcon             = "apply.png";
const char* const Shared::BreakIcon             = "break.png";
const char* const Shared::ResetIcon             = "redo.png";
const char* const Shared::TreeIcon              = "tree.png";
//...................................................................
const char* const Shared::DeletingFileMsg       = QT_TR_NOOP( "Deleting file: %1" );
const char* const Shared::FTP_Unconnected       = QT_TR_NOOP( "Unconnected." );
const char* const Shared::FTP_HostLookup        = QT_TR_NOOP( "A host name lookup." );
const char* const Shared::FTP_Connecting        = QT_TR_NOOP( "Connecting." );
const char* const Shared::FTP_Connected         = QT_TR_NOOP( "Connected." );
const char* const Shared::FTP_LoggedIn          = QT_TR_NOOP( "Session is established." );
const char* const Shared::FTP_Closing           = QT_TR_NOOP( "The connection is closing down, but it is not yet closed." );
const char* const Shared::FTP_Unknown           = QT_TR_NOOP( "Unknown status." );
const char* const Shared::MsgBoxDelErrorCaption = QT_TR_NOOP( "Deleting error" );
const char* const Shared::MsgBoxDelFileError    = QT_TR_NOOP( "Can't remove a file:\n%1" );
const char* const Shared::MsgBoxDelDirError     = QT_TR_NOOP( "Can't remove a directory:\n%1" );
const char* const Shared::NotReadableDir        = QT_TR_NOOP( "You can't read from this directory: %1.\nCheck permission access." );
const char* const Shared::MenuHelp              = QT_TR_NOOP( "Help" );
const char* const Shared::MenuAccess            = QT_TR_NOOP( "Access" );
const char* const Shared::MenuView              = QT_TR_NOOP( "View" );
const char* const Shared::MenuEdit              = QT_TR_NOOP( "Edit" );
const char* const Shared::MenuCopy              = QT_TR_NOOP( "Copy" );
const char* const Shared::MenuRename            = QT_TR_NOOP( "Rename" );
const char* const Shared::MenuMkDir             = QT_TR_NOOP( "MkDir" );
const char* const Shared::MenuDelete            = QT_TR_NOOP( "Delete" );
const char* const Shared::MenuPack              = QT_TR_NOOP( "Pack" );
const char* const Shared::MenuQuit              = QT_TR_NOOP( "Quit" );
const char* const Shared::MenuEmpty             = QT_TR_NOOP( "Empty" );
//...................................................................
const char* const Shared::AcceptBtnLabel        = QT_TR_NOOP( "&Accept" );
const char* const Shared::ApplyBtnLabel         = QT_TR_NOOP( "&Apply" );
const char* const Shared::BreakBtnLabel         = QT_TR_NOOP( "&Break" );
const char* const Shared::CancelBtnLabel        = QT_TR_NOOP( "&Cancel" );
const char* const Shared::CloseBtnLabel         = QT_TR_NOOP( "&Close" );
const char* const Shared::ContinueBtnLabel      = QT_TR_NOOP( "&Continue" );
const char* const Shared::NoBtnLabel            = QT_TR_NOOP( "&No" );
const char* const Shared::OkBtnLabel            = QT_TR_NOOP( "&Ok" );
const char* const Shared::ReloadBtnLabel        = QT_TR_NOOP( "&Reload" );
const char* const Shared::ResetBtnLabel         = QT_TR_NOOP( "&Reset" );
const char* const Shared::RunBtnLabel           = QT_TR_NOOP( "&Run" );
const char* const Shared::SaveBtnLabel          = QT_TR_NOOP( "&Save" );
const char* const Shared::YesBtnLabel           = QT_TR_NOOP( "&Yes" );
const char* const Shared::SelectBtnLabel        = QT_TR_NOOP( "&Select" );
//...................................................................
const char* const Shared::Information           = QT_TR_NOOP( "Information" );
const char* const Shared::Yes                   = QT_TR_NOOP( "Yes" );
const char* const Shared::No                    = QT_TR_NOOP( "No" );
const char* const Shared::KonsoleCall           = "konsole --workdir $dir";
//...................................................................
const QChar   Shared::Point                     = '.';
const QChar   Shared::Colon                     = ':';
const QChar   Shared::Semicolon                 = ';';
const QChar   Shared::Stern                     = '*';
const QChar   Shared::Spc                       = ' ';
const QChar   Shared::WidthMarker               = 'X';
const QChar   Shared::Ampersand                 = '&';
const QString Shared::TouchPrgName              = "touch";
const QString Shared::VersionMajor              = "2";
const QString Shared::VersionMinor              = "27";
const QString Shared::ProgramVersion            = VersionMajor + "." + VersionMinor;
const QString Shared::ProgramName               = "BSCommander";
const QString Shared::EmptyStr                  = "";

bool Shared::d_break = FALSE;

/*------- static variables:
-------------------------------------------------------------------*/
QString Shared::d_home_directory    = EmptyStr;
bool    Shared::d_file_filter_activ = TRUE;
QString Shared::d_tmp_buffer;

//*******************************************************************
// toggle_file_filter
//*******************************************************************
void Shared::toggle_file_filter()
{
	d_file_filter_activ = ( d_file_filter_activ ) ? FALSE : TRUE;
}
void Shared::toggle_file_filter( const bool in_flag )
{
	d_file_filter_activ = in_flag;
}
// end of toggleFilter

//*******************************************************************
// get_file_filter
//*******************************************************************
int Shared::get_file_filter()
{
	return ( d_file_filter_activ ) ? ( QDir::All ) : ( QDir::All | QDir::Hidden | QDir::System );
}
// end of getFileFilter

//********************************************************************
// get_home_dir
//********************************************************************
QString Shared::get_home_dir()
{
	if( TRUE == d_home_directory.isEmpty() ) {
		d_home_directory = QDir::homeDirPath();
	}
	return d_home_directory;
}
// end of get_home_dir

//*******************************************************************
// get_my_dir
//*******************************************************************
QString Shared::get_my_dir()
{
	return MyDir;
}
// end of get_my_dir

//*******************************************************************
// get_tmp_dir
//*******************************************************************
QString Shared::get_tmp_dir()
{
	return DirSep + get_my_dir() + MyTmpDir;
}
// end of get_tmp_dir

//*******************************************************************
// get_pack_dir
//*******************************************************************
QString Shared::get_pack_dir()
{
	return get_tmp_dir() + MyPackDir;
}
// end of get_pack_dir;

//*******************************************************************
// get_subpath
//-------------------------------------------------------------------
// Uzytkownik przesyla dwie sciezki katalogow.
// Katalog bazowa i jakis jego podkatalog (zaglebienie nie wazne).
// Funkcja wyodrebnia wzgledna sciezke katalogow.
// Wzgledna oczywiscie w stosunku do katalogu bazowego.
// 
// WYNIK:
// 1) "" jesli katalog bazowy w rzeczywistosci nie jest katalogiem
//    bazowym dla 'currentPath',
// 2) "/" jesli oba katalogi sa takie same,
// 3) ciag podkatalogow poczawszy od katalogu bazowego.
//*******************************************************************
QString Shared::get_subpath( const QString& in_base_path, const QString& in_current_path )
{
	QString result = EmptyStr;
	
	if( in_current_path.startsWith( in_base_path ) ) {
		result = in_current_path.mid( in_base_path.length() );
		if( result.isEmpty() ) {
			result = DirSep;
		}
	}
	return result;
}
// end of get_subpath

//*******************************************************************
// create_path
//-------------------------------------------------------------------
// Tworzenie nowego zaglebionego podkatalogu, ktorego wiele
// nadkatogow nie istnieje.
// Poczawszy od istniejacego katalogu nalezy krokowo tworzyc 
// kolejne nieistniejace katalogi.
//*******************************************************************
bool Shared::create_path( const QString& in_existing, const QString& in_subdirs )
{
	QDir dir;
	
	bool retval = dir.cd( in_existing );
	if( retval ) {
		QStringList subdir_names = QStringList::split( DirSep, in_subdirs );
		QStringList::Iterator it = subdir_names.begin();
		while(( TRUE == retval ) && (  it != subdir_names.end()) ) {
			if( FALSE == dir.exists( *it, FALSE ) ) {
				retval = dir.mkdir( *it, FALSE );
			}
			if( TRUE == retval ) {
				retval = dir.cd( *it, FALSE );
			}
			++it;
		}
	}
	return retval;
}
// end of create_path

//*******************************************************************
// remove_path_and_content
//*******************************************************************
bool Shared::remove_path_and_content( const QString& in_path, const bool in_silent, const bool in_wipe, QDialog* const in_dialog )
{
    if( d_break ) return FALSE;
    
    const QDir dir( in_path );
    const QFileInfoList* files = dir.entryInfoList( QDir::All|QDir::Hidden|QDir::System );

    Shared::idle();	
    if( files ) {
        QFileInfo* fi = 0;
        QFileInfoListIterator it( *files );
        while(( FALSE == d_break ) && (( fi = it.current() ) != 0 )) {
            idle();
            QString fname = fi->fileName();
            if( FALSE == fname.isEmpty() ) {
                if( is_regular_file( fname ) ) {
                    const QString abs_path = fi->absFilePath();
                    if( TRUE == fi->isDir() ) {
                        if( FALSE == remove_path_and_content( abs_path, in_silent, in_wipe, in_dialog )) {
                            // jesli uzytkownik zarzadzil przerwanie operacji zwracamy FALSE
                            return FALSE;
                        }
                    }
                    else {
                        if( FALSE == remove_file( abs_path, in_silent, in_wipe, in_dialog )) {
                            // jesli uzytkownik zarzadzil przerwanie operacji zwracamy FALSE.
                            return FALSE;
                        }
                    }
                }
            }
            ++it;
        }
    }
	
    /* Poniewaz katalog jest juz pusty mozna go usunac. */
    if( d_break ) return FALSE;
    bool retval = dir.rmdir( in_path );
    if( FALSE == retval ) {
        if( FALSE == in_silent ) {
            retval = can_not_delete( in_path, DIRECTORY );
        }
    }
    return retval;
}
// end of remove_dir_and_content

//*******************************************************************
// remove_file
//*******************************************************************
bool Shared::remove_file( const QString& in_fpath, const bool in_silent, const bool in_wipe, QDialog* const in_dialog )
{
    if( d_break ) return FALSE;

    if( in_wipe ) {
        WipeFile* wiper = new WipeFile( in_fpath, in_dialog );
        if( wiper ) {
            wiper->wipe();
            delete wiper;
            wiper = 0;
        }
        idle();
    }
    
    if( d_break ) return FALSE;

    if( in_dialog ) {
        BscEvent* event = new BscEvent( DeleteFileEvent );
        if( event ) {
            event->m_message = tr(DeletingFileMsg).arg( in_fpath );
            QApplication::postEvent( in_dialog, event );
            idle();
        }
    }
    
    if( d_break ) return FALSE;
    
    bool retval = TRUE;
    if( FALSE == QFile::remove( in_fpath ) ) {
        retval = FALSE;
        if( FALSE == in_silent ) {
            retval = can_not_delete( in_fpath, FILE );
        }
	}
	return retval;
}
// end of remove_file

//*******************************************************************
// can_not_delete                                            PRIVATE
//-------------------------------------------------------------------
// Dialog z informacja, ze nie mozna usunac pliku.
// Uzytkownik moze wskazac dwie reakcje na ten stan rzeczy:
// kontynuacje (zwracana wartosc TRUE ), lub przerwanie operacji
// (zwracana wartosc FALSE ).
//*******************************************************************
bool Shared::can_not_delete( const QString& in_fpath, const int in_type )
{
	int retval = -1;
		
	switch( in_type )
	{
		case FILE:
			retval = QMessageBox::critical(	0,
							tr(MsgBoxDelErrorCaption),
							tr(MsgBoxDelFileError).arg( in_fpath ),
							tr(BreakBtnLabel),
							tr(ContinueBtnLabel) );
			break;
		case DIRECTORY:
			retval = QMessageBox::critical(	0,
							tr(MsgBoxDelErrorCaption),
							tr(MsgBoxDelDirError).arg( in_fpath ),
							tr(BreakBtnLabel),
							tr(ContinueBtnLabel) );
			break;
	}
	
	return ( 1 == retval );
}
// end of can_not_delete

//*******************************************************************
// add_icon
//*******************************************************************
void Shared::add_icon( QPushButton* const in_button, const QString& in_icon_name )
{
	if( in_button ) {
		const QPixmap image = QPixmap::fromMimeSource( in_icon_name );
		const QIconSet icons( image );
		in_button->setIconSet( icons );
	}
}
// end of add_icon

//*******************************************************************
// is_binary_file                                             PUBLIC
//-------------------------------------------------------------------
// Sprawdzamy czy wskazany plik jest plikiem binarnym.
// UWAGA:
// 	Nie wykonujemy tutaj kontroli praw dostepu do pliku.
// 	Zakladamy, ze mamy prawo do jego odczytu.
//		Kontrola powinna 	byc zostac wykonana wczesniej.
//*******************************************************************
bool Shared::is_binary_file( const QString& in_fpath )
{
	bool retval = FALSE;
	
	QFile file( in_fpath );
	
	if( TRUE == file.open( IO_ReadOnly ) ) {
		QDataStream in( &file );
		char buffer[ 8 ];
		in.readRawBytes( buffer, 4 );
		retval = (( 'E' == buffer[1] ) && ( 'L' == buffer[2] ) && ( 'F' == buffer[3] ));
		file.close();
	}
	return retval;
}
// end of is_binary_file

//*******************************************************************
// is_executable_file                                         PUBLIC
//-------------------------------------------------------------------
// Za plik eykonywalny uznajemy plik binarny z naglowkiem ELF i
// ustawionym bitem wykonalnosci.
//*******************************************************************
/*
bool Shared::is_executable_file( const QString& in_path )
{
	bool retval = FALSE;
	
	if( TRUE == is_binary_file( in_path ) ) {
		const QFileInfo fi( in_path );
		retval = fi.isExecutable();
	}
	return retval;
}
*/
// end of is_executable_file

//*******************************************************************
// idle
//*******************************************************************
void Shared::idle()
{
	QApplication::eventLoop()->processEvents( QEventLoop::AllEvents );
}
// end of idle

//*******************************************************************
// make_path
//*******************************************************************
QString Shared::make_path( const QString& in_dir, const QString& in_name )
{
	QString path = in_dir;
	if( path != DirSep ) path += DirSep;
	path += in_name;
	return path;
}
// end of make_path

//*******************************************************************
// polish
//*******************************************************************
void Shared::polish( QWidget* const in_widget, const int in_width, const int in_height )
{
	const QRect screen  = qApp->desktop()->availableGeometry();
	const int   screen_width = screen.width();
	const int   screen_height = screen.height();
	const float width_percent = static_cast<float>( in_width ) / 100.0;
	const float height_percent = static_cast<float>( in_height ) / 100.0;	
	const int   dx = static_cast<int>( screen_width * width_percent );
	const int   dy = static_cast<int>( screen_height * height_percent );

	polish( in_widget, QSize( dx, dy ));
}
void Shared::polish( QWidget* const in_widget, const QSize& in_size )
{
	const int screen_width  = qApp->desktop()->availableGeometry().width();
	const int screen_height = qApp->desktop()->availableGeometry().height();
	const int x  = ( screen_width  - in_size.width()  + 1 ) >> 1;
	const int y  = ( screen_height - in_size.height() + 1 ) >>  1;
    
	in_widget->move( x, y );
	in_widget->resize( in_size.width(), in_size.height() );
}
void Shared::polish_width( QWidget* const in_widget, const int in_width )
{
	const int screen_width  = qApp->desktop()->screenGeometry( 0 ).width();
	const int screen_height = qApp->desktop()->screenGeometry( 0 ).height();
	const float width_percent = ( static_cast<float>( in_width ) ) / 100.0;
	
	QSize new_size = in_widget->size();
	new_size.setWidth( static_cast<int>( screen_width * width_percent ));
	in_widget->resize( new_size );
	
	const int x  = ( screen_width  - new_size.width()  + 1 ) >> 1;
	const int y  = ( screen_height - new_size.height() + 1 ) >>  1;
	in_widget->move( x, y );
}
// end of polish

//*******************************************************************
// num2str
//*******************************************************************
const QString& Shared::num2str( const Q_ULLONG in_value, const char sep )
{
	d_tmp_buffer = QString::number( in_value );

   unsigned int idx = d_tmp_buffer.length() - 1;
   unsigned int n   = 0;
   
   while( idx > 0 ) {
      if( n == 2 ) {
         d_tmp_buffer.insert( idx, sep );
         n = 0;
      }
      else {
         ++n;
      }
      --idx;
   }
 
   return d_tmp_buffer;
}
// end of num2str

//*******************************************************************
// clip_path
//*******************************************************************
void Shared::clip_path( const QFontMetrics& in_fm, const int in_width, QString& in_out_path )
{
	const int MAXLEN = in_width;
	
	if( in_fm.width( in_out_path ) > MAXLEN ) {
		const QString     SLASH  = "/";
		const QString     SPACER = "/.../";
		const QStringList WORDS  = QStringList::split( SLASH, in_out_path );

		int     i        = 0;
		int     j        = WORDS.count() - 1;
		QString lft_part = "";
		QString rgt_part = WORDS[j--];
		bool    lft_ok   = TRUE;
		bool    rgt_ok   = TRUE;
		QString tmp;

		while( lft_ok || rgt_ok ) {
			lft_ok = FALSE;
			if( i < j ) {
				tmp = lft_part + SLASH + WORDS[i];
				if( in_fm.width( tmp + SPACER + rgt_part ) < MAXLEN ) {
					lft_part = tmp;
					lft_ok = TRUE;
					++i;
				}
			}
			//.............................................
			rgt_ok = FALSE;
			if( i < j ) {
				tmp = WORDS[j] + SLASH + rgt_part;
				if( in_fm.width( lft_part + SPACER + tmp ) < MAXLEN ) {
					rgt_part = tmp;
					rgt_ok = TRUE;
					--j;
				}
			}
		}
		in_out_path = lft_part + SPACER + rgt_part;
	}
}
// end of clip_path

//*******************************************************************
// clip_path
//*******************************************************************
void Shared::clip_path( const QString& in_path, QString& out_dir, QString& out_name )
{
	out_dir  = EmptyStr;
	out_name = EmptyStr;

	if( FALSE == in_path.isEmpty() ) {
		const int idx = in_path.findRev( DirSep );
		if( idx != -1 ) {
			if( idx != 0 ) {
				out_dir = in_path.left( idx );
				out_name = in_path.mid( idx + 1 );
			}
			else {
				out_dir = in_path;
			}
		}
		else {
			out_name = in_path;
		}
	}
}
// end of clip_path

//*******************************************************************
// is_dir_ok
//*******************************************************************
bool Shared::is_dir_ok( const QString& in_dname )
{
	bool retval = FALSE;
	
	if( in_dname ) {
		const QFileInfo fi( in_dname );
		retval = ( fi.isDir() && fi.exists() && fi.isExecutable() && fi.isReadable() );
	}
	
	return retval;
}
// end of is_dir_ok

//*******************************************************************
// is_white_spc
//*******************************************************************
bool Shared::is_white_char( const QChar& in_char )
{
	const QChar white_chars[] = { '\a', '\b', '\f', '\n', '\r', '\t', '\v', Spc };
	const int  n = sizeof( white_chars ) / sizeof( white_chars[0] );
	bool retval = FALSE;
	
	for( int i = 0; i < n; ++i ) {
		if( white_chars[i] == in_char ) {
			retval = TRUE;
			break;
		}
	}
	
	return retval;
}
// end of is_white_spc

//*******************************************************************
// remove_white_chars
//*******************************************************************
bool Shared::remove_white_chars( QString& in_out_txt )
{
	QString result = EmptyStr;
	bool was_white_char = FALSE;
	
	for( unsigned int i = 0; i < in_out_txt.length(); ++i ) {
		const QChar current_char = in_out_txt.at( i );
		if( is_white_char( current_char ) ) {
			if( FALSE == was_white_char ) {
				result += Spc;
				was_white_char = TRUE;
			}
		}
		else {
			result += current_char;
			was_white_char = FALSE;
		}
	}
	
	in_out_txt = result.stripWhiteSpace();
	return ( FALSE == in_out_txt.isEmpty() );
}
// end of remove_white_chars

//*******************************************************************
// is_regular_file
//*******************************************************************
bool Shared::is_regular_file( const QString& in_fname )
{
	bool retval = FALSE;
	if( in_fname ) {
		retval = (( Shared::ThisDir != in_fname ) && ( Shared::ParentDir != in_fname ));
	}
	return retval;
}
// end of is_regular_file

//*******************************************************************
// auto_rename
//*******************************************************************
void Shared::auto_rename( QString& inout_path )
{
	const QFileInfo fi( inout_path );
	const QString fname = fi.baseName();
	QString fext = fi.extension();
	if( FALSE == fext.isEmpty() ) fext.insert( 0, '.' );
	QString dir = fi.dirPath( TRUE );
	if( dir != "/" ) dir += "/";

	int i = 1;
	bool found = FALSE;
	QString new_fname;
	
	while( i < 1000 ) {
		new_fname.sprintf( "%s_%03d", fname.ascii(), i++ );
		if( FALSE == QFile::exists( dir + new_fname + fext ) ) {
			found = TRUE;
			break;
		}
	}

	if( found ) {
		inout_path = dir + new_fname + fext;
	}
}
// end of auto_rename

//*******************************************************************
// is_user_root
//*******************************************************************
bool Shared::is_user_root()
{
	const QFileInfo root_dir( "/root" );
	return ( root_dir.isExecutable() && root_dir.isReadable() );	
}
// end of is_user_root

//*******************************************************************
// get_fext
//*******************************************************************
QString Shared::get_fext( const QString& in_fpath )
{
    QString result( "" );

    if( !in_fpath.isEmpty() ) {
        const int idx = in_fpath.findRev( QChar( '.' ) );
        if( idx > 0 ) {
            result = in_fpath.mid( idx + 1 );
        }
    }
    return result;
}
// end of get_fext
