# Copyright (C) 2009 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Extension types and collections of them."""

from bzrlib import registry

from bzrlib.plugins.explorer.lib import kinds


# Initial content templates. For files, a string is given.
# For directories, a dictionary mapping filenames to strings is given.
# Content strings may content the following markers:
#
# * %(title)s - the accessory title
#

_BOOKMARKS_TEMPLATE = """
<folder title="%(title)s Bookmarks">
</folder>
"""

_EDITORS_TEMPLATE = """
# Editor applications for various file extensions on various desktops.
# A DEFAULT section can be given providing cross-platform defaults.
# The recognised desktop names are gnome, kde, windows and osx.
# Within each section, the key is a comma separated list of file
# extensions and the value is the name of the application to edit
# the file with. For example:
#
#   [DEFAULT]
#   doc, xls, ppt = ooffice
# 
#   [windows]
#   vsd, vss, vst, vdx, vsx, vtx = visio
"""

_LOGOS_TEMPLATE = {
    "readme.txt": "This directory may contain 16x16 icons in png format.",
    }

_TOOLS_TEMPLATE = """
<folder title="%(title)s Tools">
</folder>
"""

_TOOLBARS_TEMPLATE = """
<folder title="%(title)s Toolbars">
    <!--
    This file contains toolbar definitions, one per folder.
    Folder titles are used as follows:

    * toolset:prefix - all toolbars start with these actions
    * toolset:suffix - all toolbars end with these actions
    * toolset:sss - a set of items that can be included in toolbars
    * kind:kkk - the toolbar for locations with kind kkk
    * bbb - a toolbar called bbb
    
    Most toolbars are lists of actions, toolsets and separators.
    You may use custom tools and folders as well. See
    http://doc.bazaar.canonical.com/explorer/en/tutorials/customization.html
    for the list of available action and kind names.
    -->
</folder>
"""


class ExtensionType(object):

    def __init__(self, label, icon_kind, filename, content_template):
        self.label = label
        self.icon_kind = icon_kind
        self.filename = filename
        self.content_template = content_template


## Registry of extension types ##

extension_type_registry = registry.Registry()

extension_type_registry.register("bookmarks", ExtensionType("Bookmarks",
    kinds.BOOKMARKS_EXT, "bookmarks.xml", _BOOKMARKS_TEMPLATE))
extension_type_registry.register("editors", ExtensionType("Editors",
    kinds.EDITORS_EXT, "editors.conf", _EDITORS_TEMPLATE))
extension_type_registry.register("logos", ExtensionType("Logos",
    kinds.LOGOS_EXT, "logos", _LOGOS_TEMPLATE))
extension_type_registry.register("tools", ExtensionType("Tools",
    kinds.TOOLS_EXT, "tools.xml", _TOOLS_TEMPLATE))
extension_type_registry.register("toolbars", ExtensionType("Toolbars",
    kinds.TOOLBARS_EXT, "toolbars.xml", _TOOLBARS_TEMPLATE))
