/*--------------------------------------------------------------------*/
/*--- Callgrind                                                    ---*/
/*---                                                       data.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Callgrind, a Valgrind tool for call tracing.

   Copyright (C) 2002-2004, Josef Weidendorfer (Josef.Weidendorfer@gmx.de)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "global.h"
#include "hash.h"

/*------------------------------------------------------------*/
/*--- Tracing Data Allocations                             ---*/
/*------------------------------------------------------------*/

typedef struct _SubEntry   SubEntry;
typedef struct _DataType   DataType;
typedef struct _DataObject DataObject;
typedef struct _TypeList   TypeList;
typedef struct _ObjectList ObjectList;

struct _SubEntry {
  DataType* type;
  UInt offset;
  UInt size;    /* Array if greater than type->size */
};

/* How a type was deduced:
 * - CLASS    : by trapped a C++ constructor call, has class name
 * - MALLOCED : allocated chunk of memory. name is call context
 * - STATIC   : static variable, with type from debug info
 * - STACK    : local variable, with type (call context, debug info)
 */
#define TYPE_UNKNOWN   0
#define TYPE_CLASS     1  
#define TYPE_MALLOCED  2
#define TYPE_STATIC    3
#define TYPE_STACK     4

struct _DataType {
  // this is an entry of a rhash
  rhash_entry* next;
  UInt hash;

  char* name;
  UInt number;

  ULong read_cost, write_cost;
  UInt allocCount, freeCount, allocSize, freeSize;

  UInt size;
  UInt subentries, capacity;
  SubEntry e[1];
};

struct _DataObject {
  DataType* type;
  Char* name;
  UInt addr, size;

  ULong* cost; /* allocated lazy */
};


/*------------------------------------------------------------*/
/*--- Data types and objects                               ---*/
/*------------------------------------------------------------*/

UInt type_counter = 0;

static UInt str_hash(const UChar *s)
{
    int hash_value = 0;
    for ( ; *s; s++)
        hash_value = (253 * hash_value + *s);
    return hash_value;
}

Bool is_type(rhash_entry* e, void* data)
{
  DataObject* o = (DataObject*) data;
  DataType* t   = (DataType*) e;
  return (VG_(strcmp)(o->name, t->name)==0);
}

rhash_entry* new_type(void* data)
{
  DataObject* o = (DataObject*) data;
  DataType* new = VG_(malloc)(sizeof(DataType));
  new->hash   = str_hash(o->name);
  new->name   = VG_(strdup)(o->name);
  new->number = type_counter++;
  new->read_cost = 0;
  new->write_cost = 0;
  new->allocCount = 0;
  new->freeCount = 0;
  new->allocSize = 0;
  new->freeSize = 0;

  CT_DEBUG(3," new_type(%s)\n", new->name);

  return (rhash_entry*) new;
}


rhash type_hash = {
  .name = "TypeHash",
  .size = 577,
  .new_entry = new_type,
  .has_key = is_type
};

static
void init_type_table()
{
  SK_(rhash_init)(&type_hash);
}
  
static
DataType* get_type(DataObject* o)
{
  return (DataType*) SK_(rhash_get)(&type_hash, str_hash(o->name), o);
}

static
void print_type(rhash_entry* e)
{
  DataType* t = (DataType*) e;
 
  CT_DEBUG(0,"T %5d /%4d Objs %6d /%5d %8llu R %8llu W %s\n",
	   t->allocCount, t->allocCount - t->freeCount,
	   t->allocSize, t->allocSize - t->freeSize,
	   t->read_cost, t->write_cost, t->name);
}

static
void print_type_table()
{
  CT_DEBUG(0,"Entries of Type Table (%d):\n", type_hash.entries);
  SK_(rhash_forall)(&type_hash, print_type);
}

static
DataObject* new_object(char* name, UInt addr, UInt size)
{
  DataObject* o = (DataObject*) VG_(malloc)(sizeof(DataObject));

  o->name  = VG_(strdup)(name);
  o->addr  = addr;
  o->size  = size;
  o->cost  = 0;
  o->type  = get_type(o);

  CT_DEBUG(3," new_object(%s): addr %p, size %d\n", o->name, o->addr, o->size);

  return o;
}


/*------------------------------------------------------------*/
/*--- Page Table                                           ---*/
/*------------------------------------------------------------*/

/*
 * The page table is used for fast mapping of an address to
 * an DataObject, which covers an address range.
 *
 * Functions:
 *  void add_object(DataObject*)
 *  void remove_object(DataObject*)
 *  DataObject* get_object(addr)
 */

typedef struct _TablePos TablePos;
struct _TablePos {
  UInt addr;
  UInt pos1, pos2, pos3;
  UInt *table2, *table3;
};

/* 2 lower bits of each table entry encode the type.
 * The pointers itself have to be 4-byte aligned.
 */
#define ENTRY_TYPE_MASK 3
#define ENTRY_INVALID   0
#define ENTRY_OBJECT    1
#define ENTRY_SUBTABLE  2

/* 3-level page table for mapping address->data structure
 * For now, only 32-bit addresses, split 12:10:10
 */
#define TABLE_LEVEL1_SIZE  4096
#define TABLE_LEVEL1_BITS  12
#define TABLE_LEVEL1_SHIFT 20

#define TABLE_LEVEL2_SIZE  1024
#define TABLE_LEVEL2_BITS  10
#define TABLE_LEVEL2_SHIFT 10

#define TABLE_LEVEL3_SIZE 1024

UInt  data_table[TABLE_LEVEL1_SIZE];

void init_page_table()
{
  Int i;

  for(i=0;i<TABLE_LEVEL1_SIZE;i++)
    data_table[i] = 0;
}

static
UInt* new_l2table()
{
  Int i;
  UInt* t = (UInt*) VG_(malloc)(sizeof(UInt)*TABLE_LEVEL2_SIZE);
  CT_ASSERT(( ((UInt)t)&3) == 0);

  for(i=0;i<TABLE_LEVEL2_SIZE;i++)
    t[i] = ENTRY_INVALID;
  
  return t;
}

static
UInt* new_l3table()
{
  Int i;
  UInt* t = (UInt*) VG_(malloc)(sizeof(UInt)*TABLE_LEVEL3_SIZE);
  CT_ASSERT( (((UInt)t)&3) == 0);

  for(i=0;i<TABLE_LEVEL3_SIZE;i++)
    t[i] = ENTRY_INVALID;
  
  return t;
}



static UInt tablePosAddr(UInt pos1, UInt pos2, UInt pos3)
{
  return (pos1 << TABLE_LEVEL1_SHIFT) |
    (pos2 << TABLE_LEVEL2_SHIFT) | pos3;
}
    

/* Set a table start position from an address
 * with minimum depth. Doesn't set subtables.
 *
 * Returns depth.
 */
static int minTableStartPos(UInt addr, TablePos* tp)
{
  int d;

  tp->addr = addr;
  tp->pos1 = addr >> TABLE_LEVEL1_SHIFT;
  tp->pos2 = (addr >> TABLE_LEVEL2_SHIFT) & (TABLE_LEVEL2_SIZE-1);
  tp->pos3 = addr & (TABLE_LEVEL3_SIZE-1);
  
  d =
    (tp->pos3 > 0) ? 3 :
    (tp->pos2 > 0) ? 2 : 
    (tp->pos1 > 0) ? 1 : 0;

  CT_DEBUG(5," minTableStartPos(addr %p): pos %d/%d/%d, depth %d\n",
	   addr, tp->pos1, tp->pos2, tp->pos3, d);

  return d;
}

static int minTableEndPos(UInt addr, TablePos* tp)
{
  int d;

  tp->addr = addr;
  tp->pos1 = addr >> TABLE_LEVEL1_SHIFT;
  tp->pos2 = (addr >> TABLE_LEVEL2_SHIFT) & (TABLE_LEVEL2_SIZE-1);
  tp->pos3 = addr & (TABLE_LEVEL3_SIZE-1);

  d =
    (tp->pos3 < TABLE_LEVEL3_SIZE-1) ? 3 :
    (tp->pos2 < TABLE_LEVEL2_SIZE-1) ? 2 : 
    (tp->pos1 < TABLE_LEVEL1_SIZE-1) ? 1 : 0;

  CT_DEBUG(5," minTableEndPos(addr %p): pos %d/%d/%d, depth %d\n",
	   addr, tp->pos1, tp->pos2, tp->pos3, d);

  return d;
}

/* Attaches subtables to a table position
 * up to the requested depth, creating subtables as
 * needed.
 * Returns the depth where attaching was possible.
 */
static int attachSubtables(int depth, TablePos* tp)
{
  UInt entry1, entry2, *table2, *table3;

  CT_DEBUG(5," attachSubtables(depth %d, pos %d/%d/%d)\n",
	   depth, tp->pos1, tp->pos2, tp->pos3);

  entry1 = data_table[tp->pos1];
  if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    tp->table2 = tp->table3 = 0;

    CT_DEBUG(5,"  => 0 [Object at L1]\n");
    return 0;
  }
  if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    if (depth <= 0) {
      tp->table2 = tp->table3 = 0;

      CT_DEBUG(5,"  => 0\n");
      return 0;
    }
    table2 = new_l2table();
    entry1 = (UInt)table2 | ENTRY_SUBTABLE;
    data_table[tp->pos1] = entry1;

    CT_DEBUG(5,"   new L2 subtable at %d: %p\n",
	     tp->pos1, table2, entry1);
  }
  else {
    table2 = (UInt*)(entry1 & ~3);
    CT_DEBUG(5,"   L2 subtable at %d: %p\n", tp->pos1, table2);
  }
  tp->table2 = table2;

  entry2 = table2[tp->pos2];
  if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    tp->table3 = 0;

    CT_DEBUG(5,"  => 1 [Object at L2]\n");
    return 1;
  }
  if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    if (depth == 1) {
      tp->table3 = 0;

      CT_DEBUG(5,"  => 1\n");
      return 1;
    }
    table3 = new_l3table();
    entry2 = (UInt)table3 | ENTRY_SUBTABLE;
    table2[tp->pos2] = entry2;
    CT_DEBUG(5,"   new L3 subtable at %d: %p\n",
	     tp->pos2, table3, entry2);
  }
  else {
    table3 = (UInt*)(entry2 & ~3);
    CT_DEBUG(5,"   L3 subtable at %d: %p\n", tp->pos2, table3);
  }
  tp->table3 = table3;

  return depth;
}

static
void compactTable2(UInt* table, int idx)
{ 
  int i;
  UInt* subtable;
  Bool isEmpty = True;
 
  if ((table[idx] & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) return;
  subtable = (UInt*)(table[idx] & ~3);
  for(i=0;i<TABLE_LEVEL3_SIZE;i++) {
    if (subtable[i] != ENTRY_INVALID)
      isEmpty = False;
  }
  if (!isEmpty) return;

  table[idx] = ENTRY_INVALID;
  VG_(free)(subtable);

  CT_DEBUG(5,"   compactTable2(./%d): deleted subtable\n", idx);
}

static
void compactTable1(UInt* table, int idx)
{ 
  int i;
  UInt* subtable;
  Bool isEmpty = True;
 
  if ((table[idx] & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) return;
  subtable = (UInt*)(table[idx] & ~3);
  for(i=0;i<TABLE_LEVEL2_SIZE;i++) {
    compactTable2(subtable,i);
    if (subtable[i] != ENTRY_INVALID)
      isEmpty = False;
  }
  if (!isEmpty) return;

  table[idx] = ENTRY_INVALID;
  VG_(free)(subtable);

  CT_DEBUG(5,"  compactTable1(%d): deleted subtable\n", idx);
}
    

/* fill all entries between start and end */
static
void changeEntry(UInt from, UInt to, TablePos* start, TablePos* end)
{
  int i;
  int start_pos1, end_pos1;
  int start_pos2, end_pos2;

  CT_DEBUG(5," changeEntry(%p -> %p, %d/%d/%d - %d/%d/%d)\n",
	   from, to,
	   start->pos1, start->pos2, start->pos3,
	   end->pos1, end->pos2, end->pos3);

  if (start->pos1 == end->pos1) {
    if (start->pos2 == end->pos2) {
      for(i = start->pos3; i<=end->pos3; i++) {
	CT_ASSERT(start->table3[i] == from);
	start->table3[i] = to;
      }
    }
    else {
      start_pos2 = start->pos2, end_pos2 = end->pos2;
      if (start->table3) {
	start_pos2++;
	for(i = start->pos3; i<TABLE_LEVEL3_SIZE; i++) {
	  CT_ASSERT(start->table3[i] == from);
	  start->table3[i] = to;
	}
      }
      if (end->table3) {
	end_pos2--;
	for(i = 0; i<= end->pos3; i++) {
	  CT_ASSERT(end->table3[i] == from);
	  end->table3[i] = to;
	}
      }
      for(i = start_pos2; i<=end_pos2; i++) {
	if (from == ENTRY_INVALID) compactTable2(start->table2,i);
	CT_ASSERT(start->table2[i] == from);
	start->table2[i] = to;
      }
    }
    return;
  }

  start_pos1 = start->pos1, end_pos1 = end->pos1;
  if (start->table2) {
    start_pos1++;
    start_pos2 = start->pos2;
    if (start->table3) {
      start_pos2++;      
      for(i = start->pos3; i<TABLE_LEVEL3_SIZE; i++) {
	CT_ASSERT(start->table3[i] == from);
	start->table3[i] = to;
      }
    }
    for(i = start_pos2; i<TABLE_LEVEL2_SIZE; i++) {
      if (from == ENTRY_INVALID) compactTable2(start->table2,i);
      CT_ASSERT(start->table2[i] == from);
      start->table2[i] = to;
    }
  }
  if (end->table2) {
    end_pos1--;
    end_pos2 = end->pos2;
    if (end->table3) {
      end_pos2--;
      for(i = 0; i<= end->pos3; i++) {
	CT_ASSERT(end->table3[i] == from);
	end->table3[i] = to;
      }
    }
    for(i = 0; i<=end_pos2; i++) {
      if (from == ENTRY_INVALID) compactTable2(end->table2,i);
      CT_ASSERT(end->table2[i] == from);
      end->table2[i] = to;
    }
  }
  for(i = start_pos1; i<=end_pos1; i++) {
    if (from == ENTRY_INVALID) compactTable1(data_table,i);
    CT_ASSERT(data_table[i] == from);
    data_table[i] = to;
  }
}

/* Sets start and end according to addr and size */
Bool setRange(UInt addr, UInt size, TablePos* start, TablePos* end)
{
  int startDepth, endDepth, d;

  CT_DEBUG(5," setRange(addr %p, size %u)\n", addr, size);

  startDepth = minTableStartPos(addr, start);
  endDepth   = minTableEndPos(addr+size-1, end);

  /* if position path goes over same subtables, depths have to
   * be adjusted */
  if (start->pos1 == end->pos1) {
    if (start->pos2 == end->pos2) {
      if (startDepth<3) startDepth=3;
      if (endDepth<3) endDepth=3;
    }
    else {
      if (startDepth<2) startDepth=2;
      if (endDepth<2) endDepth=2;
    }
  }
  
  CT_DEBUG(5,"  > adjusted: depth %d - %d\n", startDepth, endDepth);

  /* attach subtables, creating if needed.
   * return 0 if conflicts exist */
  d = attachSubtables(startDepth, start);
  if (d<startDepth) return False;
  d = attachSubtables(endDepth, end);
  if (d<endDepth) return False;

  return True;
}


static
void print_page_table()
{
  int pos1, pos2, pos3;
  UInt entry1, entry2, entry3, *table2, *table3;
  DataObject* o;
  UInt last1, last2, last3;

  CT_DEBUG(0,"Page Table:\n");

  last1 = 0;
  for(pos1=0; pos1<TABLE_LEVEL1_SIZE; pos1++) {
    entry1 = data_table[pos1];

    if ((entry1 == last1) &&
	(entry1 & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) continue;
    if (last1 != 0)
      CT_DEBUG(0,"P  ... - (%d)\n", pos1-1);
    last1 = entry1;

    if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
      o = (DataObject*)(entry1 & ~3);
      CT_DEBUG(0,"P  at L1 (%d): (%p/%u) %s\n",
	       pos1, o->addr, o->size, o->name); 
      continue;
    }
    if ((entry1 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
      CT_DEBUG(0,"P  at L1 (%d): None\n", pos1);
      continue;
    }
    last1 = 0;

    table2 = (UInt*)(entry1 & ~3);
    CT_DEBUG(0,"P  subtable (%p) at L1 %d (%p-%p)\n",
	     table2, pos1,
	     tablePosAddr(pos1,0,0),
	     tablePosAddr(pos1,TABLE_LEVEL2_SIZE-1, TABLE_LEVEL3_SIZE-1));

    last2 = 0;
    for(pos2=0; pos2<TABLE_LEVEL2_SIZE; pos2++) {
      entry2 = table2[pos2];

      if ((entry2 == last2) &&
	  (entry2 & ENTRY_TYPE_MASK) != ENTRY_SUBTABLE) continue;
      if (last2 != 0)
	CT_DEBUG(0,"P   ... - (%d/%d)\n", pos1, pos2-1);
      last2 = entry2;

      if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
	o = (DataObject*)(entry2 & ~3);
	CT_DEBUG(0,"P   at L2 (%d/%d): (%p/%u) %s\n",
		 pos1, pos2, o->addr, o->size, o->name); 
	continue;
      }
      if ((entry2 & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
	CT_DEBUG(3,"P   at L2 (%d/%d): None\n", pos1, pos2);
	continue;
      }
      last2 = 0;

      table3 = (UInt*)(entry2 & ~3);
      CT_DEBUG(0,"P   subtable (%p) at L2 %d/%d (%p-%p)\n", 
	       table3, pos1, pos2,
	       tablePosAddr(pos1,pos2,0),
	       tablePosAddr(pos1,pos2, TABLE_LEVEL3_SIZE-1));

      last3 = 0;
      for(pos3=0; pos3<TABLE_LEVEL3_SIZE; pos3++) {
	entry3 = table3[pos3];

	if (entry3 == last3) continue;
	if (last3 != 0)
	  CT_DEBUG(0,"P    ... - (%d/%d/%d)\n", pos1, pos2, pos3-1);
	last3 = entry3;

	if ((entry3 & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
	  o = (DataObject*)(entry3 & ~3);
	  CT_DEBUG(0,"P    at L3 (%d/%d/%d): (%p/%u) %s\n",
		   pos1, pos2, pos3, o->addr, o->size, o->name); 
	}
	else
	  CT_DEBUG(0,"P    at L3 (%d/%d/%d): None\n", pos1, pos2, pos3);
      }
    }
  }
}

DataObject* get_object(UInt addr)
{
  int pos1, pos2, pos3;
  UInt entry, *table2, *table3;
  DataObject* o;

  pos1 = addr >> TABLE_LEVEL1_SHIFT;
  entry = data_table[pos1];
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    o = (DataObject*)(entry & ~3);
    CT_DEBUG(5," get_object(%p): at L1 (%d)\n  => (%p/%d) %s\n",
	     addr, pos1, o->addr, o->size, o->name); 
    return o;
  }
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    CT_DEBUG(5," get_object(%p): None at L1 (%d)\n", addr, pos1); 
    return 0;
  }
  table2 = (UInt*)(entry & ~3);
  
  pos2 = (addr >> TABLE_LEVEL2_SHIFT) & (TABLE_LEVEL2_SIZE-1);
  entry = table2[pos2];
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    o = (DataObject*)(entry & ~3);
    CT_DEBUG(5," get_object(%p): at L2 (%d/%d)\n  => (%p/%d) %s\n",
	     addr, pos1, pos2, o->addr, o->size, o->name); 
    return o;
  }
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_INVALID) {
    CT_DEBUG(5," get_object(%p): None at L2 (%d/%d)\n", addr, pos1, pos2); 
    return 0;
  }
  table3 = (UInt*)(entry & ~3);

  pos3 = addr & (TABLE_LEVEL3_SIZE-1);
  entry = table3[pos3];
  if ((entry & ENTRY_TYPE_MASK) == ENTRY_OBJECT) {
    o = (DataObject*)(entry & ~3);
    CT_DEBUG(5," get_object(%p): at L3 (%d/%d/%d)\n  => (%p/%u) %s\n",
	     addr, pos1, pos2, pos3, o->addr, o->size, o->name); 
    return o;
  }
  CT_DEBUG(5," get_object(%p): None at L3 (%d/%d/%d)\n",
	   addr, pos1, pos2, pos3); 
  return 0;
}

static
void remove_object(DataObject* o)
{
  TablePos start, end;
  Bool noConflict;

  CT_DEBUG(5," remove_object(%s, addr %p, size %u)\n", 
	   o->name, o->addr, o->size);

  noConflict = setRange(o->addr,o->size,&start,&end);
  CT_ASSERT(noConflict == True);
  changeEntry( ((UInt)o) | ENTRY_OBJECT, ENTRY_INVALID, &start, &end);
}



/* Insert an object into the data table
 */
static
void add_object(DataObject* o)
{
  TablePos start, end;
  Bool noConflict;

  DataObject* o2 = get_object(o->addr);
  if (o2) {
    CT_DEBUG(0,"WARNING: removing old '%s' (%p/%d)\n         for new '%s' (%p/%d)\n",
	     o2->name, o2->addr, o2->size,
	     o->name, o->addr, o->size);
    remove_object(o2);
  }

  CT_DEBUG(5," add_object(%s, addr %p, size %u)\n",
	   o->name, o->addr, o->size);
  noConflict = setRange(o->addr,o->size,&start,&end);
  if (!noConflict) { print_page_table(); CT_ASSERT(0);  }
  changeEntry( ENTRY_INVALID, ((UInt)o) | ENTRY_OBJECT, &start, &end);
}



/*------------------------------------------------------------*/
/*--- Data handling                                        ---*/
/*------------------------------------------------------------*/

DataType* aType;

void SK_(init_data)()
{
  init_page_table();
  init_type_table();

  aType = get_type(new_object("???",0,0));
}


/* Need to get the size and subelements first time
 * from debug info.
 */
void SK_(handle_constructor)(UInt addr, Char* fullname)
{
  VG_(message)(Vg_DebugMsg, "Constructor %s for %x",
	       fullname, addr);
}

void SK_(handle_destructor)(UInt addr, Char* fullname)
{
  VG_(message)(Vg_DebugMsg, "Destructor %s for %x",
	       fullname, addr);
}

Char obj_name[BUF_LEN];
int allocated = 0, freed = 0;

void SK_(handle_malloc)(UInt addr, UInt size)
{
  DataObject* o;
  fn_node** pfn = SK_(current_fn_stack).top;
  int opos = 0, d=0;

  if (addr == 0) {
    CT_DEBUG(3," Failed allocation (addr %x, size %d)\n",
	     addr, size);
    return;
  }

  CT_DEBUG(3," Allocated (%x-%x, size %d)\n", addr, addr+size-1, size);
  while(*pfn) {
    CT_DEBUG(3,"  from %s\n", (*pfn)->name);
    if ((opos<BUF_LEN) && (++d<4))
      opos += VG_(sprintf)(obj_name+opos, "'%s", (*pfn)->name);
    pfn--;
  }
  o = new_object(obj_name+1, addr, size);
  add_object(o);

  o->type->allocCount++;
  o->type->allocSize += size;
}

Int SK_(handle_realloc)(UInt newaddr, UInt oldaddr, UInt newsize)
{
  DataObject* o;
  Int oldsize = 0;

  if (newaddr == 0) {
    CT_DEBUG(3," Failed reallocation (newaddr %x, oldaddr %x, newsize %d)\n",
	     newaddr, oldaddr, newsize);
    return 0;
  }

  CT_DEBUG(3," Reallocated (old %x, new %x-%x, size %d)\n",
	   oldaddr, newaddr, newaddr+newsize-1, newsize);

  o = get_object(oldaddr);
  if (o && (o->addr == oldaddr)) {
    CT_DEBUG(3,"  old size %d, obj %s\n", o->size, o->name);
    oldsize = o->size;
    remove_object(o);
    o->addr = newaddr;
    o->size = newsize;
    add_object(o);

    o->type->allocSize += newsize-oldsize;
  }
  else
    CT_DEBUG(3,"  nothing found to realloc.\n");

  return oldsize;
}

/* returns size of freed allocation */
Int SK_(handle_free)(UInt addr)
{
  fn_node** pfn = SK_(current_fn_stack).top-1;
  DataObject* o;
  Int res = 0;

  CT_DEBUG(3," Free %x\n", addr);
  while(*pfn) {
    CT_DEBUG(3,"  from %s\n", (*pfn)->name);
    pfn--;
  }

  o = get_object(addr);  
  if (o && (o->addr == addr)) {
    remove_object(o);
    res = o->size;

    o->type->freeCount++;
    o->type->freeSize += o->size;

    VG_(free)(o->name);
    VG_(free)(o);
  }
  else
    CT_DEBUG(3,"  nothing found to free.\n");

  return res;
}

ULong* getcost(Addr addr, Context* cxt)
{
  return 0;
}

void SK_(handle_read)(UInt addr, UInt size, int sim)
{
  DataObject* o = get_object(addr);
  DataType* t = o ? o->type : aType;

  t->read_cost++;

  if (o) {
    CT_DEBUG(3,"Read  %p: Off %d in %s (%p/%d)\n",
	     addr, addr - o->addr, o->name, o->addr, o->size);
  }
  CT_DEBUG(3,"Read (%p), reads of %s: %llu\n", addr, t->name, t->read_cost);
}

void SK_(handle_write)(UInt addr, UInt size, int sim)
{
  DataObject* o = get_object(addr);
  DataType* t = o ? o->type : aType;

  t->write_cost++;

  if (o) {
    CT_DEBUG(3,"Write %p: Off %d in %s (%p/%d)\n",
	     addr, addr - o->addr, o->name, o->addr, o->size);
  }
  CT_DEBUG(3,"Write (%p), writes of %s: %llu\n", addr, t->name, t->write_cost);
}

void SK_(finish_data)()
{
  print_type_table();
  //print_page_table();
}

void SK_(my_new_mem_startup)(Addr a, SizeT len, Bool rr, Bool ww, Bool xx)
{
  //DataObject* o;
  static char buf[64];

  if (!SK_(clo).collect_data) return;

  VG_(sprintf)(buf,"Seg[%08p-%08p/%c%c%c]", a, a+len-1,
	       rr ? 'r':'-', ww ? 'w':'-', xx ? 'x':'-');

  CT_DEBUG(0,"new_mem_startup: %s\n", buf);
  
  //o = new_object(buf, a, len);
  //add_object(o);
}

void SK_(my_new_mem_stack_signal)(Addr a, SizeT len)
{
  DataObject* o;
  static char buf[64];

  if (!SK_(clo).collect_data) return;

  VG_(sprintf)(buf,"Stack[%08p-%08p]", a, a+len-1);

  CT_DEBUG(0,"new_mem_stack_signal: %s\n", buf);
  
  o = new_object(buf, a, len);
  add_object(o);
}

void SK_(my_new_mem_brk)(Addr a, SizeT len)
{
  DataObject* o;
  static char buf[64];

  if (!SK_(clo).collect_data) return;

  VG_(sprintf)(buf,"Brk[%08p-%08p]", a, a+len-1);

  CT_DEBUG(0,"new_mem_brk: %s\n", buf);
  
  o = new_object(buf, a, len);
  add_object(o);
}

void SK_(my_new_mem_mmap)(Addr a, SizeT len, Bool rr, Bool ww, Bool xx)
{
  //DataObject* o;
  static char buf[64];

  if (!SK_(clo).collect_data) return;

  VG_(sprintf)(buf,"MMap[%08p-%08p/%c%c%c]", a, a+len-1,
	       rr ? 'r':'-', ww ? 'w':'-', xx ? 'x':'-');

  CT_DEBUG(0,"new_mem_mmap: %s\n", buf);
  
  //o = new_object(buf, a, len);
  //add_object(o);
}

void SK_(new_data_sym)(Addr a, SizeT len, SegInfo* si, Char* name)
{
  DataObject* o;
  static char buf[256];

  VG_(sprintf)(buf,"%s:%s [%08p/%d]", 
	       VG_(seg_filename)(si), name, a, len);

  CT_DEBUG(0,"new_data_sym: %s\n", buf);

  o = new_object(buf, a, len);
  add_object(o);
}

void SK_(free_seg_info)(SegInfo* si)
{
  CT_DEBUG(0,"free_seg_info: %s\n", VG_(seg_filename)(si));
}
