/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "BasicAtomProperties.h"

//----------------------- Class member init -----------------------
void BasicAtomProperties::resetUniqueIndex() {
  BasicAtomProperties::maxUniqueIndex = 0;
}

// initializing the static class member
unsigned int BasicAtomProperties::maxUniqueIndex = 0;

//----------------------- Constructors -----------------------
BasicAtomProperties::BasicAtomProperties(PhysicalModel * p)  : StructureProperties(p, StructureProperties::ATOM) {
  allocate();
  setPosition(0.0, 0.0, 0.0);
  index = maxUniqueIndex++;
}

BasicAtomProperties::BasicAtomProperties(PhysicalModel * p, xmlNodePtr node)  : StructureProperties(p, StructureProperties::ATOM) {
  allocate();

  //search the name attribute
  xmlChar *pname = xmlGetProp(node, (const xmlChar*) "name");

  if (pname)
    setName((char*)pname);

  xmlFree(pname);
  
  //search the known attributes
  xmlChar *px = xmlGetProp(node, (const xmlChar*) "x");

  xmlChar *py = xmlGetProp(node, (const xmlChar*) "y");

  xmlChar *pz = xmlGetProp(node, (const xmlChar*) "z");

  if (px && py && pz)
    setPosition(atof((char*)px), atof((char*)py), atof((char*)pz));
  else
    setPosition(0.0, 0.0, 0.0);
  xmlFree(px);
  xmlFree(py);
  xmlFree(pz);

  xmlChar *pindex = xmlGetProp(node, (const xmlChar*) "index");

  if (pindex)
    index = atoi((char*)pindex);
  else
    index = maxUniqueIndex++;
  xmlFree(pindex);
  
  //search the unknown attributes to fill the property fields map
  xmlAttr * attrs = node->properties;

  xmlNodePtr unknownAttrs = xmlNewNode(NULL, (xmlChar*)("unknownAttrs"));

  while (attrs)
  {
    const xmlChar * pname = attrs->name;
    xmlChar * pval = attrs->children->content;

    if (pname && xmlStrcmp(pname, (xmlChar*)"name")
        && xmlStrcmp(pname, (xmlChar*)"x")
        && xmlStrcmp(pname, (xmlChar*)"y")
        && xmlStrcmp(pname, (xmlChar*)"z")
        && xmlStrcmp(pname, (xmlChar*)"index")){
      xmlSetProp(unknownAttrs, pname, pval);
    }

    attrs = attrs->next;
  }

  //transform the unknown attributes to a property field map
  domToFields(unknownAttrs);

  xmlFree(unknownAttrs);
}

BasicAtomProperties::BasicAtomProperties(PhysicalModel *p, const double pos[3]) : StructureProperties(p, StructureProperties::ATOM) {
  allocate();
  setPosition(pos);
  index = maxUniqueIndex++;
}

BasicAtomProperties::BasicAtomProperties(PhysicalModel *p, const unsigned int ind)  :  StructureProperties(p, StructureProperties::ATOM) {
  allocate();
  setPosition(0.0, 0.0, 0.0);
  index = ind;

  if (ind >= maxUniqueIndex)
    maxUniqueIndex = ind + 1;
}

BasicAtomProperties::BasicAtomProperties(PhysicalModel *p, const unsigned int ind, const double pos[3]) : StructureProperties(p, StructureProperties::ATOM) {
  allocate();
  setPosition(pos);
  index = ind;

  if (ind >= maxUniqueIndex)
    maxUniqueIndex = ind + 1;
}

//----------------------- destructor -----------------------
BasicAtomProperties::~BasicAtomProperties() {
  if (allocated)
    delete [] X;
}

//----------------------- allocate -----------------------
void BasicAtomProperties::allocate() {
  X = new double[3];
  allocated = true;
}

//----------------------- setPositionPointer -----------------------
void BasicAtomProperties::setPositionPointer(double * ptr, bool update) {
  // store previous position if needed
  double pos[3];
  if (update) {
    getPosition(pos);
  }
  // delete previous memory to avoid memory leaks, if needed
  if (allocated) {
    delete [] X;
    allocated = false;
  }
  X = ptr;
  // restore if needed
  if (update) {
    setPosition(pos);
  }
}

//----------------------- beginXML -----------------------
void BasicAtomProperties::beginXML(std::ostream & o) {
  o << "<atomProperties index=\"" << index << "\"";
  o << " x=\"" << X[0] << "\" y=\"" << X[1] << "\" z=\"" << X[2] << "\"";

  if (getName() != "")
    o << " name=\"" << getName().c_str() << "\" ";
}

//----------------------- endXML -----------------------
void BasicAtomProperties::endXML(std::ostream & o) {
  o << "/>" << std::endl;
}
