/**
   \class CWebCamViewer
   
   A class for the usual webcam activity: take pictures, save them do disk
   and/or upload to a server.
 */

#define ENABLE_TRACING 1 
#if HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef INCLUDE_UNISTD_H
#include <unistd.h>
#endif
#ifdef INCLUDE_STDLIB_H
#include <stdlib.h>
#endif

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <qbitmap.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qmultilineedit.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qurl.h>

#include "CamStreamApp.h"
#include "WebCamViewer.h"


CWebCamViewer::CWebCamViewer(CVideoDevice *video, const QSize &hint_size, QWidget *parent, const char *name)
	: CCamWindow(parent, name)
{
   int r;
   QWidget *pWidget;
   QToolBar *bar;
   QStatusBar *status;
   QString nullstring;

qDebug("CWebCamViewer::CWebCamViewer(%p, %dx%d)\n", video, hint_size.width(), hint_size.height());

   /* Our central widget contains space for the view image, and optionally
      the last snapshot. The widget itself is empty (except for a background)
    */

   pVideo = video;
   pViewer = NULL;
   pLastSnapshot = NULL;
   Upload.pClient = NULL;
   Upload.State = Upload.NotConnected;
   Upload.ErrorCondition = FALSE;

   r = pVideo->Open(2);
   if (r < 0) {
     qWarning("Error opening video device: %d.\n", r);
     return;
   }
   setCaption(pVideo->GetIntfName());

   /* Create and initialize dialogs */
   pVOptions = CamApp->FindVideoOptions(pVideo->GetIntfName(), pVideo->GetNodeName(), TRUE);
   pConfiguration = new CSnapshotSettingsDlg(pVOptions);
   connect(pConfiguration->OKButton,     SIGNAL(clicked()), this, SLOT(ConfigurationDlgClosed()));
   connect(pConfiguration->CancelButton, SIGNAL(clicked()), this, SLOT(ConfigurationDlgClosed()));
   
   pSettings = new CCamDialogs(pVideo);
   connect(pSettings, SIGNAL(applyButtonPressed()), this, SLOT(SettingsDlgClosed()));

   pTimeSnapDlg = new CTimeSnapDlg();
   connect(pTimeSnapDlg->OkButton,     SIGNAL(clicked()), this, SLOT(StartTimeSnap()));
   connect(pTimeSnapDlg->CancelButton, SIGNAL(clicked()), this, SLOT(TimeSnapDlgClosed()));
   pSnapTimer = new QTimer(this);
   connect(pSnapTimer, SIGNAL(timeout()), this, SLOT(TimeSnapTick()));

   /* Connect the video device to us. You may wonder why the Notify and 
      Resized signal aren't connected to us. Well, Notify is handled
      in the Viewer widget (which updates itself), and Resized() indirectly
      through the same Viewer. Only the error condition matters to us.
    */
   connect(pVideo, SIGNAL(Error(int)), this, SLOT(DeviceError(int)));

   /* Add buttons to toolbar */
   bar = new QToolBar("main", this);

   ButtonPix[pbt_controls].load(ICON_DIR "/controls.png");
   ButtonPix[pbt_config  ].load(ICON_DIR "/configure.png");
   ButtonPix[pbt_showsnap].load(ICON_DIR "/showsnap.png");
   ButtonPix[pbt_snapshot].load(ICON_DIR "/filesave.png");
   ButtonPix[pbt_timesnap].load(ICON_DIR "/savetimer.png");

   pButton[pbt_controls] = new QToolButton(ButtonPix[pbt_controls], "various controls",                   nullstring, this, SLOT(ClickedSettingsDlg()),      bar);
   pButton[pbt_config  ] = new QToolButton(ButtonPix[pbt_config],   "configuration",                      nullstring, this, SLOT(ClickedConfigurationDlg()), bar); 
   pButton[pbt_showsnap] = new QToolButton(ButtonPix[pbt_showsnap], "show last snapshot",                 nullstring, this, SLOT(ClickedShowLastSnapshot()), bar); 
   pButton[pbt_snapshot] = new QToolButton(ButtonPix[pbt_snapshot], "take snapshot",                      nullstring, this, SLOT(TakeSnapshot()),            bar); 
   pButton[pbt_timesnap] = new QToolButton(ButtonPix[pbt_timesnap], "take snapshot at regular intervals", nullstring, this, SLOT(ClickedTimeSnapDlg()),      bar); 

   pButton[pbt_controls]->setToggleButton(TRUE);
   pButton[pbt_config  ]->setToggleButton(TRUE);
   pButton[pbt_showsnap]->setToggleButton(TRUE);
   pButton[pbt_timesnap]->setToggleButton(TRUE);
   
   /* set toolbar */
   addToolBar(bar);
   /* Create statusbar, add space for countdown timer */
   status = statusBar();
   pSnapLabel = new QLabel(status);
   pSnapLabel->setText("--:--");
   status->addWidget(pSnapLabel, 0, TRUE);

   /* Create auxilary timer & workaround for adjustSize() */
   SizeTimer = new QTimer(this);
   connect(SizeTimer, SIGNAL(timeout()), this, SLOT(CallAdjustSize()));
   connect(this, SIGNAL(endMovingToolBar(QToolBar *)), this, SLOT(CallAdjustSize()));

   /* Create an empty widget upon we place our viewer and snapshot panels */
   pWidget = new QWidget(this);
   assert(pWidget != NULL);
   pWidget->setBackgroundMode(QWidget::PaletteBase);
   pWidget->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
   setCentralWidget(pWidget);

   pViewer = new CImagePanelRGB(pVideo, pWidget);
//   pViewer = new CImagePanelYUV(pVideo, pWidget);
   connect(pViewer, SIGNAL(ChangedVisibleSize(const QSize &)), this, SLOT(DeviceChangedSize(const QSize &)));
   pLastSnapshot = new CBasicPanel("snapshot", "Last snapshot", CCamPanel::RGB, pWidget);
   
   if (!hint_size.isNull()) { /* User supplied an image size; try that */
     if (!pVideo->SetSize(hint_size))
       qWarning("Video device did not accept hint_size.\n");
   }
   if (pViewer) {
     pViewer->show();
   }
   RecalcTotalViewSize();
}


CWebCamViewer::~CWebCamViewer()
{
qDebug("CWebCamViewer::~CWebCamViewer\n");
   if (pVideo) {
     pVideo->EnableRGB(FALSE);
     pVideo->Close();
   }
   pSettings->hide();
   pConfiguration->hide();
   delete pViewer;
   delete pLastSnapshot;
   delete Upload.pClient;
}

// private


void CWebCamViewer::StartUpload(const QString &filename, bool delafterwards)
{
   QFileInfo localfile;

   if (Upload.ErrorCondition) {
     qWarning("CWebCamViewer::StartUpload() Error condition was set");
     return;
   }
   if (Upload.State == Upload.Uploading) {
     qWarning("CWebCamViewer::StartUpload() Still uploading previous image");
     return;
   }
   if (Upload.pClient == NULL) {
     Upload.pClient = new QUrlOperator();
     
     /* Keep an eye on things... */
     connect(Upload.pClient, SIGNAL(start(QNetworkOperation *)), this, SLOT(FTPStart(QNetworkOperation *)));
     connect(Upload.pClient, SIGNAL(connectionStateChanged(int, const QString &)), this, SLOT(FTPState(int, const QString &)));
     connect(Upload.pClient, SIGNAL(finished(QNetworkOperation *)), this, SLOT(FTPDone(QNetworkOperation *)));
   }

   localfile.setFile(filename);
   Upload.LocalUrl.setProtocol("file");
   Upload.LocalUrl.setPath(localfile.absFilePath());

   Upload.RemoteUrl.setProtocol("ftp");
   Upload.RemoteUrl.setUser(pVOptions->GetFTPUser());
   Upload.RemoteUrl.setPassword(pVOptions->GetFTPPass());
   Upload.RemoteUrl.setHost(pVOptions->GetFTPServer());
   Upload.RemoteUrl.setPath(pVOptions->GetFTPPath());

   Trace(FTP, 2, "StartUpload(): localurl = \"%s\", ftpurl = \"%s\"; del = %c\n", (const char *)Upload.LocalUrl.toString(), (const char *)Upload.RemoteUrl.toString(), delafterwards ? 'T' : 'F');

   Upload.State = Upload.Uploading;
   time(&Upload.StartTime);
   Upload.pClient->copy(Upload.LocalUrl, Upload.RemoteUrl, delafterwards);
}



// private slots

void CWebCamViewer::ClickedConfigurationDlg()
{
   if (pButton[pbt_config]->isOn())
     pConfiguration->show();
   else
     pConfiguration->hide();
}

void CWebCamViewer::ConfigurationDlgClosed()
{
   pButton[pbt_config]->setOn(FALSE);
}

void CWebCamViewer::ClickedSettingsDlg()
{
   if (pButton[pbt_controls]->isOn())
     pSettings->show();
   else
     pSettings->hide();
}

void CWebCamViewer::SettingsDlgClosed()
{
   pButton[pbt_controls]->setOn(FALSE);
}

void CWebCamViewer::ClickedShowLastSnapshot()
{
   if (pLastSnapshot) {
     if (pButton[pbt_showsnap]->isOn())
       pLastSnapshot->show();
     else
       pLastSnapshot->hide();
     // Only enable RGB when we might need it (and then it's still a waste of CPU)
     pVideo->EnableRGB(pButton[pbt_showsnap]->isOn());
     RecalcTotalViewSize();
   }
}




void CWebCamViewer::ClickedTimeSnapDlg()
{
qDebug("CWebCamViewer::ClickedTimeSnapDlg()");
   if (pButton[pbt_timesnap]->isOn()) 
     pTimeSnapDlg->show();
   else {
     pTimeSnapDlg->hide();
     StopTimeSnap();
   }
}

void CWebCamViewer::TimeSnapDlgClosed()
{
qDebug("CWebCamViewer::TimeSnapDlgClosed()");
   pButton[pbt_timesnap]->setOn(FALSE);
}


/* When the user clicks 'Ok' in the TimeSnap dialog, this function gets
   called and we will start the timed snapshots 
 */
void CWebCamViewer::StartTimeSnap()
{
   SnapInterval = pTimeSnapDlg->Interval->value();
   if (pTimeSnapDlg->Minutes->isOn())
     SnapInterval *= 60;
qDebug("CWebCamViewer::StartTimeSnap(): %d", SnapInterval);
   SnapCounter = 0;
   pSnapTimer->start(1000);
}


void CWebCamViewer::StopTimeSnap()
{
qDebug("CWebCamViewer::StopTimeSnap()");
   pSnapTimer->stop();
   pSnapLabel->setText("--:--");
   if (SnapInterval > 0) {
     SnapInterval = -1;
     SnapCounter = 0;
   }
}

/**
  This function gets called once a second for a countdown to the next
  snapshot.
 */
void CWebCamViewer::TimeSnapTick()
{
   int rem;
   QString TimerText;

   if (SnapInterval < 0) {
     StopTimeSnap();
     return; // not active
   }
   SnapCounter++;

   if (SnapCounter >= SnapInterval) {
     TakeSnapshot(); 
   
     SnapCounter = 0;
//     if (Snap.Single) StopTimeSnap();
   }
   // Update status bar, show remaining time
   rem = SnapInterval - SnapCounter;
   TimerText.sprintf("%02d:%02d", rem / 60 , rem % 60);
   pSnapLabel->setText(TimerText);

}


void CWebCamViewer::FTPStart(QNetworkOperation *op)
{
   Trace(FTP, 2, "FTPStart: state = %d, error = %d", op->state(), op->errorCode());
}

void CWebCamViewer::FTPState(int state, const QString &data)
{
   Trace(FTP, 2, "FTPState: state = %d, data = %s", state, (const char *)data);
}


void CWebCamViewer::FTPDone(QNetworkOperation *op)
{
   int errcode, state;
   char *StateStrings[5] = { "Waiting", "In progress", "Done", "Failed", "Stopped" };
   char *ss;
   QString errmsg;
   time_t Now;

   Upload.State = Upload.Done;
   state = op->state();
   errcode = op->errorCode();
   if (state >= 0 && state < 5)
     ss = StateStrings[state];
   else
     ss = NULL;
     
   time(&Now);
   Now -= Upload.StartTime;
   Trace(FTP, 2, "FTPDone: state = %s (%d), error = %d, elapsed time = %d", ss, state, errcode, (int)Now);
   
   if (errcode) {
     Upload.ErrorCondition = TRUE;
     switch(errcode) {
       case QNetworkProtocol::ErrHostNotFound:     QMessageBox::critical(this, "Upload failed", "The hostname of the FTP-server could not be resolved.", QMessageBox::Ok, QMessageBox::NoButton); break;
       case QNetworkProtocol::ErrPermissionDenied: QMessageBox::critical(this, "Upload failed", "Permission denied: could not write file to server.\nCheck permissions of directories and files.", QMessageBox::Ok, QMessageBox::NoButton); break;
       default: 
         errmsg.sprintf("An error code %d occured (state = %s)", errcode, ss);
         QMessageBox::critical(this, "Error", "", QMessageBox::Ok, QMessageBox::NoButton); 
         break;
     }
     Upload.ErrorCondition = FALSE;
   }
#if 0   
   delete Upload.pClient; 
   Upload.pClient = NULL;
#endif   
}

/** 
  \brief Takes an image from the video stream, saves it in the snapshot imagebuffer, starts FTP upload
*/
void CWebCamViewer::TakeSnapshot()
{
   time_t now;
   struct tm *pTM;
   char strbuf[80];
   bool stampimage, stampfile;
   bool savetodisk, ftptoserver;
   bool delafterftp;
   int sequence;

   QString SnapExtension, SnapBase, SnapFmt;
   QString savename, ftpname;
   QFileInfo savefile;

   QImage DrawImage;
   QPainter p;
   QSize viewersize = pVideo->GetSize();
   QPixmap DrawArea(viewersize);

   stampimage = pVOptions->GetTimeInImage();
   stampfile = FALSE;
   savetodisk = pVOptions->GetSaveToDisk();
   ftptoserver = pVOptions->GetFTPToServer();

qDebug("TakeSnapshot: stampimage = %c, stampfile %c, savetodisk = %c, ftptoserver = %c\n", 
       stampimage ? 'T' : 'F', 
       stampfile ? 'T' : 'F',
       savetodisk ? 'T' : 'F',
       ftptoserver ? 'T' : 'F');

   if (stampimage || stampfile) {
     time(&now);
     pTM = localtime(&now);
   }

   if (stampimage) {
     /* Paint time & text in image */
     DrawImage = pViewer->GetImage();
     DrawArea.convertFromImage(DrawImage, 0);

     strftime(strbuf, 80, "%a, %d %b %Y %H:%M:%S %z %Z", pTM);
     p.begin(&DrawArea);
     p.setPen(pVOptions->GetTextColor());
     p.setFont(pVOptions->GetTextFont());
     p.drawText(5, viewersize.height() - 5, strbuf);
     p.end();
     DrawImage = DrawArea.convertToImage();
   }
   else
     DrawImage = pViewer->GetImage().copy(); // Make deep copy of the current image 

   // Store image in LastSnapshot
   pLastSnapshot->SetSize(viewersize);
   pLastSnapshot->SetImage(0, DrawImage);
   RecalcTotalViewSize();

   /* Create filename to save to disk */
   SnapFmt = pVOptions->GetFileFormat();
   SnapExtension = CamApp->FormatStrToExtension(SnapFmt);
   SnapBase = pVOptions->GetBaseName();

   switch(pVOptions->GetSaveOption()) {
     case 1: // overwrite
        savename.sprintf("%s.%s", (const char *)SnapBase, (const char *)SnapExtension);
        break;

     case 2:
       savename.sprintf("%s-%04d%02d%02d-%02d%02d%02d.%s",
                  (const char *)SnapBase,
                  1900 + pTM->tm_year, 1 + pTM->tm_mon, pTM->tm_mday,
                  pTM->tm_hour, pTM->tm_min, pTM->tm_sec,
                  (const char *)SnapExtension);
       break;
       
     case 3:
       sequence = pVOptions->GetSequence();
       if (sequence > pVOptions->GetMaxSequence())
         sequence = 0;
       savename.sprintf("%s%03d.%s", (const char *)SnapBase, sequence, (const char *)SnapExtension);
       pVOptions->SetSequence(++sequence);
       break;
   }

   /* extract filename, withouth paths */
   savefile.setFile(savename);
   
   delafterftp = false;
   if (savetodisk) {
     qDebug("Saving image to " + savename);
     if (!DrawImage.save(savename, SnapFmt)) {
       // XXX Failed. pop up message box, or set message in status balk
       qWarning("Could not save image!");
     }
     else {
       if (ftptoserver)
         ftpname = savename;
     }
   }
   else {
     if (ftptoserver) {
       delafterftp = true;
       ftpname = CamApp->GetUploadTmpDir() + "/" + savefile.fileName();
       qDebug("Saving temp FTP image to " + ftpname);
       if (!DrawImage.save(ftpname, SnapFmt))
         qWarning("Could not save temp image for FTPing!");
     }
   }

   if (ftptoserver)
     StartUpload(ftpname, delafterftp);
}


// protected

/** 
  \brief Calculate the total required viewing area, based on video image and 'last snapshot'
*/

void CWebCamViewer::RecalcTotalViewSize()
{
   QSize asize, osize;
   int w1 = 0, w2 = 0, h;

   if (pViewer && pLastSnapshot) {
     asize = pViewer->GetVisibleSize();
     w1 = asize.width();
     h = asize.height();
  
     if (pButton[pbt_showsnap]->isOn()) {
       asize = pLastSnapshot->GetVisibleSize();
       w2 = asize.width();
       if (asize.height() > h)
         h = asize.height();
       if (w2 == 0)
         w2 = w1;
     }
     
     asize.setWidth(w1 + w2);
     asize.setHeight(h);

     osize = centralWidget()->size();
     if (asize != osize) { // Set new viewport only when necessary
qDebug("RecalcTotalViewSize: resize viewport(%dx%d)", asize.width(), asize.height());

       pViewer->move(0, 0);
       pLastSnapshot->move(w1, 0);

       centralWidget()->setFixedSize(asize);
       /* Bug or feature? At this point I cannot call adjustSize(), probably
          because we are in an event loop or something. So therefor we
          (ab)use a single shot timer to call adjustSize() for us, just
          after all current events have been processed (timeout = 0).
        */
       SizeTimer->start(0, TRUE);
     }
   }
}

// protected slots


/** Called from \ref CVideoDevice::Resized */
void CWebCamViewer::DeviceChangedSize(const QSize &new_size)
{
qDebug("CWebCamViewer::DeviceChangedSize(%dx%d)", new_size.width(), new_size.height());

   RecalcTotalViewSize();
}


void CWebCamViewer::DeviceError(int err_no)
{
   // Show pop-up box
   QString errstr;
   
   errstr.sprintf("The device experienced an error %d (%s). The window will be closed.", err_no, strerror(-err_no));
   QMessageBox::critical(this, "device error", errstr, QMessageBox::Ok, QMessageBox::NoButton);
   close(TRUE); // End!
}


/**
  \brief Work around for adjustSize()

  Because sometimes adjustSize() doesn't have the desired (or a delayed)
  effect, we call it through a single-shot timer. Unfortunately adjustSize()
  isn't a slot, so this is just a wrapper function.
*/
void CWebCamViewer::CallAdjustSize()
{
   adjustSize();
}

/// public slots

